package LCM::Configuration::ComponentArchivesConfiguration;

use strict;
use File::Basename;
use LCM::Installer;
use LCM::FileUtils;
use SDB::Install::Tools qw(askConfirmation);
use parent qw(SDB::Install::Configuration::AnyConfig);

my $SAPCAR_EXECUTABLE = 'SAPCAR';
my $TAR_EXECUTABLE    = 'tar';
my $UNRAR_EXECUTABLE  = 'unrar';

sub InitDefaults {
    my $self = shift();
    my $defaultSapcarLocation = $self->_determineDefaultSapcarLocation();
    my $shallAskInteractive = defined($defaultSapcarLocation) ? 0 : 1;

    $self->{params}->{SapcarLocation}->{set_interactive} = 1;
    $self->setDefault('SapcarLocation', $defaultSapcarLocation);
    $self->setDefault('TarExecutableLocation', '/bin/tar');
    $self->setDefault('UnrarExecutableLocation', '/usr/bin/unrar');

    return $self->SUPER::InitDefaults(@_);
}

sub setExtractTempDir {
    my ($self, $pathValue) = @_;

    return 0 if (! $self->checkExtractTempDir($pathValue));

    my $parentDir = dirname($pathValue);
    my $realParentDir = (-d $parentDir) ? MyRealPath($parentDir) : File::Spec->catfile(MyRealPath(dirname($parentDir)), basename($parentDir));
    my $extractDirName = basename($pathValue);
    my $realPath = (-d $pathValue) ? MyRealPath($pathValue) : File::Spec->catfile($realParentDir, $extractDirName);

    $self->{params}->{ExtractTempDir}->{value} = $realPath;

    return 1;
}

sub checkExtractTempDir {
    my ($self, $pathValue) = @_;
    if (!$pathValue) {
        $self->getErrMsgLst()->addError("Path cannot be empty");
        return 0;
    }
    my $parentDir = dirname($pathValue);
    my $realPath = (-e $pathValue) ? MyRealPath($pathValue) : $pathValue;
    my $realParentPath = (-e $parentDir) ? MyRealPath($parentDir) : $parentDir;

    if(!File::Spec->file_name_is_absolute($pathValue)){
        $self->getErrMsgLst()->addError("Path '$pathValue' must be absolute");
        return 0;
    }
    if (-e $realPath && !-d $realPath){
        $self->getErrMsgLst()->addError("The path '$pathValue' must point to a directory");
        return 0;
    }
    if (!-d $realPath){
        if (! -d $realParentPath){
            $self->getErrMsgLst()->addError("Parent directory '$parentDir' does not exist");
            return 0;
        }
        if (! -w $realParentPath){
            $self->getErrMsgLst()->addError("Cannot create directory '$pathValue'. Insufficient privileges");
            return 0;
        }
    }
    return 1;
}

sub setSapcarLocation {
    return $_[0]->_setExecutableLocation($_[1], 'SapcarLocation', $SAPCAR_EXECUTABLE);
}

sub setTarExecutableLocation {
    return $_[0]->_setExecutableLocation($_[1], 'TarExecutableLocation', $TAR_EXECUTABLE);
}

sub setUnrarExecutableLocation {
    return $_[0]->_setExecutableLocation($_[1], 'UnrarExecutableLocation', $UNRAR_EXECUTABLE);
}

sub _determineDefaultSapcarLocation {
    my ($self) = @_;
    my $installer = new LCM::Installer();
    my $isInstalled = $installer->isInstalled();
    my $runtimeDirectory = $installer->getRuntimeDir();
    my $sapcarPathInHDB = File::Spec->catfile($runtimeDirectory, '..', '..', 'global', 'hdb', 'saphostagent_setup', 'SAPCAR');
    my $sapcarPathInSHA = File::Spec->catfile('/', 'usr', 'sap', 'hostctrl', 'exe', 'SAPCAR');

    return MyRealPath($sapcarPathInHDB) if ($isInstalled && -x $sapcarPathInHDB);
    return $sapcarPathInSHA if (-x $sapcarPathInSHA);
    return undef;
}

sub _setExecutableLocation {
    my ($self, $pathValue, $paramId, $executable) = @_;

    if (! $self->_checkExecutableLocation($paramId, $pathValue, $executable)) {
        return 0;
    }

    my $realPath = MyRealPath($pathValue);
    $self->{params}->{$paramId}->{value} = $realPath;
    return 1;
}

sub _checkExecutableLocation {
    my ($self, $paramId, $pathValue, $executable) = @_;

    if (! -x $pathValue || ! -f $pathValue) {
        $self->getErrMsgLst()->addError("Invalid $executable executable location. The path '$pathValue' does not point to an executable.");
        return 0;
    }
    return 1;
}

1;