package LCM::Configuration::GenericStackInstallConfiguration;

use strict;
use LCM::Configuration::GenericStackAny;

use LCM::Configuration::GenericStackConfiguration;
use SDB::Install::Configuration::NewServerConfig;
use LCM::Utils::CommonUtils qw(findAutoConfiguration);
use SDB::Install::Configuration qw ($bool_false_pattern $bool_true_pattern);
use SDB::Install::Globals qw ($gDirNameAccelerator
                              $gDirNameEs
                              $gDirNameStreaming
                              $gFlavourCockpit
                              $gFlavourExpress
                              $gHostRoleAcceleratorStandby
                              $gHostRoleAcceleratorWorker
                              $gHostRoleEsWorker
                              $gHostRoleStandby
                              $gHostRoleWorker
                              $gHostRoleXS2Standby
                              $gHostRoleXS2Worker
                              $gKeynameEngine
                              $gKeynameAccelerator
                              $gKeynameXS2
                              $gMinMemGB
                              $gProductName
                              $gProductNameAccelerator
                              $gProductNameClient
                              $gProductNameCockpit
                              $gProductNameEngine
                              $gProductNameEs
                              $gProductNameStreaming
                              $gProductNameStudio
                              $gSapsysGroupIdDefault
                              $gSapsysGroupName
                              $gXSParametersConstraint
                              $gOperationHdblcmCollectInstallerInfo
                              $gSAPLocation
                              GetServerNotFoundMessage
                              GetHostRoleProperties
                              getComponentNameByHostRole);

use SAPDB::Install::System::Win32::API;
use SAPDB::Install::Hostname;
use SDB::Install::Saphostagent qw(sapadmUserExists);
use SDB::Install::SysVars;
use SDB::Install::System qw(getSysProgPath isSidadmin isAdmin getSAPDrive);
use SDB::Install::SAPSystem qw(CollectSAPSystems);

use LCM::Component;
use LCM::Configuration::CertifiedHostConfig;
use LCM::Configuration::ValueChangeListeners::Install::UseMasterPasswordListener;
use LCM::Configuration::ValueChangeListeners::Install::SelectedComponentsListener;
use LCM::Configuration::ValueChangeListeners::Install::DetectCustomConfigDir;
use LCM::Configuration::ValueChangeListeners::Install::RemoteExecutionListener;
use LCM::Configuration::ValueChangeListeners::Install::RootUserListener;
use LCM::Configuration::ValueChangeListeners::Install::RootPasswordListener;
use LCM::Configuration::ValueChangeListeners::Install::SIDListener;
use LCM::Configuration::ValueChangeListeners::FillingCertificatesListener;
use LCM::Configuration::ValueChangeListeners::Install::AdditionalHostsListener;
use LCM::Configuration::ValueChangeListeners::Install::AutoAddXsRolesListener;
use LCM::Configuration::ValueChangeListeners::Install::DbModeListener;
use LCM::Configuration::ValueChangeListeners::InstallUpdate::RecomputeXs2ApplicationParametersListener;
use LCM::Configuration::ValueChangeListeners::InstallUpdate::RecomputeXs2SpaceIsolationParameters;
use LCM::Configuration::ValueChangeListeners::InstallUpdate::XSSpaceIsolationStringListener;
use SDB::Install::Configuration::LSSUserStringListener;
use LCM::Configuration::ValueChangeListeners::InstallUpdate::HandleLSSUserAndGroupListener;
use LCM::Configuration::ValueChangeListeners::InstallUpdate::InstallXSInDefaultTenantListener;
use LCM::Configuration::ValueChangeListeners::Install::XsInitTenantDatabaseListener;
use LCM::Configuration::ValueChangeListeners::Install::XsTenantInstallationListener;
use LCM::Configuration::ValueChangeListeners::XsSpaceIsolationListener;
use LCM::Configuration::ValueChangeListeners::ReservedIDsHandler;
use LCM::Configuration::ValueChangeListeners::SignatureVerificationListenerFactory;
use LCM::Configuration::ValueChangeListeners::AddComponentsLocationListener;
use SDB::Install::Configuration::TenantUserListener;
use LCM::HostsParser;
use LCM::App::AddHostsUtils;
use LCM::Configuration::ParametersCreator;
use LCM::Configuration::PersistenceInstallManager;
use LCM::Configuration::NewDBUser::User;
use LCM::PersistencyUtils qw (existsHdblcmPendingInstall existsHdbinstPendingInstall);
use LCM::FileUtils qw(MyRealPath);
use experimental qw (smartmatch);
use File::Basename qw(dirname);
use LCM::Configuration::ValueChangeListeners::Install::CreateInitialTenantListener;
use SDB::Install::Configuration::LSSPasswordListener;
use SDB::Install::LSS::LssSidcryptUser;
use SDB::Common::Utils qw(getSidcryptName);
use LCM::Configuration::ValueChangeListeners::Install::VolumeEncryptionListener;
use LCM::Configuration::ValueChangeListeners::InstallUpdate::LocalSecureStoreListener;

our $progpath          = $isWin || $isApple ? getSysProgPath() : '/usr';
our $volumeSubDirPerHost = 'mnt00001';

our @ISA = qw (LCM::Configuration::GenericStackConfiguration SDB::Install::Configuration::NewServerConfig);

our $existing_drives = [ SAPDB::Install::System::Win32::API::GetHDDrives() ];

sub new {
    my ($class, $options, $configfile) = @_;
    my $isAutoConfigDetected = 0;
    if(!defined($configfile)) {
        my $flavour = LCM::App::ApplicationContext::getFlavour();
        $configfile = findAutoConfiguration($class->getAction(), $flavour);
        $isAutoConfigDetected = 1 if $configfile;
    }

	my $self = $class->SUPER::new($options, $configfile);
    $self->setAutoConfigDetected($isAutoConfigDetected);

	$self->defineInstallParams();
	$self->defineLSSParams();
	$self->defineStreamingParams ();
	$self->defineEsParams();
	$self->defineAcceleratorParams(1);
	$self->defineXS2Params();
	$self->addListeners();
	$self->addEventHandlers();
	$self->setPersistenceManager(new LCM::Configuration::PersistenceInstallManager($self));
	$self->getPersistenceManager()->registerParamsToPersist();
	return $self;
}

# executable required for CollectHostinfo
sub getInstallerExe {return 'hdblcm';}

sub defineInstallParams {
	my ($self) = @_;
    my $rolesConstraint = "Valid only for installation of SAP HANA components, which support host roles";
    my $studioConstraint = "Valid for Installation of $gProductNameStudio only";
	my $order = scalar keys %{ $self->{params} };

	$self->{paramIds} = undef;    # reset ordered id list cache
	my $hostagentUsernameParam = $self->getParamHostagentUserName( $order++, $ini_section_server );
	$hostagentUsernameParam->{'init_with_default'} = 1;

	my $tenantCredentialsConstraint = $self->getTenantCredentialsConstraint();
	my $installParams = {
		'UseSimpleUI'         => GetParamUseSimpleUI($order++, $ini_section_general),
		'SkipHostagentCalls'  => $self->getParamSkipHostagentCalls($order++, $ini_section_general),
		'RemoteExecution'     => $self->getParamRemoteExecution($order++, $ini_section_general, 0),
		'UseHttp'             => $self->getParamUseHttp        ($order++, $ini_section_general),
		'UseMasterPassword'   => GetParamUseMasterPassword     ($order++, $ini_section_general),
		($isWin ?
			() :
			('UsePMem'        => $self->getParamUsePMem     ($order++, $ini_section_server))
		),
		'InstallHostagent'    => $self->getParamInstallHostagent( $order++, $ini_section_server, undef, 1 ),
		'DvdPath'             => GetParamDvdPath($order++, $ini_section_general, $self->getFlavourProductName()),
		'ComponentDirs'       => GetParamComponentDirs($order++, $ini_section_general),
		'AddComponentsLocation' => GetParamAddComponentsLocation($order++, $ini_section_general),
		'ComponentFsRoot'     => GetParamComponentFsRoot($order++, $ini_section_general),
		'DbMode'              => $self->getParamDbMode ($order++, $ini_section_server),
		'DbIsolation'         => $self->getParamDbIsolation ($order++, $ini_section_server),
		'CreateInitialTenant' => $self->getParamCreateInitialTenant ($order++, $ini_section_server),
        'AutoInitializeServices'     => $self->getParamAutoInitializeServices($order++, $ini_section_addhosts,undef,$tenantCredentialsConstraint),
        'VerifySignature'     => LCM::Configuration::ParametersCreator::GetParamVerifySignature($order++, $ini_section_general),
		'SelectedComponents'  => LCM::Configuration::GenericStackConfiguration::getParamSelectedComponents( $order++, $ini_section_general ),
        'ExecutionMode'=> LCM::Configuration::ParametersCreator::GetParamExecutionMode($order++, $ini_section_general, 0, 'install'),
		(
			 ! $isWin ?
				(	'CheckMnt'     => $self->getParamCheckMnt( $order++,      $ini_section_server ),
					'Target'       => $self->getParamTarget( $order++,        $ini_section_server ),
				)
				:
				()
		),
		'HostName'        => $self->getParamHostName( $order++, $ini_section_server ),
		(
			$isWin	?
			    (
					'Drive'             => $self->getParamDrive            ( $order++, $ini_section_server ),
					'HostagentPassword' => $self->getParamHostagentPassword( $order++, $ini_section_server ),
				)
					:
				( 	# Linux
					'AddHosts'     => $self->getParamAddHosts( $order++,      $ini_section_addhosts, 1),
					'AddLocalRoles'=> GetParamAddLocalRoles( $self, $order++, $ini_section_addhosts, $rolesConstraint ),
					'InstallSSHKey'=> $self->getParamInstallSSHKey($order++,  $ini_section_server),
					'RootUser'     => $self->getParamRootUser( $order++,      $ini_section_server ),
					'RootPassword' => $self->getParamRootPassword( $order++ , $ini_section_server ),
					# HostagentPasswordSHAExecution is only needed for interactive input of AddHosts in CLI mode
					# Introduced after the fix for Bug: 166897 removed the HostagentInstallConfiguration.
					'HostagentPasswordSHAExecution' => $self->getParamHostagentPassword( $order++, $ini_section_server ),
					'HostsProperties' => GetParamHostsProperties($order++, $ini_section_addhosts),
					'AutoAddXS2Roles'  => GetParamAutoAddXS2Roles($order++, $ini_section_addhosts, $gXSParametersConstraint),
					'ImportContentXS2'=> $self->getParamImportContentXS2($order++, $ini_section_addhosts, $gXSParametersConstraint),
					'StorageConfigDir' => $self->getParamStorageConfigDir( $order++, $ini_section_server )
				)
		),
        'ListenInterface' => $self->getParamListenInterface( $order++, $ini_section_server, 1),
		'InternalNetwork' => $self->getParamInternalNetwork( $order++, $ini_section_server, 0, undef, 1, 0, 1 ),
		'SID'             => $self->getParamSID( $order++,             $ini_section_server, 1 ),
		'InstanceNumber'  => $self->getParamInstanceNumber(
			$order++,
			$ini_section_server,
			'Instance Number'
		),
		'WorkerGroup'         => $self->getParamWorkerGroup ($order++, $ini_section_server, 0),
		'SystemUsage'         => $self->getParamSystemUsage( $order++, $ini_section_server ),
		'LSSTrustUnsignedServer' => $self->getParamLSSTrustUnsignedServer( $order++, $ini_section_server),
		'VolumeEncryption'    => $self->getParamVolumeEncryption($order++, $ini_section_server),
		'BasePathDataVolumes' => $self->getParamBasePathDataVolumes(
			$order++,
			$ini_section_server,
			$isWin ? '$Drive\usr\sap\data\$SID' : '/hana/data/$SID'
		),
		'BasePathLogVolumes' => $self->getParamBasePathLogVolumes(
			$order++,
			$ini_section_server,
			$isWin ? '$Drive\usr\sap\log\$SID' : '/hana/log/$SID'
		),
		($isWin ?
			() :
			('BasePathPMem'        => $self->getParamBasePathPMem($order++, $ini_section_server))
		),
		'CustomConfigDir'     => $self->getParamCustomConfigDir($order++, $ini_section_server),
        'SecureStore'         => $self->getParamSecureStore($order++, $ini_section_server),
		'RestrictMaxMem'      => $self->getParamRestrictMaxMem($order++, $ini_section_server),
		'MaxMem'              => $self->getParamMaxMem($order++, $ini_section_server),
		'CertificatesHostmap' => $self->getParamCertificatesHostmap( $order++,      $ini_section_server ),
		($isWin ? ('HostagentUserName' => $hostagentUsernameParam,) : ()),
		'MasterPassword'      => GetParamMasterPassword($order++, $ini_section_server),
        'HostagentPassword' => $self->getParamHostagentPassword( $order++, $ini_section_server ),
		(
			$isWin 	? (	# Windows
					'Domain'            => $self->getParamDomain( $order++,            $ini_section_server ),
					'Password'          => $self->getParamPassword( $order++,          $ini_section_server )
			  		 )
					: (

					    # Unix /Linux
					'Password' => $self->getParamPassword( $order++, $ini_section_server ),
					'HomeDir' => $self->getParamHomeDir( $order++, $ini_section_server, '/usr/sap/$SID/home' ),
					'Shell' => $self->getParamShell( $order++, $ini_section_server ),
					'UID' => $self->getParamUID( $order++, $ini_section_server ),
					'GID'      => $self->getParamGID( $order++, $ini_section_server, $gSapsysGroupIdDefault, 1 ),
					'Timezone' => $self->getParamTimeZone($order, $ini_section_server),
			)
		),
		'NoStart'            => $self->getParamNoStart( $order++,            $ini_section_server, 'Do not start the instance after installation', 'Does not start the instance after installation', 1 ),
		'MachineUtilization' => $self->getParamMachineUtilization( $order++, $ini_section_server ),
		'SQLSysPasswd'       => $self->getParamSQLSysPasswd( $order++,       $ini_section_server, 'initial_passwd', 0, 'SYSTEM'),
		'AutoStart'          => $self->getParamAutoStart ( $order++,          $ini_section_server ),
		'Repository'         => $self->getParamRepository( $order++,          $ini_section_server ),
		'ISCMode'            => $self->getParamISCMode   ( $order++,          $ini_section_server),
		'ClientPath'         => $self->getParamClientPath( $order++,         $ini_section_client ),
		(
			!$isWin
			? ()
			: (
				'ODBCDriverName' => {
					'order'   => $order++,
					'opt'     => 'odbc_driver_name',
					'type'    => 'string',
					'section' => $ini_section_client,
					'value'   => undef,
					'default' => undef,
					'str'               => 'ODBC Driver Name',
                    'constraint'        => "Valid for Installation of $gProductNameClient only",
					'init_with_default' => 0,
					'set_interactive'   => 0,
					'mandatory'         => 0
				}
			)
		),
		'StudioPath' => {
			'order'             => $order++,
			'opt'               => 'studio_path',
			'type'              => 'path',
			'section'           => $ini_section_studio,
			'value'             => undef,
			'default'           => $isWin ? '$Drive\usr\sap\$SID\hdbstudio' : '$Target/$SID/hdbstudio',
			'str'               => "$gProductNameStudio Installation Path",
            'constraint'        => $studioConstraint,
			'init_with_default' => 1,
			'set_interactive'   => 0,
			'mandatory'         => 1
		},
		'StudioRepository' => {
			'order'             => $order++,
			'opt'               => 'studio_repository',
			'type'              => 'boolean',
			'section'           => $ini_section_studio,
			'value'             => undef,
			'default'           => 1,
			'str'               => "Enables copying of $gProductNameStudio repository",
            'constraint'        => $studioConstraint,
			'init_with_default' => 1,
			'set_interactive'   => 0,
			'exit_if_faulty'    => 1
		},
		'TargetRepositoryPath' => $self->getParamTargetRepositoryPath( $order++, $ini_section_studio, $studioConstraint ),
		'JavaVm'               => $self->getParamJavaVm( $order++,               $ini_section_studio, $studioConstraint ),
		'Ignore'               => $self->getParamArgIgnore( $order++,  $ini_section_general ),
        'ReferenceDataPath'    => GetParamReferenceDataPath($order++, $ini_section_refdata),
         ($isWin
                ? ()
                : ('SkipModifySudoers' => $self->getParamSkipModifySudoers($order++, $ini_section_general)),
         ),
        'InstallXSInDefaultTenant' => GetParamInstallXS2InDefaultTenant($order++, $ini_section_xsa, $gXSParametersConstraint),
        'XSTenantDatabaseName',   => GetParamXSTenantDatabaseName($order++, $ini_section_xsa, $gXSParametersConstraint),
		'TenantUser'     => $self->getParamTenantUser    ($order++, $ini_section_addhosts,1,$tenantCredentialsConstraint),
        'SQLTenantUserPassword' => $self->getParamSQLTenantUserPassword($order++, $ini_section_addhosts,1,$tenantCredentialsConstraint,'initial_passwd'),
	};
	$installParams->{TenantUser}->{hidden} = 1;
	$installParams->{SQLTenantUserPassword}->{hidden} = 1;

	# Special handling for the remoteHostsObjects is required here, because the object is
	# initialized when setting parameter AddHosts's value
	$installParams->{HostsProperties}->{custom_input} = \&LCM::App::AddHostsUtils::readHostsPropertiesForInstall;

	map { $self->{params}->{$_} = $installParams->{$_} } keys %$installParams;
	$order = $self->addPassThroughParameters( $self->{params}, $order, 'hdbinst' );

# Alternative sections are used for the sake of backward compatibility.
# They are used for config file templates only
	if(exists($self->{params}->{AddHosts})){
		my $alternativeSections = [ $ini_section_server ];
		$self->{params}->{AddHosts}->{alternative_sections} = $alternativeSections;
		$self->{params}->{AddHosts}->{custom_input} = \&LCM::App::AddHostsUtils::readInteractiveAddHostsInput;
	}

	$self->{params}->{'cmd_line_action'} = $self->getParamCmdLineAction($order);
	$self->{params}->{'cmd_line_action'}->{skip} = 1;

	$installParams->{Ignore}->{alternative_sections} = [ $ini_section_studio ]; # Added for backward compatibility
	$installParams->{WorkerGroup}->{str} = 'Local Host Worker Group';

	return $self;
}

sub setMasterPassword {
	my ($self, $value) = @_;
	my $result = 1;
	my $uiString = $self->getString('MasterPassword');
	my $msglst = new SDB::Install::MsgLst();
	my $passwordToSet = $self->_getPasswordIdsForMasterPassword();

	if(!$self->complyDefaultSqlPasswordPolicy($value, $msglst, $uiString)){
		$self->PushError ("$uiString is invalid", $msglst);
		return undef;
	}
	for my $parameterId (@$passwordToSet){
		if(!$self->hasValue($parameterId) && $self->getBatchValue($parameterId)){
			$self->getErrMsgLst()->addError("$uiString cannot be used in combination with other initial passwords");
			$self->setNoRetry('MasterPassword', 1);
			return undef;
		}
	}
	for my $parameterId (@$passwordToSet){
		$result &= $self->setValue($parameterId, $value);
	}
	if($result){
		$self->{params}->{MasterPassword}->{value} = $value;
	}
	return $result;
}

sub _getPasswordIdsForMasterPassword {
	my ($self) = @_;
	my @initialPasswordParms = grep { $_ ne 'MasterPassword' && $self->getType($_) eq 'initial_passwd' && !$self->isSkipped($_) } @{$self->getParamIds()};
	return \@initialPasswordParms;
}

sub _shouldSkipAutoAddXS2Roles {
	my ($self) = @_;
	my $componentManager = $self->getComponentManager();
	my $xs2Component = $componentManager->getComponentByKeyName($gKeynameXS2);

	return 1 if(!defined($xs2Component) || !$xs2Component->isComponentSelected());

	my $addHosts         = $self->getValue('AddHosts');
	my @hostsForAddition = (!defined $addHosts || $self->isSkipped('AddHosts')) ? () : split(',', $addHosts);
	my $isAddingXsHost = grep { $_ =~ /:role=($gHostRoleXS2Worker|$gHostRoleXS2Standby)/} @hostsForAddition;
	my $addLocalRoles    = $self->getValue('AddLocalRoles');
	my @rolesForAddition = (!defined $addLocalRoles || $self->isSkipped('AddLocalRoles')) ? () : split(',', $addLocalRoles);
	my $isAddingLocalXsRole = grep { $_ =~ /$gHostRoleXS2Worker|$gHostRoleXS2Standby/} @rolesForAddition;

	return ($isAddingXsHost || $isAddingLocalXsRole) ? 1 : 0;
}

sub fillSelectedComponentsDefaultValue {
	my $self = shift();
	$self->SUPER::fillSelectedComponentsDefaultValue(@_);
	my $defaultSelection = $self->getDefault('SelectedComponents');
	return if(length($defaultSelection) > 0); # Nothing to do - default selection is already available
# No components available for default selection. Default value "server - No additional components" is proposed
	my $serverBatchKey = $LCM::Component::componentKeynameToBatchKey->{$gKeynameEngine};
	$self->setDefault('SelectedComponents', $serverBatchKey);
}

sub isServerNoStart {
    my ($self) = @_;
    return $self->getValue('NoStart');
}

sub getParamInitialVolumeBasePath {
	my ( $self, $order, $section, $pathName, $description, $default, $footprint_str, $required_disk_space ) = @_;
	my $param = $self->getParamBasePath( $order, $section, $pathName, $description );
	$param->{default}                = $default;
	$param->{init_with_default}      = 1;
	$param->{skip}                   = 0;
	$param->{footprint_str}          = $footprint_str;
	$param->{required_disk_space}    = $required_disk_space;
	$param->{diskspace_check_subdir} = $volumeSubDirPerHost;
	return $param;
}

sub getParamTimeZone {
	my ($self, $order, $section) = @_;
	my %param = (	'order'             => $order++,
				'opt'               => 'timezone',
				'type'              => 'string',
				'section'           => $section,
				'value'             => undef,
				'str'               => 'System Administrator Timezone',
				'desc'              => "Timezone of the system administrator",
				'init_with_default' => 0,
				'set_interactive'   => 0,
				'mandatory'         => 0,
				'hidden'            => 1
				);
	return \%param;
}

sub getParamClientPath {
	my ($self, $order, $section) = @_;
	my %param = (	'order'             => $order++,
				'opt'               => 'client_path',
				'type'              => 'path',
				'section'           => $section,
				'value'             => undef,
				'default'           => $isWin ? '$Drive\usr\sap\$SID\hdbclient' : '$Target/$SID/hdbclient',
				'str'               => "$gProductNameClient Installation Path",
				'desc'              => "Installation path for the $gProductNameClient",
                'constraint'        => "Valid for Installation of $gProductNameClient only",
				'init_with_default' => 1,
				'set_interactive'   => 0,
				'mandatory'         => 1
				);
	return \%param;
}

sub CheckParams{
	my ($self, $batchMode) = @_;
	my $rc = $self->SUPER::CheckParams($batchMode);
	return $rc if(!$rc);

	my $addHostsParam = $self->{params}->{AddHosts};
	my $addHostsBatchValue = $addHostsParam->{batchValue};
	my $isGui = $self->{options}->{isGUI};
	my $isPendingInstall = $self->isPendingAction();

	if ($isPendingInstall){
		$self->setInteractive('WorkerGroup', 0);
		$self->setInteractive('SelectedComponents', 0);
		$self->setNoRetry('SelectedComponents', 1);
	}
	if ( (!$isGui) && (!$batchMode && ((!defined $addHostsBatchValue) || ($addHostsBatchValue eq ""))) && !$isPendingInstall){
		$self->setInteractive('AddHosts', 1);
		$self->setSkip('HostsProperties', 0);
		$self->setInteractive('HostsProperties', 1);
	}

	return $rc;
}

sub checkAddHostsAvailability { return 1; }
sub isPendingAction {
    return existsHdblcmPendingInstall( $_[0]->getValueOrBatchValue("SID"), $_[0]->getValueOrBatchValue("Target"));
}

sub checkAddHosts{
    my ($self, $value) = @_;

    my $rc = $self->_checkAddHosts($value);
    return $rc if defined $rc;

    return $self->SUPER::checkAddHosts( $value, undef, 1 );
}

sub _recomputeRequiredAcceleratorParams{
    my ($self) = @_;
    $self->enableAcceleratorParams ($self->isComponentSelected($gKeynameAccelerator));

    my $addLocalRolesValue = $self->getValue('AddLocalRoles');
    if(defined $addLocalRolesValue
       && ($addLocalRolesValue =~ /$gHostRoleAcceleratorStandby|$gHostRoleAcceleratorWorker/)){
        $self->setSkip('AcceleratorUser', 0);
        $self->setSkip('AcceleratorPassword', 0);
    }
}

sub InitDefaults {
	my ( $self, $kit ) = @_;
	$self->SUPER::InitDefaults($kit);

	if ($self->isPostServerResumeInstallation()) {
		$self->setType('SQLSysPasswd', 'passwd');
	}

	if(!$isWin && !isAdmin()){
		$self->resetError();
		my $flavourProductName = $self->getFlavourProductName();
		$self->setErrorMessage("$flavourProductName System can be installed with root privileges only");
		return 0;
	}

	$self->getPersistenceManager()->registerInitialParamsToPersist();
	my $physicalMemory = $self->getPhysicalMem();
	if (!defined $physicalMemory || ($physicalMemory < $gMinMemGB * 1024 * 1024 * 1024)) {
		$self->{params}->{RestrictMaxMem}->{skip} = 1;
		$self->{params}->{MaxMem}->{skip} = 1;
	} else {
		$self->setDefault('MaxMem', int($physicalMemory/(1024*1024)));
		$self->{params}->{MaxMem}->{minValue} = $gMinMemGB * 1024;
		$self->{params}->{MaxMem}->{maxValue} = int($physicalMemory/(1024*1024));
	}

	$self->getCollectSAPSystems();
	my $msglst = new SDB::Install::MsgLst();

	$self->setDefault('InstanceNumber', undef);

	my $mcm = $self->getMediumComponentManager ();
	my $flavour = $mcm->getHANAFlavour ();
	my ($isTopDown,$preferredInstanceNumber);

	if ($flavour eq $gFlavourExpress){
		$isTopDown = 1;
		$preferredInstanceNumber = 90;
	}
	elsif ($flavour eq $gFlavourCockpit){
		$isTopDown = 1;
		$preferredInstanceNumber = 96;
	}

	my $defaultInstanceNumber = $self->getFreeInstanceNumber ($isTopDown, $preferredInstanceNumber);
	$self->setDefault ('InstanceNumber', $defaultInstanceNumber);
	$self->{params}->{InstanceNumber}->{valid_values} = $self->getFreeInstanceNumbers ();
	$self->{params}->{InstanceNumber}->{f_get_console_description} = sub { '' };
	if ( !$isWin ) {
		my $gid = getgrnam($gSapsysGroupName);
		if ( defined $gid ) {
			$self->{sapsysids}->{$gid}                = [ hostname() ];
            $self->{localSapsysGID}                   = [ $gid, hostname() ];
		}
	} else {
		my $drive = getSAPDrive ($self->getMsgLst ());
		if ($drive) {
			$self->{params}->{Drive}->{value} = uc($drive);
			$self->{params}->{Drive}->{isStaticValue} = 1;
			#$self->{params}->{Target}->{value} = $path;
		}
	}
	$self->setDefault('Target', '/hana/shared');
	if(!$self->isPendingInstallation()){ # The password's type is pre-init when resuming
		$self->{params}->{Password}->{type} = 'initial_passwd';
	}

	if ( defined $self->{componentManager} ) {
		$self->selectAllComponents(0);
		my $serverComponent = $self->{componentManager}->getComponentByKeyName( $gKeynameEngine );
		if ( !defined $serverComponent ) {
			$self->setErrorMessage(GetServerNotFoundMessage("$gProductName System installation"));
			return undef;
		}

		# include additional, non-hardcoded server plugins in selection
		foreach my $component(@{$self->{componentManager}->getServerPluginComponents()}) {
			if (grep {$_ eq $component->getComponentBatchKey()} $self->{params}->{SelectedComponents}->{valid_values}) {
				next;
			}
			push (@{$self->{params}->{SelectedComponents}->{valid_values}}, $component->getComponentBatchKey());
		}

		my $consoleText = $serverComponent->getComponentName() . " version '" . $serverComponent->getVersion()
							. "' will be installed.\n\nSelect additional components for installation:\n";
		$self->{params}->{SelectedComponents}->{console_text} = $consoleText;
		$self->initializeSelectedComponents();
	}
	$self->setDefault('ISCMode', 'standard', 1);

	if(!$isWin && sapadmUserExists('sapadm')){
		$self->setType('HostagentPassword', 'passwd');
	}

	return 1;
}

sub fillValidSelectedComponentValues {
	my ($self) = @_;
	return undef if (!$self->SUPER::fillValidSelectedComponentValues());

	my $mcm = $self->getComponentManager();
	my $numDetectedComponents = scalar @{$mcm->getExternalDetectedComponents()};
	my $isOnlyServerDetected = $numDetectedComponents == 1 && defined $mcm->getComponentByKeyName($gKeynameEngine);
	if ($isOnlyServerDetected) {
		$self->setInteractive('SelectedComponents', 0) if ($self->getFlavourProductName ne $gProductNameCockpit);
	}
	$self->setSkip('AutoAddXS2Roles', 1) if($isOnlyServerDetected);

	$self->_recomputeRequiredParameters(); #This is usually called from the SelectedComponents listener
	return 1;
}

sub checkTarget {
	my ($self, $value) = @_;
	my $superCheck = $self->SUPER::checkTarget($value);

	if($superCheck && !$isWin && -d $value && MyRealPath($value) eq "/") {
		$self->PushError("The given path points to '/'");
		return 0;
	}

	return $superCheck;
}

sub pers_filename {
    my ($self) = @_;
    my $statusFileExtension = $self->getStatusFileExtension();
    if(!$self->{current_sid}){
        $self->{current_sid} = $self->getValue('SID');
    }
    return $self->getPersFilenameInstUpd ($statusFileExtension, 1);
}

sub getStatusFileExtension {
	return "hdblcm.install";
}

sub setAddLocalRoles {
	my ($self, $value) = @_;
	my $localhost = $self->getLocalHanaHost();

	return 0 if (!$self->checkHostRoles($localhost, $value, $gHostRoleWorker));

	$self->{params}->{AddLocalRoles}->{value} = $value;
	return 1;
}

sub checkTimezone {
	my ( $self, $timezone ) = @_;
	if(defined $timezone){
	   $self->addParameterWarning('Timezone',"Timezone parameter is deprecated since SP09 and will be ignored.");
	}
	return 1;
}

sub resetAddHosts {
	my $self = shift;
	undef $self->{_remoteHosts};
}

sub getSystemHosts {
	my $self        = shift;
	my $systemHosts = [ $self->getLocalHanaHost() ];
	my $remoteHosts = $self->getRemoteHosts();

	if ( defined $remoteHosts ) {
		my $remoteHostnames = $remoteHosts->getHostNames();
		push( @$systemHosts, @$remoteHostnames );
	}

	return $systemHosts;
}

sub getLocalHanaHost {
	my $self     = shift;
	my $hostname = $self->getValue('HostName');
	$hostname    = lc(hostname()) if (!defined $hostname);
	return $hostname;
}

sub isDistributedSystem {
	my ($self) = @_;
	my $isSkippedAddHosts = $self->isSkipped('AddHosts');
	return 0 if($isSkippedAddHosts); # if parameter is skipped, it doesn't matter if it has a value
	return $_[0]->getValue('AddHosts') ? 1 : 0;
}

sub getUserParams {
	my $self       = shift;
	my @userParams = $self->SUPER::getUserParams(@_);
	push( @userParams, 'Timezone' );
	return \@userParams;
}

sub getIgnoreValues { return [
                                'check_component_dependencies',
                                'check_diskspace',
                                'check_min_mem',
                                'check_platform',
                                'check_signature_file',
                                'verify_signature',
                        ];
}

sub handleSelectionOfServerComponent {
	my $self            = shift;

	my $serverComponent = $self->{componentManager}->getComponentByKeyName( $gKeynameEngine );

	if ( defined $serverComponent ) {
		$serverComponent->selectComponent();
		$serverComponent->setCanSelectComponent(0);
		$self->handleSelectionOfResidentInstaller();
	}

	return 1;
}

sub addListeners {
	my ( $self, ) = @_;
	$self->SUPER::addListeners();

	$self->addParameterListener('UseMasterPassword', LCM::Configuration::ValueChangeListeners::Install::UseMasterPasswordListener->new() );
	$self->addParameterListener('SelectedComponents', LCM::Configuration::ValueChangeListeners::Install::SelectedComponentsListener->new() );
    $self->addParameterListener('SelectedComponents', LCM::Configuration::ValueChangeListeners::SignatureVerificationListenerFactory::getSignatureVerificationListener() );
    $self->addParameterListener('SelectedComponents', LCM::Configuration::ValueChangeListeners::Install::DetectCustomConfigDir->new());
    $self->addParameterListener('SelectedComponents', LCM::Configuration::ValueChangeListeners::Install::XsTenantInstallationListener->new());
    $self->addParameterListener('SelectedComponents', LCM::Configuration::ValueChangeListeners::Install::XsInitTenantDatabaseListener->new() );
	$self->addParameterListener('RemoteExecution', LCM::Configuration::ValueChangeListeners::Install::RemoteExecutionListener->new() );
	$self->addParameterListener('RootUser', LCM::Configuration::ValueChangeListeners::Install::RootUserListener->new() );
	$self->addParameterListener('SID', LCM::Configuration::ValueChangeListeners::Install::SIDListener->new() );
	$self->addParameterListener('InstanceNumber', LCM::Configuration::ValueChangeListeners::InstallUpdate::HandleLSSUserAndGroupListener->new() );
	$self->addParameterListener('InstanceNumber', SDB::Install::Configuration::LSSUserStringListener->new(['LSSUserID', 'LSSPassword']));
	$self->addParameterListener('UID', LCM::Configuration::ValueChangeListeners::ReservedIDsHandler->new('UID', 1) );
	$self->addParameterListener('GID', LCM::Configuration::ValueChangeListeners::ReservedIDsHandler->new('GID', 0) );
	$self->addParameterListener('LSSUserID', LCM::Configuration::ValueChangeListeners::ReservedIDsHandler->new('LSSUserID', 1));
	$self->addParameterListener('LSSGroupID', LCM::Configuration::ValueChangeListeners::ReservedIDsHandler->new('LSSGroupID', 0) );
	$self->addParameterListener('XSSpaceUserIdSAP', LCM::Configuration::ValueChangeListeners::ReservedIDsHandler->new('XSSpaceUserIdSAP', 1) );
	$self->addParameterListener('XSSpaceUserIdProd', LCM::Configuration::ValueChangeListeners::ReservedIDsHandler->new('XSSpaceUserIdProd', 1) );
	$self->addParameterListener('RootPassword', new LCM::Configuration::ValueChangeListeners::FillingCertificatesListener );
    $self->addParameterListener('RootPassword', LCM::Configuration::ValueChangeListeners::Install::RootPasswordListener->new());
	$self->addParameterListener('HostagentPassword', LCM::Configuration::ValueChangeListeners::FillingCertificatesListener->new());
	$self->addParameterListener('AddHosts', LCM::Configuration::ValueChangeListeners::FillingCertificatesListener->new() );
	$self->addParameterListener('AddHosts', LCM::Configuration::ValueChangeListeners::Install::AdditionalHostsListener->new() );
	$self->addParameterListener('HostName', LCM::Configuration::ValueChangeListeners::FillingCertificatesListener->new() );
	$self->addParameterListener('AutoAddXS2Roles', LCM::Configuration::ValueChangeListeners::Install::AutoAddXsRolesListener->new());
	$self->addParameterListener('DbMode', LCM::Configuration::ValueChangeListeners::Install::DbModeListener->new());
	$self->addParameterListener('AddLocalRoles', sub {
		$self->setSkip('AutoAddXS2Roles', $self->_shouldSkipAutoAddXS2Roles());
		return 1;
	});
    $self->addParameterListener('InstallXSInDefaultTenant', LCM::Configuration::ValueChangeListeners::InstallUpdate::InstallXSInDefaultTenantListener->new());
	$self->addParameterListener('ProdSpaceName', LCM::Configuration::ValueChangeListeners::InstallUpdate::XSSpaceIsolationStringListener->new());
	$self->addParameterListener('XSSpaceIsolation', LCM::Configuration::ValueChangeListeners::InstallUpdate::RecomputeXs2SpaceIsolationParameters->new());
	$self->addParameterListener('XSSpaceIsolation', LCM::Configuration::ValueChangeListeners::XsSpaceIsolationListener->new());

	my $xs2AppsListener = LCM::Configuration::ValueChangeListeners::InstallUpdate::RecomputeXs2ApplicationParametersListener->new();
	$self->addParameterListener('SelectedComponents', $xs2AppsListener);
	$self->addParameterListener('AutoAddXS2Roles', $xs2AppsListener);
	$self->addParameterListener('AddLocalRoles', $xs2AppsListener);
	$self->addParameterListener('AddHosts', $xs2AppsListener);
	$self->addParameterListener('AddRoles', $xs2AppsListener);
	$self->addParameterListener('AddHosts',$self->getAutoInitFamilyServicesParameterHanlder('AddHosts'));
	$self->addParameterListener('AddLocalRoles',$self->getAutoInitFamilyServicesParameterHanlder('AddLocalRoles'));
	$self->addParameterListener('CreateInitialTenant',LCM::Configuration::ValueChangeListeners::Install::CreateInitialTenantListener->new());
    $self->addParameterListener('InstanceNumber', SDB::Install::Configuration::LSSPasswordListener->new());
    $self->addParameterListener('VolumeEncryption', LCM::Configuration::ValueChangeListeners::Install::VolumeEncryptionListener->new());
    $self->addParameterListener('AddComponentsLocation', LCM::Configuration::ValueChangeListeners::AddComponentsLocationListener->new());
    $self->addParameterListener('LSSTrustUnsignedServer', LCM::Configuration::ValueChangeListeners::InstallUpdate::LocalSecureStoreListener->new());
}

sub getAction {
	return INSTALL_ACTION;
}

sub getInstallerCollectOtherHostInfoOperation{
	return $gOperationHdblcmCollectInstallerInfo;
}

sub checkDbMode {
    my ($self, $value) = @_;

    if (!$self->SUPER::checkDbMode ($value)){
        return 0;
    }

    if ($value eq 'singledb') {
        return $self->_checkSingleDB();
    }

    my $componentManager = $self->getComponentManager();
    my @selectedComponents = @{ $componentManager->getSelectedComponents() };

    for my $component (@selectedComponents) {
        return 0 if (!$self->_checkIfComponentSupportsMultiDb($component));
    }
    return 1;
}

sub _checkSingleDB {
    my ($self, ) = @_;
	if ( ! $self->_handleMoreThanOneHost($gHostRoleEsWorker) ) {
		return 0;
	}
	if ( ! $self->_handleMoreThanOneHost($gHostRoleAcceleratorStandby) ) {
		return 0;
	}
	if ( ! $self->_handleMoreThanOneHost($gHostRoleAcceleratorWorker) ) {
		return 0;
	}

	return 1;
}

sub _handleMoreThanOneHost {
    my ($self, $role ) = @_;

	my $hostsWithRole = $self->getHostsListWithRole($role);
	if ( scalar (@$hostsWithRole) < 2 ) {
		return 1;
	}
 	my $roleHostsString = join(',', @$hostsWithRole);
	my $hostRoleProperties = GetHostRoleProperties();
	my $roleString = $hostRoleProperties->{$role}->{str};
	my $componentName = getComponentNameByHostRole($role);
	my $message = "Hosts $roleHostsString have been assigned with role \"$roleString\".";
	$message .= "Scale-out setup for $componentName is currently not supported.";

	$self->PushError( $message );
	return 0;
}

sub getSystemHostsWithRole {
	return [];
}

sub getHostsListWithRole {
    my ($self, $role) = @_;
	my @hostsWithRole = ();

	my $addLocalRoles = $self->getValue('AddLocalRoles');
	if (defined $addLocalRoles) {
		my @localRoles = split(',', $addLocalRoles);
		if ( $role ~~ @localRoles ) {
			push (@hostsWithRole, $self->getValue('HostName'));
		}
	}

	my $addHostsListWithRole = $self->getAddHostsWithRole( $role );
	push ( @hostsWithRole, @$addHostsListWithRole );

	return \@hostsWithRole;
}

sub _shouldSkipHostagentPassword {
	my ($self) = @_;
	return (!$self->_shouldInstallSHAOnHosts());
}

sub checkXSTenantDatabaseName {
# Do not call the method in the parent class
    return 1;
}

sub checkHostagentPassword {
	my ($self, $password) = @_;

	my $rc = $self->SUPER::checkHostagentPassword($password);
	if (!$rc) {
		return undef;
	}

	if (!$self->isUseSAPHostagent()) {
		return $rc;
	}

	my $addHostsValue = $self->getValue('AddHosts');
	if (! defined $addHostsValue) {
		return $rc;
	}

	my ($addHostsMap, $errMsgLst) = LCM::HostsParser::ParseHostStringToMap($addHostsValue);
	if (! defined $addHostsMap) {
		$self->PushError("Error in parsing 'addhosts' parameter value", $errMsgLst);
		return undef;
	}

	my @addHosts = keys %$addHostsMap;
	if (scalar @addHosts == 0) {
		return $rc;
	}

	my $sapAdmUser = $self->_createSapAdmUser();
	$rc = $sapAdmUser->verifyPasswordSHA($password, $self, \@addHosts);

	if (! defined $rc) {
		$self->{params}->{HostagentPassword}->{no_retry} = 1;
	}

	return $rc;
}

sub _createSapAdmUser {
	my ($self) = @_;

	my $userName = $self->getValue('HostagentUserName');
	if ( ! defined $userName ) {
		$userName = 'sapadm';
	}

	require LCM::Configuration::SapAdmUser;

	return new LCM::Configuration::SapAdmUser($userName);
}

sub checkPassword{
	my ($self, $value) = @_;
	if($self->getType('Password') eq 'initial_passwd'){
		return $self->SUPER::checkPassword($value);
    }
	if(!$value || length($value) == 0) {
		$self->PushError ("Password cannot be empty.");
		return 0;
	}

	my $sidadmUser = new LCM::Configuration::NewDBUser::User( $self->getSysAdminUserName($self->getValue('SID')));
	my $savedCtx = $sidadmUser->setMsgLstContext([$self->getMsgLst()]);
	my $rc = $sidadmUser->verifyPassword($value, $self);
	if (!defined $rc){
		$self->getMsgLst()->appendMsgLst ($sidadmUser->getErrMsgLst());
		$self->getMsgLst()->addMessage ("Ignoring error => password validation skipped.");
		return 1;
	}
	if($rc != 1 && !isSidadmin($self->getSID())) {
		$self->appendErrorMessage ('Unknown user password combination');
	}
	return $rc;
}


sub getHiddenIgnoreValues{
    return [qw (scan_password_database check_hostname check_pmem)];
}

sub _recomputeRequiredRootCredentials {
	my ($self) = @_;
	if ($self->isUseSAPHostagent()) {
		$self->setSkip('RootUser', 1);
		$self->setSkip('RootPassword', 1);
	}
}

sub pers_keys {
	my ($self) = @_;

	for my $paramId ( keys %{$self->{params}} ) {
		if (  $self->_shouldPersistParameter( $paramId ) ) {
			$self->setPersStep($paramId, 1);
		}
	}
    return [@{$self->SUPER::pers_keys()},
            'pers_install_steps'];
}

sub pers_getstepname {
    my $self = shift();
    return $self->getStepName();
}

sub setPersistedSteps{
    my($self,$components) = @_;
    $self->{'pers_install_steps'} = $components;
}

sub getPersistedSteps{
    my ($self) = @_;
    return $self->{'pers_install_steps'};
}

sub _shouldPersistParameter {
	my ($self, $paramId) = @_;
	if (!defined $self->getValue($paramId)
	    || ( $paramId eq 'SID' )
	    || $self->isSkipped( $paramId )
	    || ( defined $self->getType($paramId) && ($self->getType($paramId) =~ /passwd/) )) {
	    return 0;
	}
	return 1;
}

sub handlePendingInstall {
	my ($self, $value) = @_;
	return $self->getPersistenceManager()->handlePendingInstall($value);
}

sub getAutoAssignXs2RolesMap {
	my ($self) = @_;
	my $localRolesString = ($self->isSkipped('AddLocalRoles') || !defined $self->getValue('AddLocalRoles'))
	                       ? $gHostRoleWorker
	                       : join(',', $gHostRoleWorker, split(',', $self->getValue('AddLocalRoles')));
	my $additionalHostsString = $self->isSkipped('AddHosts') ? '' : $self->getValue('AddHosts');
	my $localHostString = sprintf("%s:role=%s", $self->getLocalHanaHost(),  join(':role=', split(',', $localRolesString)));
	my $allHostString = (defined $additionalHostsString)
	                    ? [ $localHostString, split(',', $additionalHostsString) ]
	                    : [ $localHostString ];
	my $resultMap = {};

	for my $hostString (@{$allHostString}) {
		if($hostString =~ /^([^:]+)/) {
			$resultMap->{$1} = $self->_determineApplicableXs2Role($hostString);
		}
	}
	return $resultMap;
}

sub isXSWorkerHostAvailable {
	my ($self) = @_;
	my $autoAddXS2RolesValue = $self->hasValue('AutoAddXS2Roles') ? $self->getValue('AutoAddXS2Roles') : 1;
	my $addHostsValue = $self->hasValue('AddHosts') && !$self->isSkipped('AddHosts') ? $self->getValue('AddHosts') : '';
	my $additionalLocalRoles = $self->hasValue('AddLocalRoles') && !$self->isSkipped('AddLocalRoles') ? $self->getValue('AddLocalRoles') : '';

	return 1 if(!$self->isSkipped('AutoAddXS2Roles') && $autoAddXS2RolesValue);
	return 1 if($addHostsValue =~ /:role=$gHostRoleXS2Worker/);
	return 1 if($additionalLocalRoles =~ /$gHostRoleXS2Worker/);
	return 0;
}

sub isAutoInitializeServicesRequired {
     my ($self)                = @_;
     return $self->isSystemInCompatibilityMode();
}

sub isPostServerResumeInstallation {
	my ($self, $sidValue) = @_;
	my $sid = $sidValue // $self->getValue('SID');
	return $self->isPendingInstallation() && !existsHdbinstPendingInstall($sid, $self->getValue('Target'));
}

sub getPersFilenameInstUpd {
    my ($self, $statusFileExtension, $isInstallation) = @_;

    my $filename = $self->SUPER::getPersFilenameInstUpd ($statusFileExtension, $isInstallation);
    if (! defined $filename) {
        my $sid = $self->getSID() || $self->{current_sid};
        my $file = $sid.'.'.$statusFileExtension;
        $filename = File::Spec->catfile($gSAPLocation,$file);
        return $filename;
    }
    my($vol,$filenameDir,$file) = File::Spec->splitpath($filename);
    if(! -d $filenameDir) {
        $filename = File::Spec->catfile($gSAPLocation,$file);
    }

    return $filename;
}

sub checkUID {
    my ($self, $value, $msglst) = @_;
    return 0 if (! $self->SUPER::checkUID($value, $msglst));

    my $sapsysGroup = new SDB::Install::Group($gSapsysGroupName);
    if ($sapsysGroup->exists() && !$self->checkSelectedComponentsPermissions(undef, $value, $sapsysGroup->id())){
        return 0;
    }
    return $self->checkReservedOSUserIDs($value, 'UID');
}

sub checkGID {
    my ($self, $value, $msglst) = @_;
    return 0 if (! $self->SUPER::checkGID($value, $msglst));

    my $sapsysGroup = new SDB::Install::Group($gSapsysGroupName);
    my $user = new SDB::Install::NewDBUser($self->getSID());
    if (!$user->exists() && !$sapsysGroup->exists()){
        return $self->checkSelectedComponentsPermissions(undef, $self->getValue('UID'), $value);
    }
    return 1;
}

sub checkSID {
    my ($self, $value) = @_;
    if ($self->isPostServerResumeInstallation($value)) {
        return 1;
    }
    if (!$self->SUPER::checkSID($value)){
        return undef;
    }
    if (!$self->isUseSAPHostagent()) {
        return 1;
    }
    my $checkMnt = $self->getValue('CheckMnt');
    if (defined $checkMnt) {
        my ($sapmnt, $checkMntSid) = ($checkMnt =~ /(.*)\/+([^\/]+)\/*$/);
        if ($value ne $checkMntSid) {
            $self->getErrMsgLst()->addMessage("$self->{params}->{SID}->{str} '$value' doesn't match value of parameter '"
                                             . $self->getOpt('CheckMnt')
                                             . "=$checkMnt'. For SAPHostControl "
                                             . $self->getOpt('CheckMnt')
                                             . " pattern is ('<Installation Path>/<SID>'.");
            return undef;
        }
    }
    return 1;
}

sub setHostagentPasswordSHAExecution {
    my ($self, $value) = @_;
    return undef if(!$self->setHostagentPassword($value));
    $self->{params}->{HostagentPasswordSHAExecution}->{value} = $value;
    return 1;
}

sub setHostagentPassword {
    my ($self, $value) = @_;
    if (!$self->SUPER::setHostagentPassword($value)) {
        return 0;
    }
    if ($self->getType('HostagentPassword') eq 'initial_passwd') {
# Bug: 180280 - Prevents a missleading message that 'sapadm_password' won't be used
        $self->deleteParamById('HostagentPasswordSHAExecution');
    }
    return 1;
}

sub _enableMultipleHostHostagentPassword {
    my ($self) = @_;
    my $appContext = LCM::App::ApplicationContext->getInstance();
    my $app = (defined $appContext) ? $appContext->getApplication() : undef;
    if (defined $app && !$app->isGUIApplication()) {
        $self->setSkip('HostagentPasswordSHAExecution', 0);
    }
    $self->setSkip('HostagentPassword', 0);
    $self->setType('HostagentPasswordSHAExecution', 'passwd');
    $self->setType('HostagentPassword', 'passwd');
    $self->setSkip('HostagentUserName', 0);
    my $remoteHosts = $self->getRemoteHosts();
    if (defined $remoteHosts && $remoteHosts->isHostctrl()) {
        $remoteHosts->useHostagentUser(); # resets useSidadm
    }
}

sub checkDataAndLogPathExistenceOnHost { return 1 };

sub getSystemUsage {
    my ($self) = @_;
    return $self->getValue('SystemUsage');
}

# Override
sub setLSSPasswordType {
    my ($self) = @_;
    my $sidcryptUser = SDB::Install::LSS::LssSidcryptUser->new(getSidcryptName($self->getValue('SID')));
    if(!$sidcryptUser->exists()) {
        $self->setType('LSSPassword', 'initial_passwd');
    }
}

# Override
sub shouldUseOptimizedExecution {
    my ($self) = @_;
    return 0 if !$self->getFeatureManager()->isOptimizedInstallationSupported();
    return 1 if $self->SUPER::shouldUseOptimizedExecution();
    return $self->getComponentManager()->isHDBServerPluginSelected();
}

# Override
sub getAllDbHosts {
    my ($self, $instance) = @_;
    my $dbHosts = $self->SUPER::getAllDbHosts($instance);
    unshift(@$dbHosts, $self->getLocalHanaHost());
    return $dbHosts;
}

sub canMigrateToLSS {
    return 1;
}

1;
