package LCM::Configuration::Hosts::AddHosts::AddHostConfiguration;

use strict;
use File::stat;
use File::Spec;
use SDB::Install::Configuration::NewServerConfig;
use LCM::Configuration::Hosts::AddHosts::AddHostsBaseConfiguration;
use SDB::Install::SysVars qw ($isWin $path_separator);
use SDB::Install::Globals qw ($gHostRoleWorker
                              $gHostRoleStandby
                              $gHostRoleAcceleratorStandby
                              $gHostRoleAcceleratorWorker
                              $gProductNameEngine
                              $gPlatform);
use SDB::Install::System qw(readLink);
use LCM::FileUtils;

require SDB::Install::Configuration::AddHost;
use LCM::Configuration::Hosts::AddHosts::Validators::LSS::OSUserCheck;
use LCM::Configuration::Hosts::AddHosts::Validators::LSS::InstallationPathCheck;

use base qw( SDB::Install::Configuration::AddHost LCM::Configuration::Hosts::AddHosts::AddHostsBaseConfiguration);

sub new {
    my $self = shift()->SUPER::new (@_);
    my $order = 0;
    my $section = $LCM::Configuration::Hosts::AddHosts::AddHostsBaseConfiguration::ini_section_addhosts;

    $self->{params} = {
        'InstallHostagent' => $self->getParamInstallHostagent ($order++, $section, undef, 1),
        'SID'       => $self->getParamSID      ($order++, $section),
        'HostName'  => $self->getParamHostName ($order++, $section),
        $isWin ? (
            'Drive' => $self->getParamDrive ($order++, $section)
        ) : (),
        'Target' => $self->getParamTarget ($order++, $section),
        'HostRoles'              => $self->getParamHostRoles             ($order++, $section),
        'HostFailoverGroup'      => $self->getParamHostFailoverGroup     ($order++, $section),
        'StoragePartitionNumber' => $self->getParamStoragePartitionNumber($order++, $section),
        'WorkerGroup'            => $self->getParamWorkerGroup           ($order++, $section),
        'Password' => $self->getParamPassword($order++, $section),
        'NoStart' => $self->getParamNoStart ($order++, $section, 'Do not start the instance after installation', 'Does not start the instance after installation', 0),
        ($isWin  ? ('HostagentUserName' => $self->getParamHostagentUserName($order++, $section)) : ()),
        'HostagentPassword' => $self->getParamHostagentPassword($order++, $section),
        'ImportContentXS2' => $self->getParamImportContentXS2($order++, $section),
        'SystemUser'          => $self->getParamSystemUser         ($order++, $section, 1),
        'SQLSysPasswd'        => $self->getParamSQLSysPasswd       ($order++, $section, 'passwd', 1),
        'AcceleratorUser'     => $self->getParamAcceleratorUser    ($order++, $section),
        'AcceleratorPassword' => $self->getParamAcceleratorPassword($order++, $section),
    };
    $self->setDefault('InstallHostagent', 1);
    $self->addListeners();
    return $self;
}

sub getNumberOfExpectedOutputLines {
    return 50;
}

sub checkWorkerGroup {
    my ($self, $workerGroupCSV) = @_;
    my @workerGroups = split(',', $workerGroupCSV);

    for my $group (@workerGroups){
        return undef if(!$self->SUPER::checkWorkerGroup($group));
    }
    return 1;
}

#-------------------------------------------------------------------------------
# Checks the host roles and enables the system user and accelerator user.

sub checkHostRoles {
    my ($self, $roleList) = @_;
	foreach my $currRole (split (',', $roleList)) {
		if (!$self->checkRole($currRole)) {
			return 0;
		}
	}

	my $shouldSkipAcceleratorParameters = 1;
    foreach my $currRole (split (',', $roleList)) {
        if ($currRole =~ /$gHostRoleAcceleratorStandby|$gHostRoleAcceleratorWorker/){
			$shouldSkipAcceleratorParameters = 0;
        	last;
        }
        if ($currRole =~ /^$gHostRoleWorker|^$gHostRoleStandby/) {
            return 0 if (!$self->checkSharedVolumePaths());
        }
    }

    $self->setSkip('SystemUser', $shouldSkipAcceleratorParameters);
	$self->setSkip('SQLSysPasswd', $shouldSkipAcceleratorParameters);
	$self->setSkip('AcceleratorUser', $shouldSkipAcceleratorParameters);
	$self->setSkip('AcceleratorPassword', $shouldSkipAcceleratorParameters);
    return 1;
}

sub checkSharedVolumePaths {
    my ($self) = @_;

    my $instance = $self->getOwnInstance();
    if (!$instance->isBasePathShared()) {
        return 1;
    }

    my $errlst = $self->getErrMsgLst();
    my $datapath = $instance->getDataPath(1);
    if (!defined $datapath) {
        $errlst->addError("Failed to get data volume path from global.ini");
        return 0;
    }
    if (!$self->_checkSharedLocalVolumePath($datapath)) {
        return 0;
    }

    my $logpath  = $instance->getLogPath(1);
    if (!defined $logpath) {
        $errlst->addError("Failed to get log volume path from global.ini");
        return 0;
    }

    if (!$self->_checkSharedLocalVolumePath($logpath)) {
        return 0;
    }

    return 1;
}

# This only tries to find the familiar directory structure
# inherent to shared data/log volumes. It does not actually check
# if this belongs to the proper system or any other more fullproof checks.
# Implemented like this because attempts to use hdbnsutil -printTopology to get
# the actual system's volume paths depended on creating a temprary sidadm user
# which we decided to avoid.
sub _checkSharedLocalVolumePath {
    my ($self, $volumePath) = @_;

    my $errlst = $self->getErrMsgLst();
    if (!defined $volumePath) {
        $errlst->addError("Volume path is undefined");
        return 0;
    }

    my $genericErrString = "Volume path ‘$volumePath’ is not shared with the existing database hosts of the system";
    if (!File::stat::stat($volumePath)) {
        $errlst->addError("$genericErrString. It is missing from the local host.");
        return 0;
    }

    my $isDirEmpty = isEmptyDirectory($volumePath, $errlst);
    if (!defined $isDirEmpty || $isDirEmpty) {
        $errlst->addError("$genericErrString. It is empty.");
        return 0;
    }

    my $volumePathContents = listDirectory($volumePath, $errlst);
    my @mntSubDirs = grep {$_ =~ /^mnt\d+/} @$volumePathContents;
    if (!@mntSubDirs) {
        $errlst->addError("$genericErrString. It does not contain any persistence volume sub directories.");
        return 0;
    }

    my $mntSubDir = File::Spec->catfile($volumePath, $mntSubDirs[0]);
    $isDirEmpty = isEmptyDirectory($mntSubDir, $errlst);
    if (!defined $isDirEmpty || $isDirEmpty) {
        $errlst->addError("$genericErrString. '$mntSubDir' is empty.");
        return 0;
    }

    my $volumeSubDirContents = listDirectory($mntSubDir, $errlst);
    my @hdbSubDirs = grep {$_ =~ /^hdb\d+/} @$volumeSubDirContents;
    if (!@hdbSubDirs) {
        $errlst->addError("$genericErrString. '$mntSubDir' does not contain the expected persistence volume sub directories.");
        return 0;
    }

    return 1;
}

#-------------------------------------------------------------------------------
# Checks if the specified SID and the system administrator user
#
# Parameter: string $wantedSID
#
# Returns int retCode

sub checkSID{
    my ($self, $wantedSID) = @_;

    my $properties = $self->{params}->{SID};

    if (!$self->checkAdditionalSID($wantedSID, $properties->{str})) {
        return 0;
    }

    my $user = new SDB::Install::NewDBUser ($wantedSID);
    my $userName = $user->getSidAdmName ();
    if ($user->exists()){
        $self->AddProgressMessage ("The defined user '$userName' already exists on the system. Neither the password, nor any other attribute of the user will be changed.");
        $self->AddProgressMessage ('Verify that the user is correct.');
    }

    $self->changePasswordStr($userName);
    $self->enableHostagentPasswordIfRequired();
    return 1;
}


#-------------------------------------------------------------------------------
# Checks the path consisting of the given target directory and SID
#
# Parameter: string $targetDirectory
#
# Returns int retCode

sub checkTarget{
    my ($self, $targetDirectory) = @_;
    my $sid = $self->{params}->{SID}->{value};
    my $path = $targetDirectory . $path_separator . $sid;

    if (!-d $path){
        $self->PushError ("Directory '$path' doesn't exist");
        return 0;
    }

	$self->{params}->{Target}->{no_retry} = 1;

	my $globalIni = $self->_getGlobalIni($path);

	if (!defined $globalIni) {
		return undef;
	}

	my $listenInterface = $globalIni->getValue ('communication','listeninterface');
	my $internalNetworkPrefix = $globalIni->getValue ('communication','internal_network');
	my $errMsg1 = "Inter-Service communication is set to 'local'.";
	my $errMsg1Append1 = "Please reconfigure your master node:";
	my $errMsg1Append2 = "Perform 'hdblcm --action=configure_internal_network' on master node.";
	my $errMsg2 = "Inter-Service communication is set to 'internal', but internal network is unknown.";

	if (! $self->validateListenInterface($listenInterface, $internalNetworkPrefix, $errMsg1, $errMsg1Append1, $errMsg1Append2, $errMsg2)) {
		return 0;
	}

    return $self->check_path_access($targetDirectory, 0);
}


sub _getGlobalIni {
	my ($self, $path) = @_;

	if (!defined $path) {
		return undef;
	}

	my $instanceDirLink = join ($path_separator, $path, 'exe',
			$gPlatform, $isWin ? () : ('hdb'));
	my $instanceDirAbsolute = readLink ($instanceDirLink);
	my $configDir = $instanceDirAbsolute . $path_separator . 'config';
	my $hostname = $self->getValue('HostName');
	my $layeredCfg = new SDB::Install::LayeredConfig(undef, $path, $hostname, undef, $configDir);

	if ($layeredCfg->ErrorState ()){
		$self->PushError ('Error initializing layered configuration', $layeredCfg);
		return undef;
	}

	$layeredCfg->resetMsgLstContext();
	my $globalIni = $layeredCfg->getIniFile ('global.ini');

	if (!defined $globalIni){
		$self->PushError ('Cannot get global.ini', $layeredCfg);
		return undef;
	}

	$globalIni->resetMsgLstContext();

	if (!defined $globalIni->readValues ()){
		$self->PushError ('Cannot read global.ini', $globalIni);
		return undef;
	}

	return $globalIni;
}


#-------------------------------------------------------------------------------
# Checks the specified sidadm password
#
# Parameter: string $sidadmPassword
#
# Returns int retCode

sub checkPassword{
    my ($self, $sidadmPassword) = @_;
    if (!$sidadmPassword){
        $self->PushError ("Empty system administrator password not allowed");
        return 0;
    }
    return 1;
}

sub addListeners {
    my ($self) = @_;
    $self->SUPER::addListeners();
    $self->addParameterListener( 'HostRoles', LCM::Configuration::Hosts::AddHosts::Validators::LSS::OSUserCheck->new('HostRoles'));
    $self->addParameterListener( 'HostRoles', LCM::Configuration::Hosts::AddHosts::Validators::LSS::InstallationPathCheck->new('HostRoles'));
}

sub setHostRoles {
    my($self, $value) = @_;
    return 0 if (!$self->checkHostRoles($value));
    $self->{params}->{HostRoles}->{value} = $value;
    return 1;
}

1;
