package LCM::Configuration::Hosts::AddHosts::Validators::HostagentPasswordValidator;

use strict;

use LCM::HostsUtility qw (IsHostPartOfSystem);
use LCM::HostsParser;

# Table Legend: 
# LH - LocalHost
# RH - RemoteHost(s)
# (0) - Installed SHA on all RemoteHosts that will be added
# (1) - If LH does not have SHA installed: type=initial_passwd
# (2) - Validate passwd only on LH
# (3) - SHA is not mandatory for LH
# (4) - Validate passwd on RH
# (5) - Validate passwd on LH if --install_hostagent=1 and SHA is not installed on LH

# Remote Execution	|	SSH		|	SAPHOSTAGENT

# Prerequisite		| 	No		| 	(0)
# Add Only LH		| 	(1)		| 	(5)
# Add LH + RH		| 	(1)(2) 	| 	(5) (4)
# Add Only RH		| 	(3) 	| 	(4)


sub new {
    my ( $class, $config ) = @_;
    
    my $self = {
    	"execution" => $config->getValue("RemoteExecution"),
    	"isHostPartOfSystem" => IsHostPartOfSystem(),
    	"isHostAgentInstalledOnLocalHost" => $config->isHostagentInstalled(),
    	"installHostagentValue" => $config->getValue('InstallHostagent'),
    };
    
    bless $self, $class;
    $self->_computeHostsFilters();
    return $self;
}

sub _computeHostsFilters {
    my ( $self ) = @_;

	if ( $self->{"execution"} eq "saphostagent" ) {
		$self->{shouldCheckLocalHost} = (!$self->{isHostPartOfSystem} && $self->{installHostagentValue} && !$self->{isHostAgentInstalledOnLocalHost}) ? 1 : 0; 
		$self->{shouldCheckRemoteHosts} = 1;
		return;
	}

	$self->{shouldCheckLocalHost} = 1;
	$self->{shouldCheckRemoteHosts} = 0;
}

sub validate {
	my ( $self, $value, $config ) = @_;

	if (  $self->{shouldCheckLocalHost} and ! $self->_checkLocalHostagentPassword( $value, $config ) ) {
		return undef;
	}

	if (  $self->{shouldCheckRemoteHosts} and ! $self->_checkRemoteHostsHostagentPassword( $value, $config ) ) {
		return undef;
	}
	
	return 1;
}

sub _checkLocalHostagentPassword {
    my ($self, $value, $config) = @_;
    return $config->checkLocalHostagentPassword( $value );
}

sub _checkRemoteHostsHostagentPassword {
	my ( $self, $value, $config ) = @_;

	my $addHostsValue = $config->getValue('AddHosts');
	if (! defined $addHostsValue) {
		return 1;
	}

	my ($localHost, $remoteHosts) = LCM::HostsParser::ExtractLocalAndRemoteHosts($addHostsValue, $config);
	if (! defined $remoteHosts) {
		return 1;
	}

	my ($remoteHostsMap, $errMsgLst) = LCM::HostsParser::ParseHostStringToMap($remoteHosts);
	if (! defined $remoteHostsMap) {
		$config->PushError("Error in parsing 'addhosts' parameter value: '$remoteHosts'", $errMsgLst);
		return undef;
	}

	my @addRemoteHosts = keys %{$remoteHostsMap};
	my $sapAdmUser = $self->_createHostagentUser($config);
	my $rc = $sapAdmUser->verifyPasswordSHA($value, $config, \@addRemoteHosts);

	if (! defined $rc) {
		$config->{params}->{HostagentPassword}->{no_retry} = 1;
	}

	return $rc;
}

sub _createHostagentUser {
	my ( $self, $config ) = @_;

    my $userName = $config->getValue('HostagentUserName');
    if ( ! defined $userName ) {
        $userName = 'sapadm';
    }

	require LCM::Configuration::SapAdmUser;

	return new LCM::Configuration::SapAdmUser($userName);
}

1;
