package LCM::Configuration::PersistenceManager;

use strict;

use experimental qw (smartmatch);

use SDB::Install::SAPSystem qw ( CollectSAPSystems );
use SDB::Install::Globals qw ($gKeynameEngine $gKeynameInstaller $gProductNameEngine $gProductNameInstaller);
use SDB::Install::System qw ( getAllUsersAppDataPath );
use SDB::Install::SysVars qw ($isWin);
use SAPDB::Install::Hostname qw (hostname);

use LCM::Task::TaskStatus qw (FINISHED);
use LCM::Component qw ( ALL PREPARE_PHASE );
use LCM::Task::GenericStackTask::ComponentTask;
use LCM::PersistencyUtils qw ( 	existsPendingComponentsUpdate existsHdblcmPendingInstall );

use base qw ( Exporter );

sub new {
	my ($class, $configuration) = @_;

	my $self = {};
	bless $self, $class;

	$self->setConfiguration($configuration);

	return $self;
}

sub getConfiguration {
	return $_[0]->{_configuration};
}

sub setConfiguration {
	$_[0]->{_configuration} = $_[1];
}

sub handlePendingAction {
	my ($self, $sid) = @_;

	my $config = $self->getConfiguration();
	if ( $config->isUpdate() ) {
		my $selectedCmpsBatchValue = $config->getBatchValue('SelectedComponents');

		# Don't support resume of pending update in case of hdblcm self-update (done during sidadm update).
		if (defined $selectedCmpsBatchValue && ($selectedCmpsBatchValue =~ m/^hdblcm$/)) {
			return 1;
		}
	}

	if ( ! $self->existsHdblcmPendingAction($sid, $config->getValueOrBatchValue('Target')) && ! $self->existsHdbPendingAction($sid) ) {
		return 1;
	}

	if ($self->existsHdbPendingActionOnly($sid)) {
		return $self->handleHdbPendingAction($sid);
	}

	return $self->handleHdblcmPendingAction($sid);
}

sub handleHdbPendingAction {
	my ($self, $sid) = @_;
	my $config = $self->getConfiguration();
	my $mcm = $config->getMediumComponentManager();

	if (!defined $mcm) {
		return undef;
	}

	my $detectedServerCmp = $mcm->getComponentByKeyName($gKeynameEngine);
	if (!defined $detectedServerCmp) {
		$config->getErrMsgLst()->addError("Pending " .  $config->getAction() . " of $gProductNameEngine cannot be resumed, " .
			"because $gProductNameEngine server component was not detected from the provided resources.");
		return undef;
	}

	my $detectedServerLocation = $detectedServerCmp->getPath();
	require SDB::Install::Kit;
	my $kit = new SDB::Install::Kit($detectedServerLocation);
	my $hdbConfig = $self->_createHdbConfiguration($sid);

	if (!$hdbConfig->InitDefaults($kit)) {
# Very nasty workaround for the problem described in Bug 87925 (see SDB::Install::Configuration::Upgrade sub InitDefaults)
# Ignore the very specific error when hdblcm is not started as root and is an official release
		my $manifest = $kit->getManifest();
		if ($isWin || $> == 0 || !$manifest->$hdbConfig()) {
			$config->getErrMsgLst()->addError(undef, $hdbConfig->getErrMsgLst());
			return undef;
		}
	}

	if (!$self->validateHdbPersistency($hdbConfig)) {
		return undef;
	}

	if (!$config->setValue('SelectedComponents', 'server')) {
		return undef;
	}

	$config->{params}->{'CertificatesHostmap'}->{'set_interactive'} = 0;
	my $skipIdsArray = ['SID'];
	if ( $config->isUpdate() ) {
		$self->_copyCommonParamsFromHdbupdIntoHdblcmConfig($hdbConfig, $skipIdsArray);
	}

	my $serverCmpTask = new LCM::Task::GenericStackTask::ComponentTask($config, $detectedServerCmp);
	my $pendingStepName = $serverCmpTask->getName();

	$config->setStepName($pendingStepName);
	if (!$config->getIgnore('check_pending_upgrade')) {
		$config->AddProgressMessage ("Resuming pending " .  $config->getAction() . " at step '${pendingStepName}'");
	}

	return 1;
}

sub validateHdbPersistency {
	my ($self, $hdbConfig) = @_;

	my $config = $self->getConfiguration();
	my $rc = $hdbConfig->pers_exists ();

	if (!defined $rc) {
		$config->getErrMsgLst()->addError(undef, $hdbConfig->getErrMsgLst());
		return undef;
	}
	if ($rc) {
		if (!defined $hdbConfig->validatePersistency()) {
			$config->getErrMsgLst()->addError(undef, $hdbConfig->getErrMsgLst());
			return undef;
		}
	} else {
		# Checking for deprecated hdbupd status file in /usr/sap
		my $hdbOldPersistenceData = $hdbConfig->validateOldLocalPersistencyFile ();
		if (defined $hdbOldPersistenceData) {
			if (!defined $hdbConfig->validatePersistency ($hdbOldPersistenceData)) {
				$config->getErrMsgLst()->addError(undef, $hdbConfig->getErrMsgLst());
				return undef;
			}
		}
	}

	return 1;
}

sub handleHdblcmPendingAction {
	my ($self, $sid) = @_;

	my $config = $self->getConfiguration();
	my $rc = $config->pers_exists();

	if (!defined $rc) {
		$config->AddError("Runtime error: status file name is unknown");
		return undef;
	}

    my $persistenceData = $config->pers_load();
	if (!$self->existsPendingAction($sid, $persistenceData)) {
		my $statusFileName = $config->pers_filename(); 
		$config->AddMessage("Deleting hdblcm status file $statusFileName : no component status file existing");
		if (!defined $config->pers_remove($statusFileName)) {
			$config->AddError('Unable to remove persistence file ', $config->getErrMsgLst());
			return 0;
		}
		return 1;
	}
	for my $parameterId ( keys %$persistenceData ) {
		$config->setPersStep($parameterId, 0);
	}
	if ($rc && !defined $config->validatePersistency($persistenceData)){
		return undef;
	}

	return 1;
	
}

sub existsPendingAction {
	my ($self, $sid, $persistenceData) = @_;
	return $self->existsHdbPendingAction($sid) || existsPendingComponentsUpdate($sid); 
}

sub getParamIdsToSkip {
    my ($self) = @_;
    return qw ( SID cmd_line_action CheckOnly PrepareUpdate Phase ContinueUpdate DvdPath ComponentFsRoot ExecutionMode );
}

sub registerParamsToPersist {
	my ($self) = @_;

	my $config = $self->getConfiguration();
	my @paramIdsToSkip = $self->getParamIdsToSkip();
	my @parametersIdsWithSetterOnDeserialization = qw (Scope RootUser ListenInterface AddHosts DbMode RestrictMaxMem AutoAddXS2Roles);
	for my $paramId (keys %{$config->{params}}) {
		my $param = $config->{params}->{$paramId};
		if (($param->{type} =~ /passwd/) || (grep {$paramId eq $_} ( @paramIdsToSkip ))) {
			next;
		}
		
		if ($paramId ~~ @parametersIdsWithSetterOnDeserialization) {
			$param->{shouldUseSetterOnDeserialization} = 1;
		}
		$param->{persStep} = 1;
	}

	return 1;
}

sub getPersistenceFilename {
	my ($self) = @_;

	my $config = $self->getConfiguration();
	my $sid = $config->{current_sid};

	if (!defined $sid){
		return undef;
	}

	my $action = $config->getAction();

	if ($config->isa ('LCM::Configuration::GenericStackUpdateResidentConfiguration')) {
		$action = LCM::Configuration::GenericStackUpdateConfiguration::getAction();
	}

	my $persName  = "$sid.hdblcm.$action";
	my $filename = undef;

	if ($isWin) {
		$filename = $self->_getPersistenceFilenameForWindows($persName);
	} else {
		$filename = $self->_getPersistenceFilenameForLinux($persName, $sid);
	}

	return $filename;
}

sub _getPersistenceFilenameForWindows {
	my ($self, $persName) = @_;

	my $config = $self->getConfiguration();
	my $msglst = new SDB::Install::MsgLst ();
	my $appData = getAllUsersAppDataPath ($msglst);

	if (!$appData){
		$config->AddError ('Cannot get Common AppData path', $msglst);
		return undef;
	}
	my $path = File::Spec->catfile($appData, '.hdb');
	if (!-e $path){
		my $cfg = {'mode' => 0775};
		if (!defined makedir ($path, $cfg)){
			$config->AddError("Cannot create directory $path", $cfg);
			return undef;
		}
	}

	return File::Spec->catfile($path, $persName);
}

sub _getPersistenceFilenameForLinux {
	my ($self, $persName, $sid) = @_;
	my $config = $self->getConfiguration();
	my $allSystems = CollectSAPSystems();
	my $sapSys = $allSystems->{$sid};

	return undef if (!defined $sapSys);

	my $instPath = $sapSys->get_target();
	if(defined($instPath) && length($instPath) > 0){
		# If system is installed under /usr/sap/ get_target returns undef
		return File::Spec->catfile($instPath, $sid, $persName);
	}
	return File::Spec->catfile($sapSys->getUsrSapSid(), $persName);
}

sub registerHostnameToPersist {
	my ($self) = @_;

	my $config = $self->getConfiguration();
	my $localHostname = $self->_getLocalHostname();

	if (defined $localHostname) {
		$config->{pers_hostname} = $localHostname;
	}
}

sub registerTimeoutsToPersist {
    my ($self) = @_;

    my $config = $self->getConfiguration();

    my $optTimeout = $config->getOptionTimeout();
    if (defined $optTimeout){
        my $timeoutArgs = $optTimeout->getSetKeysValidFor ($optTimeout);
        my $timeoutStr  = $optTimeout->getStringFromArgList ($timeoutArgs);
        if ($timeoutStr) {
            $config->{pers_timeouts} = $timeoutStr;
        }
    }
}

sub registerIgnoreOptionsToPersist {
    my ($self) = @_;
    my $config = $self->getConfiguration();
    my $optIgnore = $config->getOptionIgnore();

    if (defined($optIgnore)){
        my $ignoreArgs = $optIgnore->getSetKeysValidFor($optIgnore);
        my @updatedIgnoreArgs = grep(!/check_pending_upgrade/,@{$ignoreArgs});
        my $ignoreStr = $optIgnore->getStringFromArgList(\@updatedIgnoreArgs);
        $config->{pers_ignore} = $ignoreStr if defined($ignoreStr);
    }
}

sub _getLocalHostname {
	my ($self)= @_;

	my $config = $self->getConfiguration();
	my $instance = $config->getOwnInstance();

	return (defined $instance) ? $instance->get_host() : hostname();
}

sub validate {
	my ($self, $persData) = @_;

	my $config = $self->getConfiguration();

	if ((exists($persData->{'Scope'}) && ($persData->{'Scope'} eq 'instance')) ||
		(!exists($persData->{'Scope'}) && $config->isScopeInstance())) {
		return undef if (!$self->validateExecutionHost($persData));
	}

	return undef if (!$self->validateHdblcmKitVersion($persData));
	return 1;
}

sub validateHdblcmKitVersion {
	my ($self, $persData) = @_;
	my $config = $self->getConfiguration();
	my $detectedHdblcmVersion = $config->{pers_hdblcm_kitversion};
	my $persistedHdblcmVersion = $persData->{pers_hdblcm_kitversion};

	if (!defined $detectedHdblcmVersion) {
		$config->setErrorMessage("Hdblcm installation kit version is not defined.");
		return undef;
	}

	if ($persistedHdblcmVersion ne $detectedHdblcmVersion) {
		my $pendingErrorMessage = $self->_getPendingErrorMessage($gProductNameInstaller, $detectedHdblcmVersion, $persistedHdblcmVersion);

		my $mcm = $config->getMediumComponentManager();
		my $installerComponent = (defined $mcm) ? $mcm->getComponentByKeyName($gKeynameInstaller) : undef;
		if (defined $installerComponent && !$installerComponent->hasPersistenceFile($config)) {
			$pendingErrorMessage = $self->_getHdblcmKitVersionErrorMessage($detectedHdblcmVersion, $persistedHdblcmVersion);
		}

		my $msg = $config->setErrorMessage($pendingErrorMessage);
		$msg->getSubMsgLst()->addError("$gProductNameEngine installation kit '$persistedHdblcmVersion' required.");

		if (!$config->getIgnore ('check_pending_upgrade') && !$config->getIgnore ('check_version')) {
			return undef;
		}

		$config->ResetError ();
		$config->getMsgLst()->addMessage ("Ignoring error due to command line switch '--ignore'");
	}

	return 1;
}

sub validateDetectedServerKitVersion {
	my ($self) = @_;
	my $config = $self->getConfiguration();
	my $mcm = $config->getMediumComponentManager();

	if (!defined $mcm) {
		return undef;
	}

	return 1 if (!$mcm->isHDBServerComponentSelected());

	my $detectedServerCmp = $mcm->getComponentByKeyName($gKeynameEngine);
	if (!$detectedServerCmp->hasPersistenceFile($config)) {
		return 1;
	}

	my $xml = $detectedServerCmp->getPersistenceXMLObject($config);
	if (!defined $xml) {
		return 1;
	}

	my $persistedServerKitVersion = $xml->getTargetVersionString();
	if (!defined $persistedServerKitVersion) {
		$config->setErrorMessage("$gProductNameEngine server installation kit version is not persisted.");
		return undef;
	}

	my $detectedServerVersion = $detectedServerCmp->getVersion();
	my $persistedVersion = new SDB::Install::Version(split('\.', $persistedServerKitVersion));
	my $detectedVersion  = new SDB::Install::Version(split('\.', $detectedServerVersion));
	return ( ! $self->_validateDetectedServerKitVersionPerAction($persistedVersion, $detectedVersion) ) ? undef : 1;
}

sub _getPendingErrorMessage {
	my ($self, $detectedComponentName, $detectedComponentVersion, $persistedComponentVersion) = @_;
	my $configuration = $self->getConfiguration();
	my $action = $configuration->isUpdate() ? 'update' : 'installation';
	my $msg = "Pending $action of component $detectedComponentName";

	if (defined $persistedComponentVersion) {
		$msg .= " to version '$persistedComponentVersion'";
	}
	$msg .= " cannot be resumed";
	if (defined $detectedComponentVersion) {
		$msg .= " with version '$detectedComponentVersion'";
	}
	return $msg;
}

sub _getHdblcmKitVersionErrorMessage {
	my ($self, $detectedComponentVersion, $persistedComponentVersion) = @_;
	my $configuration = $self->getConfiguration();
	my $action = $configuration->isUpdate() ? 'update' : 'installation';
	my $msg = "Pending $action started with $gProductNameInstaller";

	if (defined $persistedComponentVersion) {
		$msg .= " version '$persistedComponentVersion'";
	}
	$msg .= " cannot be resumed";
	if (defined $detectedComponentVersion) {
		$msg .= " with version '$detectedComponentVersion'";
	}
	return $msg;
}

sub validateExecutionHost {
	my ($self, $persData) = @_;

	my $config = $self->getConfiguration();
	my $localHostname = $self->_getLocalHostname();
	my $persistentHostname = $persData->{pers_hostname};

	if ($persistentHostname ne $localHostname) {
		my $msg = $config->setErrorMessage("hdblcm tool is started from wrong host.");
		$msg->getSubMsgLst()->addError("Start hdblcm tool from host '$persistentHostname'.");

		if (!$config->getIgnore ('check_pending_upgrade') && !$config->getIgnore ('check_resume_hostname')){
			return undef;
		}

		$config->ResetError ();
		$config->getMsgLst()->addMessage ("Ignoring error due to command line switch '--ignore'");
	}

	return 1;
}

sub _copyMapParam {
	my ($self, $hdbupdConfig, $id) = @_;

	my $hdblcmConfig = $self->getConfiguration();
	my $hdbupdParam = $hdbupdConfig->{params}->{$id};
	my $hdbupdValue = $hdbupdParam->{value};
	my $param = $hdblcmConfig->{params}->{$id};

	$param->{origin_values} = $hdbupdParam->{origin_values};

	foreach my $key (keys (%{$hdbupdValue})) {
		if ($param->{shouldUseSetterOnDeserialization} ) {
			$hdblcmConfig->setMapValueItem($id, $key, $hdbupdValue->{$key});
		}
		$param->{value}->{$key} = $hdbupdValue->{$key};
		$hdblcmConfig->notifyParameterListeners($id, $hdbupdValue->{$key}, $key);
	}
}

sub _copyNotMapParam {
	my ($self, $hdbupdConfig, $id) = @_;

	my $hdblcmConfig = $self->getConfiguration();
	my $hdbupdParam = $hdbupdConfig->{params}->{$id};
	my $hdbupdValue = $hdbupdParam->{value};
	my $param = $hdblcmConfig->{params}->{$id};

	if ($param->{shouldUseSetterOnDeserialization}) {
		$hdblcmConfig->setValue($id, $hdbupdValue);
	} else {
		$param->{value} = $hdbupdValue;
		$hdblcmConfig->notifyParameterListeners($id, $hdbupdValue);
	}
}

sub registerInitialParamsToPersist {
	my ($self) = @_;

	my $config = $self->getConfiguration();
	my $installer = LCM::Installer->new();
	$config->{pers_hdblcm_kitversion} = $installer->GetVersion();

	my $action = $config->getAction();
	if (defined $action) {
		$config->{pers_action} = $action;
	}
}

1;
