package LCM::Configuration::ValueChangeListeners::XSUserIsolationBase;

use strict;
use SDB::Install::User;
use LCM::HostsParser qw(IsLocalHost);

sub new {
    my ($class, $parameterId) = @_;
    my $self = { _parameterId => $parameterId };
    return bless($self, $class);
}

sub _getHostsList {
    my ($self, $configuration) = @_;
    ...
}

sub onValueChange {
    my ($self, undef, $configuration) = @_;
    my $parameterId = $self->{_parameterId};
    my $additionalXsHosts = $self->_getHostsList($configuration);
    my $userConfig = $configuration->getSAPSystem()->getUserConfig();
    my $keys = ['xs_space_user_sap', 'xs_space_user_prod'];
    my $remoteUIDs = $configuration->{remoteUids} || {};
    my $sapsysGid = getgrnam('sapsys');

    $configuration->clearParameterWarnings($parameterId);

    for my $key (@{$keys}){
        my @hostsWithExistingUser = ();
        my ($targetUserName, $targetUid) = ($userConfig->{$key}, $userConfig->{"${key}_id"});
        my $errorMessagePrefix = "Checking XS Advanced OS user '$targetUserName' failed";
        my $canBeInSapsys = ($key ne 'xs_space_user_prod');

        next if(!defined($targetUserName) || !defined($targetUid));

        for my $targetHost (@{$additionalXsHosts}){
            if(IsLocalHost($targetHost, $configuration)){
                my $user = new SDB::Install::User($targetUserName);
                my $existingUserName = getpwuid($targetUid);
                my $isExistingUser = $user->exists();

                if(defined($existingUserName) && $existingUserName ne $targetUserName){
                    $configuration->getErrMsgLst()->AddError("$errorMessagePrefix. User ID '$targetUid' is already in use by user '$existingUserName'");
                    return 0;
                }
                if($isExistingUser && $user->id() ne $targetUid){
                    my $actualUid = $user->id();
                    $configuration->getErrMsgLst()->AddError("$errorMessagePrefix. Existing user has wrong ID '$actualUid' (expected '$targetUid')");
                    return 0;
                }
                if($isExistingUser && !$canBeInSapsys && $user->gid() eq $sapsysGid){
                    $configuration->getErrMsgLst()->AddError("$errorMessagePrefix. Existing user must not belong to group 'sapsys'");
                    return 0;
                }
                if($isExistingUser){
                    push(@hostsWithExistingUser, $targetHost);
                }
                next;
            }

            for my $userId (keys(%{$remoteUIDs})){
                my $userData = $remoteUIDs->{$userId};
                for my $entry (@{$userData}){
                    my ($host, $remoteUser, $group) = @{$entry};
                    my $isInSapsys = defined($sapsysGid) && $sapsysGid == $group;
                    my $isSameUserName = $targetUserName eq $remoteUser;

                    next if($host ne $targetHost);

                    if($isSameUserName && $targetUid != $userId){
                        $configuration->getErrMsgLst()->AddError("$errorMessagePrefix. User '$targetUserName' has wrong ID '$userId' on host '$host' (expected '$targetUid')");
                        return 0;
                    }
                    if($isSameUserName && $isInSapsys && !$canBeInSapsys){
                        $configuration->getErrMsgLst()->AddError("$errorMessagePrefix. User '$targetUserName' belongs to group 'sapsys' on host '$host'");
                        return 0;
                    }
                    if(!$isSameUserName && $targetUid == $userId) {
                        $configuration->getErrMsgLst()->AddError("$errorMessagePrefix. User ID '$targetUid' is already in use by user '$remoteUser' on host '$host'");
                        return 0;
                    }
                    if($isSameUserName){
                        push(@hostsWithExistingUser, $targetHost);
                    }
                }
            }
        }
        if(scalar(@hostsWithExistingUser) > 0){
            my $prefix = sprintf('User \'%s\' already exists on host(s) %s.', $targetUserName, join(', ', @hostsWithExistingUser));
            my $middle = 'No changes to its attributes and configurations will be made. Verify that the user is correct.';
            my $suffix = sprintf('Refer to SAP Note 2243156 for information how to properly configure user \'%s\'.', $targetUserName);
            $configuration->addParameterWarning($parameterId, sprintf('%s %s %s', $prefix, $middle, $suffix));
        }
    }
    return 1;
}

1;