package LCM::FileUtils;

use strict;
use SDB::Install::SysVars;
use Cwd qw (realpath);
use Fcntl ':mode';
use File::Spec;
use File::Basename;
use File::stat;

our @ISA = qw (Exporter);
our @EXPORT = qw (	MyRealPath
					isEmptyDirectory
					createDirectory
					listDirectory
					readFile
					findInFile
					writeToFile
					deleteFile
					changeFileOwnership
					changeFilePermissions
					GetPermissions
					hasFileExecutePermissions);

sub myRealPathWin {
    my $path = $_[0] ? Cwd::realpath( $_[0] ) : "";
    $path =~ s/\//\\/g;
    return $path;
}

sub MyRealPath;
*MyRealPath = $isWin ? \&myRealPathWin : \&Cwd::realpath;

sub isEmptyDirectory {
	my ($absoluteDirPath, $errorMessageList) = @_;
	my $dirHandle = undef;
	
	unless(-d $absoluteDirPath){
		_addErrorMessage("Invalid value '$absoluteDirPath'. Path '$absoluteDirPath' does not point to a directory", $errorMessageList);
		return undef;
	}
	
	unless(opendir($dirHandle, $absoluteDirPath)){
		_addErrorMessage("Failed to open directory '$absoluteDirPath' for reading : " . $!, $errorMessageList);
		return undef;
	}
	
	my @fileList = readdir($dirHandle);
	closedir($dirHandle);
	return scalar(@fileList) == 2 ? 1 : 0;	
}

# $fileMode must be an octal number (for example 0755) but NOT a string starting with zero
# does not modify properties of existing directories along the path
sub createDirectory {
	my ($absoluteDirPath, $fileMode, $uid, $gid, $isRecursive, $errorMessageList) = @_;

	my $parentDir = dirname($absoluteDirPath);
	if($isRecursive && ! -d $parentDir){
		return undef unless(createDirectory($parentDir, $fileMode, $uid, $gid, $isRecursive, $errorMessageList));
	}

	my $wasCreated = mkdir($absoluteDirPath);
	my $mkdirErrorMessage = $!;
	if (! -d $absoluteDirPath){
		_addErrorMessage("Failed to create directory '$absoluteDirPath' : ${mkdirErrorMessage}", $errorMessageList);
		return 0;
	}
	if ( !$wasCreated || !chmod($fileMode, $absoluteDirPath) ){
		_addErrorMessage("Failed to change permissions of directory '$absoluteDirPath'", $errorMessageList);
		return 0;
	}
	if ( !$wasCreated || !chown($uid, $gid, $absoluteDirPath) ){
		_addErrorMessage("Failed to change ownership of directory '$absoluteDirPath'", $errorMessageList);
		return 0;
	}

	return 1;
}

sub listDirectory {
	my ($directory, $errorMessageList) = @_;
	my $dirHandle = undef;
	if(!opendir($dirHandle, $directory)){
		_addErrorMessage("Failed to open directory '$directory': " . $!, $errorMessageList);
		return undef;
	}
	my @fileList = readdir($dirHandle);
	closedir($dirHandle);
	return \@fileList;
}

sub readFile {
	my ($filePath, $errorMessageList) = @_;
	my $fileHandle = undef;
	if(!open($fileHandle, '<',  $filePath)){
		_addErrorMessage("Failed to open file '$filePath' for reading: " . $!, $errorMessageList);
		return undef;
	}
	my @lines = <$fileHandle>;
	close($fileHandle);
	return \@lines;
}

sub findInFile {
	my ($filePath, $pattern, $errorMessageList) = @_;
	my $lines = readFile($filePath, $errorMessageList);
	if (!defined $lines) {
		_addErrorMessage("Failed to read file '$filePath': " . $!, $errorMessageList);
		return undef;
	}
	my $match = undef;
	for my $line (@$lines) {
		$match = $line if ($line =~ /$pattern/);
		last if (defined $match);
	}
	return $match;
}

sub deleteFile {
	my ($path, $errorMessageList) = @_;
    if(!unlink($path)){
    	_addErrorMessage("Deleting file '$path' failed: $!", $errorMessageList);
        return 0;
    }
    return 1;
}

sub changeFileOwnership {
	my ($filePath, $uid, $gid, $errorMessageList) = @_;

	if (!defined $uid) {
		$uid = -1;
	}

	if (!defined $gid) {
		$gid = -1;
	}

	if (!chown($uid, $gid, $filePath)) {
		_addErrorMessage("Could not chown '$filePath' to uid=$uid and gid=$gid: $!", $errorMessageList);
		return undef;
	}

	return 1;
}

# $fileMode must be an octal numer (for example 0755), but NOT a string
sub changeFilePermissions {
	my ($filePath, $fileMode, $errorMessageList) = @_;

	if (!defined $fileMode) {
		return undef;
	}

	if (!chmod($fileMode, $filePath)) {
		my $errMsg = sprintf("Could not chmod '%s' to mode '%o': %s", $filePath, $fileMode, $!);
		_addErrorMessage($errMsg, $errorMessageList);
		return undef;
	}

	return 1;
}

sub writeToFile {
	my ($filePath, $content, $mode, $uid, $gid, $errorMessageList) = @_;
	my $fileHandle = undef;

	if(!open($fileHandle, '>',  $filePath)){
		_addErrorMessage("Failed to open file '$filePath' for writing: " . $!, $errorMessageList);
		return undef;
	}
	if(! print $fileHandle $content) {
		_addErrorMessage("Cannot write into file '$filePath': " . $!, $errorMessageList);
		return undef;
	}
	if(! close($fileHandle)) {
		_addErrorMessage("Cannot close created file '$filePath': " . $!, $errorMessageList);
		return undef;
	}
	if(! changeFileOwnership($filePath, $uid, $gid, $errorMessageList)) {
		return undef;
	}
	if(! changeFilePermissions($filePath, $mode, $errorMessageList)) {
		return undef;
	}
	return 1;
}

sub _addErrorMessage {
	my ($message, $errorMessageList) = @_;
	if(defined($errorMessageList)){
		$errorMessageList->addError($message);
	}
}

# Format: 777
sub GetPermissions {
	my ($path) = @_;
	my $file = File::stat::stat($path);
	return undef if (!$file);
	my $mode = $file->mode();
	return sprintf "%03o", S_IMODE($mode);
}

sub hasFileExecutePermissions {
    my ($filePath, $uid, $gid) = @_;
    return 0 if(!$filePath);
    return 1 if(defined $uid && $uid == 0);
    my $file = File::stat::stat($filePath);
    return 0 if(! defined $file);
    my $mode = $file->mode();
    my $isExecutable = 0;

    if(defined $uid && $file->uid() == $uid){
        $isExecutable = ($mode & S_IXUSR);
    } elsif (defined $gid && $file->gid() == $gid){
        $isExecutable = ($mode & S_IXGRP);
    } else {
        $isExecutable = ($mode & S_IXOTH);
    }

    return $isExecutable;
}

1;
