package LCM::Hosts::HostsDetector;

use strict;
use base qw(SDB::Install::Base);
use SDB::Install::SAPSystem qw (CollectSAPSystems);
use LCM::HostsUtility qw (IsHostPartOfSystem);
use SDB::Install::SAPInstance::TrexInstance;
use SDB::Install::Globals qw($gHostRoleWorker $gProductNameSystem GetHostRoleProperties $gProductName);
use SDB::Install::System qw (isSidadmin);
use experimental qw (smartmatch);
use SDB::Install::Globals qw($gHostRoleWorker);
use LCM::App::AddHostsUtils;

our @EXPORT_OK = qw( createHostsDetector );

sub new {
	my ($class, $installer, $msgLstContext, $instance) = @_;
	my $self = $class->SUPER::new ();

	$self->_setInstaller($installer);
	$self->_setInstance($instance);
	$self->setMsgLstContext($msgLstContext);

	if(!defined($self->_retrieveHostRolesFromIni($self->_getInstance()))){
		$self->_setHasParsingErrors(1);
	}
	my $numberOfHanaWorkers = $self->_getNumberOfHostsWithRole($gHostRoleWorker);
	if ($numberOfHanaWorkers == 0) {
# At least one hana worker should exist in the landscape, otherwise it's an error
		$self->_setHasParsingErrors(1);
	}

	return $self;
}

sub hasParsingErrors {
	my ($self) = @_;
	return $self->{_hasParsingErrors};
}

sub getHostsWithRoles {
	my($self, $roles) = @_;
	my $systemHosts = $self->_getAllSystemHosts();
	my $hostsWithRoles = [];

	for my $host (@{$systemHosts}) {
		my $result = grep { $self->isHostInRole($host, $_) } @{$roles};
		push(@{$hostsWithRoles}, $host) if($result > 0);
	}

	return $hostsWithRoles; 
}

sub getHostRoles {
	my ($self, $hostName) = @_;
	return $self->{_hostRolesMap}->{$hostName} // [];
}

sub getHostsFromUserInput {
	my ($self, $hostsForRemoveUserInputString, $doLog) = @_;
	my $hostMap = $self->getHostRolesMapFromUserInput($hostsForRemoveUserInputString, $doLog);
	return [ keys(%{$hostMap}) ];
}

sub getHostRolesMapFromUserInput {
	my ($self, $hostsInputString, $doLog) = @_;
	my @hostNames = split (',', $hostsInputString);
	my $systemHosts = $self->_getAllSystemHosts();
	my @invalidHostNames = grep {! ($_ ~~ @{$systemHosts}) } @hostNames;

	if (scalar(@invalidHostNames) > 0) {
		if($doLog) {
			for my $invalidHostName (@invalidHostNames) {
				$self->getErrMsgLst()->addWarning("User provided host value '$invalidHostName' is not part of the $gProductNameSystem.");
			}
		}
		return {};
	}

	my %hostRolesMap = map { $_ => $self->getHostRoles($_) } @hostNames;
	return \%hostRolesMap;
}

sub userInputContainsLocalhost {
	my ($self, $userInput) = @_;
	my $selectedHosts = $self->getHostsFromUserInput($userInput);

	return $self->_getLocalHostName() ~~ @{$selectedHosts} ? 1 : 0;
}

sub userInputContainsRemoteHosts {
	my ($self, $userInput) = @_;
	my $selectedRemoteHosts = $self->getSelectedRemoteHosts($userInput);

	return scalar(@{$selectedRemoteHosts}) > 0 ? 1 : 0;
}

sub isComponentHost{
	my($self, $host, $component) = @_;
	my $validComponentRoles = $component->getHostRoles() // [];

	for my $componentSpecificRole (@{$validComponentRoles}) {
		return 1 if ($self->isHostInRole($host, $componentSpecificRole));
	}
	return 0;
}

sub getSelectedRemoteHosts {
	my ($self, $userInput) = @_;
	my $hostsMap = $self->getHostRolesMapFromUserInput($userInput);

	delete ($hostsMap->{$self->_getLocalHostName()});

	return [ keys(%{$hostsMap}) ];
}

sub hasStandByRole{
    my($self,$host) = @_;	
    my $hostRoles = $self->getHostRoles($host);
    my $roleProperties = GetHostRoleProperties();
    for my $role (@{$hostRoles}) {
       if($roleProperties->{$role}->{isStandby}){
        return 1;
       }
    }
    return 0;
}

sub  areAllWorkersInUserInput{
    my ($self,$inputString) = @_;
    my $numberOfHanaWorkers = $self->_getNumberOfHostsWithRole($gHostRoleWorker);
    my $hostsForRemoval = $self->getHostsFromUserInput($inputString, 1);
    my $numberOfWorkersForRemoval = scalar( grep { $self->isHostInRole($_, $gHostRoleWorker) } @{$hostsForRemoval} );

    if ($self->hasParsingErrors()) {
        $self->getMsgLst()->addWarning("Could not retrieve host roles from file 'nameserver.ini'.");
    return undef;
    }
    return $numberOfWorkersForRemoval == $numberOfHanaWorkers; 
}
# Only private subs bellow

sub _getInstance {
	my ($self) = @_;

	return $self->{_instance} if ($self->{_instance});

	my $installer = $self->_getInstaller();
	my $sid = $installer->getSid();

	if (IsHostPartOfSystem()) {
		my $allSystems = CollectSAPSystems( undef, 1 );
		my $sapSys = (defined $sid) ? $allSystems->{$sid} : undef;
		$self->{_instance} = $sapSys->getNewDBInstances()->[0];
	} else {
		my $sapSidDir = substr($installer->getRuntimeDir(), 0, -18); # trim away hdblcm/instruntime
		my $instanceDir = LCM::HostsUtility::GetInstanceDirOnHostNotPartOfSystem(undef, $sapSidDir);

		if ( defined $instanceDir ) {
			my $instance = new SDB::Install::SAPInstance::TrexInstance( $instanceDir, $sid );
			$self->{_instance} = $instance;
		}
	}

	return $self->{_instance};
}

sub _setInstance {
	$_[0]->{_instance} = $_[1];
}

sub _getLocalHostName {
	my ($self) = @_;
	return $self->_getInstance()->get_host();
}

sub _getAllSystemHosts {
	my ($self) = @_;
	return $self->_getInstance()->get_allhosts();
}

sub _setHostRoles {
	my ($self, $hostName, $rolesArray) = @_;
	$self->{_hostRolesMap}->{$hostName} = $rolesArray;
}

sub _getNumberOfHostsWithRole {
	my ($self, $role) = @_;
	my $hostsArray = $self->_getAllSystemHosts();

	return scalar( grep { $self->isHostInRole($_, $role) } @{$hostsArray} );
}

sub isHostInRole {
	my ($self, $host, $role) = @_;
	my $hostRoles = $self->getHostRoles($host);

	return (defined($hostRoles) && ($role ~~ @{$hostRoles})) ? 1 : 0;
}

sub filterExistingLocalRoles {
    my ($self, $addLocalRolesUserInputString) = @_;
    return undef if ! defined $addLocalRolesUserInputString;

    my $addLocalRolesString = undef;
    my $localHost = $self->_getLocalHostName();
    $addLocalRolesString = join(',',
        grep { ! $self->isHostInRole($localHost, $_) } split(',', $addLocalRolesUserInputString) );
    return $addLocalRolesString;
}

sub filterExistingRoles {
    my ($self, $addRolesMap) = @_;

    return {} if ! defined $addRolesMap || ! %$addRolesMap;

    my $rolesToAddMap = {};
    for my $host (keys %$addRolesMap){
        my @rolesToAdd = ();
        for my $role (split(',', $addRolesMap->{$host})){
            next if $self->isHostInRole($host, $role);
            push (@rolesToAdd, $role);
        }
        $rolesToAddMap->{$host} = join(',', @rolesToAdd) if scalar(@rolesToAdd) > 0;
    }
    return $rolesToAddMap;
}

sub filterExistingHostsCmd {
    my ($self, $addHostsList, $otherExistingHostsList) = @_;

    return undef if ! defined $addHostsList || scalar($addHostsList) == 0;

    my $otherExistingHosts = undef;
    if (defined $otherExistingHostsList) {
       my $hostsParser = LCM::HostsParser->new();
       $hostsParser->parse(join(',', @$otherExistingHostsList));
       $otherExistingHosts = $hostsParser->getHosts();
    }

    my $addfilteredHostsList = ();
    my $existingHosts = $self->_getInstance()->get_allhosts();
    my $hostsParser = LCM::HostsParser->new();
    $hostsParser->parse(join(',', @$addHostsList));
    my $userInputHosts = $hostsParser->getHosts() || [];
    for my $host (@{$userInputHosts}){
        my $isPendingHost = LCM::App::AddHostsUtils::isPendingAddHost($self->_getInstance(), $host);
        next if $isPendingHost;
        next if ! ($host ~~ @$existingHosts);
        next if defined $otherExistingHosts && ! ($host ~~ @$otherExistingHosts);
        $hostsParser->setSkipHost($host, 1);
    }
    return $hostsParser->generateAddHostsCmd();
}

sub filterExistingHosts {
    my ($self, $addHostsList, $otherExistingHostsList) = @_;
    my $addfilteredHostsCmd = $self->filterExistingHostsCmd($addHostsList, $otherExistingHostsList);
    return defined $addfilteredHostsCmd ? [ split(',', $addfilteredHostsCmd) ] : [];
}

sub filterExistingRolesInHostsAndRoles {
    my ($self, $addHostsList, $addRolesMap) = @_;

    if ( (! defined $addHostsList || scalar($addHostsList) == 0) &&
        (! defined $addRolesMap || ! %$addRolesMap) ) {
           return ( $addHostsList, $addRolesMap );
    }

    my $rolesToAddMap = $self->filterExistingRoles($addRolesMap);

    if(defined $addHostsList && scalar($addHostsList) > 0) {
        my $addfilteredHostsList = $self->filterExistingHosts($addHostsList);

        my $existingHosts = $self->_getInstance()->get_allhosts();
        my $hostsParser = new LCM::HostsParser();
        $hostsParser->parse(join(',', @$addHostsList));
        my $userInputHosts = $hostsParser->getHosts() || [];
        my @existingHostsWithMissingRoles = grep { ($_ ~~ @$existingHosts) } @$userInputHosts;
        return $rolesToAddMap if scalar(@existingHostsWithMissingRoles) == 0;
        
        for my $host (@existingHostsWithMissingRoles){
            my @rolesToAdd = ();
            my $hostRoles = $hostsParser->getRoles($host);
            for my $role (@{$hostRoles}){
                next if $self->isHostInRole($host, $role);
                push (@rolesToAdd, $role);
            }
            if ( scalar(@rolesToAdd) > 0){
               if ( ! exists $rolesToAddMap->{$host} ){
                   $rolesToAddMap->{$host} = join(',', @rolesToAdd);
               } else {
                   my @roles = split(',', $rolesToAddMap->{$host});
                   my $mergedRoles = [@roles, grep { not ($_ ~~ @roles) } @rolesToAdd];
                   $rolesToAddMap->{$host} = join(',', @$mergedRoles);
               }
            }
        }
    }

    return $rolesToAddMap;
}

sub _retrieveHostRolesFromIni {
	my($self, $instance) = @_;
	my $hostRolesMap = $instance->getHostRolesInfo();

	return undef if(!defined($hostRolesMap));

	for my $hostName (keys %{$hostRolesMap}) {
		my $rolesArray = [ grep { length($_) > 0 } split(/\s+/, $hostRolesMap->{$hostName}) ];
		$self->_setHostRoles($hostName, $rolesArray);
	}
	return 1;
}

sub _setInstaller {
	my ($self, $installer) = @_;
	$self->{_installer} = $installer;
}

sub _getInstaller {
	my ($self) = @_;
	return $self->{_installer};
}

sub _setHasParsingErrors {
	my ($self, $hasErrors) = @_;
	$self->{_hasParsingErrors} = $hasErrors;
}

1;