package LCM::HostsParser;

use strict;

use SDB::Install::System qw( $ipv4_regex is_local_address );
use SDB::Install::Configuration::HdbModify;
use SAPDB::Install::Hostname;
use SDB::Install::NewDBUser;
use SDB::Install::System;
use SDB::Install::Configuration::AnyMultiHostConfig qw($validHostRoles $hostRoleDefaultValue);
use SDB::Install::Globals qw($gHostRoleWorker $gHostRoleStandby);

use base qw( SDB::Install::Base Exporter );

our @EXPORT_OK = qw( ExtractLocalAndRemoteHosts ParseHostStringToMap ParseHostStringToList IsLocalHost appendDefaultValuesToAddHostsCmd );

sub new{
    my $self  = shift->SUPER::new (@_);
    my (%options) = @_;

    $self->setIgnoreDefaultRoleValue($options{ignoreDefaultRoleValue});
	$self->{_hostsCmdStr} = '';
	$self->{_hostsMap} = {};

	return $self;
}

sub setIgnoreDefaultRoleValue {
	my ($self, $value) = @_;
	$self->{ignoreDefaultRoleValue} = $value;
}

sub getIgnoreDefaultRoleValue {
	my ($self) = @_;
	return $self->{ignoreDefaultRoleValue};
}

sub appendDefaultValuesToAddHostsCmd {
	my ($self, $addHostsCmdStr) = @_;

	my ($rc, $parserErrMsgLst) = $self->parse($addHostsCmdStr);
	if (!defined $rc) {
		return (undef, $parserErrMsgLst);
	}

	my $addHostsCmd = $self->generateAddHostsCmd();
	if (!defined $addHostsCmd) {
		return (undef,undef);
	}

	return ($addHostsCmd, undef);
}

sub parse {
	my ($self, $addHostsCmdStr) = @_;

	if (!defined $addHostsCmdStr) {
		return (undef, undef);
	}

	$self->setHostsCmdStr($addHostsCmdStr);

	my $ignoreDefaultRoleValue = $self->getIgnoreDefaultRoleValue();
	my ($addHostsMap, $parserErrMsgLst) = ParseHostStringToMap($addHostsCmdStr, $ignoreDefaultRoleValue);
	if (!defined $addHostsMap) {
		return (undef, $parserErrMsgLst);
	}

	$self->setHostsMap($addHostsMap);

	return (1, undef);
}

sub generateAddHostsCmd {
	my ($self) = @_;

	my $additionalHostnames = $self->getHosts();
	if (!defined $additionalHostnames) {
		return undef;
	}

	my $addHostsCmd = '';
	my $i = 0;
	for my $currHost (@{$additionalHostnames}) {
		next if (exists $self->{_hostsMap}->{$currHost}->{skip} && $self->{_hostsMap}->{$currHost}->{skip});

		if ($i > 0) {
			$addHostsCmd .= ',';
		}
		$addHostsCmd .= $currHost;

		my $roles = $self->getRoles($currHost);
		if (defined $roles) {
			for my $role (@{$roles}) {
				$addHostsCmd .= ":role=" . $role;
			}
		}

		my $group = $self->getGroup($currHost);
		if (defined $group) {
			$addHostsCmd .= ":group=" . $group;
		}

		my $sp = $self->getStoragePartition($currHost);
		if (defined $sp) {
			$addHostsCmd .= ":storage_partition=" . $sp;
		}

		my $wg = $self->getWorkerGroups($currHost);
		if (defined($wg)) {
			my @workerGroups = split(',', $wg);
			$addHostsCmd .= ":workergroup=" . join(':workergroup=', @workerGroups);
		}

		$i++;
	}

	return $addHostsCmd;
}

sub getHostsCmdStr {
	return $_[0]->{_hostsCmdStr};
}

sub setHostsCmdStr {
	$_[0]->{_hostsCmdStr} = $_[1];
}

sub getHostsMap {
	return $_[0]->{_hostsMap};
}

sub setHostsMap {
	$_[0]->{_hostsMap} = $_[1];
}

sub getHosts {
	my ($self) = @_;

	my $hostsMap = $self->{_hostsMap};

	if (!defined $hostsMap || (ref($hostsMap) !~ /HASH/)) {
		return undef;
	}

	my @hosts = keys %$hostsMap;

	if (scalar @hosts == 0) {
		return undef;
	}

	return \@hosts;
}

sub setSkipHost {
	my ($self, $host, $skip) = @_;	
	$self->{_hostsMap}->{$host}->{skip} = $skip ? 1 : 0;
}

sub getRoles {
    my ($self, $wantedHost) = @_;

	my $roles = $self->{_hostsMap}->{$wantedHost}->{roles};

	if (!defined $roles && !$self->getIgnoreDefaultRoleValue()) {
			$roles = [$hostRoleDefaultValue];
	}

    return $roles;
}

sub getGroup {
    my ($self, $wantedHost) = @_;

	my $group = $self->{_hostsMap}->{$wantedHost}->{group};

	if (!defined $group) {
		my $roles = $self->getRoles($wantedHost);
		if (!defined $roles || scalar @{$roles} == 1) {
			$group = $validHostRoles->{$roles->[0]}->{defaultGroup};
		} else {
			$group = $validHostRoles->{worker}->{defaultGroup};
		}
	}

    return $group;
}

sub getStoragePartition {
    my ($self, $wantedHostname) = @_;

    my $selectedHost = $self->{_hostsMap}->{$wantedHostname};

    return (defined $selectedHost)
           ? $selectedHost->{storage_partition}
           : undef;
}

sub getWorkerGroups {
    my ($self, $wantedHostname) = @_;
    my $selectedHost = $self->{_hostsMap}->{$wantedHostname} || {};
    my $workerGroups = $selectedHost->{workergroup} || [];
    my $workerGroupsString = join(',', @{$workerGroups});

    return length($workerGroupsString) > 0 ? $workerGroupsString : undef;
}

sub ParseHostStringToList {
	my ($hostsStr) = @_;

	require SDB::Install::Configuration::AddHostsParser;
    my $parser = new SDB::Install::Configuration::AddHostsParser ();

    if (!defined $parser->parse ($hostsStr)){
        return (undef, $parser->getErrMsgLst());
    }

    my @hostsList;
    foreach my $host (@{$parser->getHosts()}) {
        if (!$host){
            next;
        }

		push @hostsList, $host;
    }

	return \@hostsList;
}

sub ParseHostStringToMap {
	my ($addHostsStr, $ignoreDefaultRoleValue) = @_;

	require SDB::Install::Configuration::AddHostsParser;
    my $parser = new SDB::Install::Configuration::AddHostsParser ();

    if (!defined $parser->parse ($addHostsStr, undef, undef, $ignoreDefaultRoleValue)){
        return (undef, $parser->getErrMsgLst());
    }

    my %hostsMap;
    foreach my $host (@{$parser->getHosts()}) {
        if (!$host){
            next;
        }

		$hostsMap{$host} = {
			roles => $parser->getRoles($host),# Array ref
			group => $parser->getGroup($host),
			storage_partition => $parser->getStoragePartition($host),
			workergroup => $parser->getWorkerGroups ($host)
		};
    }

	return \%hostsMap;
}

sub ExtractLocalAndRemoteHosts {
	my ($addHostsInputString, $msglst) = @_;
	my $remoteHostsInputString = "";
	my $localHostInputString;

	foreach my $hostString (split (',', $addHostsInputString)){
		if (IsLocalHost($hostString, $msglst)){
			$localHostInputString = $hostString;
			next;
		}
		$remoteHostsInputString .= "$hostString,";
	}
	chop ($remoteHostsInputString);
	
	return ($localHostInputString, ( ($remoteHostsInputString eq "") ? undef : $remoteHostsInputString ));
}

sub IsLocalHost {
	my ($hostString, $msglst) = @_;

	my $hostname;
	($hostname) = split (':', $hostString);

	if ($hostname !~ /$ipv4_regex/){
		my $lcLocalHostname = lc (hostname());
		if (lc($hostname) =~ m/^$lcLocalHostname$/){
			return 1;
		}

		my $hostAddress = _getAddressFromHostname($hostname, $msglst);
		if (not defined $hostAddress){
			return undef;
		}
		$hostname = $hostAddress;
	}

	return _isLocalHost($hostname, $msglst);
}

sub HasLocalHost {
	my ($hostString, $msglst) = @_;

	my $hostArgsMap = ParseHostStringToMap($hostString);
	if (not defined $hostArgsMap){
		return undef;
	}

	my @hostnames = (keys %$hostArgsMap);
	for my $hostname (@hostnames) {
		if (IsLocalHost($hostname, $msglst)){
			return 1;
		}
	}

	return 0;
}

sub _getAddressFromHostname {
	my ($hostname, $msglst) = @_;

	my $nslookup = nslookup ($hostname, $msglst);
	if (not defined $nslookup){
		$msglst->AddWarning ("Error calling nslookup") if defined $msglst;
		return undef;
	}
	if (not defined $nslookup->{address}){
		$msglst->AddWarning ("Cannot resolve host name '$hostname': $nslookup->{msg}") if defined $msglst;
		return undef;
	}

	return $nslookup->{address};
}

sub _isLocalHost {
	my ($hostAddress, $msglst) = @_;

	my $rc = is_local_address ($hostAddress, $msglst);
	if (not defined $rc){
		$msglst->AddWarning ("Error checking ip address is local", $msglst) if defined $msglst;
		return undef;
	}

	if ($rc) {
		return 1;
	}
	return 0;
}

sub getLocalHanaHost{
	my ($self) = @_;
    my $localHanaHost;

    foreach my $host (@{$self->getHosts()}){
        next if(! IsLocalHost($host));
        if(grep {$_ =~ /$gHostRoleWorker|$gHostRoleStandby/} @{$self->getRoles($host)}){
            $localHanaHost = $host;
        }
    }
    return $localHanaHost;
}

1;
