package LCM::Landscape::DescriptionXML;

use strict;
use File::Spec;
use SDB::Install::User;
use SDB::Install::MsgLst;
use LCM::Landscape::PDXml;
use parent 'SDB::Install::Base';
use LCM::Landscape::ProductFactory;
use SDB::Install::SysVars qw($isWin);
use LCM::Landscape::Properties::CurrentStack;
use LCM::ComponentManager::SystemComponentManager;
use LCM::Landscape::Properties::LandscapeVariables;
use LCM::FileUtils qw(writeToFile readFile createDirectory);
use SDB::Common::Utils qw(createXMLParser createSysAdminUserName);
use LCM::Landscape::PDXmlDetector qw(getListOfComponentDirs);
use SDB::Install::Globals qw($gPlatform $gDirNameEs $gDirNameStreaming $gDirNameAccelerator $gDirNameXS2 $gKeynameClient $gKeynameStudio);

my $LANDSCAPE_DESCRIPTION_XML_TEMPLATE= '<?xml version="1.0" encoding="UTF-8"?><landscapeDescription xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:noNamespaceSchemaLocation="landscapeDescription.xsd"><version>1.09</version><includeVariables><path>landscapeVariables.properties</path></includeVariables><includeCurrentStack><path>currentStack.properties</path></includeCurrentStack><productSystem><sid>$(SID)</sid><host>$(centralserver)</host><productSystemLandscapeProduct></productSystemLandscapeProduct><applicationSystemInLandscape><databaseSystem><sid>$(SID)</sid><dBName>$(SID)</dBName><dBTypeForSAP>HDB</dBTypeForSAP><host>$(centralserver)</host><installationPath>/usr/sap/$(SID)</installationPath><installationNumber>$(installationNumberHDB)</installationNumber><installedProductOnApplicationSystem></installedProductOnApplicationSystem></databaseSystem></applicationSystemInLandscape></productSystem><computerSystemsInLandscape></computerSystemsInLandscape></landscapeDescription>';
my $LANDSCAPE_DESCRIPTION_XML_FILE = 'landscapeDescription.xml';
my $FILE_MODE = 0755;

sub new {
	require XML::LibXML;

	my ($class, $messageListContext, $sapSystem, $isExtended,$withIncludes) = @_;
	my $self = $class->SUPER::new();

	$self->setMsgLstContext($messageListContext) if(defined($messageListContext));

	my $manifestDir = $sapSystem->getManifestDir();
	my $landscapeVariables = new LCM::Landscape::Properties::LandscapeVariables($self->getMsgLstContext(), $sapSystem);
	my $componentManager = new LCM::ComponentManager::SystemComponentManager();
	my $installationPath = $landscapeVariables->getLandscapeInstallationPath();

	$componentManager->detectComponentsBySapmntSid($installationPath, $sapSystem->get_sid());
	$self->_setSapSystem($sapSystem);

	my $serverPdXml = new LCM::Landscape::PDXml($componentManager, $manifestDir, $self);
	my $xmlTree = $self->_buildLandscapeDescriptionXmlTree($landscapeVariables, $serverPdXml, $componentManager, $isExtended,$withIncludes);

	$self->_setXmlTree($xmlTree);
	return $self;
}

sub store {
	my ($self, $sid, $targetDirectory) = @_;
	my ($uid, $gid) = (undef, undef);
	my $errorList = new SDB::Install::MsgLst();
	my $filePath = File::Spec->catfile($targetDirectory, $LANDSCAPE_DESCRIPTION_XML_FILE);
	my $fileContent = $self->toString();

	if(!$isWin){
		my $sidadmUserName = createSysAdminUserName($sid);
		my $sidadmUser = new SDB::Install::User($sidadmUserName);
		($uid, $gid) = ($sidadmUser->id(), $sidadmUser->gid());
	}
	if(!-d $targetDirectory && !createDirectory($targetDirectory, $FILE_MODE, $uid, $gid, undef, $errorList)){
		$self->getErrMsgLst()->addError("Failed to create directory '$targetDirectory'", $errorList);
		return undef;
	}
	if(!writeToFile($filePath, $fileContent, $FILE_MODE, $uid, $gid, $errorList)){
		$self->getErrMsgLst()->addError("Failed to write file '$filePath' to the file system", $errorList);
		return undef;
	}
	return 1;
}

sub toString {
	my ($self, $replacePlaceholders) = @_;
	my $xmlTree = $self->_getXmlTree();
	my $xmlString = $xmlTree->toString(1);

	return $replacePlaceholders ? $self->_replacePlaceHolders($xmlString) : $xmlString;
}

sub _replacePlaceHolders {
	my ($self, $xmlString) = @_;
	my $sapSystem = $self->_getSapSystem();
	my $landscapeVariables = new LCM::Landscape::Properties::LandscapeVariables($self->getMsgLstContext(), $sapSystem);
	my $componentManager = new LCM::ComponentManager::SystemComponentManager();
	my $installationPath = $landscapeVariables->getLandscapeInstallationPath();

	$componentManager->detectComponentsBySapmntSid($installationPath, $sapSystem->get_sid());

	my $stackProperties = new LCM::Landscape::Properties::CurrentStack($componentManager, $self->getMsgLstContext(), $sapSystem);
	my $stackId = $stackProperties->getStackId();
	my $stackCaption = $stackProperties->getStackCaption();
	my $stackReleaseDate = $stackProperties->getStackReleaseDate();
	my $stackInstallDateTime = $stackProperties->getStackInstallDate();
	my $sid = $landscapeVariables->getLandscapeSid();
	my $installationNumber = $landscapeVariables->getLandscapeInstallationNumber();
	my $centralServer = $landscapeVariables->getLandscapeCentralServer();

	$xmlString =~ s/\$\(stackID\)/$stackId/g;
	$xmlString =~ s/\$\(stackCaption\)/$stackCaption/g;
	$xmlString =~ s/\$\(stackReleaseDate\)/$stackReleaseDate/g;
	$xmlString =~ s/\$\(stackInstallDateTime\)/$stackInstallDateTime/g;
	$xmlString =~ s/\$\(SID\)/$sid/g;
	$xmlString =~ s/\$\(installationNumberHDB\)/$installationNumber/g;
	$xmlString =~ s/\$\(centralserver\)/$centralServer/g;
	$xmlString =~ s/\$\(sapmntPath\)/$installationPath/g;

	return $xmlString;
}

sub _getSapSystem {
	my ($self) = @_;
	return $self->{_sapSystem};
}

sub _setSapSystem {
	my ($self, $sapSystem) = @_;
	$self->{_sapSystem} = $sapSystem;
}

sub _getXmlTree {
	my ($self) = @_;
	return $self->{_xmlTree};
}

sub _setXmlTree {
	my ($self, $xmlTree) = @_;
	$self->{_xmlTree} = $xmlTree;
}

sub _buildLandscapeDescriptionXmlTree {
	my ($self, $landscapeVariables, $serverPdXml, $componentManager, $isExtended,$withIncludes) = @_;
	my $xmlParser = createXMLParser();
	my $centralServer = $landscapeVariables->getLandscapeCentralServer();
	my $landscapeDescriptionTree = $xmlParser->parse_string($LANDSCAPE_DESCRIPTION_XML_TEMPLATE);
	my $installedProductVersionNode = $self->_buildMainInstalledProductVersionNode($serverPdXml);
	my $installedProductVersionNodes = $self->_buildInstalledProductVersionNodes($componentManager, $landscapeVariables, $isExtended);
	my $studioComponent = $componentManager->getComponentByKeyName($gKeynameStudio);
	my $clientComponent = $componentManager->getComponentByKeyName($gKeynameClient);
	my $studioNode = defined($studioComponent) ? $self->_buildHANAStudioNode($serverPdXml, $isExtended ? $studioComponent : undef) : undef;
	my $clientNode = defined($clientComponent) ? $self->_buildHANAClientNode($serverPdXml, $isExtended ? $clientComponent : undef) : undef;
	my $computerSystemNodes = $self->_buildComputerSystemNodes($centralServer);
	my $installedProductVersionNodeParent = $landscapeDescriptionTree->findnodes('/landscapeDescription/productSystem/productSystemLandscapeProduct')->get_node(1);
	my $installedProductVersionNodesParent = $landscapeDescriptionTree->findnodes('/landscapeDescription/productSystem/applicationSystemInLandscape/databaseSystem/installedProductOnApplicationSystem')->get_node(1);
	my $studioNodeParent = $landscapeDescriptionTree->findnodes('/landscapeDescription/productSystem/applicationSystemInLandscape')->get_node(1);
	my $clientNodeParent = $studioNodeParent;
	my $computerSystemNodesParent = $landscapeDescriptionTree->findnodes('/landscapeDescription/computerSystemsInLandscape')->get_node(1);

	$installedProductVersionNodeParent->appendChild($installedProductVersionNode);
	$installedProductVersionNodesParent->appendChild($_) for (@{$installedProductVersionNodes});
	$studioNodeParent->appendChild($studioNode) if(defined($studioNode));
	$clientNodeParent->appendChild($clientNode) if(defined($clientNode));
	$computerSystemNodesParent->appendChild($_) for (@{$computerSystemNodes});

	if($isExtended){
		my $landscapeDescriptionNode = $landscapeDescriptionTree->findnodes('/landscapeDescription')->get_node(1);
		my $includeVariablesNode = $landscapeDescriptionTree->findnodes('/landscapeDescription/includeVariables')->get_node(1);
		my $includeCurrentStackNode = $landscapeDescriptionTree->findnodes('/landscapeDescription/includeCurrentStack')->get_node(1);
		my $databaseSystemNode = $landscapeDescriptionTree->findnodes('/landscapeDescription/productSystem/applicationSystemInLandscape/databaseSystem')->get_node(1);
		my $installationNumberNode = $landscapeDescriptionTree->findnodes('/landscapeDescription/productSystem/applicationSystemInLandscape/databaseSystem/installationNumber')->get_node(1);
		my $systemNumber = $self->_getSystemNumber();
        if(!$withIncludes){
		  $landscapeDescriptionNode->removeChild($includeVariablesNode);
		  $landscapeDescriptionNode->removeChild($includeCurrentStackNode);
		  $databaseSystemNode->insertAfter($self->_createTextElement('systemNumber', $systemNumber), $installationNumberNode);
        }
	}
	return $landscapeDescriptionTree;
}

sub _getSystemNumber {
	my ($self) = @_;
	my $sapSystem = $self->_getSapSystem();
	my $trexInstance = $sapSystem->getNewDBInstances()->[0];
	my $metadataDir = $trexInstance->get_globalMetadataDir();
	my $installPropertiesFile = File::Spec->catfile($metadataDir, 'installation.properties');
	my $lines = readFile($installPropertiesFile, new SDB::Install::MsgLst()) || []; # Ignore errors

	for my $line (@{$lines}){
		if($line =~ /^\s*systemno\s*=\s*(.+)$/){
			my $systemNumber = $1;
			$systemNumber = s/\s//g;
			return $systemNumber;
		}
	}
	return '000000000000000000';
}

sub _buildComputerSystemNodes {
	my ($self, $centralServer) = @_;
	my $sapSystem = $self->_getSapSystem();
	my $trexInstance = $sapSystem->getNewDBInstances()->[0];
	my $computerSystemNodes = [];

	for my $hanaHost (@{$trexInstance->get_allhosts()}){
		my $hostName = ($hanaHost eq $centralServer) ? '$(centralserver)' : $hanaHost;
		my $computerSystemTag = XML::LibXML::Element->new('computerSystem');

		$computerSystemTag->appendChild($self->_createTextElement('host', $hostName));
		$computerSystemTag->appendChild($self->_createTextElement('sharedDirectory', '$(sapmntPath)/$(SID)/trans'));

		push(@{$computerSystemNodes}, $computerSystemTag);
	}
	return $computerSystemNodes;
}

sub _buildInstalledProductVersionNodes {
	my ($self, $systemComponentManager, $landscapeVariables, $isExtended) = @_;
	my $sapSystem = $self->_getSapSystem();
	my $sid = $sapSystem->get_sid();
	my $installationPath = $landscapeVariables->getLandscapeInstallationPath();
	my $pluginsDir = File::Spec->catdir($installationPath, $sid, 'exe', $gPlatform, 'hdb', 'plugins');
	my $newDBManifestDir = File::Spec->catfile($installationPath, $sid, 'exe', $gPlatform, 'hdb');
	my $componentDirectories = getListOfComponentDirs($pluginsDir, $self);

	for my $componentDir ($gDirNameEs, $gDirNameStreaming, $gDirNameAccelerator){
		my $candidateDir = File::Spec->catfile($installationPath, $sid, $componentDir);
		if(-r File::Spec->catfile($candidateDir, 'PD.XML')){
			push(@{$componentDirectories}, $candidateDir);
		}
	}
	unshift(@{$componentDirectories}, $newDBManifestDir);

	my $productFactory = new LCM::Landscape::ProductFactory($systemComponentManager, $componentDirectories, $self);
	my $detectedProducts = $productFactory->getProducts(); # Only non-empty products are returned
	my $nonEmptyProducts = [];

	for my $product (@{$detectedProducts}){
		my $name = $product->getName();
		my $version = $product->getVersion();
		$self->getMsgLst()->addMessage("Detected product information for product '$name $version'");
		push(@{$nonEmptyProducts}, $product->createXmlElement($isExtended));
	}
	return $nonEmptyProducts;
}

sub _buildMainInstalledProductVersionNode {
	my ($self, $serverPdXml) = @_;
	my $serverProduct = $serverPdXml->getProduct();
	my $installedProductVersionTag = XML::LibXML::Element->new('installedProductVersion');
	my $installedSPStackTag = XML::LibXML::Element->new('installedSPStack');

	$installedSPStackTag->appendChild($self->_createTextElement('identifyingNumber', '$(stackID)'));
	$installedSPStackTag->appendChild($self->_createTextElement('caption', '$(stackCaption)'));
	$installedSPStackTag->appendChild($self->_createTextElement('releaseDate', '$(stackReleaseDate)'));
	$installedSPStackTag->appendChild($self->_createTextElement('installDateTime', '$(stackInstallDateTime)'));
	$installedProductVersionTag->appendChild($self->_createTextElement('identifyingNumber', $serverProduct->getPPMSId()));
	$installedProductVersionTag->appendChild($self->_createTextElement('name', $serverProduct->getName()));
	$installedProductVersionTag->appendChild($self->_createTextElement('vendor', $serverProduct->getVendor()));
	$installedProductVersionTag->appendChild($self->_createTextElement('version', $serverProduct->getVersion()));
	$installedProductVersionTag->appendChild($self->_createTextElement('caption', $serverProduct->getCaption()));
	$installedProductVersionTag->appendChild($installedSPStackTag);

	return $installedProductVersionTag;
}

sub _buildHANAClientNode {
	my ($self, $serverPdXml, $clientComponent) = @_;
	my $sid = 'IMCC';
	my $technologyType = 'IMDBClient';
	my $installationPath = '$(sapmntPath)/$(SID)/hdbclient';
	my $caption = 'SAP HANA Client';
	my $instanceKey = '1';
	my $identifyingNumber = '73554900104900002962';

	return $self->_buildUnspecificStandaloneApplicationSystemNode($sid, $technologyType, $installationPath, $caption, $instanceKey, $identifyingNumber, $serverPdXml, $clientComponent);
}

sub _buildHANAStudioNode {
	my ($self, $serverPdXml, $studioComponent) = @_;
	my $sid = 'IMCS';
	my $technologyType = 'IMDBStudioRepository';
	my $installationPath = '$(sapmntPath)/$(SID)/hdbstudio_update';
	my $caption = 'SAP HANA Studio';
	my $instanceKey = '3';
	my $identifyingNumber = '73554900104900002963';

	return $self->_buildUnspecificStandaloneApplicationSystemNode($sid, $technologyType, $installationPath, $caption, $instanceKey, $identifyingNumber, $serverPdXml, $studioComponent);
}

sub _buildUnspecificStandaloneApplicationSystemNode {
	my ($self, $sid, $technologyType, $installationPath, $caption, $instanceKey, $identifyingNumber, $serverPdXml, $component) = @_;
	my $serverProduct = $serverPdXml->getProduct();
	my $unspecificSystemTag = XML::LibXML::Element->new('unspecificStandaloneApplicationSystem');
	my $installedProductOnSystemTag = XML::LibXML::Element->new('installedProductOnApplicationSystem');
	my $installedProductVersionTag = XML::LibXML::Element->new('installedProductVersion');
	my $collectedSoftwareFeaturesTag = XML::LibXML::Element->new('collectedSoftwareFeatures');
	my $installedSWFeatureTag = XML::LibXML::Element->new('installedSWFeature');

	$unspecificSystemTag->appendChild($self->_createTextElement('sid', $sid));
	$unspecificSystemTag->appendChild($self->_createTextElement('host', '$(centralserver)'));
	$unspecificSystemTag->appendChild($self->_createTextElement('technologyType', $technologyType));
	$unspecificSystemTag->appendChild($self->_createTextElement('installationPath', $installationPath));
	$installedProductVersionTag->appendChild($self->_createTextElement('identifyingNumber', $serverProduct->getPPMSId()));
	$installedProductVersionTag->appendChild($self->_createTextElement('name', $serverProduct->getName()));
	$installedProductVersionTag->appendChild($self->_createTextElement('vendor', $serverProduct->getVendor()));
	$installedProductVersionTag->appendChild($self->_createTextElement('version', $serverProduct->getVersion()));
	$installedProductVersionTag->appendChild($self->_createTextElement('caption', $serverProduct->getCaption()));
	$installedSWFeatureTag->appendChild($self->_createTextElement('caption', $caption));
	$installedSWFeatureTag->appendChild($self->_createTextElement('instanceKey', $instanceKey));
	$installedSWFeatureTag->appendChild($self->_createTextElement('identifyingNumber', $identifyingNumber));

	if(defined($component)){
		my $manifest = $component->getManifest();
		$installedSWFeatureTag->appendChild($self->_buildSoftwareComponentsNode($manifest));
	}

	$collectedSoftwareFeaturesTag->appendChild($installedSWFeatureTag);
	$installedProductVersionTag->appendChild($collectedSoftwareFeaturesTag);
	$installedProductOnSystemTag->appendChild($installedProductVersionTag);
	$unspecificSystemTag->appendChild($installedProductOnSystemTag);

	return $unspecificSystemTag;
}

sub _buildSoftwareComponentsNode {
	my ($self, $manifest) = @_;
	my $softwareComponentsTag = XML::LibXML::Element->new('software-components');
	my $softwareComponentTag = XML::LibXML::Element->new('software-component');

	my $caption = $manifest->getValue('keycaption') || $manifest->getValue('keyname');
	my $release = $manifest->getValue('hostagent release') || $manifest->getValue('release');
	my $platform = $manifest->getValue('hostagent os') || $manifest->getValue('platform');

	my $spPatchLevel = $manifest->getValue('sp-patchlevel');
	my $revPatchLevel = $manifest->getValue('rev-patchlevel');
	my $shaPatchLevel = $manifest->getValue('hostagent patch number');
	my $revChangeList = $manifest->getValue('rev-changelist');
	my $patchLevel = defined($shaPatchLevel) ? $shaPatchLevel : defined($spPatchLevel) ? $spPatchLevel : $revPatchLevel;

	my $spLevel = $manifest->getValue('sp-number');
	my $revLevel = $manifest->getValue('rev-number');
	my $shaLevel = $manifest->getValue('hostagent patch number');
	my $level = defined($shaLevel) ? $shaLevel : defined($spLevel) ? $spLevel : $revLevel;

	$softwareComponentTag->appendChild($self->_createTextElement('name', $manifest->getValue('keyname')));
	$softwareComponentTag->appendChild($self->_createTextElement('vendor', $manifest->getValue('keyvendor')));
	$softwareComponentTag->appendChild($self->_createTextElement('caption', $caption));
	$softwareComponentTag->appendChild($self->_createTextElement('ppms-number', $manifest->getValue('compversion-id')));
	$softwareComponentTag->appendChild($self->_createTextElement('patch-level', $patchLevel));
	$softwareComponentTag->appendChild($self->_createTextElement('version', $release));
	$softwareComponentTag->appendChild($self->_createTextElement('os-set', $platform));
	$softwareComponentTag->appendChild($self->_createTextElement('sp-level', $level));
	$softwareComponentTag->appendChild($self->_createTextElement('rev-changelist',defined $revChangeList ? $revChangeList : 0));
	my $compilationMode = $manifest->getValue('compilation mode');
	if(defined $compilationMode){
	 $softwareComponentTag->appendChild($self->_createTextElement('compilation-mode',$compilationMode)); #relevant only for the SHA
	}
	$softwareComponentsTag->appendChild($softwareComponentTag);

	return $softwareComponentsTag;
}

sub _createTextElement {
	my ($self, $elementName, $text) = @_;
	my $element = XML::LibXML::Element->new($elementName);
	$element->appendTextNode($text);
	return $element;
}

1;