use strict;
use XML::Parser;
use File::Spec;

package LCM::Mid::DUParser;

#static members
my %Expat_Map = (
	"data-unit" => [ "mid", "data-carrier", "data-units", ],
	"name"      => [
		"mid", "data-carrier", "data-units", "data-unit", "logical-data-unit",
	],
	"path" => [ "mid", "data-carrier", "data-units", "data-unit", ],
);

sub _isOfInterest {
	my ( $expat, $element ) = @_;
	if ( $Expat_Map{$element} ) {
		my $context  = $expat->{"Context"};
		my $expected = $Expat_Map{$element};
		if ( _arraysEqual( $expected, $context ) ) {
			return 1;
		}
	}
	return 0;
}

sub _arraysEqual {
	my ( $expected, $given ) = @_;
	if ( scalar @{$expected} != scalar @{$given} ) {
		return 0;
	}

	foreach my $i ( 0 .. scalar @{$expected} - 1 ) {
		if ( $expected->[$i] eq $given->[$i] ) {
			next;
		}
		return 0;
	}
	return 1;
}

#object members

sub new {
	my ( $class, $dvdPath ) = @_;
	my $self = bless {
		"dvdPath"   => $dvdPath,
		"dataUnits" => {},
		"parser"    => 0,
	}, $class;
	my $parser = new XML::Parser(
		Handlers => {
			Start => sub { $self->_parseStartElement(@_) },
			Char  => sub { $self->_parseElementValue(@_) },
			End   => sub { $self->_parseEndOfElement(@_) },
		}
	);
	$self->{"parser"} = $parser;
	return $self;
}

sub _parseStartElement {
	my ( $self, $expat, $element ) = @_;
	if ( _isOfInterest( $expat, $element ) ) {
		if ( $element eq "data-unit" ) {
			$self->_newCurrentDataUnit();
		}
		else {
			$self->_setCurrentElement($element);
		}
	}
}

sub _parseElementValue {
	my ( $self, $expat, $value ) = @_;
	my @context         = @{ $expat->{Context} };
	my $element         = $context[-1];
	my $expectedElement = $self->_getCurrentElement();
	if ( $expectedElement && $expectedElement eq $element ) {
		$self->_addCurrentUnitProperty( $element, $value );
	}
}

sub _parseEndOfElement {
	my ( $self, $expat, $element ) = @_;
	if ( _isOfInterest( $expat, $element ) ) {
		if ( $element eq "data-unit" ) {
			$self->_addCurrentToDataUnits();
			$self->_clearCurrentDataUnit();
			$self->_clearCurrentElement();
		}
		elsif ( $element eq $self->_getCurrentElement() ) {
			$self->_clearCurrentElement();
		}
		else {

			#do nothing
		}
	}
}

sub _addDataUnit {
	my ( $self, $unit ) = @_;
	my $name = $unit->{"name"};
	$self->{"dataUnits"}->{$name} = $unit;
}

sub _getCurrentDataUnit {
	my ($self) = @_;
	return $self->{"currentDataUnit"};
}

sub _newCurrentDataUnit {
	my $self = $_[0];
	$self->{"currentDataUnit"} = {};
}

sub _clearCurrentDataUnit {
	my $self = $_[0];
	$self->{"currentDataUnit"} = undef;
}

sub _addCurrentToDataUnits {
	my $self    = shift;
	my $current = $self->_getCurrentDataUnit();
	$self->_addDataUnit($current);
}

sub _addCurrentUnitProperty {
	my ( $self, $property, $value ) = @_;
	if ( $property eq "path" ) {
		$value = substr $value, 1;
		$value = File::Spec->catdir( ( $self->{"dvdPath"}, $value ) );
	}
	$self->_getCurrentDataUnit()->{$property} = $value;
}

sub _getCurrentElement {
	return $_[0]->{"currentElement"};
}

sub _setCurrentElement {
	$_[0]->{"currentElement"} = $_[1];
}

sub _clearCurrentElement {
	$_[0]->{"currentElement"} = undef;
}

sub parse {
	my $self        = shift;
	my @dvdPathDirs = File::Spec->splitpath( $self->{"dvdPath"} );
	my $midPath     = File::Spec->catfile( @dvdPathDirs, "MID.XML" );
	my $parser      = $self->{"parser"};

	eval {
		#$SIG{__DIE__} is handler, which is called for die and croak
		my $oldDieSig = $SIG{__DIE__};
		$SIG{__DIE__} = sub {
			my ($errMsg) = @_;
			$errMsg =~ s/ at .*? line \d+.$/\n/;
			die $errMsg;
		};

		$parser->parsefile($midPath);

		$SIG{__DIE__} = $oldDieSig;
	};
	if ( defined $@ && length $@ ) {
		die "$@\n";
	}
}

sub getProperty {
	my ( $self, $duName, $duProperty ) = @_;
	return $self->{"dataUnits"}->{$duName}->{$duProperty};
}

sub getDataUnits {
	my $self = $_[0];
	return $self->{"dataUnits"};
}

return 1;
