package LCM::SLD::SldregUtility;

use strict;
use parent qw(SDB::Install::Base);
use LCM::ProcessExecutor;
use SDB::Install::System qw(isSidadmin);
use SDB::Common::Utils qw(createSysAdminUserName);
use File::Spec;

use constant SLD_CONFIG_FILE => "slddest.cfg";

our %SLD_PARAM_MAPPING = (
	'host_param'	=> 'SLDHostname',
	'port_param'	=> 'SLDPort',
	'user_param'	=> 'SLDUsername',
	'https_param'	=> 'UseHTTPS',
);

sub new {
	my ($class, $config) = @_;
	my $self = bless {}, $class;

	$self->setConfiguration($config);

	return $self;
};

sub setConfiguration {
	my ($self, $config) = @_;
	$self->{configuration} = $config;
}

sub getConfiguration {
	my ($self) = @_;
	return $self->{configuration};
}

sub setOutput {
	my ($self, $outputLines) = @_;
	$self->{output} = $outputLines;
}

sub getOutput {
	my ($self) = @_;
	return $self->{output};
}

sub setOutputHandler {
	my ($self, $progressHandler) = @_;
	$self->{progressHandler} = $progressHandler;
}

sub getOutputHandler {
	my ($self) = @_;
	return $self->{progressHandler};
}

sub setDelayOnWrite {
	my ($self, $delay) = @_;
	$self->{delayOnWrite} = $delay;
}

sub getDelayOnWrite {
	my ($self) = @_;
	return $self->{delayOnWrite};
}

sub getExecutable {
	my ($self) = @_;
	my $configuration = $self->getConfiguration();
	my $instance = $configuration->getOwnInstance();
	my $exeDir = $instance->get_instanceExeDir();
	return File::Spec->catfile($exeDir, 'sldreg');
}

sub executeSldreg {
	my ($self, $args, $input) = @_;
	my $executable = $self->getExecutable();
	my $executor = LCM::ProcessExecutor->new($executable, $args, $input);
	$executor->setMsgLstContext($self->getMsgLstContext());

	if (defined $self->getOutputHandler()) {
		$executor->setOutputHandler($self->getOutputHandler());
	}

	if (defined $self->getDelayOnWrite()) {
		$executor->setDelayOnWrite($self->getDelayOnWrite());
	}

	my $rc;
	eval {
		$rc = isSidadmin() ? $self->_executeAsSidadm($executor) : $self->_executeAsRoot($executor);
	};

	$self->getErrMsgLst()->addError($@) if ($@);
	$self->setOutput($executor->getOutputLines());
	return (!defined $rc || $rc) ? 0 : 1;
}

sub getSLDConfigFile {
	my ($self) = @_;
	my $configuration = $self->getConfiguration();
	my $instance = $configuration->getOwnInstance();
	my $globalDir = $instance->get_globalDir();
	return File::Spec->catfile($globalDir, SLD_CONFIG_FILE);
}

sub configureSLD {
	my ($self) = @_;
	my $args = $self->_buildArgs();
	my $passwordInput = $self->_buildPasswordInput();

	return $self->executeSldreg($args, $passwordInput);
}

sub _buildArgs{
	my ($self) = @_;
	my $configuration = $self->getConfiguration();
	my $destFile = $self->getSLDConfigFile();
	my $usernameValue = $configuration->getUsername();
	my $hostValue = $configuration->getHostname();
	my $portValue = $configuration->getPort();
	my @args = (
		'-noninteractive', '-usekeyfile', '-configure', $destFile,
		'-user', $usernameValue,
		'-host', $hostValue,
		'-port', $portValue
	);
	push(@args, '-usehttps') if ($configuration->isHTTPS());
	return \@args;
}

sub _buildPasswordInput {
	my ($self) = @_;
	my $configuration = $self->getConfiguration();
	my $password = $configuration->getPassword();
	return [$password];
}

sub getStoredSLDProperties {
	my ($self) = @_;
	my $destFile = $self->getSLDConfigFile();
	my $args = [ '-showconnect', $destFile ];

	$self->executeSldreg($args);
# No need to check for returnCode or existence of the destination file here
# If something went wrong, the values of the params will just be empty
	return $self->_parseSldProeprties($self->getOutput());
}



# -------------------------------------------------------
# Only private subs below this line
# -------------------------------------------------------

sub _parseSldProeprties {
	my ($self, $output) = @_;
	my $paramRegex = "(".(join("|", keys(%SLD_PARAM_MAPPING))).")=(['\"]*)(.+?)\\2\$"; # Strip the quotes around the value
	my $result = {};

	for my $line ( @{$output} ) {
		my ($param, undef, $value) = ($line =~ qr/$paramRegex/);
		my $configParamID = $SLD_PARAM_MAPPING{$param};
		$result->{$configParamID} = $value if ($configParamID);
	}

	return $result;
}

sub _executeAsSidadm {
	my ($self, $executor) = @_;
	return $executor->executeProgram(1);
}

sub _executeAsRoot {
	my ($self, $executor) = @_;
	my $configuration = $self->getConfiguration();
	my $instance = $configuration->getOwnInstance();
	my $sid = $instance->get_sid();
	my $sidadm = createSysAdminUserName($sid);
	my $exeDir = $instance->get_instanceExeDir();

	# Preset environment variable needed for the execution of sldreg as root
	local %ENV = %ENV;
	$ENV{LD_LIBRARY_PATH} = $ENV{LD_LIBRARY_PATH} . ":$exeDir";

	# Set uid and gid of sidadm to switch to when executing sldreg
	my (undef, undef, $uid, $gid) = getpwnam($sidadm);
	$executor->{uid} = $uid;
	$executor->{gid} = $gid;

	return $executor->executeProgram(1);
}

1;