package LCM::Slpp::Serializer;

use strict;
use base qw(Exporter);
use SDB::Common::Utils qw(createXMLParser);
#use JSON;

our @EXPORT_OK = qw(serialize deserialize);

our $LibXML_loaded = 0;

sub XML_TEXT_NODE;

sub _load_LibXML{
    return if $LibXML_loaded;
    require XML::LibXML;
    import XML::LibXML qw (XML_TEXT_NODE);
    $LibXML_loaded = 1;
}

sub serialize {
	my ($structure, $type) = @_;
	_load_LibXML();
	if(!defined($type)) {
		$type = 'XML'; 
	}
#	my $methodName = "_serializeTo${type}";
	return _toString($type, _serializeToXML($structure))
}

sub deserialize {
	my ($rawString, $type) = @_;
	_load_LibXML();
	if(!defined($type)) {
		$type = 'XML'; 
	}
#	my $methodName = "_deserializeFrom${type}";
	my $root = _fromString($type, $rawString);
	return {$root->nodeName, _deserializeFromXML($root)};
}

sub _fromString {
	my ($type, $rawString) = @_;
	if ($type eq 'XML') {
		my $parser = createXMLParser();
		my $tree = $parser->parse_string($rawString);
		return $tree->getDocumentElement();
	}
	return $rawString;
}

sub _deserializeFromXML {
	my ($node) = @_;

	if ( _containsOnlyTextNode($node) ) {
		return $node->textContent;
	} 
	
	if ( _isArray($node) ) {
		my $objectArray; 		
		for my $child ($node->childNodes()) {
			$objectArray = [] if(!defined $objectArray);
			if ($child->nodeType == XML_TEXT_NODE) {
				next;
			}			
			push(@$objectArray, { $child->nodeName => _deserializeFromXML($child) });
		}
		return $objectArray;
	}
	
	my $objectHash = {};
	for my $child ($node->childNodes()) {
		if ($child->nodeType == XML_TEXT_NODE) {
			next;
		}
		$objectHash->{$child->nodeName} = _deserializeFromXML($child);
	}
	return $objectHash;
}

sub _containsOnlyTextNode {
	my ($node) = @_;
	my @childNodes = $node->childNodes();
	if ( (scalar(@childNodes) == 1) && ($childNodes[0]->nodeType == XML_TEXT_NODE) ) {
		return 1;
	} else {
		return 0;
	}
}

sub _isArray {
	my ($node) = @_;
	my $childName = undef;
	for my $child ($node->childNodes()) {
		if (_containsOnlyTextNode($child)) {
			return 0;
		}
		if ($child->nodeType == XML_TEXT_NODE) {
			next;
		}
		if (not defined $childName) {
			$childName = $child->nodeName;
			next;
		}
		if ($childName ne $child->nodeName) {
			return 0;
		}
	}	
	return 1;
}

sub _toString {
	my ($type, $result) = @_;
	if ($type eq 'XML') {
		my $document = XML::LibXML::Document->new('1.0', 'utf-8');
		$document->setDocumentElement($result->[0]);
		$result->[0]->setNamespace('http://www.sap.com/lmsl/slp') if(defined $result->[0]);
		return $document->toString(1);
	}
	return $result;
}

sub _serializeToXML {
	my ($structure) = @_;
	my $refType = ref($structure);
	my $xmlElements = [];
	
	if($refType =~ /HASH/){
		for my $key (keys(%{$structure})) {
			my $elementName = $key;
			$elementName =~ s/_CDATA$//;
			my $element = XML::LibXML::Element->new($elementName);
			if(!ref($structure->{$key})) {
				if($key =~ /^.*_CDATA$/){
					my $cdataNode = XML::LibXML::CDATASection->new($structure->{$key});
					$element->appendChild($cdataNode);
				} else {
					$element->appendTextNode($structure->{$key});
				}
			} else {
				my $childElements = _serializeToXML($structure->{$key});
				for my $childElement (@$childElements) {
					$element->appendChild($childElement);	
				}
			}
			push(@$xmlElements, $element);
		}
	} elsif ($refType =~ /ARRAY/) {
		for my $innerStrcuture (@$structure) {
			my $result = _serializeToXML($innerStrcuture);
			for my $element (@$result) {
				push(@$xmlElements, $element);
			}
		}
	}
	return $xmlElements;
}

#sub _serializeToJSON {
#	my ($structure) = @_;
#	return JSON->new->utf8 ( 1 )->pretty ( 1 )->encode ($structure);
#}

1;