package LCM::Slpp::SlppAdapter;

use strict;
use base qw(Exporter);

use Data::Dumper;
use LCM::Configuration::SLD::RegistrationConfiguration;
use LCM::Configuration::GenericStackUpdateConfiguration;
use LCM::Slpp::HdblcmModelFactory;
use LCM::Slpp::SlppExecutionHandler;
use SDB::Install::Log;
use SDB::Install::Globals qw ($gTmpDir $gFlavourCockpit);
use SDB::Install::SysVars qw($path_separator);
use SDB::Install::Configuration qw($bool_true_pattern);
use LCM::Slpp::SlppAdapterFactory  qw(createParameterAdapter);
use LCM::Slpp::ExecutionState qw(STATE_ERROR);
use integer; # important due to the timestamps
use Time::HiRes qw(time);
use feature 'state';
use LCM::Slpp::FileBrowser;
use experimental qw (smartmatch);

my $new = sub { return bless({}, $_[0]); };

my $instance = $new->('LCM::Slpp::SlppAdapter');

sub getInstance {
	return $instance;
}

sub init {
    my ($self, $sid, $scenario, $shoudVerifySignature, $application, $errorState) = @_;
    $self->{scenario} = $scenario;
    $self->{sid} = $sid;    
    $self->{application} = $application;

    $self->{instconfig} = $self->_createConfiguration($self->{scenario}, $application);
    $self->{instconfig}->InitDefaults();
    $self->{FileBrowser} = new LCM::Slpp::FileBrowser();
    if (!$errorState) {
    	$self->setParameterIfExists('SID', $sid);
        $self->handleVerifySignature($shoudVerifySignature);
	}
}

sub setParameterIfExists {
	my ($self, $paramId, $value) = @_;
	if ($paramId ~~ @{$self->{instconfig}->getParamIds()}) {
	    $self->{instconfig}->setValue($paramId, $value);
	}
}

sub isUpdateSystem {
    my ($self) = @_;
    return ($self->getScenario() eq 'update_system') ? 1 : 0;
}

sub handleVerifySignature {
    my ($self, $verifySignatureArg) = @_;
    return if !$self->isUpdateSystem();

    my $config = $self->getConfiguration();
    my $shouldVerifySignature = ($verifySignatureArg =~ /$bool_true_pattern/) ? 1 : 0;
    $config->setDefault('VerifySignature', $shouldVerifySignature, 1);
}

sub getSid {
	return $_[0]->{sid};
}

sub getScenario {
	return $_[0]->{scenario};
}

sub getConfiguration {
	return $_[0]->{instconfig};
}

sub _createConfiguration {
    my ( $self, $scenario, $application) = @_;
    my $configuration = LCM::Slpp::HdblcmModelFactory::createConfiguration($scenario);
    if ($self->isUpdateSystem()) {
        my $isCockpitInstallationKit = ($application->getFlavour() eq $gFlavourCockpit) ? 1 : 0;
        $configuration->setIsCockpitConfiguration($isCockpitInstallationKit);
    }

    my $appMsgLstContext = $application->getMsgLstContext();
    my $cfgMsgLstContext = $configuration->getMsgLstContext();
    $appMsgLstContext->[$_]->appendMsgLst($cfgMsgLstContext->[$_]) for ( 0..1 );
    $configuration->setMsgLstContext($appMsgLstContext);

    my $applicationContext = LCM::App::ApplicationContext::getInstance();
    $applicationContext->setConfiguration($configuration);
    return $configuration; 
}

sub getParameters {
    my ($self) = @_;
    my $parameters = [];
    my $configuration = $self->{instconfig};

    for my $key (keys %{$configuration->{params}}) {
    	my $parameter = $self->getParameter($key);
    	if ($parameter) {
    		push @$parameters, $self->getParameter($key);
    	}
    }

    return {config => $parameters};
}

sub getPendingParameters {
    my ($self) = @_;
    my $parameters = [];
    my $pendingConfiguration = $self->{instconfig}->getPendingConfiguration();

    return undef if(!defined($pendingConfiguration));

    for my $key (keys(%{$pendingConfiguration->{params}})) {
        next if $pendingConfiguration->isSkipped($key);
        next if $pendingConfiguration->isHidden($key);
        my $parameterAdapter = $self->_createPendingParameterAdapter($pendingConfiguration, $key);
        if(defined($parameterAdapter)){
            push(@{$parameters}, $parameterAdapter->getSlppParameter());
        }
    }

    my $optIgnore = $pendingConfiguration->getOptionIgnore();
    if (defined($optIgnore)){
        my $ignoreArgs = $optIgnore->getSetKeysValidFor($optIgnore);
        my $ignoreStr  = $optIgnore->getStringFromArgList($ignoreArgs);

        $pendingConfiguration->setValue('Ignore', $ignoreStr) if ($ignoreStr);
    }

    my $optTimeout = $pendingConfiguration->getOptionTimeout();
    if (defined $optTimeout){
        LCM::Slpp::HdblcmModelFactory::addTimeouts($pendingConfiguration);
        my $parameterAdapter = $self->_createPendingParameterAdapter($pendingConfiguration, 'Timeouts');
        my $timeoutArgs = $optTimeout->getSetKeysValidFor ($optTimeout);
        my $timeoutStr  = $optTimeout->getStringFromArgList ($timeoutArgs);
        if ($timeoutStr) {
            for my $timeout (split(',', $timeoutStr)) {
                my ($timeoutKey, $timeoutValue) = ($timeout =~ /(.+):(.+)/);
                $parameterAdapter->setTimeouts($timeoutKey, $timeoutValue);
            }
        }
    }

    return {config => $parameters};
}

sub _createPendingParameterAdapter {
	my ($self, $pendingConfiguration, $paramId) = @_;
	my $hdblcmRepresentation = $pendingConfiguration->{params}->{$paramId};
	return createParameterAdapter($paramId, $hdblcmRepresentation, $pendingConfiguration);
}

sub getParameter {
	my ($self, $paramId) = @_;

	if($paramId eq 'FileBrowser'){
		return $self->{FileBrowser}->getSlppParameter();
	}

	if (! $self->existsParam($paramId)) {
		return undef;
	}
	
	if ($self->isHiddenParam($paramId)) {
		return undef;
	}

	if ($self->isSkippedParam ($paramId)) {
		return undef;
	}

	$self->ensureInitializedAdapter($paramId);
	
	# Ugly hack to resolve problems with substituttion of parameters in default values
	if($self->{instconfig}->{params}->{$paramId}->{default} =~ /\$/){
		my $paramDefault = $self->{instconfig}->getDefault($paramId);
		$self->{instconfig}->setDefault($paramId, $paramDefault);
	}
	return $self->{parameterAdapters}->{$paramId}->getSlppParameter();
}

sub existsParam {
	my ($self, $paramId) = @_;
	return defined($self->{instconfig}->{params}->{$paramId});
}

sub isHiddenParam {
	my ($self, $paramId) = @_;
	return $self->{instconfig}->{params}->{$paramId}->{hidden};
}

sub isSkippedParam {
	my ($self, $paramId) = @_;
	return $self->{instconfig}->isSkipped($paramId);
}

sub updateParameter {
    my ( $self, $paramId, $value ) = @_;
	my $configuration = $self->getConfiguration();
	
	if($paramId eq 'FileBrowser'){
		return $self->{FileBrowser}->setValue($value);
	}
	
    if (! $self->existsParam($paramId)) {
		return undef;
	}
	
	if ($configuration->isSkipped($paramId)) {
		return 1;
	}
	
	$self->ensureInitializedAdapter($paramId);
	$configuration->resetError();
	
	my $parameterAdapter = $self->{parameterAdapters}->{$paramId};
    my $isBatchValueSet = defined($value->{Parameter}->{batchvalue});
    my $internalValue = $parameterAdapter->parseInternalValue($value->{Parameter}, $isBatchValueSet);
    my $shouldSetBatchValue = $isBatchValueSet && $paramId ne 'Timeouts' && $paramId ne 'Ignore';
    my $rc = $shouldSetBatchValue ? $self->setBatchValue($paramId, $internalValue) : $self->setValue($paramId, $internalValue);

    $parameterAdapter->setValidationStatus($rc, ${$self->{instconfig}->getErrMsgLst()->getMsgLstString()});
    
   	$configuration->resetError();
    
    return $rc;
}

sub setBatchValue {
	my ($self, $paramId, $internalValue) = @_;
	my $configuration = $self->getConfiguration();
	my $paramType = $configuration->{params}->{$paramId}->{type};
	# Reset the parameter, so setBatchValueOfRequiredParams can set it again and call the listeners in the right moment
	$configuration->resetParam($paramId);

	if ($paramType =~ /map/) {
		for my $key (keys (%{$internalValue})){
			$configuration->{params}->{$paramId}->{batchValue}->{$key} = $internalValue->{$key};
		}
	} else {
		$configuration->{params}->{$paramId}->{batchValue} = $internalValue;
	}
	return 1;
}

sub setValue {
	my ($self, $paramId, $internalValue) = @_;

	my $configuration = $self->getConfiguration();
	my $paramType = $configuration->{params}->{$paramId}->{type};

	if ($paramType =~ /map/) {
		my $rc = 1;
	    for my $key (keys (%{$internalValue})){
	    	if($paramId eq 'Timeouts'){
	    	   $rc = $self->{parameterAdapters}->{Timeouts}->setTimeouts($key, $internalValue->{$key}) && $rc;
	    	} else {
	           $rc = $configuration->setMapValueItem($paramId, $key, $internalValue->{$key}) && $rc;
	    	}
		}
		if (!$configuration->checkAllMapValues($paramId)) {
			$configuration->resetParam($paramId);
			return 0;
		}
		return $rc;
	}

	return $configuration->setValue($paramId, $internalValue);
}

sub ensureInitializedAdapter {
	my ($self, $paramId) = @_;
	if (not exists $self->{parameterAdapters}->{$paramId}){
		my $hdblcmRepresentation = $self->{instconfig}->{params}->{$paramId};
		my $parameterAdapter = createParameterAdapter($paramId, $hdblcmRepresentation, $self->{instconfig});
		if(!defined($parameterAdapter)){
#			TODO HANDLE THIS EXCEPTIONAL STATE
			die("This should not happen to this awseome tool !\n");
		}
		$self->{parameterAdapters}->{$paramId} = $parameterAdapter;
	}
}

sub updateParameters {
    my ( $self, $value) = @_;
    my $rc = 1;
    my @filteredParameters = grep {$self->existsParam($_->{Parameter}->{id})} @{$value->{config}};
    my @sortedFilteredParameters = sort {$self->_parametersSortingFunction($a, $b)} @filteredParameters;

    if(scalar @sortedFilteredParameters) {
		my $config = $self->getConfiguration();
		my $paramIdOfFirst = $sortedFilteredParameters[0]->{Parameter}->{id};
        my $hasBatchParams = grep { defined($_->{Parameter}->{batchvalue}) } @sortedFilteredParameters;
# Ugly hack to avoid setting the default value of the Advanced Config parameter instead of the batch value
# We do not want to call setBatchValueOfRequiredParams if any of the parameters has defined {batchValue}
# This would mean that this POST to the /config resource comes from the advanced config, instead from one
# of the dialogs
		$config->setBatchValueOfRequiredParams($paramIdOfFirst) if !$hasBatchParams;
	}

    for my $parameterHash (@sortedFilteredParameters) {
    	my $paramId = $parameterHash->{Parameter}->{id};
		my $paramRc = $self->updateParameter($paramId, $parameterHash);
    	$rc = $rc && $paramRc;
    }
    
    return $rc;
}

sub _parametersSortingFunction {
	my ($self, $first, $second) = @_;
	my $paramIdFirst = $first->{Parameter}->{id};
	my $paramIdSecond = $second->{Parameter}->{id};
	my $params = $self->getConfiguration()->{params};

	return $params->{$paramIdFirst}->{order} <=> $params->{$paramIdSecond}->{order};
}

sub getSlppTasklist {
	my $executionHandler = LCM::Slpp::SlppExecutionHandler::getInstance();
	my @slppSteps = $executionHandler->getSlppSteps();
	return { tasklist => [ $_[0]->getSlppTask(), @slppSteps ] };
}

sub getSlppMonitor {
	my $executionHandler = LCM::Slpp::SlppExecutionHandler::getInstance();
	my @slppSteps = $executionHandler->getSlppFinishedSteps();
	return { monitor => [ $_[0]->getSlppTask(), @slppSteps ] };
}

sub getSlppTask {
	my ($self) = @_;
	state $startedAt = int(time());
	my $executionHandler = LCM::Slpp::SlppExecutionHandler::getInstance();
	return { 
		Task => {
			id 				 => $self->getScenario(),
			type 			 => 'slp.task.type.PROCESS',
			technicalName 	 => $self->getScenario(),
			status 			 => $executionHandler->getState(),
			startedAt 		 => $startedAt,
			finishedAt 		 => $executionHandler->getFinishedAt(),
			progress 		 => int($executionHandler->getProgress()),
			refreshRate 	 => '1',
			logs 			 => '',
			error			 => '',
			progressMessages => $self->getProgressMessages(),
			displayName 	 => $self->getScenario(),
			description 	 => $self->getScenario(),
			externalInfo 	 => 'help.sap.com'
		}
	};
}

sub getSlppError {
	my ($self) = @_;
	my $executionHandler = LCM::Slpp::SlppExecutionHandler::getInstance();

	return { Error => {} } if ($executionHandler->getState() ne LCM::Slpp::ExecutionState::STATE_ERROR);

	my $errorState = $executionHandler->getErrorState();
	return { 
		Error => {
			id 				 => $errorState->getId(),
			code 			 => $errorState->getCode(),
			displayName		 => $errorState->getDisplayName(),
			description_CDATA 	 => $errorState->getDescription(),
		}
	};
}

sub getProgressMessages {
	my $self = shift();
	my $progressMessagesArray = [];
	my $progressMessages = LCM::Slpp::SlppExecutionHandler::getInstance()->getProgressMessages();
	
	my $idCounter = 1;
    for my $progressMessage (@{$progressMessages}) {
    	push @$progressMessagesArray, {
    		ProgressMessage => {
    			id => $idCounter++,
    			message => $progressMessage
    		}
    	};
    }

	return $progressMessagesArray;
}

sub getLogs {
	my ($self) = @_;
	my $executionHandler = LCM::Slpp::SlppExecutionHandler::getInstance();
	my $logFileURIsHash = $executionHandler->getLogFileURIs();
	my $logsElementsArray = [];
	for my $logId (keys %{$logFileURIsHash}){
		my $logFileLocation = $logFileURIsHash->{$logId};
		next unless($logFileLocation);
		push(@{$logsElementsArray}, { Log => { id => $logId, content => $logFileLocation, format => 'slp.log.format.TEXT' } });
	}
	return { logs => $logsElementsArray };
}

sub getLogElementById {
	my ($self, $logId) = @_;
	my $executionHandler = LCM::Slpp::SlppExecutionHandler::getInstance();
	my $logFileURIsHash = $executionHandler->getLogFileURIs();
	
	return undef unless (exists($logFileURIsHash->{$logId}));
	return undef unless ($logFileURIsHash->{$logId});
	
	return { Log => { id => $logId, content => $logFileURIsHash->{$logId}, format => 'slp.log.format.TEXT' } };
}

1;
