package LCM::Task::CommonTask::AssignAdditionalLocalRolesTask;

use strict;
use parent qw (LCM::Task::SHASupportedTask);
use SDB::Install::System qw (isSidadmin);
use LCM::HostsParser;
use SDB::Install::Configuration::AnyMultiHostConfig qw($validHostRoles);
use SDB::Install::Globals qw ($gHostRoleAcceleratorWorker $gHostRoleAcceleratorStandby $gLogDir);
use SDB::Install::SysVars qw($path_separator);  
use LCM::Utils::CommonUtils qw(getHdbmodifyTimeoutValues buildTimeoutsString);
use LCM::Task::TaskProgressHandler::AddHostRolesTaskProgressHandlerFactory;

# These constants are here because the GSI configuration
# does not have 'system_user' and 'add_roles' parameters
my $HOSTCTRL_OPERATION = 'hdblcm_assign_roles_v8';
my $ADD_ROLES_OPTION = 'add_roles';
my $SYSTEM_USER_OPTION = 'system_user';
my $PROGRESS_MESSAGE = 'Assigning Additional Roles to the Local Host';
my $PASSWORD_KEYS = ['Password', 'AcceleratorPassword', 'OrgManagerPassword', 'SQLSysPasswd', 'SQLTenantUserPassword'];

our @EXPORT = qw ($PROGRESS_MESSAGE);

sub getId {
	return 'assign_additional_local_roles';
}

sub getName { 
	return "Assign Additional Roles to the Local Host"; 
}

sub getExecutionName {
    return $PROGRESS_MESSAGE;
}

sub _getNumberOfExpectedOutputLines{
	return 100;
}

sub _executeInternal {
    my $self = shift();
    my $sid = $self->_getConfiguration()->getSID();
    my $message = $self->getMsgLst()->addProgressMessage($self->getExecutionName() . '...');
    my $saveContext = $self->setMsgLstContext([$message->getSubMsgLst()]);
    $self->getMsgLst()->getProgressHandler()->incrementIndentationDepth();

    my $returnCode = isSidadmin($sid) ? $self->_executeSHAOperation() : $self->_executeAsRoot();

    if($returnCode) {
        $self->getStatus()->_setFinishedState();
        $message->endMessage (undef, sprintf("%s finished", $self->getName()));
    } else {
        $self->getStatus()->_setErrorState();
        $message->endMessage (undef, sprintf("%s failed", $self->getName()));
    }

    $self->getMsgLst()->getProgressHandler()->decrementIndentationDepth();
    $self->setMsgLstContext($saveContext);
}

sub _executeSHAOperation {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    
    my $localHost = $self->_createRemoteHostsHostctrlObject();
    $localHost->useSidadm();
    $localHost->setMsgLstContext( $self->getMsgLstContext() );

    my $rc = 1;
    my $optionsMap = $self->buildSHAOptionsMap();
    my $exitCode = $localHost->executeHostctrlParallel($HOSTCTRL_OPERATION, $configuration, undef, $PASSWORD_KEYS, undef, undef, undef,
                                                        undef, undef, $optionsMap, undef, undef, undef, 0);
    my $outputBuffer = $localHost->getOutputBuffer();
    my $executionOutput = [ split('\n', $outputBuffer->[0]) ];
    $self->_setLogLocation($self->_parseLogFileLocation($executionOutput));
    
    if (!defined $exitCode || ($exitCode != 0)) {
        $self->setErrorMessage($self->getFailedExecutionMessage(), $localHost->getErrMsgLst());
        $rc = undef;
    }
    return $rc;
}

sub _createRemoteHostsHostctrlObject {
    my ($self) = @_;
    my $progressHandler  = $self->getMsgLst()->getProgressHandler();
    my $configuration = $self->_getConfiguration();
    my @localHost = ($configuration->getLocalHanaHost());
    my $localExecutionHost = new SDB::Install::RemoteHostctrlHosts(@localHost);
    $localExecutionHost->setOutputHandler($localHost[0], $progressHandler);
    return $localExecutionHost;
}

sub buildSHAOptionsMap{
    my $self = shift();
    my $result = $self->SUPER::buildSHAOptionsMap();
    my $rolesString = $self->_getLocalHostRolesString();
    my $configuration = $self->_getConfiguration();
    $result->{SID} = $self->getSID();
    $result->{INSTLOG_DIR} = $gLogDir;
    $result->{ROLES} = $rolesString;
    if($rolesString =~ /$gHostRoleAcceleratorWorker|$gHostRoleAcceleratorStandby/){
        $result->{ASE_USER} = $configuration->getValue('AcceleratorUser');
    }
    $result->{SYSTEM_USER} = $configuration->getValue('SystemUser') || 'SYSTEM';
    if (!$configuration->isSkipped('AutoInitializeServices')) {
        $result->{AIS} = $configuration->getValue('AutoInitializeServices');
    }
    if (!$configuration->isSkipped('TenantUser')) {
        $result->{TENANT_USER} = $configuration->getValue('TenantUser');
    }

    if(!$configuration->isSkipped('OrgManagerUser')) {
        my $orgManagerUser = $configuration->getValue('OrgManagerUser');
        $result->{ORG_MANAGER_USER} = $orgManagerUser;
    }
    my $importContentXS2 = $configuration->getValue('ImportContentXS2');
    if (defined $importContentXS2) {
        my $key         = $configuration->getHostctrlOpt('ImportContentXS2');
        $result->{$key} = ($importContentXS2) ? 'on' : 'off';
    }
    my $timeoutString = buildTimeoutsString($configuration, getHdbmodifyTimeoutValues(), 0); # 0 - without CLI option
    $result->{TIMEOUTS} = $timeoutString if $timeoutString;

    return $result;
}

sub getParameterToOptionMapping {
    return {
        'Target' => 'SAPMNT',
        'SID' => 'SID',
        'SkipModifySudoers' => 'SUD',
    };
}

sub setLocalHostRolesString {
    my ($self, $localHostRolesString) = @_;
    $self->{local_roles} = $localHostRolesString;
}

sub _getLocalHostRolesString {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $localHostRolesString = defined $self->{local_roles} ? $self->{local_roles} : $configuration->getAddLocalRolesString();
    my $rolesString = length($localHostRolesString) > 0 ? join(':role=', split(',', $localHostRolesString)) : '';
    return '' if $rolesString eq '';

    my $localHost = $configuration->getLocalHanaHost();
    return sprintf('%s:role=%s', $localHost, $rolesString);
}

sub _executeAsRoot {
    my ($self) = @_;
    my $executor = $self->_createProcessExecutor();
     my $saveCntxt = $self->getMsgLstContext();
    $executor->setMsgLstContext( $self->getMsgLstContext() );
    my $rc = $executor->executeProgram( 1 );
    $self->_setLogLocation($self->_parseLogFileLocation($executor->getOutputLines()));
    $self->setMsgLstContext( $saveCntxt );

    return 1 if (defined $rc && $rc == 0);

    $self->getErrMsgLst()->addError($self->getFailedExecutionMessage());
    return undef;
}

sub _createProcessExecutor {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $command = $self->_getExecutable();
    my $args = $self->_buildArgs();
    my $executor; 
    if ( $self->_isPasswordArgsExists() ){
        my $xmlPasswordStream = $configuration->getXmlPasswordStream();
        $executor = new LCM::ProcessExecutor( $command, $args, $xmlPasswordStream );
    } else {
    	$executor = LCM::ProcessExecutor( $command, $args );
    }
    $executor->setOutputHandler( $self->getMsgLst()->getProgressHandler() );
    $executor->setProcessEnvironment (LCM::TraceLoggingEnvironment::PrepareProcessExecutorHdbEnvironment($self->_getExecutable()));
    return $executor;
}

sub _buildArgs{
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $localRolesString = $self->_getLocalHostRolesString();
    my $localRolesArgument = "--${ADD_ROLES_OPTION}=$localRolesString";
    my $args = ['-b', $localRolesArgument, '--read_password_from_stdin=xml',"--instlog_dir=$gLogDir"];

    if($localRolesString =~ /$gHostRoleAcceleratorWorker|$gHostRoleAcceleratorStandby/){
        my $acceleratorUserOpt = $configuration->getOpt('AcceleratorUser');
        my $acceleratorUser = $configuration->getValue('AcceleratorUser');
        my $acceleratorUserArgument = "$acceleratorUserOpt=$acceleratorUser";
        push(@{$args}, $acceleratorUserArgument);
    }
    my $systemUser = $configuration->getValue('SystemUser') || 'SYSTEM';
    my $systemUserArgument = "--${SYSTEM_USER_OPTION}=$systemUser";
    push(@{$args}, $systemUserArgument);
    if (!$configuration->isSkipped('AutoInitializeServices')) {
        my $autoInitOpt = $configuration->getOpt('AutoInitializeServices');
        my $autoInit = $configuration->getValue('AutoInitializeServices');
        my $autoInitArgument = "${autoInitOpt}=$autoInit";
        push(@{$args}, $autoInitArgument);
    }
    if (!$configuration->isSkipped('TenantUser')) {
        my $tenantUserOpt = $configuration->getOpt('TenantUser');
        my $tenantUser = $configuration->getValue('TenantUser');
        my $tenantUserArgument = "${tenantUserOpt}=$tenantUser";
        push(@{$args}, $tenantUserArgument);
    }
    if(!$configuration->isSkipped('OrgManagerUser')) {
        my $orgManagerUserOpt = $configuration->getOpt('OrgManagerUser');
        my $orgManagerUser = $configuration->getValue('OrgManagerUser');
        my $orgManagerUserArgument = "$orgManagerUserOpt=$orgManagerUser";
        push(@{$args}, $orgManagerUserArgument);
    }
    my $importContentXS2 = $configuration->getValue('ImportContentXS2');
    if (defined($importContentXS2)) {
        my $option = $configuration->getOpt('ImportContentXS2');
        my $value = $importContentXS2 ? 'on' : 'off';
        push(@{$args}, sprintf('%s=%s', $option, $value));
    }
    if ($configuration->hasValue('SkipModifySudoers')) {
        my $value = $configuration->getValue('SkipModifySudoers') ? 'on' : 'off';
        my $option = $configuration->getOpt('SkipModifySudoers');
        push(@{$args}, sprintf('%s=%s', $option, $value));
    }
    my $timeoutString = buildTimeoutsString($configuration, getHdbmodifyTimeoutValues());
    push(@{$args}, $timeoutString) if $timeoutString;

    return $args;
}

sub _isPasswordArgsExists{
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    return 1 if($configuration->getValue("Password"));
    return 1 if($configuration->getValue("AcceleratorPassword"));
    return 1 if($configuration->getValue("SQLSysPasswd"));
    return 1 if($configuration->getValue("OrgManagerPassword"));

    return 0;
}

sub _getExecutable {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();

    my $hdbExecutable =
           join($path_separator,
                $configuration->getSAPSystem()->get_globalTrexInstallDir(),
                'bin',
                'hdbmodify');

    return $hdbExecutable;
}

sub initProgressHandler {
    my ($self) = @_;
    my $progressHandler = LCM::Task::TaskProgressHandler::AddHostRolesTaskProgressHandlerFactory::createAddHostRolesTaskProgressHandler($self);
    $self->getMsgLst()->setProgressHandler($progressHandler);
}

sub getSlppLogFileName{
	return 'assign_local_roles.log';
}

sub shouldPersistFinishedStep {
    return 1;
}

1;
