package LCM::Task::DownloadComponents::DownloadSingleComponentTask;

use strict;
use parent 'LCM::Task';
use SDB::Install::User;
use SDB::Install::System qw(loadSSLRequiringPackage);
use LCM::FileUtils qw(createDirectory MyRealPath);
use LCM::DownloadComponents::DownloadableArchive;
use File::stat;

my $PROGRESS_MESSAGE = "Downloading component archives...";
my $SUCCESS_END_MESSAGE = "Finished downloading component archives";
my $FAIL_END_MESSAGE = "Downloading of component archives failed";

# Override
sub new{
    my ($class, $downloadInformation) = (shift(), shift());
    my $self = $class->SUPER::new(@_);
    my $configuration = $self->_getConfiguration();
    my $downloadDir = $configuration->getValue('DownloadDir');
    $self->_setDownloadInformation($downloadInformation);

    for my $file (@{$downloadInformation->{'files'}}) {
        push(@{$self->{archives}}, LCM::DownloadComponents::DownloadableArchive->new($file, $configuration));
    }

    return $self;
}

# Override
sub getId {
    my ($self) = @_;
    my $downloadInformation = $self->_getDownloadInformation();

    return sprintf('download_archive_%s', $downloadInformation->{ppmsId});
}

# Override
sub getName {
    my ($self) = @_;
    my $downloadInformation = $self->_getDownloadInformation();

    return sprintf('Download %s Archives', $downloadInformation->{name});
}

# Override
sub getExecutionName {
    my ($self) = @_;
    my $downloadInformation = $self->_getDownloadInformation();

    return sprintf('Downloading %s Archives', $downloadInformation->{name});
}

# Override
sub _executeInternal {
    my $self = shift();
    my $configuration = $self->_getConfiguration();
    my $downloadTarget = $configuration->getValue('DownloadDir');
    my $downloadDirObject = File::stat::stat($downloadTarget);
    my $rc = 1;

    $rc &&= $self->_createDir($downloadTarget) if (!-d $downloadDirObject);
    $rc &&= $self->_downloadComponentArchives($configuration);

    if (!$rc) {
        $self->getStatus()->_setErrorState();
        $self->setErrorMessage($FAIL_END_MESSAGE, $self->getErrMsgLst());
    } else {
        $self->getStatus()->_setFinishedState();
    }
    return $rc;
}

# Override
sub _getNumberOfExpectedOutputLines{
    return 150;
}

sub _getDownloadInformation {
    my ($self) = @_;
    return $self->{_downloadInformation};
}

sub _setDownloadInformation {
    my ($self, $downloadInformation) = @_;
    $self->{_downloadInformation} = $downloadInformation;
}

sub _getArchives {
    my ($self) = @_;
    return $self->{archives};
}

# Important - this sub must always match the implementation of
# sub _constructUserAgent in CleanupDownloadDirectoryTask.pm
sub _constructUserAgent {
    my ($self) = @_;
    my $downloadedBytes = 0;
    my $configuration = $self->_getConfiguration();
    my $caPath = MyRealPath($configuration->getValue('CAPath'));
    my $proxyHost = $configuration->isSkipped('ProxyHost') ? undef : $configuration->getValue('ProxyHost');
    my $proxyPort = $configuration->isSkipped('ProxyPort') ? undef : $configuration->getValue('ProxyPort');
    my $proxyUser = $configuration->isSkipped('ProxyUser') ? undef : $configuration->getValue('ProxyUser');
    my $proxyPassword = $configuration->isSkipped('ProxyPassword') ? undef : $configuration->getValue('ProxyPassword');
    my $smpUser = $configuration->getValue('SMPUser');
    my $smpPassword = $configuration->getValue('SMPUserPassword');
    my $userAgent = new LCM::Utils::TCPUserAgent($caPath);

    $userAgent->setUser($smpUser);
    $userAgent->setPassword($smpPassword);
    $userAgent->setProxyHost($proxyHost);
    $userAgent->setProxyPort($proxyPort);
    $userAgent->setProxyUser($proxyUser);
    $userAgent->setProxyPassword($proxyPassword);

    return $userAgent;
}

sub _downloadComponentArchives {
    my ($self, $configuration) = @_;
    my $downloadInformation = $self->_getDownloadInformation();
    my $downloadTarget = $configuration->getValue('DownloadDir');
    my $message = $self->getMsgLst()->addProgressMessage($PROGRESS_MESSAGE);
    my $saveContext = $self->setMsgLstContext([$message->getSubMsgLst()]);

    if (!loadSSLRequiringPackage ('LWP::Protocol::https', $configuration->getMsgLst (), $configuration->getErrMsgLst())){
        $message ->endMessage(undef, $FAIL_END_MESSAGE);
        $self->setMsgLstContext($saveContext);
        return undef;
    }

    require LCM::Utils::TCPUserAgent;

    my $parentProcessTask = $self->getParentTask();
    my $returnCode = 1;
    for my $archive (@{$self->_getArchives()}) {
        $archive->setUserAgent($self->_constructUserAgent());
        $returnCode &&= $self->_downloadArchive($archive);
        $parentProcessTask->addDownloadedArchiveName($archive->getName());

        last if !$returnCode;
    }

    my $endMessage = $returnCode ?  $SUCCESS_END_MESSAGE : $FAIL_END_MESSAGE;
    $message ->endMessage(undef, $endMessage);
    $self->setMsgLstContext($saveContext);
    return $returnCode;
}

sub _downloadArchive {
    my ($self, $archive) = @_;
    my $url = $archive->getUrl();
    my $msg = $self->getMsgLst()->addProgressMessage("Initiating download via URL '$url'...");
    my $subMsgLst = $msg->getSubMsgLst();
    my $saveContext = $archive->setMsgLstContext([ $subMsgLst, $subMsgLst ]);

    my $returnCode = $archive->download($self->_getResponseDataCallback($archive));

    $self->setMsgLstContext($saveContext);
    return $returnCode;
}

sub _createDir {
    my ($self, $path) = @_;
    my $currentUser = new SDB::Install::User();
    my ($fileMode, $uid, $gid, $isRecursive) = (0755, $currentUser->id(), $currentUser->gid(), 0);
    my $errlst = new SDB::Install::MsgLst();
    if (! createDirectory($path, $fileMode, $uid, $gid, $isRecursive, $errlst)) {
        my $errorMessage = $errlst->getMsgLstString();
        $self->setErrorMessage($$errorMessage, undef);
        return 0;
    }
    return 1;
}

sub _getResponseDataCallback {
    my ($self, $archive) = @_;
    my $targetSize = $archive->getSize();

    return sub {
        my ($downloadedBytes) = @_;
        my $newProgress = ($downloadedBytes > $targetSize) ? 99 : int( ($downloadedBytes * 100) / $targetSize);
        $self->_setProgress($newProgress);
        $self->_notifyTaskChanged();
    }
}

1;
