package LCM::Task::RenameSystemProcessTask;

use strict;
use parent qw(LCM::Task::SLProcessTask);

use LCM::App::ApplicationContext;
use LCM::Task qw($gFlavourProductName $gActionScope);
use LCM::Task::RenameTask::RenameSystemTask;
use LCM::Task::CommonTask::UpdateRemoteHostsTask;
use LCM::Task::RenameTask::CreatePersistenceFileTask;
use LCM::Task::RenameTask::DeletePersistenceFileTask;
use LCM::Task::RenameTask::UpdateLocalHostAfterRenameTask;
use LCM::Task::RenameTask::UpdateLocalHostBeforeRenameTask;
use LCM::Task::RenameTask::UpdateRemoteHostsAfterRenameTask;
use LCM::Task::RenameTask::UpdateRemoteHostsBeforeRenameTask;

my $ACTION_DONE_MESSAGE               = "$gFlavourProductName $gActionScope renamed";
my $ACTION_FAILED_MESSAGE             = "Renaming of $gFlavourProductName $gActionScope failed.";
my $ACTION_NOTHING_TO_DO_MESSAGE      = "There is nothing to rename. Change some properties of $gFlavourProductName System to be able to rename the system.";
my $ACTION_ABORTED_MESSAGE            = "Renaming of $gFlavourProductName $gActionScope aborted.";
my $ACTION_DONE_WITH_WARNINGS_MESSAGE = "$gFlavourProductName $gActionScope renamed with warnings.";

my $CHECK_DONE_MESSAGE                = "Rename checks for the $gFlavourProductName $gActionScope passed.";
my $CHECK_FAILED_MESSAGE              = "Rename checks for the $gFlavourProductName $gActionScope failed.";
my $CHECK_ABORTED_MESSAGE             = "Rename checks for the $gFlavourProductName $gActionScope aborted.";


sub _formatFlavourMessage {
    my($self, $message) = @_;
    my $configuration = $self->_getConfiguration();
    my $flavourProductName = LCM::App::ApplicationContext::getFlavourProductName();
    $message =~ s/\Q$gFlavourProductName\E/$flavourProductName/g;
    return $message;
}

sub _initActionMessages {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $status = $self->getStatus();

    if ($configuration->isCheckOnly()) {
        $status->_setActionDoneMessage($self->_formatMessage($CHECK_DONE_MESSAGE));
        $status->_setActionDoneWithWarningsMessage(""); # No execution => no execution warnings
        $status->_setActionFailedMessage($self->_formatMessage($CHECK_FAILED_MESSAGE));
        $status->_setActionAbortedMessage($self->_formatMessage($CHECK_ABORTED_MESSAGE));
    } else {
        $status->_setActionDoneMessage($self->_formatMessage($ACTION_DONE_MESSAGE));
        $status->_setActionDoneWithWarningsMessage($self->_formatMessage($ACTION_DONE_WITH_WARNINGS_MESSAGE));
        $status->_setActionFailedMessage($self->_formatMessage($ACTION_FAILED_MESSAGE));
        $status->_setActionAbortedMessage($self->_formatMessage($ACTION_ABORTED_MESSAGE));
    }
}

# Override
sub getId {
    return "rename_system_process_task";
}

# Override
sub getName {
    return "Rename System Process Task";
}

# Override
sub _executeInternal {
    my $self = shift() ;
    my $configuration = $self->_getConfiguration();
    my $status = $self->getStatus();

    $self->_initActionMessages();

    if ( !$configuration->{isContinueOption} && !$configuration->shouldRunHdbRename() && !$configuration->isAnyCertificateChanged() ) {
        $status->_setActionDoneMessage($self->_formatMessage($ACTION_NOTHING_TO_DO_MESSAGE));
        $self->getStatus()->_setFinishedState();
        return;
    }

    if ($configuration->isCheckOnly()) {
        $self->getStatus()->_setFinishedState();
        return;
    }

    if (!defined $self->{_subtask}) {
        $self->_createSubtasks();
    }

    $self->SUPER::_executeInternal();
}

sub _createSubtasks {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $shallUnregisterComponents = $configuration->shallUnregisterComponents();
    my $isContinueAction          = $configuration->{isContinueOption};
    my $scope                     = $configuration->getValue('Scope');    
            
    if($shallUnregisterComponents){
        $self->addSubtask(new LCM::Task::RenameTask::UpdateLocalHostBeforeRenameTask($configuration));
        $self->addSubtask(new LCM::Task::RenameTask::UpdateRemoteHostsBeforeRenameTask($configuration)) if($configuration->shallUnregisterComponentsOnRemoteHosts());
    }
    $self->addSubtask(new LCM::Task::RenameTask::RenameSystemTask($configuration));    
    $self->addSubtask(new LCM::Task::RenameTask::UpdateLocalHostAfterRenameTask($configuration)) if ( $configuration->shallUpdateLocalHostAfterRename() );
    $self->addSubtask(new LCM::Task::RenameTask::UpdateRemoteHostsAfterRenameTask($configuration)) if ( $configuration->shallUpdateRemoteHostsAfterRename() );
    $self->addSubtask(new LCM::Task::RenameTask::CreatePersistenceFileTask($configuration)) if ( !$isContinueAction && $scope eq 'instance' && $configuration->isDistributedSystem(1) );
    $self->addSubtask(new LCM::Task::RenameTask::DeletePersistenceFileTask($configuration)) if ( $isContinueAction );
}

1;