package LCM::Task::UpdateSystemBootstrapProcessTask;

use strict;
no warnings 'exec';
use parent 'LCM::Task::SLProcessTask';
use LCM::FileUtils qw(MyRealPath);
use LCM::Task::CLIProcessTaskMetadata;
use LCM::DevelopmentTrace;
use LCM::App::ApplicationContext;
use LCM::Task::TaskProgressHandler::UpdateSystemBootstrapTaskProgressHandler;
use SDB::Install::Globals qw($gLogDir $gKeynameInstaller);
use SDB::Install::SysVars qw ($isWin);
use File::stat;

sub new {
    my ($class) = shift();
    my $self = $class->SUPER::new(@_);
    $self->_setCLIMetadata(LCM::Task::CLIProcessTaskMetadata->new({
        requiresInput           => 1,
        requiresSummary         => 0,
        requiresConfirmation    => 0,
        requiresLog             => 1,
        requiresHeaderMessage   => 1,
        messageBeforeConfirm    => undef,
    }));
    $self->initProgressHandler();
    return $self;
}

sub getId {
    return "bootstrap_resident_update_system_task";
}

sub getExecutionName {
    return "Starting hdblcm from the installation kit";
}

sub getName {
    return "Start hdblcm from the installation kit";
}

sub initProgressHandler {
    my ($self) = @_;
    my $progressHandler = LCM::Task::TaskProgressHandler::UpdateSystemBootstrapTaskProgressHandler->new($self);
    $self->getMsgLst()->setProgressHandler($progressHandler);
}

sub _executeInternal {
    my ($self) = @_;
    my $message = $self->getMsgLst()->addProgressMessage($self->getExecutionName() . '...');
    my $saveContext = $self->setMsgLstContext([$message->getSubMsgLst()]);
    $self->getMsgLst()->getProgressHandler()->incrementIndentationDepth();
# Execute the resident hdblcm after the validation
# If we return from this function, it means that something went wrong
    $self->_executeExternalHdblcm($message, $saveContext);

    $message->endMessage(undef, "Failed to start external hdblcm");
    $self->getMsgLst()->getProgressHandler()->decrementIndentationDepth();
    $self->getStatus()->_setErrorState();
    $self->setMsgLstContext($saveContext);
    return 0;
}

sub _executeExternalHdblcm {
    my ($self, $progressMessage, $saveContext) = @_;
    my $configuration = $self->_getConfiguration();
    my $app = $self->_getApp();
    if (!defined($app)) {
        $self->getErrMsgLst()->addError("Failed to execute non-resident hdblcm: Couldn't fetch the app from the context");
        return 0;
    }
# We don't need to save the context of the config here
# because we won't need it anymore - either we'll execute the external hdblcm
# or we will fail here

# Do not use the valud of the ExternalExecutablePath just in case someone finds a way to change its value
# from outside :)
    my $executablePath = $configuration->getExternalHdblcmPath();

    $self->getMsgLst()->addProgressMessage("Delegating execution to '$executablePath'...");
    $progressMessage->endMessage();
    $self->setMsgLstContext($saveContext);
    $self->_writeLogs($app);

    exec($executablePath, @{$self->_buildHdblcmArgs($configuration, $app)});
# If we've come to this point, the exec call has failed
    $self->getErrMsgLst()->addError("Execution of external hdblcm executable has failed: $!");
    return 0;
}

sub _getApp {
    my ($self) = @_;
    my $appContext = LCM::App::ApplicationContext->getInstance();
    return $appContext ? $appContext->getApplication() : undef;
}

sub _writeLogs {
    my ($self, $app) = @_;
    my $errorMessage = "Failed to write resident hdblcm logs";
    LCM::DevelopmentTrace::RemoveTempDevelopmentTrace();

    $app->handleExecutorMsgLst();

    my $log = $app->{log};
    if (defined($log)) {
        $log->setIndicateLocationOnStdout(0);
        if(!$log->writeLogs()) {
            $self->getErrMsgLst()->addError("$errorMessage: Couldn't write the logs on the filesystem");
            return 0;
        }
    }
    return 1;
}

sub _buildHdblcmArgs {
    my ($self, $configuration, $app) = @_;
    my $sid = $configuration->getSID();
    my $arguments = $app->getCLIParser()->getLeftoverArguments();

    push @{$arguments}, "--read_password_from_stdin=xml" if $configuration->isReadingPasswordsFromXml();
    push @{$arguments}, "--batch" if $app->isBatchMode();
    push @{$arguments}, "--action=update";
    push @{$arguments}, "--sid=$sid";
    push @{$arguments}, "--instlog_dir=$gLogDir";
    push @{$arguments}, "--verify_signature=".($self->_shouldVerifySignature($configuration));
    push @{$arguments}, "--bootstrap=1";

# Add the --ignore switches
    push @{$arguments}, @{$self->_buildIgnoreParams($configuration)};
# Add the --timeouts values
    push @{$arguments}, @{$self->_buildTimeoutParams($configuration)};
# Add the component_root, component_dirs and component_medium params
    push @{$arguments}, @{$self->_buildComponentPathParams($configuration, $app)};

    return $arguments;
}

sub _shouldVerifySignature {
    my ($self, $configuration) = @_;
    my $installer = $configuration->getComponentManager()->getComponentByKeyName($gKeynameInstaller);
    my $signature = $installer ? $installer->getSignature() : undef;
    my $isInstallerSecure = $signature ? $signature->isValid() : 0;
    return $configuration->getValue('VerifySignature') ? $isInstallerSecure : 0;
}

sub _buildComponentPathParams {
    my ($self, $configuration, $app) = @_;
    my $callerDir = $app->getAbsPath();
    my $pathParams = [];

    for my $paramId ('DvdPath', 'ComponentFsRoot') {
        my $value = $configuration->getValue($paramId);
        next if !$value;

        my $absolutePath = $self->_toAbsolutePath($callerDir, $value);
        push @{$pathParams}, ($configuration->getOpt($paramId)."=$absolutePath");
    }

    my $componentDirs = $self->_buildComponentDirsParam($configuration, $callerDir);
    push @{$pathParams}, $componentDirs if $componentDirs;

    return $pathParams;
}

sub _buildComponentDirsParam {
    my ($self, $configuration, $callerDir) = @_;
    my $componentDirs = $configuration->getValue('ComponentDirs');
    return undef if !$componentDirs;

    my @absoluteDirs = ();
    for my $dir (split ',', $componentDirs) {
        my $absoluteDir = $self->_toAbsolutePath($callerDir, $dir);
        push @absoluteDirs, $absoluteDir;
    }
    return @absoluteDirs ? ($configuration->getOpt('ComponentDirs')."=".(join ',', @absoluteDirs)) : undef;
}

sub _buildTimeoutParams {
    my ($self, $configuration) = @_;
    my @timeouts = ();
    for my $timeoutArg (@{$configuration->getTimeoutValues()}) {
        my $timeoutValue = $configuration->getTimeout($timeoutArg);
        push @timeouts, "$timeoutArg:$timeoutValue" if $timeoutValue;
    }
    return @timeouts ? [ "--timeouts=".(join ',', @timeouts) ] : [];
}

sub _buildIgnoreParams {
    my ($self, $configuration) = @_;
    my @ignores = ();
    for my $ignoreValue (@{$configuration->getIgnoreValues()}) {
        if ($configuration->getIgnore($ignoreValue)) {
            push @ignores, $ignoreValue;
        }
    }
    return @ignores ? [ "--ignore=".(join ',', @ignores) ] : [];
}

sub _toAbsolutePath {
    my ($self, $callerDir, $path) = @_;
    my $pattern = $isWin ? qr/^([a-zA-Z]:)?\\/ : qr/^\//;
    my $result = $path =~ $pattern ? $path : MyRealPath(File::Spec->catfile($callerDir, $path));
    return $result;
}

1;