package LCM::TaskHandler;

use strict;
use LCM::App::ConsoleUtils ':all';
use SDB::Install::Tools qw (askConfirmation);
use LCM::ComponentProgressHandler;
use LCM::App::ApplicationContext;

use base qw (SDB::Install::Base);

sub new {
	my ($class, $configuration, $executor, $commandLineArgumentsParser, $systemInformation, $console) = @_;
	my $self = {};
	bless ($self, $class);
	$self->_setConfiguration($configuration);
	$self->_setExecutor($executor);
	$self->_setCommandLineArgumentsParser($commandLineArgumentsParser);
	$self->_setSystemInformation($systemInformation);
	$self->_setConsole($console);

	return $self;
}

sub process {
	my ($self, $batchMode) = @_;
	my $configuration = $self->_getConfiguration();
	if(defined $configuration){
		$self->getMsgLst()->addMessage(undef, $configuration->getMsgLst());
		$self->getErrMsgLst()->addMessage(undef, $configuration->getErrMsgLst());
		$configuration->setMsgLstContext($self->getMsgLstContext()) ;
	}

	my $executor = $self->_getExecutor();
	if(!defined $executor){
		return;
	}

	$self->_setExecutorProgressHandler($executor);
	if ($executor->getCLIMetadata()->requiresInput()) {
		if ( !$configuration->InitDefaults() ) {
			$self->setErrorMessage(undef, $configuration->getErrMsgLst());
			return undef;
		}

		if ( !$self->collectInput($batchMode,0) ) {
			return undef;
		}

		if ( !$configuration->CheckParams(1) ){
			$self->setErrorMessage( undef, $configuration->getErrMsgLst() );
			return undef;
		}
	}

	$configuration->dumpConfigFile('hdblcm');

	if ($executor->getCLIMetadata()->requiresSummary()) {
		my $summaryTree = $configuration->createSummaryTree();
		printSummary( $summaryTree );

		my $warnings    = $configuration->getWarningList();
		if (@$warnings) {
			print "\n";
		}
		foreach my $currWng (@$warnings) {
			print "Note: $currWng\n";
		}
		my $infoMsg    = $configuration->{infoMsgs}?$configuration->{infoMsgs}:[];
        if (@$infoMsg) {
            print "\n";
        }
        foreach my $currMsg (@$infoMsg) {
            print "Info: $currMsg\n";
        }
	}

	if ( !$batchMode and $executor->getCLIMetadata()->requiresConfirmation()) {
		my $messageBeforeConfirm = $executor->getCLIMetadata()->getMessageBeforeConfirm();
		print "\n".$messageBeforeConfirm . "\n" if(defined $messageBeforeConfirm);
        print "\n";
		my $userConfirms = askUserConfirmation();
		print "\n";
		if ( !$userConfirms ) {
			$self->setErrorMessage('Procedure was canceled by the user');
			return undef;
		}
	}

	$self->_setExecutionStarted(1);
	return $self->execute();
}

sub isExecutionStarted {
	my ($self) = @_;
	return $self->{_isStarted};
}

sub _setExecutionStarted {
	my ($self, $isStarted) = @_;
	$self->{_isStarted} = $isStarted;
}

sub _setExecutorProgressHandler {
	my ($self, $executor) = @_;

	my $executorMsgLst = $executor->getMsgLst();
	my $executorErrMsgLst = $executor->getErrMsgLst();

	unless($executorMsgLst->getProgressHandler()) {
		$executorMsgLst->setProgressHandler($self->getMsgLst()->getProgressHandler());
	}

	unless($executorErrMsgLst->getProgressHandler()) {
		$executorErrMsgLst->setProgressHandler($self->getErrMsgLst()->getProgressHandler());
	}

	return 1;
}

sub collectInput {
	my ($self, $batchMode, $optionsPassThrough) = @_;
	my $configuration = $self->_getConfiguration();
	my $returnCode = $self->_getCommandLineArgumentsParser()->parse($optionsPassThrough);
	if (!defined $returnCode) {
		return undef;
	}
	if (!$self->_checkSystemRequierements()){
		return undef;
	}
	$returnCode = $configuration->CheckParams ($batchMode);

	if ( $batchMode ) {
		if ( !$returnCode ) {
			if ( defined $returnCode ) {
				$self->setErrorMessage( "Running in batch mode", $configuration->getErrMsgLst() );
			} else {
				$self->setErrorMessage( 'Configuration error (batch mode):', $configuration->getErrMsgLst() );
			}
			return undef;
		}
	} else {
		if ( !$returnCode && $configuration->ErrorState() ) {
			$self->setErrorMessage( 'Configuration error:', $configuration->getErrMsgLst() );
			return undef;
		}

		my $console = $self->_getConsole();

		if ( !defined $console->ConfigureInteractive($configuration) ) {
			$self->setErrorMessage( 'Configuration error: ', $console->getErrMsgLst() );
			return undef;
		}
	}
	return $returnCode;
}

sub askUserConfirmation{
	return askConfirmation();
}

sub _checkSystemRequierements{
	my ($self) = @_;
	my $configuration = $self->_getConfiguration();
	my $systemInformation = $self->_getSystemInformation();

    if (defined $configuration){
    	my $oldcontext = $configuration->setMsgLstContext($self->getMsgLstContext());
	    if (!defined $configuration->checkSystemRequirements($systemInformation)){
	    	$configuration->setMsgLstContext($oldcontext);
			$self->setErrorMessage (undef, $configuration->getErrMsgLst());
			return undef;
		}
		$configuration->setMsgLstContext($oldcontext);
    }

	return 1;
}

sub _getConfiguration{
	return $_[0]->{_configuration};
}

sub _setConfiguration{
	my ($self, $configuration) = @_;
	$self->{_configuration} = $configuration;
}

sub _getExecutor{
	return $_[0]->{_executor};
}

sub _setExecutor{
	my ($self, $executor) = @_;
	$self->{_executor} = $executor;
}

sub _getCommandLineArgumentsParser{
	return $_[0]->{_commandLineArgumentsParser};
}

sub _setCommandLineArgumentsParser{
	my ($self, $commandLineArgumentsParser) = @_;
	$self->{_commandLineArgumentsParser} = $commandLineArgumentsParser;
}

sub _getSystemInformation{
	return $_[0]->{_systemInformation};
}

sub _setSystemInformation{
	my ($self, $systemInformation) = @_;
	$self->{_systemInformation} = $systemInformation;
}

sub _getConsole{
	return $_[0]->{_console};
}

sub _setConsole{
	my ($self, $console) = @_;
	$self->{_console} = $console;
}

sub execute{
    my $self = shift();
    my $executor = $self->_getExecutor();
    $executor->execute();

    if($executor->getStatus()->isInFinishedState()){
    	return 1;
    } elsif($executor->getStatus()->isInAbortedState()){
        return 2;
    }

    return undef;
}

1;