package SDB::Common::Utils;

use strict;
use parent 'Exporter';
use SAPDB::Install::Hostname;
use SDB::Common::BuiltIn;
use SDB::Install::SysVars qw($isWin);
use SDB::Install::Globals qw($gSapsysGroupName);
use utf8;
use MIME::Base64;
use Encode;

our @EXPORT = qw(createXSSpaceSAPUserName
                 createXSSpaceProdUserName
                 createSysAdminUserName
                 createXMLParser
                 toBase64
                 trim
                 getSidcryptName
                 checkOsUserID
                 checkOsGroupID
                 createUsername
                 createGroupname
                 isValidUnquotedAbsolutePath);

sub trim {
    my $string = shift();
    $string =~ s/^\s+//;
    $string =~ s/\s+$//;
    return $string;
}

# Only for Linux for now
sub isValidUnquotedAbsolutePath {
    my ($absolutePath) = @_;
    my $isPathValid = ($absolutePath =~ /^\/$|^(\/[^\/\0]+)+(\/)?$/);
    return $isPathValid ? 1 : 0;
}

sub toBase64 {
    my ($string) = @_;
    if (utf8::is_utf8($string)) {
        $string = Encode::encode_utf8($string);
    }
    return encode_base64($string,'');
}

sub createXSSpaceSAPUserName {
    my ($sid) = @_;

    return defined($sid) ? sprintf('sap%sxsa', lc($sid)) : undef;
}

sub createXSSpaceProdUserName {
    my ($sid) = @_;

    return defined($sid) ? sprintf('%sxsa', lc($sid)) : undef;
}

sub createSysAdminUserName {
    my ($sid, $domainParam) = @_;
    my $name = sprintf('%sadm', lc($sid));
    my $domain = $isWin && !defined($domainParam) ? hostname() : $domainParam;

    return $isWin ? sprintf('%s\\%s', $domain, $name) : $name;
}

sub getSidcryptName {
    my ($sid) = @_;
    return undef if (! defined $sid);
    return lc($sid).'crypt';
}

sub createUsername {
    my ($paramID, $sid) = @_;

    if ($paramID eq 'UID'){
        return createSysAdminUserName($sid);
    } elsif ($paramID eq 'LSSUserID') {
        return getSidcryptName($sid);
    } elsif ($paramID eq 'XSSpaceUserIdSAP') {
        return createXSSpaceSAPUserName($sid);
    } elsif($paramID eq 'XSSpaceUserIdProd'){
        return createXSSpaceProdUserName($sid);
    }
    return $paramID;
}

sub createGroupname {
    my ($paramID, $sid) = @_;

    if ($paramID eq 'GID'){
        return $gSapsysGroupName;
    } elsif ($paramID eq 'LSSGroupID') {
        return getSidcryptName($sid);
    }
    return $paramID;
}

sub createXMLParser {
    require XML::LibXML;
    my $parser = XML::LibXML->new({
        load_ext_dtd => 0,
        expand_entities => 0,
        ext_ent_handler => sub {},
    });

    return $parser;
}

sub checkOsUserID {
    my ($config, $parameterId, $osUserName, $value, $canBeInSapsys, $noLocalHostCheck, $specificHosts) = @_;
    my $userName = SDB::Common::BuiltIn->get_instance()->getpwuid($value);
    my $sapsysGid = SDB::Common::BuiltIn->get_instance()->getgrnam('sapsys');
    my @hostsWithExistingUser = ();

    $config->clearParameterWarnings($parameterId);
    if ($value !~ /^\d+$/){
        $config->appendErrorMessage("User ID '$value' has to be a decimal number");
        return 0;
    }
    if ($value < 1){
        $config->appendErrorMessage("User ID '$value' has to be greater than zero");
        return 0;
    }
    if (defined($userName) && ($userName ne $osUserName)) {
        my $errMsg = "User ID '$value' is already in use by user '$userName'";
        my $freeID = $config->getFreeOsUserID(undef, $config->getReservedUserIDs());
            if (defined $freeID) {
                $errMsg .= " (available ID: $freeID)";
        }
        $config->appendErrorMessage($errMsg);
        return 0;
    }
    my $user = new SDB::Install::User($osUserName);
    my $localHost = _getLocalHost($config);
    if (!$noLocalHostCheck && $user->exists()) {
        if($user->id() != $value){
            $config->appendErrorMessage(_constructWrongIDMessage($value, $user->id(), $localHost, 'User', $osUserName));
            return 0;
        }
        push(@hostsWithExistingUser, $localHost);
    }
    if(!$noLocalHostCheck && !$canBeInSapsys && $user->exists() && $user->gid() == $sapsysGid){
        $config->getErrMsgLst()->addError("User '$osUserName' belongs to group 'sapsys' on the local host");
        return 0;
    }
    my $remoteUIDs = $config->{remoteUids} || {};
    for my $userId (keys(%{$remoteUIDs})){
        my $userData = $remoteUIDs->{$userId};
        for my $entry (@{$userData}){
            my ($host, $remoteUser, $group) = @{$entry};

            my $shouldCheckHost = !defined($specificHosts) || grep { $host eq $_ } @{$specificHosts};
            next if !$shouldCheckHost;

            if($osUserName eq $remoteUser){
                if($value != $userId){
                    $config->appendErrorMessage(_constructWrongIDMessage($value, $userId, $host, 'User', $osUserName));
                    return 0;
                }
                if(defined($sapsysGid) && $sapsysGid == $group && !$canBeInSapsys){
                    $config->getErrMsgLst()->addError("User '$osUserName' belongs to group 'sapsys' on host '$host'");
                    return 0;
                }
                push(@hostsWithExistingUser, $host);
            } elsif($value == $userId) {
                my $errMsg = "User ID '$value' is already in use by user '$remoteUser' on host '$host'";
                my $freeID = $config->getFreeOsUserID(undef, $config->getReservedUserIDs());
                if (defined $freeID) {
                    $errMsg .= " (available ID: $freeID)";
                }
                $config->appendErrorMessage($errMsg);
                return 0;
            }
        }
    }
    if(scalar(@hostsWithExistingUser) > 0){
        my $prefix = sprintf('User \'%s\' already exists on host(s) %s.', $osUserName, join(', ', @hostsWithExistingUser));
        my $middle = 'No changes to its attributes and configurations will be made. Verify that the user is correct.';
        my $xsaSuffix = _isXSAUser($osUserName) ? sprintf('Refer to SAP Note 2243156 for information how to properly configure user \'%s\'.', $osUserName) : '';
        $config->addParameterWarning($parameterId, sprintf('%s %s %s', $prefix, $middle, $xsaSuffix));
    }
    return 1;
}

sub checkOsGroupID {
    my ($config, $paramName, $wantedGroupName, $wantedGroupID, $noLocalHostCheck, $specificHosts) = @_;

    $config->clearParameterWarnings($paramName);
    if (!defined $wantedGroupID || $wantedGroupID eq ''){
        $config->setErrorMessage (($paramName ? $paramName : 'Group ID')." is empty");
        return 0;
    }
    if ($wantedGroupID =~ /\D/){
        $config->setErrorMessage (($paramName ? $paramName : 'Group ID')." '$wantedGroupID' has to be a decimal number");
        return 0;
    }
    if ($wantedGroupID < 1){
        $config->setErrorMessage (($paramName ? $paramName : 'Group ID')." '$wantedGroupID' has to be greater than zero");
        return 0;
    }

    my $localHost = _getLocalHost($config);
    my $existingGroupID = getgrnam ($wantedGroupName);
    if (defined $existingGroupID && !$noLocalHostCheck) {
        if ($existingGroupID != $wantedGroupID) {
            $config->setErrorMessage(_constructWrongIDMessage($wantedGroupID, $existingGroupID, $localHost, 'Group', $wantedGroupName));
            return 0;
        }
    }
    my @hostsWithExistingGroup = ();
    my $existingGroupName = getgrgid($wantedGroupID);
    if (defined $existingGroupName && !$noLocalHostCheck) {
        if($existingGroupName ne $wantedGroupName) {
            my $errMsg = "Group ID '$wantedGroupID' for '$wantedGroupName' is already in use by group '$existingGroupName'";
            my $freeID = $config->getFreeGroupID();
            if (defined $freeID) {
                $errMsg .= " (available ID: $freeID)";
            }
            $config->setErrorMessage($errMsg);
            return 0;
        }
        push(@hostsWithExistingGroup, $localHost);
    }

    my $remoteGIDs = $config->{remoteGids} || {};
    for my $groupId (keys(%{$remoteGIDs})){
        my $userData = $remoteGIDs->{$groupId};
        for my $entry (@{$userData}){
            my ($host, $remoteGroup) = @{$entry};

            my $shouldCheckHost = !defined($specificHosts) || grep { $host eq $_ } @{$specificHosts};
            next if !$shouldCheckHost;

            if($wantedGroupName eq $remoteGroup){
                if($wantedGroupID != $groupId){
                    $config->setErrorMessage(_constructWrongIDMessage($wantedGroupID, $groupId, $host, 'Group', $wantedGroupName));
                    return 0;
                }
                push(@hostsWithExistingGroup, $host);
            } elsif($wantedGroupID == $groupId) {
                $config->setErrorMessage("Group ID '$wantedGroupID' for '$wantedGroupName' is already in use by group '$remoteGroup' on host '$host'");
                return 0;
            }
        }
    }
    if(scalar(@hostsWithExistingGroup) > 0){
        my $warning = sprintf('Group \'%s\' already exists on host(s) %s. No changes to its attributes and configurations will be made. Verify that the group is correct. ', $wantedGroupName, join(', ', @hostsWithExistingGroup));
        $config->addParameterWarning($paramName, sprintf('%s', $warning)) if(defined $paramName);
    }

    return 1;
}

sub _getLocalHost {
    my ($config) = @_;
    my $instance = $config->getOwnInstance();
    my $hostname = $config->getValue('HostName');
    if (!$hostname && defined($instance)) {
        $hostname = $instance->get_host();
    }
    return $hostname || 'localhost';
}

sub _constructWrongIDMessage {
    my ($expectedID,    # ID, which the entity is expected to have
        $existingID,    # ID, which the entity actually has
        $host,          # Hostname, where the ID has wrong value
        $entity,        # 'Group' or 'User'
        $entityName     # '<sid>crypt', 'sapsys', etc..
        ) = @_;
    my $hostString = $host eq 'localhost' ? 'the local host' : "host '$host'";
    my $msgTemplate = "%s '%s' has wrong ID '%s' on %s (expected '%s')";
    return sprintf($msgTemplate, $entity, $entityName, $existingID, $hostString, $expectedID);
}

sub _isXSAUser {
    my ($userName) = @_;
    if($userName =~ /^sap[a-z][a-z\d]{2}xsa$/){
        return 1;
    } elsif ($userName =~ /^[a-z][a-z\d]{2}xsa$/){
        return 1;
    }
    return 0;
}

1;
