#!/usr/bin/perl
#
# Desc: add a new host to the HDB system

package SDB::Install::App::Console::AddHost;

use base SDB::Install::App::Console;
use SDB::Install::SAPSystem;
use SDB::Install::Globals qw ($gLogDir
                              $gProductNameEngine
                              $gProductNameInstaller
                              $gSapsysGroupName
                              $gShortProductNameEngine
                              $gShortProductNameXS2);
use SDB::Install::SysVars qw ($isWin $path_separator);
use Getopt::Long;
use SDB::Install::System qw (copy_file
                             copy_tree
                             copySelectedFiles
                             deltree
                             getDirectoryFilenames
                             makedir);
use SAPDB::Install::Hostname;
use SDB::Install::Configuration::AddHost;
use SDB::Install::Log;
use SDB::Install::Tools qw (askConfirmation);
use IO::Dir;

use strict;

our $programVers = '0025';


#-------------------------------------------------------------------------------
# Constructor
sub new {
    my $self = shift->SUPER::new ();
    return $self;
}


#-------------------------------------------------------------------------------
# Returns the name of this program

sub getProgramName {
    return 'hdbaddhost';
}


#-------------------------------------------------------------------------------
# Initializes this application
# Without parameters
# Returns int 1

sub InitApp{
    my ($self) = @_;
    $self->SUPER::InitApp();
    return 1;
}


#-------------------------------------------------------------------------------
sub init{
    my ($self) = @_;
    my $rc = 1;

    if (!$self->checkIsAdmin($self->getProgramName())){
        return undef;
    }

    my $instconfig = $self->getInstconfig();

    my $oldcontext = $instconfig->setMsgLstContext($self->getMsgLstContext());
    if (!$self->isHelp() && !defined $instconfig->checkSystemRequirements ($self->GetSysInfo())){
        $instconfig->setMsgLstContext($oldcontext);
        $self->setErrorMessage ("Checking system requirements failed ", $instconfig->getErrMsgLst ());
        if ($instconfig->getIgnore ('check_hardware')){
            $self->getMsgLst ()->addMessage ("Ignoring error due to command line switch '--ignore=check_hardware'");
        }
        else{
            return undef;
        }
    }
    $instconfig->setMsgLstContext($oldcontext);

    my $msglst = $self->getMsgLst ();
    $self->warnIfShouldWarnIfCalledStandalone();
    my $caption = "$gProductNameInstaller - $gShortProductNameEngine Addhost " .$self->GetInstaller->{version};
    $msglst->addProgressMessage ("\n\n".$caption . "\n" . ('*' x length ($caption)) . "\n\n");
    $self->addStartDateMsg($msglst, $self->getProgramName() . "  $programVers");
    
    $msglst->appendMsgLst($instconfig->getMsgLst());
    $instconfig->resetMsgLstContext ();

    my $msg = $msglst->addMessage ("Checking configuration");
    $msg->setEndTag ('Configuration Check');
    $rc = $instconfig->CheckParams ($self->{batch_mode});

    my $cfgDumpRc = $self->handleOptionDumpConfigFileTemplate();
    if (!defined $cfgDumpRc){
        $self->{return} = 1;
        return undef;
    }
    if ($cfgDumpRc){
        $self->{return} = 0;
        return 1;
    }

    if($self->{batch_mode}){
        if (!$rc){
            if (defined $rc){
                $self->setErrorMessage ("Running in batch mode", $instconfig->getErrMsgLst ());
                #$self->PrintUsage ();
            }
            else{
                $self->setErrorMessage ('Configuration error (batch mode):', $instconfig->getErrMsgLst ());
            }
            $msg->getSubMsgLst ()->appendMsgLst ($instconfig->getMsgLst ());
            return undef;
        }
    }
    else{
        if (!$rc && $instconfig->ErrorState ()){
            $self->ShowErrorMsg ('Configuration error:',$instconfig->getErrMsgLst ());
        }
        if (!defined $self->ConfigureInteractive ($instconfig)){
            return undef;
        }   
    }

    if (!$instconfig->tryAddHostAutoAddXS2()) {
        $self->setErrorMessage ("Autoadd $gShortProductNameXS2", $instconfig->getErrMsgLst ());
        return undef;
    }

    $msg->getSubMsgLst ()->appendMsgLst ($instconfig->getMsgLst ());
    $instconfig->resetMsgLstContext();

    my $hostname = $instconfig->getValue('HostName');

    if ($instconfig->getStep() > 0) {
        $msglst->addProgressMessage
            ("Resuming installation of additional host '$hostname' at step '"
            . $instconfig->pers_getstepname($instconfig->getStep()) . "'.");
    }

    if (!$self->defineLog('addhost',
                          $self->getProgramName(),
                          $instconfig->isCheckOnly(),
                          $instconfig->getSID())) {
        return undef;
    }

    $instconfig->dumpConfigFile($self->getProgramName());

    $instconfig->displayParameterSummary();
    $self->printWarnings();
    print "\n";
    $msg->getSubMsgLst()->appendMsgLst($instconfig->getMsgLst());
    $instconfig->resetMsgLstContext();

    if (!$self->{batch_mode} && !$instconfig->isCheckOnly()
                       && !$self->{options}->{noprompt} && !askConfirmation()) {
        $msglst->addProgressMessage('Cancelled');
        $self->{return} = 0;
        return 1;
    }

    my $existingLogFiles = getDirectoryFilenames($gLogDir);

    $rc = $self->executeAddHost($msg, $msglst);

    if (defined $self->{options}->{action_id}) {

        my $sapSys   = $instconfig->getSAPSystem();
        my $instance = $instconfig->getOwnInstance();

        if (defined $sapSys && defined $instance && defined $hostname) {

            copySelectedFiles($gLogDir,
                         $instance->get_hostNameTraceDir($hostname),
                         $msglst,
                         $self->{options}->{action_id},
                         $existingLogFiles,
                         $sapSys->getUID(),
                         $sapSys->getGID());
        }
    }

    return $rc;
}


#-------------------------------------------------------------------------------
sub executeAddHost {
    my ($self, $msg, $msglst) = @_;

    my $rc         = 1;
    my $instconfig = $self->getInstconfig();
    my $sid        = $instconfig->getSID();

    $msg = $msglst->addMessage ("Checking system requirements");
    $msg->setEndTag ('System Requirements Check');
    my $sapSys = $instconfig->getSAPSystem ();
    if (!defined $instconfig->checkSystemRequirementsAfterConfig ($self->GetSysInfo(), $sapSys->getManifest ())){
        $self->setErrorMessage ("Checking system requirements failed ", $instconfig->getErrMsgLst ());
        if ($instconfig->getIgnore ('check_hardware')){
            $msglst->addMessage ("Ignoring error due to command line switch '--ignore=check_hardware'");
        }
        else{
             $msg->getSubMsgLst ()->appendMsgLst ($instconfig->getMsgLst ());
            return undef;
        }
    }
    $msg->getSubMsgLst ()->appendMsgLst ($instconfig->getMsgLst ());

    my $errlst = new SDB::Install::MsgLst ();
    
    my $sapmnt = $instconfig->getValue ('Target');

    my $hostname = $instconfig->getValue ('HostName');

    my $globalSapSid = "$sapmnt$path_separator$sid"; 
    
    if (!-d $globalSapSid){
        $self->setErrorMessage ("Configuration error: directory '$globalSapSid' not found");
        return undef;
    }
    
    
    if (!$sapSys->hasNewDB){
        $self->setErrorMessage ("SAP system has no $gProductNameEngine instance", $sapSys->getErrMsgLst ());
        return undef;
    }

    my $uid = undef;
    my $gid = undef;
    if (!$isWin){
        my $cfg = $sapSys->getUserConfig ();
        if (!defined $cfg){
            $self->setErrorMessage (undef, $sapSys->getErrMsgLst ());
            return undef;
        }
        $uid = $cfg->{sidadm_id};
        $gid = getgrnam ($gSapsysGroupName);
        
        my $error = 0;
        if (defined $gid && $gid != $cfg->{sapsys_groupid}){
            $self->appendErrorMessage ("Group '$gSapsysGroupName' has wrong id $gid (expected: $cfg->{sapsys_groupid})!");
            $error = 1;
        }
        my $sidadm_id = $sapSys->getUser()->uidUx();
        my $sidadmName = $sapSys->getUser()->getSidAdmName();
        if (defined $sidadm_id && $sidadm_id != $cfg->{sidadm_id}){
            $self->appendErrorMessage ("User $sidadmName has wrong id ($sidadm_id, expected $cfg->{sidadm_id} from master host)!");
            $error = 1;
        }
        if (!defined $sidadm_id){
            my $username = getpwuid ($cfg->{sidadm_id});
            if ($username){
                $self->appendErrorMessage ("Required uid for user $sidadmName is already in use by user $username");
                $error = 1;
            }
        }
        if ($error){
            return undef;
        }
    }
    $self->addLogToTraceDir($sapSys,
                            $self->getProgramName(),
                            0, # not check only
                            $hostname, $uid, $gid);
    $sapSys->SetProgressHandler ($msglst->getProgressHandler ());

    $msg = $msglst->addProgressMessage ("Checking master nameserver");

    $msg->setEndTag ('Master nameserver check');
    
    my $instance = $sapSys->getNewDBInstances ()->[0];
    
    $instance->SetProgressHandler ($msglst->getProgressHandler ());

    my $requiredActionMsgLsg = new SDB::Install::MsgLst ();

    $rc = $self->_isNameServerAlive ($instance, $msg->getSubMsgLst(), $requiredActionMsgLsg);

    $msg->endMessage ();

    if (!defined $rc){
        $self->setErrorMessage ("Cannot check master nameserver",
            $instance->getErrMsgLst() );
        return undef;
    }
    
    if (!$rc){
        $self->setErrorMessage ('AddHost not possible.', $requiredActionMsgLsg);
        return undef;
    }

    my $infoPrefix = ($instconfig->isCheckOnly())
               ? "Checking additional host '$hostname' of"
               : "Add new host '$hostname' to";

    $msg = $msglst->addProgressMessage ("$infoPrefix $gProductNameEngine system '$sid'");
    
    $msg->setEndTag ('Add host');
    
    my $nr = $instance->get_nr();
    
    if (($instconfig->getStep() == 0) && ExistsInstanceNumber ($nr)){
        $self->setErrorMessage ("Instance number '$nr' is already occupied");
        $msglst->addMessage ('Occupied numbers are: ' . join (', ', @{OccupiedInstanceNumbers()}));
        return undef;
    }

    $msg->getSubMsgLst()->appendMsgLst($instconfig->getMsgLst());
    $instconfig->resetMsgLstContext();

    if ($instconfig->isCheckOnly()) {
        return 1;
    }

    $msg->getSubMsgLst()->appendMsgLst($instconfig->getMsgLst());
    $instconfig->resetMsgLstContext();

    if ($rc) {
        $sapSys->resetMsgLstContext ();
        $rc = $sapSys->addHost ($hostname,
                                      $instconfig->getValue ('Password'),
                                      $instconfig, $self->GetSysInfo());
        if (!$rc) {
            $self->setErrorMessage (undef, $sapSys->getErrMsgLst ());
        }

        $msg->getSubMsgLst ()->appendMsgLst($sapSys->getMsgLst());
        $sapSys->resetMsgLstContext();
    }

    if ($rc) {
        $rc = $instconfig->tryAddHostLoadXS2InitialContent();

        if (!$rc) {
            $self->setErrorMessage(undef, $instconfig->getErrMsgLst());
        }
        $msg->getSubMsgLst()->appendMsgLst($instconfig->getMsgLst());
        $instconfig->resetMsgLstContext();
    }

    return $rc;
}


#-------------------------------------------------------------------------------
#
# Workaround to get layered configuration enabled during addHost.
# Layered configuration is enabled if directory 
# $(DIR_GLOBAL)/trex/custom/config exists.
# But 'sapgparam (DIR_GLOBAL)' returns '/usr/sap/...'
# Setting DIR_INSTALL in profile solves this problem.
#
# Better workaround is to touch this directory locally and remove it afterwards.
#

sub fakeInstallation{
    my ($self, $instance, $msglst) = @_;
    my $instconfig = $self->getInstconfig();

    my $sid = $instconfig->getSID();
    my @baseList = ('usr', 'sap', $sid);
    my @list = (@baseList, 'SYS', 'global', 'hdb', 'custom');
    
    my $path = $isWin ? ((-d '\\\\' . hostname() . $path_separator . 'saploc')
                         ? $instconfig->getValue ('Drive') : 'C:')
                         : '';

    my $auxExtension       = '.~aux' . $$;
    my $dirHDB             = 'HDB' . $instance->get_nr();
    $self->{_usrSapSidHDB} = join ($path_separator, $path, @baseList, $dirHDB);
    $self->{_usrSapSidSYS} = join ($path_separator, $path, @baseList, 'SYS');

    if (-d $self->{_usrSapSidHDB}) {
        $self->{_auxUsrSapSidHDB} = $self->{_usrSapSidHDB} . $auxExtension;

        if (!CORE::rename($self->{_usrSapSidHDB}, $self->{_auxUsrSapSidHDB})) {
            $self->setErrorMessage ("Cannot rename '$self->{_usrSapSidHDB}'"
                                    . " => '$self->{_auxUsrSapSidHDB}': $!");
            return undef;
        }
    }

    if (-d $self->{_usrSapSidSYS}) {
        $self->{_auxUsrSapSidSYS} = $self->{_usrSapSidSYS} . $auxExtension;

        if (!CORE::rename($self->{_usrSapSidSYS}, $self->{_auxUsrSapSidSYS})) {
            $self->setErrorMessage ("Cannot rename '$self->{_usrSapSidSYS}'"
                                    . " => '$self->{_auxUsrSapSidSYS}': $!");
            return undef;
        }
    }

    my $fs_join_root = $path;
    my ($i,$x);
    my $exists = 1;
    my $target = join ($path_separator,$fs_join_root,@list,'config');
    my @secBaseList = @list;
    pop @secBaseList;
    my $securityTarget = join ($path_separator,$fs_join_root,@secBaseList,'security');
    pop @secBaseList;
    my $sysSecurityTarget = join ($path_separator,$fs_join_root,@secBaseList,'security');
    foreach $i (0..$#list){
        $path .= $path_separator . $list[$i];
        $x = $i;
        if  (!-d $path){
            $x++;
            $exists = 0;
            last;
        }       
    }

    if ($exists){
        $msglst->addWarning ("Local system directory already exists\n");
        return '';
    }

    my $msg = $msglst->addMessage ('Creating local custom directory');
    my $cfg = {'mode' => 0755,'msg_lst' => $msg->getSubMsgLst ()->{msg_lst}};

    if (!defined makedir ($path . ($x == $#list ? '' : ($path_separator . join ($path_separator, @list[$x..$#list]))), $cfg)){
        $self->setErrorMessage ("Cannot create directory '".($path . ($x == $#list ? '' : ($path_separator . join ($path_separator, @list[$x..$#list]))))."'", $msg->getSubMsgLst ());
        return undef;
    }
    if ($isWin){
        $msg = $msglst->addMessage ('Copying ini files to local custom folder');
        if (!copy_tree ($instance->get_globalHdbCustomConfigDir(), $target, $msg->getSubMsgLst())){
            $self->setErrorMessage ("Cannot copy sytem ini files", $msg->getSubMsgLst());
        }
    }
    else{
        # create symlink /usr/sap/SID/SYS/global/hdb/custom/config =>
        # /sapmnt/SID/global/hdb/custom/config
        my $srcOfLink = $instance->get_globalHdbCustomConfigDir();
        $msglst->addMessage ("Creating symlink $target => $srcOfLink");
        if (!symlink ($srcOfLink, $target)){
            $self->setErrorMessage ("Cannot create symlink '$target': $!");
        }
        # create symlink /usr/sap/SID/SYS/global/hdb/security =>
        # /sapmnt/SID/global/hdb/security
        $srcOfLink = $instance->get_globalTrexDir() . '/security';
        $msglst->addMessage ("Creating symlink $securityTarget => $srcOfLink");
        if (!symlink ($srcOfLink, $securityTarget)){
            $self->setErrorMessage ("Cannot create symlink '$securityTarget': $!");
        }
        # create symlink /usr/sap/SID/SYS/global/security =>
        # /sapmnt/SID/global/security
        $srcOfLink = $instance->get_globalDir() . '/security';
        $msglst->addMessage ("Creating symlink $sysSecurityTarget => $srcOfLink");
        if (!symlink ($srcOfLink, $sysSecurityTarget)){
            $self->setErrorMessage ("Cannot create symlink '$sysSecurityTarget': $!");
        }
    }

    # DEFAULT.PFL
    #my $profileDir = join ($path_separator, $fs_join_root, @baseList, 'SYS', 'profile');
    #if (!-e $profileDir){
    #    delete $cfg->{msg_lst};
    #    if (!defined makedir ($profileDir, $cfg)){
    #        $self->setErrorMessage ("Cannot fake installation", $cfg);
    #    }
    #}
    #my $default_profile = (new SDB::Install::SAPProfiles ($instance->get_profileDir()))->get_DefaultProfile ();
    #symlink ($default_profile, $profileDir);

    $instance->{_host} = hostname ();
    my $usrsapsid = join ($path_separator, $fs_join_root, @baseList);
    $instance->resetMsgLstContext ();
    $instance->set_localUsrSapSidDir($usrsapsid);
    $instance->setMsgLstContext ([$msglst]);
    $instance->{_fsMan} = new SDB::Install::FsManager ($usrsapsid, $usrsapsid);
    $instance->{_fsMan}->set_instanceNr ($instance->get_nr ());
    $instance->createHostNameDirs ();
    my $instanceDir  = $instance->get_instanceDir();
    my $dir = new IO::Dir();
    if(!$dir->open($instanceDir)){
        $self->setErrorMessage ("Cannot open directory '$instanceDir': $!");
    }
    else{
        my ($ent,$srcOfLink,$link);
        while(defined ($ent = $dir->read())){
            if ($ent eq '.' || $ent eq '..'){
                next;
            }
            $srcOfLink = "$instanceDir/$ent";
            $link = "$self->{_usrSapSidHDB}/$ent";
            $msglst->addMessage ("Creating symlink $link => $srcOfLink");
            if (!symlink ($srcOfLink, $link)){
                $self->setErrorMessage ("Cannot create symlink '$link': $!");
            }
        }
    }
    $instance->{_fsMan} = undef;
    $instance->installSapprofileIni (0644);
    $instance->setMsgLstContext ([new SDB::Install::MsgLst()]);
    return $path;
}


sub _copyPingTraces{
    my ($self, $traceDir, $msglst) = @_;
    my $dir = new IO::Dir();
    my $msg = $msglst->addMessage("Copying ping trace files");
    my $submsglst = $msg->getSubMsgLst();
    if(!$dir->open($traceDir)){
        $submsglst->addWarning("Cannot open faked trace directory '$traceDir': $!");
        return 0;
    }
    my ($path, $stat, $ent);
    my $rc = 0;
    my $copyCfg = {};
    $submsglst->injectIntoConfigHash($copyCfg);
    require File::stat;
    while (defined ($ent = $dir->read())){
        if ($ent eq '.' || $ent eq '..'){
            next;
        }
        $path = $traceDir . $path_separator . $ent;
        $stat = File::stat::stat($path);
        if (defined $stat && -f $stat){
            $submsglst->addMessage("Copying '$ent'");
            if (!defined copy_file($path, $gLogDir, $copyCfg)){
                $rc = 0;
            }
        }
    }
    return $rc;
}

#-------------------------------------------------------------------------------
sub _isNameServerAlive{
    my ($self, $instance, $msglst, $requiredActionMsgLsg) = @_;

    if ($isWin){
        # not yet implemented for windows
        return 1;
    }

    my @nsMasters;

    if (!defined $msglst){
        $msglst = $self;
    }
    my $rc;
    my $originalHost = $instance->get_host ();
    my $msg = $msglst->addMessage ('Preparing tool environment');
    my $fakeDir = $self->fakeInstallation ($instance, $msg->getSubMsgLst ());
    my $ipcDir = join ($path_separator, '', 'var', 'lib', 'hdb', $instance->{_sid}, 'ipc');

    if (!defined $fakeDir){
        return undef;
    }
    $msg = $msglst->addMessage ('Checking whether master nameserver is alive');
    my $returnCode = 0;
    if (!defined $instance->pingMasterNameServer ($msg->getSubMsgLst())){
        $msg = $msglst->addMessage ('Master nameserver is not available');
        $requiredActionMsgLsg->appendMsg ($msg);
        $requiredActionMsgLsg->addMessage ("Start $gProductNameEngine on master host and run addHost again.")
    }
    else{
        $msglst->addMessage ('Master namesever is running');
        $returnCode = 1;
    }

    if ($returnCode){
        $msg = $msglst->addMessage ('Checking time difference to master host');
        $returnCode = 0;
        my $allowedOffset = 3 * 60;
        my $timeOffset = $instance->getTimeOffset ($msg->getSubMsgLst());
        if (!defined $timeOffset){
            $self->setErrorMessage ("Cannot get time difference", $msg->getSubMsgLst());
            $returnCode = undef;
        }
        elsif (abs ($timeOffset) > $allowedOffset){
            $msg = $self->setErrorMessage ("Time difference to master host is ${timeOffset}s. (> ${allowedOffset}s)");
            $requiredActionMsgLsg->appendMsg ($msg);
            $requiredActionMsgLsg->addMessage ("Please adjust system time!");
            $returnCode = 0;
        }
        else{
            if ($timeOffset){
                $msglst->addMessage ("Time difference (${timeOffset}s) is ok. (<= ${allowedOffset}s)");
            }
            else{
                $msglst->addMessage ('There is no time difference.');
            }
            $returnCode = 1;
        }

        if (!$returnCode && $self->getInstconfig()->getIgnore ('check_timediff')){
            $self->getMsgLst ()->addMessage ("Ignoring error due to command line switch '--ignore=check_timediff'");
            $self->resetError ();
            $returnCode = 1;
        }
    }

    $self->_copyPingTraces($instance->get_hostNameTraceDir(), $msglst);

    $instance->set_localUsrSapSidDir (undef);

    if ($fakeDir){
        $msg = $msglst->addMessage ("Cleaning up");

        if (-d $self->{_usrSapSidHDB}) {
            deltree ($self->{_usrSapSidHDB}, $msg->getSubMsgLst());
        }

        if (-d $self->{_usrSapSidSYS}) {
            deltree ($self->{_usrSapSidSYS}, $msg->getSubMsgLst());
        }

        if (-d $ipcDir) {
            deltree ($ipcDir, $msg->getSubMsgLst());
        }

        if (exists $self->{_auxUsrSapSidHDB}) {
            if (!CORE::rename($self->{_auxUsrSapSidHDB}, $self->{_usrSapSidHDB})) {
                $self->setErrorMessage ("Cannot rename '$self->{_auxUsrSapSidHDB}'"
                                        . " => '$self->{_usrSapSidHDB}': $!");
                return undef;
            }
        }

        if (exists $self->{_auxUsrSapSidSYS}) {
            if (!CORE::rename($self->{_auxUsrSapSidSYS}, $self->{_usrSapSidSYS})) {
                $self->setErrorMessage ("Cannot rename '$self->{_auxUsrSapSidSYS}'"
                                        . " => '$self->{_usrSapSidSYS}': $!");
                return undef;
            }
        }
    }
    $instance->{_host} = $originalHost;
    return $returnCode;
}


#-------------------------------------------------------------------------------
# Initializes the arguments of the command line
#
# Parameters string-array arguments
#
# Returns a reference to a string array containing the arguments
#         or undef in case of an error

sub InitCmdLineArgs{
    my ($self,$args) = @_;
    $self->{instconfig} = new SDB::Install::Configuration::AddHost
        ($self->{options}, $self->{configfile});

    return $self->SUPER::InitCmdLineArgs ($args);
}



#===============================================================================
#   M A I N

sub main{
    my $app = new __PACKAGE__;

    my $rc;
    $app->{stackBacktraceMsglst} = new SDB::Install::MsgLst ();
    eval{ 

        if (!defined $app->InitCmdLineArgs (\@_)){
           return undef;
        }

        if (defined $app->{return}){
            LCM::DevelopmentTrace::RemoveTempDevelopmentTrace();
            return $app->{return};
        }
        
        $rc = $app->init ();
        
    };
    
    if (defined $app->{return}){
        LCM::DevelopmentTrace::RemoveTempDevelopmentTrace();
        return $app->{return};
    }   
    my $msglst = $app->getMsgLst ();
    if ($@){
        my $signalInfoText;
        if ($@ =~ /^__SIGINT__/){
            $msglst->addMessage ('User canceled addHost with Ctrl + c');
            $rc = 2;
        }
        elsif ($@ =~ /^__SIGPIPE__/){
            $app->setErrorMessage ('Broken pipe', $app->{stackBacktraceMsglst});
            $signalInfoText = $app->getSignalInfoText ();
            undef $rc;
        }
        elsif ($@ =~ /^__SIG(\w+)__/){
            $app->setErrorMessage ("Caught signal $1", $app->{stackBacktraceMsglst});
            $signalInfoText = $app->getSignalInfoText ();
            undef $rc;
        }
        else{
            $app->setErrorMessage ('unhandled exception: '. $@, $app->{stackBacktraceMsglst});
            $rc = undef;
        }
        if ($signalInfoText){
            $app->appendErrorMessage ($signalInfoText);
        }
    }

    
    if (defined $rc){
        if ($rc == 2){
            print "\n";
            $msglst->addProgressMessage($app->getProgramName() . 'aborted');
        }
        elsif ($app->getInstconfig()->isCheckOnly()) {
            $msglst->addProgressMessage($app->getProgramName() . ' checked');
        }
        else {
            $msglst->addProgressMessage($app->getProgramName() . ' done');
        }

    }
    else{
        $app->ShowErrorMsg ('Add host failed',$app->getErrMsgLst());
    }

    $app->CleanUp ();   
    
    undef $app;
    
    return defined $rc ? 0 : 1; 
}

sub shouldWarnIfCalledStandalone{
    return 1;
}



1;
