#!/usr/bin/perl
#
# Desc: checks a SAP HANA system

package SDB::Install::App::Console::HdbCheck;

use base SDB::Install::App::Console;

use SDB::Install::Configuration::HdbCheck;
use SDB::Install::Globals qw ($gProductName
                              $gProductNameInstaller
                              $gShortProductNameXS2);
use SDB::Install::Tools   qw (askConfirmation);
use SDB::Install::SAPSystem;
use SDB::Install::SysVars qw ($isWin $path_separator);
use Getopt::Long;

use strict;

our $programVers = '0007';


#-------------------------------------------------------------------------------
# Constructor
sub new {
    my $self = shift->SUPER::new ();
    return $self;
}

#-------------------------------------------------------------------------------

sub CleanUp{
    my ($self) = @_;

    my $instconfig = $self->getInstconfig();
    my $sapSys     = (defined $instconfig) ? $instconfig->getSAPSystem()
                                           : undef;
    if (defined $sapSys) {
        # now added trace directories exist
        $self->addLogToTraceDir($sapSys,
                                $self->getProgramName(),
                                $instconfig->isCheckOnly());
    }
    $self->SUPER::CleanUp();
}


#-------------------------------------------------------------------------------
# Returns the name of this program

sub getProgramName {
    return 'hdbcheck';
}


#-------------------------------------------------------------------------------
sub getScope {
    my ($self) = @_;
    my $scope = $self->SUPER::getScope();
    return (defined $scope) ? $scope : 'system';
}


#-------------------------------------------------------------------------------
# Initializes this application
# Without parameters
# Returns int 1

sub InitApp{
    my ($self) = @_;
    $self->SUPER::InitApp();
    return 1;
}


#-------------------------------------------------------------------------------
sub init{
    my ($self) = @_;

    my $strProgramName = $self->getProgramName();

    if ($isWin){
        $self->setErrorMessage
                   ("$strProgramName is not yet implemented for this platform");
        return undef;
    }

    if (!$self->checkIsAdmin($strProgramName)){
        return undef;
    }

    my $msglst     = $self->getMsgLst ();
    my $instconfig = $self->getInstconfig();
    $instconfig->setMsgLstContext   ([$msglst]);
    $instconfig->SetProgressHandler ($msglst->getProgressHandler());

    my $strHeadline = "$gProductNameInstaller - $strProgramName  "
                    . $self->GetInstaller->{version};

    my $info = "\n\n" . $strHeadline . "\n" . ('*' x length($strHeadline)) . "\n";

    $msglst->addProgressMessage($info);

    $self->addStartDateMsg($msglst, "$strProgramName  $programVers");

    my $msg       = $msglst->addMessage ("Checking parameters...");
    my $rc        = $instconfig->CheckParams ($self->{batch_mode});
    my $cfgDumpRc = $self->handleOptionDumpConfigFileTemplate();
    if (!defined $cfgDumpRc){
        $self->{return} = 1;
        return undef;
    }
    if ($cfgDumpRc){
        $self->{return} = 0;
        return 1;
    }

    if($self->{batch_mode}){
        if (!$rc){
            $self->setErrorMessage ("Running in batch mode",
                                    $instconfig->getErrMsgLst());
            return undef;
        }
    }
    else{
        if (!$rc && $instconfig->ErrorState ()){
            $self->ShowErrorMsg ('Invalid parameter:',
                                 $instconfig->getErrMsgLst());
        }
        if (!defined $self->ConfigureInteractive ($instconfig)){
            return undef;
        }
    }

    if (!$self->defineLog($strProgramName,
                          $strProgramName,
                          $instconfig->isCheckOnly(),
                          $instconfig->getSID())) {
        return undef;
    }

    $instconfig->dumpConfigFile($strProgramName);
    $instconfig->displayParameterSummary();
    if (!$self->{batch_mode} && !$self->{options}->{noprompt}
                                                        && !askConfirmation()) {
        $msglst->addProgressMessage($self->getActionProgressive() . ' '
                     . $gProductName . ' ' . $self->getScope() . ' cancelled');
        $self->{return} = 0;
        return 1;
    }

    $msg = $msglst->addProgressMessage("Checking $gProductName "
                                       . $self->getScope() . " '"
                                       . $instconfig->getSID() . "'...");

    $rc = $instconfig->performSystemCheck($self);
    if (!$rc) {
        $self->setErrorMessage('Check system:', $instconfig->getErrMsgLst());
    }
    return $rc;
}


#-------------------------------------------------------------------------------
# Initializes the arguments of the command line
#
# Parameters string-array arguments
#
# Returns a reference to a string array containing the arguments
#         or undef in case of an error

sub InitCmdLineArgs{
    my ($self,$args) = @_;
    $self->{instconfig} =
            new SDB::Install::Configuration::HdbCheck ($self->{options});

    $self->{action}            = "Check";
    $self->{actionProgressive} = "Checking";
    $self->{actionDone}        = "checked";

    if (!$self->additionalInit()) {
        return undef;
    }

    return $self->SUPER::InitCmdLineArgs ($args);
}


#-------------------------------------------------------------------------------
# Additional initialization - may be overridden

sub additionalInit {
    return 1;
}


#-------------------------------------------------------------------------------
# Returns true if the SAP HANA system should be stopped in case of changing
# listen interface.

sub isStopSystem {
    return 1;
}



#===============================================================================
#   M A I N

sub main{
    my $app = new __PACKAGE__;
    return $app->mainFunction(\@_);
}


#-------------------------------------------------------------------------------
# This function can also be called from other classes:
#     require SDB::Install::App::Console::HdbModify;
#     my $modifyMultiHost = new SDB::Install::App::Console::HdbModify();
#     $modifyMultiHost->mainFunction(\@ARGV);

sub shouldWarnIfCalledStandalone{
    return 0;
}

1;
