#!/usr/bin/perl
#
# Desc: reconfigures the landscape of a SAP HANA system
# Switches the hostname resolution to 'global', 'internal' or 'local'.
# Removes remote hosts before adding remote hosts in parallel.

package SDB::Install::App::Console::HdbModify;

use base SDB::Install::App::Console;

use SDB::Install::Configuration::HdbModify;
use SDB::Install::Globals qw ($gProductNameSystem
                              $gProductNameInstaller
                              $gShortProductNameXS2);
use SDB::Install::Tools   qw (askConfirmation);
use SDB::Install::SAPSystem;
use SDB::Install::System qw (nslookup is_local_address);
use Getopt::Long;

use strict;

our $programVers = '0032';


#-------------------------------------------------------------------------------
# Constructor
sub new {
    my $self = shift->SUPER::new ();
    return $self;
}

#-------------------------------------------------------------------------------

sub CleanUp{
    my ($self) = @_;

    my $instconfig = $self->getInstconfig();
    my $sapSys     = (defined $instconfig) ? $instconfig->getSAPSystem()
                                           : undef;
    if (defined $sapSys) {
        # now added trace directories exist
        $self->addLogToTraceDir($sapSys,
                                $self->{ModifyName},
                                $instconfig->isCheckOnly());
    }
    $self->SUPER::CleanUp();
}


#-------------------------------------------------------------------------------
sub getActionDone {
	my ($self) = @_;
	my $instconfig = $self->getInstconfig();

    if (!defined $instconfig || !$instconfig->isReconfigured()) {
        return 'not modified';
    }

    my $scope = $instconfig->getValue('ScopeModify');

    return (defined $scope && ($scope eq 'instance'))
           ? undef  # suppress end message (outstanding remote hosts)
           : 'reconfigured';
}


#-------------------------------------------------------------------------------
# Returns the name of this program

sub getProgramName {
    return 'hdbmodify';
}


#-------------------------------------------------------------------------------
sub getScope {
	return 'system';
}


#-------------------------------------------------------------------------------
# Initializes this application
# Without parameters
# Returns int 1

sub InitApp{
    my ($self) = @_;
    $self->SUPER::InitApp();
    return 1;
}

sub _isAddRolesToLocalHost{
    my ($self) = @_;
    my $instconfig     = $self->getInstconfig();
    my $hostRoles = $instconfig->getBatchValue('AddRoles');
    if (!defined $hostRoles){
        return 0;
    }
    my @hostRoles = split (',', $hostRoles);
    if (@hostRoles != 1){
        return 0;
    }
    my ($hostname) = (split (':', $hostRoles));
    my $nslookup = nslookup ($hostname);

    if (!defined $nslookup || !defined $nslookup->{address}){
        return 0;
    }
    my $rc = is_local_address ($nslookup->{address});
    if (!defined $rc){
        return 0;
    }
    return $rc;
}



#-------------------------------------------------------------------------------
sub init{
    my ($self) = @_;

    my $instconfig     = $self->getInstconfig();
    my $msglst         = $self->getMsgLst ();
    $self->{ModifyName}= $self->getProgramName();

    if (defined $instconfig->getBatchValue('AddRoles') ||
        defined $instconfig->getBatchValue('AddLocalRoles')) {
       if ($self->_isAddRolesToLocalHost ()){
            $self->{ModifyName} .= '_addlocalroles';
        }
        elsif ($self->{ModifyName} !~ /addroles/) {
            $self->{ModifyName} .= '_addroles';
        }
    }
    elsif (defined $instconfig->getBatchValue('AddHosts')) {
        $self->{ModifyName} .= '_addhosts';
    }
    elsif (defined $instconfig->getBatchValue('RemoveHosts')) {
        $self->{ModifyName} .= '_removehosts';
    }
    elsif (defined $instconfig->getBatchValue('RemoveRoles') ||
           defined $instconfig->getBatchValue('RemoveLocalRoles')) {
        $self->{ModifyName} .= '_removeroles';
    }
    elsif (defined $instconfig->getBatchValue('LoadInitialContentXS2')) {
        $self->{ModifyName} .= '_loadcontent';
    }
    elsif (defined $instconfig->getBatchValue('ISCMode')) {
        $self->{ModifyName} .= '_isc_mode';
    }
    elsif (defined $instconfig->getBatchValue('ConfigureLSS')) {
        $self->{ModifyName} .= '_configure_lss';
    }

    my $strHeadline = "$gProductNameInstaller - $self->{ModifyName}  "
                    . $self->GetInstaller->{version};
    $instconfig->SetProgressHandler ($msglst->getProgressHandler());

    $self->warnIfShouldWarnIfCalledStandalone();
    my $info = "\n\n" . $strHeadline . "\n" . ('*' x length($strHeadline)) . "\n";

    $msglst->addProgressMessage($info);

    $self->addStartDateMsg
            ($msglst, $self->getProgramName() . "  $programVers");

    my $msg = $msglst->addMessage ("Checking configuration");
    $instconfig->setMsgLstContext([$msg->getSubMsgLst ()]);
    my $rc = $instconfig->CheckParams ($self->{batch_mode});

    my $cfgDumpRc = $self->handleOptionDumpConfigFileTemplate();
    if (!defined $cfgDumpRc){
        $self->{return} = 1;
        return undef;
    }
    if ($cfgDumpRc){
        $self->{return} = 0;
        return 1;
    }

    if($self->{batch_mode}){
        if (!$rc){
            if (defined $rc){
                $self->setErrorMessage ("Running in batch mode", $instconfig->getErrMsgLst ());
            }
            else{
                $self->setErrorMessage ('Configuration error (batch mode):', $instconfig->getErrMsgLst ());
            }
            return undef;
        }
    }
    else{
        if (!$rc && $instconfig->ErrorState ()){
            $self->ShowErrorMsg ('Configuration error:',$instconfig->getErrMsgLst ());
        }
        if (!defined $self->ConfigureInteractive ($instconfig)){
            return undef;
        }
    }

    if ((defined $instconfig->{addRemoveLocalRoles}||
         defined $instconfig->getBatchValue('ISCMode') ||
         defined $instconfig->getBatchValue('LoadInitialContentXS2'))
        &&
        !$self->checkIsAdmin($self->{ModifyName})) {
        return undef;
    }

    if (!$self->defineLog($self->{ModifyName},
                          $self->{ModifyName},
                          $instconfig->isCheckOnly(),
                          $instconfig->getSID())) {
        return undef;
    }

    $msg = $msglst->addProgressMessage('Modifying configuration of '
                   . "$gProductNameSystem '" . $instconfig->getSID() . "'...");

    $instconfig->setMsgLstContext ([$msg->getSubMsgLst()]);

    $instconfig->dumpConfigFile($self->{ModifyName});
    $instconfig->displayParameterSummary();
    if (!$self->{batch_mode} && !$self->{options}->{noprompt}
                                                        && !askConfirmation()) {
        $msglst->addProgressMessage($self->getActionProgressive()
                                            . " $gProductNameSystem cancelled");
        $self->{return} = 0;
        return 1;
    }

    $rc = $instconfig->sendRemoveHostCommands();
    if (!$rc) {
        $self->setErrorMessage('Remove hosts:', $instconfig->getErrMsgLst());
    }

    if ($rc) {
        $rc = $instconfig->tryChangeListenInterface();
        if (!$rc) {
            $self->setErrorMessage('Change listen interface:',
                                                   $instconfig->getErrMsgLst());
        }
    }

    if ($rc) {
        $rc = $instconfig->addLocalRoles();
        if (!$rc) {
            $self->setErrorMessage('Add local host roles:',
                                   $instconfig->getErrMsgLst());
        }
    }

    if ($rc) {
        $rc = $instconfig->removeLocalRoles();
        if (!$rc) {
            $self->setErrorMessage('Remove local host roles:',
                                   $instconfig->getErrMsgLst());
        }
    }

    if ($rc) {
        $rc = $instconfig->sendAddRemoveRolesCommands();
        if (!$rc) {
            $self->setErrorMessage('Add remote host roles:',
                                   $instconfig->getErrMsgLst());
        }
    }

    if ($rc) {
        $rc = $instconfig->sendAddRemoteHostCommands();
        if (!$rc) {
            $self->setErrorMessage('Add hosts:', $instconfig->getErrMsgLst());
        }
    }

    if ($rc) {
        $rc = $instconfig->tryLoadInitialContentXS2();
        if (!$rc) {
            $self->setErrorMessage(undef, $instconfig->getErrMsgLst());
        }
    }

    if ($rc) {
        $rc = $instconfig->tryChangeISCMode();
        if (!$rc) {
            $self->setErrorMessage(undef, $instconfig->getErrMsgLst());
        }
    }

    if ($rc) {
        $rc = $instconfig->configureLSS();
        if (!$rc) {
            $self->setErrorMessage(undef, $instconfig->getErrMsgLst());
        }
    }
    $msg->endMessage ();
    return $rc;
}


#-------------------------------------------------------------------------------
# Initializes the arguments of the command line
#
# Parameters string-array arguments
#
# Returns a reference to a string array containing the arguments
#         or undef in case of an error

sub InitCmdLineArgs{
    my ($self,$args) = @_;
    $self->{instconfig} =
            new SDB::Install::Configuration::HdbModify ($self->{options});

    $self->{action}            = 'Reconfigure';
    $self->{actionProgressive} = 'Reconfiguring';
    $self->{actionDone}        = 'reconfigured';

    if (!$self->additionalInit()) {
        return undef;
    }

    return $self->SUPER::InitCmdLineArgs ($args);
}


#-------------------------------------------------------------------------------
# Additional initialization - may be overridden

sub additionalInit {
    return 1;
}


#-------------------------------------------------------------------------------
# Returns true if the SAP HANA system should be stopped in case of changing
# listen interface.

sub isStopSystem {
    return 1;
}



#===============================================================================
#   M A I N

sub main{
    my $app = new __PACKAGE__;
    return $app->mainFunction(\@_);
}


#-------------------------------------------------------------------------------
# This function can also be called from other classes:
#     require SDB::Install::App::Console::HdbModify;
#     my $modifyMultiHost = new SDB::Install::App::Console::HdbModify();
#     $modifyMultiHost->mainFunction(\@ARGV);

sub shouldWarnIfCalledStandalone{
    return 1;
}


1;
