package SDB::Install::App::Console::LSS::RenameInstanceLss;
use parent qw(SDB::Install::App::Console);
use warnings;
use strict;

use Getopt::Long;
use SDB::Install::Globals qw($gProductNameLSS $gProductName);
use SDB::Install::LSS::LssInstance;
use SDB::Common::Utils qw(getSidcryptName);
use SDB::Install::LSS::Installer;
use SDB::Install::LSS::LssSidcryptUser;
use SDB::Install::Group;
use File::Spec;
use File::stat;

use SDB::Install::System qw(changeOwn);
use SDB::Install::SAPSystem qw(CollectSAPSystems);

use SDB::Install::Configuration::AnyConfig;
use SDB::Install::Configuration::LSSConfiguration;
use SDB::Install::Configuration::AnyMultiHostConfig;
use SDB::Install::Configuration::AnyRename;

sub new {
    my $self = shift->SUPER::new ();
    return $self;
}

sub getAction {
    return "Rename";
}

sub getActionScope {
    return $gProductNameLSS;
}

sub getActionDone {
    return "renamed";
}

sub getActionProgressive {
    return "Renaming";
}

sub getProgramName {
    return 'lssrenamehost';
}

sub createConfiguration {
    my ($self) = @_;
    my $config = SDB::Install::Configuration::LSSConfiguration->new($self->{options});
    my $section = 'main';
    my $order = 1;
    $config->{params} = {
        'IsRegister'     => SDB::Install::Configuration::AnyRename::getParamIsRegister($order++, $section, 0),
        'SID'            => SDB::Install::Configuration::AnyConfig::getParamSID($order++, $section),
        'newSID'         => SDB::Install::Configuration::AnyRename::getParamNewSID($order++, $section),
        'LSSPassword'    => SDB::Install::Configuration::AnyMultiHostConfig::getParamTargetLSSPassword(undef, $order++, $section, 0),
        'LSSUserID'      => $config->getParamLSSUserID($order++, $section, 0),
        'LSSGroupID'     => $config->getParamLSSGroupID($order++, $section, 0),
        'LSSUserHomeDir' => $config->getParamLSSUserHomeDir($order++, $section, undef, 0),
        'LSSUserShell'   => $config->getParamLSSUserShell($order++, $section, 0),
    };
    $config->{params}->{SID}->{opt} = 'source_sid';
    $config->{params}->{SID}->{str} = 'Source System ID';
    $config->setMsgLstContext($self->getMsgLstContext());

    $self->setInstconfig($config);
}

sub InitCmdLineArgs{
    my $self = shift();
    $self->createConfiguration();

    return $self->SUPER::InitCmdLineArgs (@_);
}

sub init {
    my ($self) = @_;

    my $installer = SDB::Install::LSS::Installer->new();
    my $sid = $installer->getSid();

    if(!$self->defineLog($self->getProgramName(), $self->getProgramName(), undef, $sid)){
        return undef;
    }
    return undef if(!$sid);

    $installer->setMsgLstContext($self->getMsgLstContext());

    my $configuration = $self->getInstconfig();
    if (!$configuration->CheckParams(1)) {
        $self->setErrorMessage("Failed to check configuration", $configuration->getErrMsgLst());
        return undef;
    }

    return $self->_renameInstance($installer);
}

sub _renameInstance {
    my ($self, $installer) = @_;
    my $config         = $self->getInstconfig();
    my $isRegister     = $config->getValue('IsRegister');
    my $oldSid         = $config->getValue('SID');
    my $targetSid      = $config->getValue('newSID');
    my $isSidChanged   = $oldSid ne $targetSid;

    if (!$isRegister && $isSidChanged) {
        $self->_deleteOldCryptUsers($oldSid);
    }
    if ($isSidChanged) {
        return undef if (!$installer->renameSharedDir($targetSid));
    }
    my $lssInstance = $installer->getLssInstance();
    $lssInstance->setMsgLstContext($self->getMsgLstContext());

    return undef if (!$self->_adaptLssUsersAndLocalFiles($lssInstance));
    return 1;
}

sub _deleteOldCryptUsers {
    my ($self, $oldSid) = @_;
    $self->getMsgLst()->addProgressMessage("Deleting old $gProductNameLSS crypt group and user...");

    my $sidcryptName = getSidcryptName($oldSid);
    my $oldUser = SDB::Install::LSS::LssSidcryptUser->new($sidcryptName);
    my $isDefaultGroupName = $oldUser->group() eq $sidcryptName ? 1 : 0;
    $oldUser->setMsgLstContext($self->getMsgLstContext());
    $oldUser->delete();

    return 1 if(!$isDefaultGroupName);
    my $oldGroup = SDB::Install::Group->new($sidcryptName);
    $oldGroup->setMsgLstContext($self->getMsgLstContext());
    $oldGroup->delete();
}

sub _adaptLssUsersAndLocalFiles {
    my ($self, $lssInstance) = @_;
    my $config = $self->getInstconfig();
    my $isRegister     = $config->getValue('IsRegister');
    my $uid            = $config->getValue('LSSUserID');
    my $gid            = $config->getValue('LSSGroupID');
    my $homeDir        = $config->getValue('LSSUserHomeDir');
    my $userShell      = $config->getValue('LSSUserShell');
    my $oldSid         = $config->getValue('SID');
    my $targetSid      = $config->getValue('newSID');
    my $isSidChanged   = $oldSid ne $targetSid;

    my $msg = $self->getMsgLst()->addProgressMessage("Adapting $gProductNameLSS file layout and crypt group and user...");
    my $saveContext = $lssInstance->setMsgLstContext([$msg->getSubMsgLst()]);
    if (!$isRegister) {
        if ($isSidChanged) {
            my $group = $lssInstance->getLssCryptGroup($gid);
            my $isDefaultGroupName = $group->{name} eq getSidcryptName($targetSid);
            return undef if ($isDefaultGroupName && !$group->create(undef, $gid));
            return undef if (!$lssInstance->getLssCryptUser()->create($uid, $config->getValue('LSSPassword'), undef, $gid, $homeDir, $userShell));
        # Adjust LSS local dir @ /usr/sap/<SID>
            my $lssRootDir = $lssInstance->getLocalLssRootDir();
            $lssInstance->createExeLink();
            $lssInstance->createSharedLink();
            $self->getMsgLst()->addMessage("Changing ownership of '$lssRootDir' to $uid:$gid...");
            return undef if (!changeOwn(undef, $uid, undef, $gid, $lssRootDir, $self->getErrMsgLst(), $self->getMsgLst(), undef, undef, 1));
        }
    } else {
        return undef if (!$lssInstance->configureHost($uid, $config->getValue('LSSPassword'), $userShell, $homeDir, $gid));
        return undef if (!$lssInstance->initialize(undef, 1));
    }
    $lssInstance->setMsgLstContext($saveContext);
    return undef if (!$self->_adaptLssSharedOwnership($lssInstance, $uid, $gid));
    return 1;
}

sub _adaptLssSharedOwnership {
    my ($self, $lssInstance, $uid, $gid) = @_;

    my $lssSidDir = $lssInstance->getLssSidDir();
    my $stat = File::stat::stat($lssSidDir);
    my $oldInstallDirUid = $stat->uid();
    my $oldInstallDirGid = $stat->gid();

    my $msg = $self->getMsgLst()->addProgressMessage("Checking ownership of '$lssSidDir'...");
    my $saveContext = $lssInstance->setMsgLstContext([$msg->getSubMsgLst()]);
    if (defined $oldInstallDirUid && $oldInstallDirUid != $uid) {
        $msg->getSubMsgLst()->addProgressMessage("Adjusting onwership of '$lssSidDir'");
        return undef if (!changeOwn($oldInstallDirUid, $uid, $oldInstallDirGid, $gid, $lssSidDir, $self->getErrMsgLst(), $msg->getSubMsgLst(), undef, undef, 1));

        return undef if !$lssInstance->adaptSpecialExecutable('localsecurestore', undef, undef, 04750, 1);
        return undef if !$lssInstance->adaptSpecialExecutable('lssvalidator',     undef, undef, 04555, 1);
    } else {
        $msg->getSubMsgLst()->addProgressMessage("Ownership of '$lssSidDir' is already adapted");
    }
    $lssInstance->setMsgLstContext($saveContext);
    return 1;
}

 sub main {
    my $app = new __PACKAGE__;
    $app->mainFunction(\@_);
}

sub shouldWarnIfCalledStandalone {
    return 1;
}

sub CleanUp{
    my ($self) = @_;
    my $installer = SDB::Install::LSS::Installer->new();
    my $sid = $installer->getSid();
    my $systems = CollectSAPSystems();
    my $sapSystem = $systems->{$sid};

    if (defined $sapSystem) {
        $self->addLogToTraceDir($sapSystem, $self->getProgramName());
    }
    $self->SUPER::CleanUp();
}

1;