#!/usr/bin/perl
#
# Desc: remove a host from the HDB system

package SDB::Install::App::Console::RemoveHost;

use base SDB::Install::App::Console;

use SAPDB::Install::Hostname qw(hostname);
use SDB::Install::SAPSystem;
use SDB::Install::Globals qw ($gProductNameInstaller
                              $gShortProductNameEngine);
use SDB::Install::SysVars qw ($isWin $path_separator);
use Getopt::Long;
use SDB::Install::System qw (isAdmin);
use SDB::Install::Configuration::RemoveHost;
use SDB::Install::Log;
use SDB::Install::Tools qw (askConfirmation);

use strict;


sub new {
    my $self = shift->SUPER::new ();
    return $self;
}

sub getProgramName {
    return 'hdbremovehost';
}

sub InitApp{
    my ($self) = @_;
    $self->SUPER::InitApp();
    return 1;
}

sub init{
    my ($self) = @_;
    my $rc = 1;

    if (!isAdmin ()){
        $self->setErrorMessage ( $isWin ? 'Please start hdbremovehost as local administrator!' :
                                    'Please restart hdbremovehost as user root!');
        return undef;
    }
    $self->warnIfShouldWarnIfCalledStandalone();

    my $msglst = $self->getMsgLst ();

    my $instconfig = $self->{instconfig};

    $msglst->appendMsgLst($instconfig->getMsgLst());
    $instconfig->resetMsgLstContext();

    my $msg = $msglst->addMessage ("Checking configuration");
    $msg->setEndTag ('Configuration Check');
    $rc = $instconfig->CheckParams ($self->{batch_mode});

    my $cfgDumpRc = $self->handleOptionDumpConfigFileTemplate();
    if (!defined $cfgDumpRc){
        $self->{return} = 1;
        return undef;
    }
    if ($cfgDumpRc){
        $self->{return} = 0;
        return 1;
    }

    if($self->{batch_mode}){
        if (!$rc){
            if (defined $rc){
                $self->setErrorMessage ("Running in batch mode", $instconfig->getErrMsgLst ());
                #$self->PrintUsage ();
            }
            else{
                $self->setErrorMessage ('Configuration error (batch mode):', $instconfig->getErrMsgLst ());
            }
            $msg->getSubMsgLst ()->appendMsgLst ($instconfig->getMsgLst ());
            return undef;
        }
    }
    else{
        if (!$rc && $instconfig->ErrorState ()){
            $self->ShowErrorMsg ('Configuration error:',$instconfig->getErrMsgLst ());
        }
        if (!defined $self->ConfigureInteractive ($instconfig)){
            return undef;
        }
    }
    $msg->getSubMsgLst ()->appendMsgLst ($instconfig->getMsgLst ());

    my $sid = $instconfig->getValue ('SID');
    my $sapSys = $instconfig->getSAPSystem();

    my $errlst = new SDB::Install::MsgLst ();

    if (!$self->defineLog('removehost',
                          'hdbremovehost',
                          $instconfig->isCheckOnly(),
                          $sid)) {
        return undef;
    }
    my $instances = $sapSys->getNewDBInstances();
    my $instance  = undef;
    if(defined $instances && @$instances) {
        $instance = $instances->[0];
        $self->addLogToTraceDir($sapSys,
                                'hdbremovehost',
                                $instconfig->isCheckOnly());
    }
    my $host = (defined $instance) ? $instance->get_host() : undef;
    $host    = lc(hostname()) if (!defined $host);
    $sapSys->SetProgressHandler ($msglst->getProgressHandler);

	$sapSys->GenPackageList();
	$sapSys->SelectAllPackages();
    my $force = $instconfig->getValue ('Force');
    if (!$self->{batch_mode}){
       my $question = "\nDo you want to remove the host '$host' from the following installation?\n\n"
                    . $sapSys->asString(undef, 1, 1, 1);
       if ($force){
            $question .= "\nAll data volumes and all log volumes will be removed!\n"
                      .  "Data and log backup files in directory '"
                      . $sapSys->get_globalSidDir()
                      . "' and its subdirectories will be removed as well.\n";
        }
        $question .= "\nConfirm ";
        if (!askConfirmation($question)) {
            $msglst->addProgressMessage("Removal of host '$host' cancelled.\n");
            $self->{return} = 0;
            return 1;
        }
    }
    if(not $force && defined $instance) {
        my $rc = $instance->pingMasterNameServer ($self->getMsgLst());
        if(!$rc) {
            $self->setErrorMessage ("Master name server not reachable.");
            return undef;
        }
    }
    my $isDeleteUser = ($instconfig->getValue('KeepUser')) ? 0 : 1;
    $sapSys->setMsgLstContext ([$msglst]);
	$rc = $sapSys->removeSAPSystem($isDeleteUser,
	                               $force,
	                               $instconfig);
    if (!$rc) {
		$self->setErrorMessage ("Removal of host '$host' from system " . $instconfig->getValue('SID') . " failed", $sapSys->getErrMsgLst ());
		return undef;
	}

    return $rc;
}


sub InitCmdLineArgs{
    my ($self,$args) = @_;

    $self->{options} = {};

    $self->{instconfig} = new SDB::Install::Configuration::RemoveHost
        ($self->{options}, $self->{configfile});

    if (!$self->{instconfig}->InitDefaults ()){
        $self->setErrorMessage (undef, $self->{instconfig}->getErrMsgLst ());
        return undef;
    }

    my $rc = $self->SUPER::InitCmdLineArgs ($args);

    if (!defined $rc){
        return undef;
    }

    local @ARGV = @$rc;

    my $optctrl =  $self->{instconfig}->GetOptionCtrl();

    if (!defined $self->getOptions ($optctrl, 0)){
        $self->{return} = -1;
        return undef;
    }
    return \@ARGV;
}


#-------------------------------------------------------------------------------
sub main{
    my $app = new __PACKAGE__;

    my $rc;
    $app->{stackBacktraceMsglst} = new SDB::Install::MsgLst ();
    eval{

        if (!defined $app->InitCmdLineArgs (\@_)){
           return undef;
        }

        if (defined $app->{return}){
            LCM::DevelopmentTrace::RemoveTempDevelopmentTrace();
            return $app->{return};
        }

        $rc = $app->init ();

    };

    if (defined $app->{return}){
        LCM::DevelopmentTrace::RemoveTempDevelopmentTrace();
        return $app->{return};
    }
    my $msglst = $app->getMsgLst ();
    if ($@){
        my $signalInfoText;
        if ($@ =~ /^__SIGINT__/){
            $msglst->addMessage ('User canceled removal of host with Ctrl + c');
            $rc = 2;
        }
        elsif ($@ =~ /^__SIGPIPE__/){
            $app->setErrorMessage ('Broken pipe', $app->{stackBacktraceMsglst});
            $signalInfoText = $app->getSignalInfoText ();
            undef $rc;
        }
        elsif ($@ =~ /^__SIG(\w+)__/){
            $app->setErrorMessage ("Caught signal $1", $app->{stackBacktraceMsglst});
            $signalInfoText = $app->getSignalInfoText ();
            undef $rc;
        }
        else{
            $app->setErrorMessage ('unhandled exception: '. $@, $app->{stackBacktraceMsglst});
            $rc = undef;
        }
        if ($signalInfoText){
            $app->appendErrorMessage ($signalInfoText);
        }
    }


    if (defined $rc){
        if ($rc == 2){
            print "\n";
            $msglst->addProgressMessage('Removal of host aborted');
        }
        else{
            $msglst->addProgressMessage('Removal of host done');
        }

    }
    else{
        $app->ShowErrorMsg ('Removal of host failed',$app->getErrMsgLst ());
    }

    $app->CleanUp ();

    undef $app;

    return defined $rc ? 0 : 1;
}

sub shouldWarnIfCalledStandalone{
    return 1;
}


1;
