
#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#
# Desc: uninstallation application class

package SDB::Install::App::Console::Uninstallation;

use strict;
use SDB::Install::App::Console;
use SDB::Install::SAPSystem;
use SDB::Install::PackageManagerCollection;
use SDB::Install::Globals qw ($gLogDir
                              $gProductName
                              $gProductNameClient
                              $gProductNameEngine
                              $gProductNameInstaller
                              $gProductNameStudio
                              $gProductNameSystem
                              $gSapsysGroupName
                              $gTmpDir
                              gPrepareLogPath
    );
use SDB::Install::SysVars qw ($isWin $path_separator);
use Getopt::Long;
use SDB::Install::System qw (isPASE isAdmin normalizePath isSameFile removeEmptyDirsAtExit);
use SDB::Install::Tools qw (askConfirmation printTableToArrayOfLines);
use SDB::Install::Configuration::NewDB;
use SDB::Install::Configuration::Uninstallation;
use SDB::Install::Installation::DatabaseStudio;
use SDB::Install::Installation::Client;
use SDB::Install::Log;
use SAPDB::Install::Hostname qw(hostname);


our @ISA = qw (SDB::Install::App::Console);

sub new {
    my $self = shift->SUPER::new ();
    return $self;
}


sub InitApp{
    my ($self) = @_;
    $self->SUPER::InitApp();
    return 1;
}



sub selectPackages{
    my ($self,$installation) = @_;
    
    my $rc = 1;
    my $instconfig = $self->getInstconfig();
    my $paramPackageValue = $instconfig->getBatchValue ('Package');

    if (defined $paramPackageValue){
        $installation->GenerateTree ();
        
        foreach my $package_name (split(',', $paramPackageValue)){
            my $package = $installation->GetPackageByName ($package_name);
            if (!defined $package){
                $self->appendErrorMessage ("Package \"$package_name\" not found");
                undef $rc;
                next;
            }
            $installation->SelectPackage ($package);
        }
    }
    else{
        $installation->GenPackageList ();
        $installation->SelectAllPackages ();
    }
    return $rc;
}


our ($systems, $studio_installations, $client_installations, $pending_sids);

sub _findInstallations{
    my ($self, $nocache) = @_;
    if (!defined $systems || $nocache){
        $systems = CollectSAPSystems();
        my $errlist = new SDB::Install::MsgLst ();
        $studio_installations = SDB::Install::Installation::DatabaseStudio::EnumDBStudioInstallations ($errlist);

        if (!defined $studio_installations){
            $self->setErrorMessage ("Cannot get $gProductNameStudio installations", $errlist);
        }

        $client_installations = SDB::Install::Installation::Client::EnumClientInstallations ($errlist);


        if (!defined $client_installations){
            $self->setErrorMessage ("Cannot get $gProductNameClient installations", $errlist);
        }

        my $newdb_instconfig = new SDB::Install::Configuration::NewDB();
        $pending_sids = {};
        foreach my $sid (sort keys %$systems){
            $newdb_instconfig->{current_sid} = $sid;
            $newdb_instconfig->{sapSys}      = $systems->{$sid};
            if ($newdb_instconfig->pers_exists ()){
                $pending_sids->{$sid} = 1;
            }
        }
    }
    return ($systems, $studio_installations, $client_installations, $pending_sids);
}

sub _selectInstallation{
    my ($self, $installation) = @_;
    my $instconfig = $self->getInstconfig();
    $self->{installation} = $installation;
    $instconfig->initInstallation ($installation);
    my $args = $self->{cmdLineArgsAfterInit};
    delete $self->{cmdLineArgsAfterInit};
    $args = $self->SUPER::InitCmdLineArgs ($args);
    if ($self->isHelp()) {
        $self->PrintUsage();
        $self->{return} = 0;
        return undef;
    }
    return $args;
}


sub _initConfiguration{
    my ($self,$installation,$instconfig) = @_;
    my $genericConfigClass;

    if ($installation->isa ('SDB::Install::Installation::GenericServerPlugin')){
        $genericConfigClass = 'SDB::Install::Configuration::Uninstallation::GenericServerPlugin';
    }
    elsif($installation->isa ('SDB::Install::Installation::Generic')){
        $genericConfigClass = 'SDB::Install::Configuration::GenericUninstallation';
    }

    if (defined $genericConfigClass && $instconfig->isa($genericConfigClass)){
        return 1;
    }

    if (defined $genericConfigClass){
        eval "require $genericConfigClass";
        bless ($instconfig, $genericConfigClass);
        $instconfig->setInstallation ($installation);
        $instconfig->setMsgLstContext([$self->getMsgLst()]);
        $instconfig->parseInstallParamsXml ();
        my $customConfigurationClass = $instconfig->getCustomConfigurationClass ();
        if (defined $customConfigurationClass){
            local @INC = @INC;
            push @INC, $installation->getCustomModulesDir();
            eval ("require $customConfigurationClass;");
            if ($@){
                $self->setErrorMessage ("Configuration class '$customConfigurationClass' not implemented: " . $@);
                return undef;
            }
            bless ($instconfig,$customConfigurationClass);
            if (!$instconfig->isa($genericConfigClass)){
                $self->setErrorMessage ("Custom configuration class '$customConfigurationClass' has to be derived from '$genericConfigClass'");
                return undef;
            }
        }
    }
    $instconfig->prependParams();
    $instconfig->appendParams();
    return 1;
}


sub init{
    my ($self) = @_;
    
    my $allowNonRoot = 0;
    require SDB::Install::Installation::Client;
    require SDB::Install::Installation::DatabaseStudio;
    require SDB::Install::Installation::Generic;
    my $installation = $self->{installation};
    if(defined $installation) {
        if($installation->isa('SDB::Install::Installation::Client')) {
            $allowNonRoot = 1;
        }
        elsif($installation->isa('SDB::Install::Installation::DatabaseStudio')) {
            $allowNonRoot = 1;
        }
        elsif($installation->isa('SDB::Install::Installation::Generic')) {
            $allowNonRoot = 1;
        }
    }

    $self->_findInstallations ();

    my $instconfig = $self->getInstconfig();
    my $noexclusive = $instconfig->getBatchValue ('NoExclusive');
    my $ignoreNotExisting = $instconfig->getBatchValue ('IgnoreNotExisting');

    if (!isAdmin () && !$allowNonRoot){
        $self->setErrorMessage ( $isWin ? 'Start uninstallation as local administrator!' :
                                ( isPASE() ? 'Start uninstallation as user QSECOFR!' :
                                             'Start uninstallation as user root!'));
        return undef;
    }

    $self->warnIfShouldWarnIfCalledStandalone();
    my $caption = "$gProductNameInstaller - Uninstallation " .$self->GetInstaller->{version}; 
    my $msglst = $self->getMsgLst();
    $msglst->addProgressMessage ("\n\n".$caption . "\n" . ('*' x length ($caption)) . "\n\n");
    

    my $rc = 1;

    my $packageManagerCollection = $self->{packageManagerCollection} = new SDB::Install::PackageManagerCollection ();

    if (!defined $installation){
        #
        # try to detect installation via argv0 of installer:
        #
        $installation = $self->GetInstaller ()->getInstallation ();

        if (!defined $installation){

            #
            # populate list of studio, client, and engine installations
            #

            my @insts;
            my @outlist;
            my $instPath;
            my $i = 0;
            if (defined $studio_installations){
                foreach $instPath (sort keys (%$studio_installations)){
                    push @insts, [0,$instPath,$studio_installations->{$instPath}];
                    push @outlist, ["$i:", "$gProductNameStudio " . $studio_installations->{$instPath}."\n\t$instPath"];
                    $i++;
                }
            }
        
            if (defined $client_installations){
                foreach $instPath (sort keys (%$client_installations)){
                    push @insts, [1,$instPath,$client_installations->{$instPath}];
                    push @outlist, ["$i:", "$gProductNameClient " . $client_installations->{$instPath}."\n\t$instPath"];
                    $i++;
                }
            }
            foreach my $sid (sort grep { $systems->{$_}->hasNewDB() || exists $pending_sids->{$_} } keys (%$systems)){
                my $systemInfo = "$gProductNameEngine " . $systems->{$sid}->asString(undef,1,1,1);
                my $pendingInfo = $systems->{$sid}->getRegisterOrRenamePending();
                if(not $pendingInfo) {
                    	my $newdb_instconfig = new SDB::Install::Configuration::NewDB ();
		                $newdb_instconfig->{current_sid} = $sid;
		                $newdb_instconfig->{sapSys}      = $systems->{$sid};
        			    $pendingInfo = getPendingInfoFromPersFile('Installation',
                                                                  'of',
                                                                  $newdb_instconfig,
                                                                  \@STEP_NAMES);
                }
                if(not $pendingInfo) {
                        my $updConfig  = new SDB::Install::Configuration::Upgrade();
	                    $updConfig->{current_sid} = $sid;
	                    $updConfig->{sapSys}      = $systems->{$sid};
            			$pendingInfo = getPendingInfoFromPersFile('Update',
                                                                  'to',
                                                                  $updConfig,
                                                                  \@UPGRADE_STEP_NAMES);
                }
                $systemInfo .= "\t$pendingInfo\n" if (defined $pendingInfo);
                push @insts, [2,$sid];
                push @outlist, ["$i:", $systemInfo];
                $i++;
            }
            
            push @outlist, ["$i:", 'None (Abort uninstallation)'];
            
            
            if (scalar (@outlist) == 1){
                $msglst->addProgressMessage ("No $gProductName installation found");
                $self->{return} = 0;
                return 1;
            }
            
            
            print "Installed $gProductName installations:\n\n";
            
            my $tableLines = printTableToArrayOfLines(\@outlist,
                                                      '   ', # column separator
                                                      undef,
                                                      undef,
                                                      undef,
                                                      1);    # space between rows

            print join ("\n", @$tableLines) . "\n";


            if ($self->{batch_mode}){
                $self->setErrorMessage ("Running in batch mode: No installation selected");
                return undef;
            }
            print "\nEnter $gProductName installation Id [". ($i)."]: ";
            my $selection;
            while (<STDIN>){
                chomp;
                if (!/\S/){
                    undef $selection;
                    last;
                }
                if (!/\D/){
                    $selection = int $_;
                    if ($selection >= 0 && $selection <= ($i - 1)){
                        last;
                    }
                    if ($selection == $i){
                        undef $selection;
                        last;
                    }
                }
                print "\nEnter $gProductName installation Id [". ($i)."]: ";
            }
            if (defined $selection){
                if ($insts[$selection]->[0] == 0){
                    $installation = 
                        new SDB::Install::Installation::DatabaseStudio (
                            $insts[$selection]->[1]);
                }
                elsif ($insts[$selection]->[0] == 1){
                    $installation = 
                        new SDB::Install::Installation::Client (
                            $insts[$selection]->[1]);
                }
                elsif ($insts[$selection]->[0] == 2){
                    $installation = $systems->{$insts[$selection]->[1]};
                }
            }
            else{
                $msglst->addProgressMessage("Uninstallation aborted.\n");
                $self->{return} = 0;
                return 1;
            }
        }
        my $args = $self->_selectInstallation ($installation);
        if (!defined $args){
            return undef;
        }
        if (defined $self->{return}){
            return 1;
        }
    }

    if (!defined $self->_initConfiguration($installation, $instconfig)){
        return undef;
    }

    my $cfgDumpRc = $self->handleOptionDumpConfigFileTemplate ();

    if (!defined $cfgDumpRc){
        $self->{return} = 1;
        return undef;
    }
    if ($cfgDumpRc){
        $self->{return} = 0;
        return 1;
    }

    my $removeUnreachableHost = $instconfig->getBatchValue ('RemoveUnreachableHost');
    if ($removeUnreachableHost && !$installation->isa ('SDB::Install::SAPSystem')){
        $self->setErrorMessage ("Wrong usage: option '--remove_unreachable_host' can be used with $gProductNameEngine installations only");
        return undef;
    }
	$msglst->appendMsgLst ($installation->getMsgLst());
    my $errlst = new SDB::Install::MsgLst ();
    if (!$removeUnreachableHost && $installation->isa ('SDB::Install::SAPSystem')){
        my $msg = $msglst->addMessage ("Checking configuration");
        my $rc = $self->{instconfig}->CheckParams ($self->{batch_mode});
        if (!$rc && $self->{batch_mode}){
            $self->setErrorMessage ("Configuration error", $self->{instconfig}->getErrMsgLst());
            $msg->getSubMsgLst()->appendMsgLst ($self->{instconfig}->getMsgLst());
            return undef;
        }
        if (!$self->{batch_mode}){
            if (!$rc && $self->{instconfig}->ErrorState ()){
                if (!defined $rc){
                    $self->setErrorMessage ('Configuration error:',$self->{instconfig}->getErrMsgLst ());
                    $msg->getSubMsgLst()->appendMsgLst ($self->{instconfig}->getMsgLst());
                    return undef;
                }
                $self->ShowErrorMsg ('Configuration error:', $self->{instconfig}->getErrMsgLst ());
            }
            if (!$self->ConfigureInteractive ($self->{instconfig})){
                $self->setErrorMessage ("Configuration error", $self->{instconfig}->getErrMsgLst ());
                $msg->getSubMsgLst()->appendMsgLst ($self->{instconfig}->getMsgLst());
                return undef;
            }
        }
        $msg->getSubMsgLst()->appendMsgLst ($self->{instconfig}->getMsgLst());
        if ($self->{instconfig}->isShowPending()) {
            $self->{instconfig}->showRemainingHosts($msglst, 1);
            $self->{return} = 0;
            return 1;
        }
        if (!gPrepareLogPath($installation->get_sid() ,'uninstall', $errlst,
                                              $self->{options}->{instlog_dir})){
            $self->setErrorMessage ('Cannot get log directory', $errlst);
            return undef;
        }
    }
    else{
        my $msg = $msglst->addMessage ("Checking configuration");
        $instconfig->setMsgLstContext ([$msg->getSubMsgLst ()]);
        my $rc = $instconfig->CheckParams ($self->{batch_mode});
        if (!$rc && $self->{batch_mode}){
            $self->setErrorMessage ("Configuration error", $instconfig->getErrMsgLst());
            return undef;
        }
        if (!$self->{batch_mode}){
            if (!$rc && $instconfig->ErrorState ()){
                if (!defined $rc){
                    $self->setErrorMessage ('Configuration error:',$instconfig->getErrMsgLst ());
                    return undef;
                }
                $self->ShowErrorMsg ('Configuration error:', $instconfig->getErrMsgLst ());
            }
            if (!$self->ConfigureInteractive ($instconfig)){
                $self->setErrorMessage ("Configuration error", $instconfig->getErrMsgLst ());
                return undef;
            }
        }
    }

   if (!$self->{batch_mode}){
        #
        # confirm
        #
        my $question;
        if ($removeUnreachableHost) {
            $question = "\nDo you want to remove the following host?\n\n"
                      . $removeUnreachableHost;
        }
        else {
            $question = "\nDo you want to uninstall the following installation?\n\n"
                      . $installation->asString(undef, 1, 1, 1);
        }
        if (!$removeUnreachableHost && $installation->isa ('SDB::Install::SAPSystem')){
            my $instance = $installation->getNewDBInstances()->[0];
            if (defined $instance){
                $question .= "\nAll data volumes and all log volumes will be removed!\n"
                           . "Data and log backup files in directory '"
                           . $installation->get_globalSidDir()
                           . "' and its subdirectories will be removed as well.\n";
            }
        }
        $question .= "\nConfirm ";
        if (!askConfirmation($question)) {
            $msglst->addProgressMessage("Uninstallation cancelled.\n");
            $self->{return} = 0;
            return 1;
        }
    }

    $errlst = new SDB::Install::MsgLst ();
    my $logname = 'hdbuninst';
    my $msg;
    if ($installation->isa ('SDB::Install::Installation::DatabaseStudio')){
        $msg = $msglst->addProgressMessage ("Uninstalling $gProductNameStudio");
        if (!gPrepareLogPath('studio','uninstall', $errlst, $self->{options}->{instlog_dir})){
            $self->setErrorMessage ('Cannot get log directory', $errlst);
            return undef;
        }
        $logname = 'hdbuninst_studio';
    }
    elsif ($installation->isa ('SDB::Install::Installation::Client')){
        $msg = $msglst->addProgressMessage ("Uninstalling $gProductNameClient...");
        if (!gPrepareLogPath('client','uninstall', $errlst, $self->{options}->{instlog_dir})){
            $self->setErrorMessage ('Cannot get log directory', $errlst);
            return undef;
        }
        $logname = 'hdbuninst_client';
    }
    elsif ($installation->isa ('SDB::Install::Installation::GenericServerPlugin')){
        $msg = $msglst->addProgressMessage ("Uninstalling " . $installation->getProductName() . "...");
        if (!gPrepareLogPath($installation->getProductKey(),'uninstall', $errlst,
                                              $self->{options}->{instlog_dir})){
            $self->setErrorMessage ('Cannot get log directory', $errlst);
            return undef;
        }
        $logname = 'hdbuninst_plugin_' . $installation->getProductKey();
    }
    elsif ($installation->isa ('SDB::Install::Installation::Generic')){
        $msg = $msglst->addProgressMessage ("Uninstalling " . $installation->getProductName() . "...");
        if (!gPrepareLogPath($installation->getProductKey(),'uninstall', $errlst,
                                              $self->{options}->{instlog_dir})){
            $self->setErrorMessage ('Cannot get log directory', $errlst);
            return undef;
        }
        $logname = 'hdbuninst_' . $installation->getProductKey();
    }
    elsif ($installation->isa ('SDB::Install::SAPSystem')){
        if ($removeUnreachableHost){
            $msg = $msglst->addProgressMessage ("Removing host '$removeUnreachableHost' from $gProductNameEngine...");
            print $installation->asString(undef, undef, 1, 1) . "\n";
            if (!gPrepareLogPath($installation->get_sid() ,'removehost', $errlst,
                                              $self->{options}->{instlog_dir})){
                $self->setErrorMessage ('Cannot get log directory', $errlst);
                return undef;
            }
            $logname     = 'hdbuninst_removehost';
            my $instance = $installation->getNewDBInstances()->[0];
            my $hostname = (defined $instance) ? $instance->get_host() : undef;
            $logname    .= '_' . $hostname if (defined $hostname);
        }
        else{
            $msg = $msglst->addProgressMessage ("Uninstalling $gProductNameEngine...");
            if (!gPrepareLogPath($installation->get_sid() ,'uninstall', $errlst,
                                              $self->{options}->{instlog_dir})){
                $self->setErrorMessage ('Cannot get log directory', $errlst);
                return undef;
            }
        }
        my $uid = undef;
        my $gid = undef;
        if (!$isWin){
            my $cfg = $installation->getUserConfig ();
            if(defined $cfg) {
                $uid = $cfg->{sidadm_id};
            }
            $gid = getgrnam ($gSapsysGroupName);
        }
        my $instances = $installation->getNewDBInstances();
        if(defined $instances && @$instances) {
        $self->addLogToTraceDir($installation,
                                $logname,
                                0,     # not check only
                                undef, # hostname
                                $uid, $gid);
        }
    }

    my $logUID = $self->{options}->{instlog_uid};
    my $logGID = $self->{options}->{instlog_gid};
    $self->{log}->addLogFileLocation(LOG_FORMAT_PLAIN,   $gLogDir, 1, "$logname.log", LOG_INDICATE_LOCATION_ON_STDOUT,      $logUID, $logGID);
    $self->{log}->addLogFileLocation(LOG_FORMAT_MSGLIST, $gLogDir, 1, "$logname.msg", LOG_DONT_INDICATE_LOCATION_ON_STDOUT, $logUID, $logGID);
    $self->{log}->addLogFileLocation(LOG_FORMAT_PLAIN,   $gTmpDir, 1, "$logname.log", LOG_DONT_INDICATE_LOCATION_ON_STDOUT, $logUID, $logGID);
    $self->{log}->addLogFileLocation(LOG_FORMAT_MSGLIST, $gTmpDir, 1, "$logname.msg", LOG_DONT_INDICATE_LOCATION_ON_STDOUT, $logUID, $logGID);
    my $no_package_registry = 0;

    my $force    = $instconfig->getValue ('Force');
    my $keepUser = $instconfig->getValue ('KeepUser');


    if ($installation->isa ('SDB::Install::SAPSystem') && $removeUnreachableHost){
        #
        # remove unreachable host
        #
        my $rHost        = $removeUnreachableHost;
        my $lcRemoveHost = lc($rHost);
        my $instance     = $installation->getNewDBInstances ()->[0];

        if (($lcRemoveHost eq lc(hostname())) # is standard local host
                || $instance->isHostLocal($lcRemoveHost)) {
            $self->setErrorMessage ("Local host '$rHost' cannot be removed. Use hdbremovehost to remove this host.");
            return undef;
        }

        my $remoteHosts = $instance->get_hosts ();
        my $found = 0;

        foreach my $host (@$remoteHosts){
            if (lc($host) eq $lcRemoveHost){
                $found = 1;
                last;
            }
        }
        if (!$found){
            $self->setErrorMessage ("Host '$rHost' is not a member of $gProductNameSystem");
            if (!$ignoreNotExisting){
                return undef;
            }
        }
        $instance->setMsgLstContext ([$msg->getSubMsgLst ()]);
        if (!defined $instance->removeUnreachableHost ($rHost, $force)){
            $self->setErrorMessage ("Removing host '$rHost' failed!", $instance->getErrMsgLst ());
            return undef;
        }
        return 1;
    }
    $installation->setMsgLstContext ([$msg->getSubMsgLst ()]);
    if (!$noexclusive || !$installation->isa ('SDB::Install::SAPSystem')){
        if (!$installation->OpenExclusive ()){
            $self->setErrorMessage ("Cannot open installation exclusively", $installation->getErrMsgLst());
            if (-e $installation->getProgramPath()){
                return undef;
            }
            $installation->Unregister ();
            return 1;
        }
    }

    if ($noexclusive && $installation->isa ('SDB::Install::SAPSystem')){
        $installation->setNoLock (1);
    }

    if (!defined $self->selectPackages ($installation)){
        return undef;
    }
    $packageManagerCollection->setMsgLstContext ([$msg->getSubMsgLst()]);
    $packageManagerCollection->addPackageManagerObject ('KEY', $installation);

    $self->{_uninstRoot} = $installation->isa ('SDB::Install::SAPSystem') ?
                        $installation->get_globalSidDir ()
                            :
                        $installation->getProgramPath ();

    if (!defined $packageManagerCollection->Uninstall (
        $keepUser ? 0 : 1,
        $force,
        $self->{instconfig})){
            $self->getErrMsgLst()->appendMsgLst ($packageManagerCollection->getErrMsgLst ());
            undef $rc; 
    }

    delete $self->{packageManagerCollection};
    $packageManagerCollection->CleanUp();

    if (($rc == 1) && $installation->isa ('SDB::Install::SAPSystem')) {
        $self->{instconfig}->showRemainingHosts($self->getMsgLst());
    }

    return $rc;
}

#-------------------------------------------------------------------------------
# Initializes the arguments of the command line
#
# Parameters string-array arguments
#
# Returns a reference to a string array containing the arguments
#         or undef in case of an error

sub InitCmdLineArgs{
    my ($self,$args) = @_;

    my $installation;

    my $instconfig = new SDB::Install::Configuration::Uninstallation
        ($self->{options}, $self->{configfile});

    $self->{instconfig} = $instconfig;
    $self->{_two_step_config} = 1;
    my @copy = @$args;
    $args = $self->SUPER::InitCmdLineArgs ($args, 1);
    delete $self->{_two_step_config};
    if (!defined $args){
        return undef;
    }

    $self->_findInstallations ();

    $instconfig->CheckParams ();

    my $sid = $instconfig->getBatchValue ('SID');
    my $path = $instconfig->getBatchValue ('PATH');
    my $ignoreNotExisting = $instconfig->getBatchValue ('IgnoreNotExisting');

    if (defined $sid){
        if (!defined $systems->{$sid}){
            $self->setErrorMessage ("$gProductNameEngine '$sid' not found");
            if ($ignoreNotExisting){
                print "$gProductNameEngine '$sid' not found\n";
                $self->{return} = 0;
                return 1;
            }
            return undef;
        }
        if (!$systems->{$sid}->hasNewDB() && !exists $pending_sids->{$sid}){
            $self->setErrorMessage ("SAP System '$sid' is no $gProductNameEngine instance");
            return undef;
        }
        $installation = $systems->{$sid};
        $instconfig->_initHDBSystem ($installation);
    }
    elsif (defined $path){
        my @statbuf = stat ($path);
        my $nPath;
        if ($isWin){
            $nPath = normalizePath ($path);
        }

        if (!@statbuf){
            if (defined $studio_installations){
                foreach my $p (keys (%$studio_installations)){
                    if (normalizePath ($p) eq normalizePath ($path)){
                        $installation = new SDB::Install::Installation::DatabaseStudio ($p);
                        last;
                    }
                }
            }
            if (!defined $installation && defined $client_installations){
                foreach my $p (keys (%$client_installations)){
                    if (normalizePath ($p) eq normalizePath ($path)){
                        $installation = new SDB::Install::Installation::Client ($p);
                        last;
                    }
                }
            }
        }
        else{
            if (defined $studio_installations){
                foreach my $p (keys (%$studio_installations)){
                    if (isSameFile ($p, $nPath, \@statbuf)){
                        $installation = new SDB::Install::Installation::DatabaseStudio ($p);
                        last;
                    }
                }
            }
            if (!defined $installation && defined $client_installations){
                foreach my $p (keys (%$client_installations)){
                    if (isSameFile ($p, $nPath, \@statbuf)){
                        $installation = new SDB::Install::Installation::Client ($p);
                        last;
                    }
                }
            }
        }

        if (!defined $installation){
            #
            # checking if this is a client via manifest file
            #
            $installation = new SDB::Install::Installation ($path);
            if (ref ($installation) eq 'SDB::Install::Installation'){
                undef $installation;
            }
        }

        if (!defined $installation){
            $self->setErrorMessage ("No installation found in path '$path'");
            if ($ignoreNotExisting){
                print "No installation found in path '$path'\n";
                $self->{return} = 0;
                return 1;
            }
            return undef;
        }
    } # if (defined $sid) ... elsif (defined $path)


    if (!defined $installation){
        $installation = $self->GetInstaller ()->getInstallation ();
    }
    if (defined $installation){
        if (!defined $self->_initConfiguration($installation, $instconfig)){
            return undef;
        }
    }

    $self->{instconfig} = $instconfig;

    if (defined $installation){
        $args = $self->_selectInstallation ($installation);
    }
    elsif ($self->isHelp()) {
        $self->PrintUsage();
        $self->{return} = 0;
        return undef;
    }

    return $args;
}

#-------------------------------------------------------------------------------
sub main{
    my $app = new __PACKAGE__;

    my $rc;
    my $any_key_already_asked = 0;
    $app->{stackBacktraceMsglst} = new SDB::Install::MsgLst ();
    eval{ 

        $rc = $app->InitCmdLineArgs (\@_);
    
    
        if (defined $app->{return}){
            LCM::DevelopmentTrace::RemoveTempDevelopmentTrace();
            if ($^O =~ /mswin/i && !$app->{batch_mode}
                                              && !$app->{options}->{noprompt}) {
                SDB::Install::App::Console->waitForKeyPress();
                $any_key_already_asked = 1;
            }
            return $app->{return};
        }
        if (defined $rc){
            $rc = $app->init ();
        }
        
    };
    my $msglst = $app->getMsgLst ();
    if (defined $app->{return}){
        LCM::DevelopmentTrace::RemoveTempDevelopmentTrace();
        if ($^O =~ /mswin/i && !$app->{batch_mode}
                   && !$app->{options}->{noprompt} && !$any_key_already_asked) {
            SDB::Install::App::Console->waitForKeyPress();
        }
        return $app->{return};
    }   

    if ($@){
        my $signalInfoText;
        if ($@ =~ /^__SIGINT__/){
            $msglst->addMessage ('User canceled uninstallation with Ctrl + c');
            $rc = 2;
        }
        elsif ($@ =~ /^__SIGPIPE__/){
            $app->setErrorMessage ('Broken pipe', $app->{stackBacktraceMsglst});
            $signalInfoText = $app->getSignalInfoText ();
            undef $rc;
        }
        elsif ($@ =~ /^__SIG(\w+)__/){
            $app->setErrorMessage ("Caught signal $1", $app->{stackBacktraceMsglst});
            $signalInfoText = $app->getSignalInfoText ();
            undef $rc;
        }
        else{
            $app->setErrorMessage ('unhandled exception: '. $@, $app->{stackBacktraceMsglst});
            $rc = undef;
        }
        if ($signalInfoText){
            $app->appendErrorMessage ($signalInfoText);
        }
    }

    
    if (defined $rc){
        if ($rc == 2){
            print "\n";
            $msglst->addProgressMessage('Uninstallation aborted!');
        }
        else{
            $msglst->addProgressMessage('Uninstallation done.');
        }

    }
    else{
        $app->ShowErrorMsg ('Uninstallation failed!',$app->getErrMsgLst);
    }

    $app->CleanUp ();
    
    if ($^O =~ /mswin/i && !$app->{batch_mode} && !$app->{options}->{noprompt}) {
        SDB::Install::App::Console->waitForKeyPress();
    }

    my $rootDir = $app->{_uninstRoot};
    undef $app;

    #
    # try to remove directories, which couldn't be deleted due to .nfs* files
    # caused by busy installer objects
    #

    if (!$isWin && defined $rootDir && -d $rootDir){
        removeEmptyDirsAtExit ($rootDir, defined $rc ? 0 : 1);
    }

    return defined $rc ? 0 : 1; 
}

sub shouldWarnIfCalledStandalone{
    return 1;
}

1;
