# AddHostsParser
#
# Example of internal result after parsing:
#
# $self->{hosts}->{lu111333}->{role }->['worker', 'streaming', 'extended_storage_worker']
#              |           |->{group}->'DEFAULT'
#              |           |->{storage_partition}->2
#              |           |->{workergroup}->['abc', 'def', 'ghi']
#			   |           |->{order}->'0'
#              |
#              |->{lu222444}->{role }->['standby', 'extended_storage_standby']
#                          |->{group}->'DEFAULT'
#			               |->{order}->'1'


package SDB::Install::Configuration::AddHostsParser;

use base SDB::Install::BaseLegacy;
use SDB::Install::Tools qw (trim);
use SDB::Install::Configuration::AnyMultiHostConfig qw ($hostRoleDefaultValue
                                                        $validHostRoles);
use SDB::Install::Globals qw ($gHostRoleXS2Standby
                              $gHostRoleXS2Worker
                              $gHostRoleComputeServer);
use strict;

sub trim ($);


#-------------------------------------------------------------------------------
# constructor

sub new {
    my $self = shift->SUPER::new (@_);
    $self->{hosts} = {};
    $self->{_roleMap} = {};
    return $self;
}


#-------------------------------------------------------------------------------

sub _isValidValue {
	my ($self, $value, $knownValues) = @_;
    foreach my $currVal (@$knownValues) {
        if ($value eq $currVal){
            return 1;
        }
    }
    return 0;
}

#-------------------------------------------------------------------------------
# Parses the arguments of '--addhosts', '--removehosts' or '--add_roles'.
# The hostname and host role names are converted into lower case.
#
# Returns 1 or undef
#
# addhosts Syntax
#    <addhost>[,<addhost>]...
#    <addhost> ::= <host_name>[:role=<role1>[:role=<role2>]...][:group=<name>]
#                      [:storage_partition=<number>][:workergroup=<workergroup1>[:workergroup=<workergroup2>]...]
#
# removehots Syntax
#    <removehost>[,<removehost>]...
#    <removehost> ::= <host_name>[:role=<role1>[:role=<role2>]...]
#
# add_roles Syntax
#    <addrole>[,<addrole>]...
#    <addrole> ::= <host_name>[:role=<role1>[:role=<role2>]...]

sub parse {
    my ($self,
        $string,  # entry list that should be parsed
        $paramID, # valid values: 'AddHosts'    (default)
                  #               'RemoveHosts'
                  #               'AddRoles'
                  #               'RemoveRoles'
        $isAutoAddXS2, # in case of 'AddHosts' or 'AddRoles'
                       # the role 'xs_worker'/'xs_standby'
                       # is added to 'worker'/'standby'
        $noUseDefaultRoleValue
       ) = @_;

    my $rc = 1;

    $paramID      = 'AddHosts' if (!defined $paramID);
    my $validArgs = ($paramID eq 'AddHosts')
                    ? ['role', 'group', 'storage_partition', 'workergroup']
                    : ['role'];

    $self->{string} = $string;
    $self->{hosts}  = {};
	my $index = 0;
    foreach my $hostEntry (split (',', $string)){

        my @hostArgs = split (':', $hostEntry);
        my $hostname = shift @hostArgs;

        if (!defined $hostname || (length($hostname) == 0)) {
            $self->PushError ("Host name required in front of '$hostEntry'");
            $rc = undef;
            next;
        }

        trim (\$hostname);
        $hostname = lc($hostname);

        if (exists $self->{hosts}->{$hostname}) {
            $self->PushError ("Duplicate host name '$hostname' found"
                                                          . " in '$string'");
            $rc = undef;
            next;
        }

        my $args = $self->_parseHostArguments($validArgs,
                                              $hostEntry,
                                              \@hostArgs,
                                              $paramID,
                                              $isAutoAddXS2,
                                              $noUseDefaultRoleValue);
        if (defined $args) {
            $self->{hosts}->{$hostname} = $args;
        }
        else {
            $rc = undef;
        }
       	$self->{hosts}->{$hostname}->{order} = $index;
       	$index ++;
    }
    return $rc;
}

#-------------------------------------------------------------------------------
# Parses the keywords of one host entry
#
# Returns a hash map containing the keywords with the values
# or undef in case of an error

sub _parseHostArguments {

    my ($self,
        $knownKeys, # an array containing the valid keywords
        $hostEntry, # e.g. lu000111:group=default:role=worker:role=streaming
        $hostArgs,  # e.g. ['group=default', 'role=worker', 'role=streaming]
        $paramID,   # 'AddHosts' or 'RemoveHosts' or 'AddRoles'
        $isAutoAddXS2, # in case of 'AddHosts' or 'AddRoles'
                       # the role 'xs_worker'/'xs_standby'
                       # is added to 'worker'/'standby'
        $noUseDefaultRoleValue # do not add the default host role value
       ) = @_;

    my $result         = {};
    my $hasGroup       = 0;
    my $hasPartition   = 0;
    my $hasWorkerGroup = 0;

    foreach my $currArg (@$hostArgs){

        my @list = split ('=', $currArg);
        if (@list != 2){
            $self->PushError ("Invalid expression '$currArg' in '$hostEntry'"
                              . ' (<keyword>=<value> required)');
            return undef;
        }
        my ($key,$value) = @list;
        trim (\$key);
        if (!$self->_isValidValue($key, $knownKeys)){
            $self->PushError("Unknown keyword '$key' found in '$hostEntry'");
            return undef;
        }
        trim (\$value);

        if ($key eq 'role') {

            $value = lc($value);
            if (!exists $validHostRoles->{$value}) {
                $self->PushError("Unknown host role '$value' found in '$hostEntry'");
                return undef;
            }

            if ((($paramID eq 'AddRoles') || ($paramID eq 'RemoveRoles'))
                && $validHostRoles->{$value}->{isColumnStore}) {
                $self->PushError("Host role '$value' is not allowed");
                return undef;
            }

            if (defined $validHostRoles->{$value}->{defaultGroup}) {
                $hasGroup = 1;
            }

            if (defined $validHostRoles->{$value}->{hasPartition}) {
                $hasPartition = 1;
            }

            if (defined $validHostRoles->{$value}->{hasWorkerGroup}) {
                $hasWorkerGroup = 1;
            }

            if (defined $result->{$key}) {
                # check additional role

                my $isStandby = $validHostRoles->{$value}->{isStandby};

                foreach my $currRole (@{$result->{$key}}) {

                    if ($value eq $currRole) {
                         $self->PushError
                            ("Duplicate host role '$value' found in '$hostEntry'");
                         return undef;
                    }

                    if($gHostRoleComputeServer =~ /$value|$currRole/){
                        $self->appendErrorMessage("Host role '$gHostRoleComputeServer' cannot be combined with other host roles");
                        return undef;
                    }

                    my $currStandby = $validHostRoles->{$currRole}->{isStandby};

                    if (( $isStandby && !$currStandby) ||
                        (!$isStandby &&  $currStandby)) {
                        $self->PushError ("Host role '$value' is not allowed in"
                            . " combination with host role '$currRole' in '$hostEntry'");
                        return undef;
                    }
                }
                push @{$result->{$key}}, $value;
            }
            else {
                $result->{$key} = [$value];
            }
            if ($validHostRoles->{$value}->{isAccelerator}) {
                $self->{isAnyAccelerator} = 1;
            }
        } elsif ($key eq 'workergroup') {
            $result->{$key} = $result->{$key} || [];
            push(@{$result->{$key}}, $value);
        }
        else {
            if (($key eq 'storage_partition')
                           && (($value !~ m/^([0-9]+)$/) || ($value < 2))) {
                $self->PushError("Storage partition '$value' has to be a"
                                 . " number greater than one in '$hostEntry'");
                return undef;
            }
            $result->{$key} = $value;
        }
    }

    if (!$noUseDefaultRoleValue && !defined $result->{role}) {

        $result->{role} = [$hostRoleDefaultValue];

        if (defined $validHostRoles->{$hostRoleDefaultValue}->{defaultGroup}) {
            $hasGroup = 1;
        }
        if (defined $validHostRoles->{$hostRoleDefaultValue}->{hasPartition}) {
            $hasPartition = 1;
        }
        if (defined $validHostRoles->{$hostRoleDefaultValue}->{hasWorkerGroup}) {
            $hasWorkerGroup = 1;
        }
    }

    if ($isAutoAddXS2
        && (($paramID eq 'AddHosts') || ($paramID eq 'AddRoles'))) {

        my $workerFound  = 0;
        my $standbyFound = 0;
        my $xs2Found     = 0;

        foreach my $currRole (@{$result->{role}}) {

            if ($validHostRoles->{$currRole}->{isColumnStore}) {
                if ($validHostRoles->{$currRole}->{isStandby}) {
                    $standbyFound = 1;
                }
                else {
                    $workerFound = 1;
                }
            }
            elsif ($validHostRoles->{$currRole}->{isXS2}) {
                $xs2Found = 1;
            }
        }

        if ($workerFound && !$xs2Found) {
            push @{$result->{role}}, $gHostRoleXS2Worker;
        }
        elsif ($standbyFound && !$xs2Found) {
            push @{$result->{role}}, $gHostRoleXS2Standby;
        }
    }

    if (!$hasGroup && defined $result->{group}) {
        $self->PushError("Keyword 'group' is not allowed in '$hostEntry'");
        return undef;
    }

    if (!$hasPartition && defined $result->{storage_partition}) {
        $self->PushError
            ("Keyword 'storage_partition' is not allowed in '$hostEntry'");
        return undef;
    }

    if (!$hasWorkerGroup && defined $result->{workergroup}) {
        $self->appendErrorMessage(
            "Keyword 'workergroup' is not allowed in '$hostEntry'");
        return undef;
    }

    return $result;
}


#-------------------------------------------------------------------------------
# Returns the group of the specified host

sub getGroup {
    my ($self, $wantedHostname) = @_;

    my $selectedHost = $self->{hosts}->{$wantedHostname};

    return (defined $selectedHost)
           ? $selectedHost->{group}
           : undef;
}


#-------------------------------------------------------------------------------
# Returns a reference to an array containing the host names

sub getHosts{
    return [keys %{$_[0]->{hosts}}];
}

#-------------------------------------------------------------------------------
# Returns a reference to an array containing the host names of all hosts with
# role 'worker' or 'standby'

sub getHanaHosts{
    my @hanaHosts;
    my $hosts = $_[0]->{hosts};
    my $role;
    foreach my $host (keys (%$hosts)){
        foreach $role (@{$hosts->{$host}->{role}}){
            if (defined $validHostRoles->{$role} &&
                    $validHostRoles->{$role}->{isColumnStore}){
                push @hanaHosts, $host;
            }
        }
    }
    return \@hanaHosts;
}

#-------------------------------------------------------------------------------
# Returns a reference to an array of role names the specified host

sub getRoles {
    my ($self, $wantedHost) = @_;

    return (defined $self->{hosts}->{$wantedHost})
           ? $self->{hosts}->{$wantedHost}->{role}
           : undef;
}


#-------------------------------------------------------------------------------
# Returns the storage partition number of the specified host

sub getStoragePartition {
    my ($self, $wantedHostname) = @_;

    my $selectedHost = $self->{hosts}->{$wantedHostname};

    return (defined $selectedHost)
           ? $selectedHost->{storage_partition}
           : undef;
}

#-------------------------------------------------------------------------------
# Returns the worker groups of the specified host

sub getWorkerGroups {
    my ($self, $wantedHostname) = @_;

    my $selectedHost = $self->{hosts}->{$wantedHostname};

    return (defined $selectedHost)
           ? $selectedHost->{workergroup}
           : undef;
}

#-------------------------------------------------------------------------------
# Returns the specified value (e.g. 'storage_partition')
# of a specified host.
# In case of 'role', a column store role (i.e. worker or standby) is returned.

sub getValue {
    my ($self,
        $wantedHostname,
        $wantedKey, # see @knownAddhostKeys
       ) = @_;

    my $value = undef;
    my $selectedHost = $self->{hosts}->{$wantedHostname};

    if (defined $selectedHost) {
        if ($wantedKey eq 'role') {
            foreach my $currRole (@{$selectedHost->{$wantedKey}}) {
                if ($validHostRoles->{$currRole}->{isColumnStore}) {
                    $value = $currRole;
                    last;
                }
            }
        }
        else {
            $value = $selectedHost->{$wantedKey};
        }
    }
    return $value;
}

#-------------------------------------------------------------------------------
# Returns 1 if any additional host with an accelerator should be added.

sub isAcceleratorFound {
    return ($_[0]->{isAnyAccelerator}) ? 1 : 0;
}

sub getOrder{
	my($self,$wantedHost) = @_;
    return (defined $self->{hosts}->{$wantedHost})
           ? $self->{hosts}->{$wantedHost}->{order}
           : -1;
}
1;
