# Order and Dependencies of Parameters used for Multiple Host Systems
#
# Parameter ID                  | HdbModify | HdbReg | NewDB | Rename | Upgrade | ServerUninstallation
# ------------------------------+-----------+--------+-------+--------+---------+---------------------
# RemoteExecution               | X         |  X     | X     |  X     |  X      |  1) (a)
# UseHttp                      s| 2)        |  2)    | 2)    |  2)    |  2)     |  2)
# InstallHostagent              | X         |  X     | X     |  X     |  X      |
# SID                           | X [C1] (b)|  X (c) |       |  X (c) |  X  (c) |
# AddRoles    [C]               | X  (d)    |        |       |        |         |
# RemoveHosts [C]               | X  (d)    |        |       |        |         |
# AddHosts    [C]               | X  (d)    |        | X  (d)|        |         |
# HostMap                       |           |  X  (e)|       |  X  (e)|         |
# Scope            [C2]        s|           |  1) (d)|       |  1) (d)|  1) (b) |  1) (d)
# ScopeInteractive [C2]      i s|           |  4) (d)|       |  4) (d)|  4) (b) |  4) (d)
# Win:   HostagentUserName   i s|  5)       |  6)    |  7)   |  6)    |  8)     |  6)
# Linux: InstallSSHKey         s|  9)       | 10)    | 11)   | 10)    | 10)     | 10)
# Linux: RootUser    [C1][H] i s|  9)       | 10)    | 11)   | 10)    | 10)     | 10)
# Linux: RootPassword[C1][H] i s| 12)       | 13)    | 14)   | 13)    | 13)     | 13)
# SkipHostagentPw              s| 15)       |        |       | 16)    |  X      |
# HostagentPassword      [H] i s| 17)       |  6)    |  7)   | 18)    | 19)     |  6)
# SID                           |           |        |  X (f)|        |         |
# ListenInterface            i  |  X        |  X     |       |  X     |         |
# InternalNetwork            i  |  X        |  X     | 20)   |  X     |         |
# UID / GID                     |           |  X (g) |  X (g)|  X (g) |         |
# Win: RetryHostagentUserName   |           |        |  X (h)|        |         |
# RetryHostagentPassword        |           |        |  X (i)|        |         |
#
#  i - 'set_interactive' is set per default during initialization
#  s - 'skip'            is set per default during initialization
#
#  X - parameter is used
#
#   1) if any remote host exists
#   2) if RemoteExecution='saphostagent'
#   4) if any remote host exists AND Scope is not specified on the command line
#   5) if InstallHostagent OR (RemoteExecution='saphostagent' AND add/remove host is specified)
#   6) if InstallHostagent OR (RemoteExecution='saphostagent' AND isRemoteScope)
#   7) if InstallHostagent OR (RemoteExecution='saphostagent' AND add host is specified)
#   8) if InstallHostagent
#   9) if RemoteExecution='ssh'
#  10) if RemoteExecution='ssh' AND isRemoteScope
#  11) if RemoteExecution='ssh' AND add host is specified
#  12) if RemoteExecution='ssh' AND SSH key does not exist
#  13) if RemoteExecution='ssh' AND SSH key does not exist AND isRemoteScope
#  14) if RemoteExecution='ssh' AND SSH key does not exist AND add host is specified
#  15) if InstallHostagent AND  RemoteExecution='ssh'
#  16) if InstallHostagent AND (RemoteExecution='ssh' OR Scope='instance')
#  17) if NOT SkipHostagentPw AND (InstallHostagent OR (RemoteExecution='saphostagent' AND add/remove host is specified)
#  18) if NOT SkipHostagentPw AND (InstallHostagent OR (RemoteExecution='saphostagent' AND isRemoteScope))
#  19) if NOT SkipHostagentPw AND InstallHostagent
#  20) if add host is specified
#
#
#  [C]  calls handleAddOrRemoveHosts
#       - creates the class RemoteHosts or RemoteHostctrlHosts containing added/removed hosts
#       - connections to the additional hosts are established in case of SSH
#
#  [C1] calls initRemoteHosts
#       - creates the class RemoteHosts or RemoteHostctrlHosts containing existing hosts
#         if $instconfig->{_remoteHosts} is undefined
#       - connections to the existing hosts are established in case of SSH
#
#  [C2] calls initRemoteHosts if RemoteExecution='saphostagent' and Scope='System'
#       - creates the class RemoteHostctrlHosts containing existing hosts
#
#  [H]  calls CollectOtherHostInfo
#
#
#  (a) - Before checking parameters the subroutine ServerUninstallation::initDefaults
#        detects remote hosts and enables the parameters RemoteExecution
#        that is skipped during initialization.
#
#      - subroutine setRemoteExecution
#        -- enables the parameter Scope and ScopeInteractive
#        -- Scope is set to 'system' if the command line contains '--root_user'
#
#  (b) calls enableMultipleHostParameters in case of remote execution
#       - SSH parameters or SAPHostagent parameters with sidadm are enabled
#       - if add/remove host is detected later, HostagentUser/sapadm will be
#         used instead of sidadm in case of SAPHostagent
#
#  (c) - calls tryEnableScopeParams (if not is slave)
#        -- detects remote hosts and enables the parameters Scope and ScopeInteractive
#        -- Scope is set to 'system' if the command line contains '--root_user'
#
#      - calls enableHostagentPasswordIfRequired
#        -- enables the parameters HostagentUserName and HostagentPassword
#           if a local SAPHostagent is not yet installed
#
#  (d) calls enableMultipleHostParameters in case of remote execution
#       - SSH parameters or SAPHostagent parameters hostagent user/password are enabled
#
#  (e) initRemoteHosts uses new hostnames according to the parameter HostMap
#
#  (f) - calls enableHostagentPasswordIfRequired
#        -- enables the parameters HostagentUserName and HostagentPassword
#           if a local SAPHostagent is not yet installed
#
#      - CollectOtherHostInfo has to be called before in order
#        to check free SID
#
#  (g) CollectOtherHostInfo has to be called before in order
#      to set default UID/GID and to check free UID/GID
#
#  (h) retries the parameter setting of HostagentUserName if not already set
#
#  (i) retries the parameter setting of HostagentPassword if not already set

package SDB::Install::Configuration::AnyConfig;

use strict;

use SDB::Common::BuiltIn;
use SDB::Install::Configuration;
use SDB::Install::Globals qw(
    $gDriveWin
    $gOperationCollectHostInfo
    $gOperationCollectInstallerInfo
    $gProductName
    $gProductNameAccelerator
    $gProductNameEs
    $gProductNameStudio
    $gProductNameSystem
    $gSapsysGroupIdDefault
    $gMaxSapsysGroupIDDefault
    $gSapsysGroupName
    $gShortProductNameAccelerator
    $gShortProductNameEs
    $gShortProductNameLSS
    $gShortProductNameXS2
    $gSignatureManifestName
    $gNameAccelerator
    $gShortProductNameTenant
    $gProductNameXS2
    $gProductNameLSS
    $gProductNameEngine
    $gHostRoleXS2Worker
    $gProductNameSHA
    GetAutoInitializeServicesRolesComponentMap
    determineSignedManifestPath
    $gShortProductNameLSS
    $gSecureStoreTypeSSFS
    $gSecureStoreTypeLSS
);
use SDB::Install::Installer;
use SDB::Install::LayeredConfig qw (CFG_LAYER_SYSTEM);
use SDB::Install::RemoteHostctrlHosts;
use SDB::Install::RemoteHosts;
use SDB::Install::Saphostagent qw(sapadmUserExists getSaphostexecPath);
use SDB::Install::SAPSystem qw(CleanupSAPSystemCache
                               CollectSAPSystems
                               isForbiddenSID
                               matchesPatternSID
                               $STEP_CREATE_INSTANCE);
use SDB::Install::System qw (copy_file
                             deltree
                             find
                             FIND_TYPE_FILE
                             getAllUsersAppDataPath
                             getFilesTimestampPID
                             makedir
                             nslookup
                             isSidadmin
                             getSAPDrive
                             $hostname_regex
                             $ipv4_regex
                             is_local_address
                             isAdmin
                             is_loopback_address);
use SDB::Install::SysVars   qw($isWin $path_separator $installerIs64bit);
use SAPDB::Install::Hostname;
use SAPDB::Install::MachInfo;
use SDB::Install::DirectoryWalker;
use SDB::Install::PasswordPolicy;
use SDB::Install::Version;
use File::Basename qw (dirname);
use SDB::Install::Tools qw (readini);
use SDB::Install::Configuration::EffectiveModulesRemoteCheck;
use SDB::Install::Sql::SqlConnectionProvider;
use SDB::Install::CloudEditionUpdateCheck;
use SDB::Common::Utils qw(getSidcryptName );
use experimental qw(smartmatch);
use Digest::SHA qw(sha256_hex);

our @ISA = qw (SDB::Install::Configuration Exporter);

our $remoteTmpDir = sprintf ('hdbinst_remote_check_%s_%d', hostname(), $$);


# runtime files required for CollectHostinfo
our @instruntime_files;
{
    my ($instruntime_file);
    foreach my $module (@SDB::Install::Configuration::EffectiveModulesRemoteCheck::effective_modules){
        (undef, $instruntime_file) = ($module =~ /instruntime\/(.+)/);
        if ($instruntime_file){
            push @instruntime_files, $instruntime_file;
        }
    }
}

our @systemUsageValidValues = qw (production test development custom);
our @systemUsageUIValues = ('System is used in a production environment',
                            'System is used for testing, not production',
                            'System is used for development, not production',
                            'System usage is neither production, test nor development');

# executable required for CollectHostinfo
our $installerExe = 'hdbinst';

our $MAX_INSTANCE_NUMBER = 97;

our @EXPORT = qw (@instruntime_files $installerExe $mcdErrorMessageTemplate);

#-------------------------------------------------------------------------------
# executable required for CollectHostinfo
sub getInstallerExe {return $installerExe;}


#-------------------------------------------------------------------------------
# Constructor
sub new{
    my $self = shift->SUPER::new (@_);

    return $self;
}


#-------------------------------------------------------------------------------
# Checks if the specified SID is valid and not already used
#
# Parameters: string $additionalSID
#             string $paramName     # e.g. $self->{params}->{SID}->{str}
#
# Returns int retCode

sub checkAdditionalSID {
    my ($self, $additionalSID, $paramName) = @_;

    if (!$self->matchesPatternSID($additionalSID)){
        $self->PushError ("Invalid $paramName $additionalSID");
        return 0;
    }

    if ($self->isForbiddenSID($additionalSID, $self)) {
        return 0;
    }

    my $systems = $self->getCollectSAPSystems();
    if (exists $systems->{$additionalSID} && defined $systems->{$additionalSID}) {
        if(not defined $self->{_dontFailOnAlreadyInUseParameter} or !$self->{_dontFailOnAlreadyInUseParameter})   # hack
        {
            $self->PushError ("$paramName \"$additionalSID\" "."is already in use");
            return 0;
        }
    }
    return 1;
}

#-------------------------------------------------------------------------------
# Checks the options '--ets_administrator'

sub checkAcceleratorUser {
    my ($self, $username) = @_;

    if (length($username) == 0) {
        $self->PushError ("SAP ASE Administrator User value is empty");
        return 0;
    }

    my $passwordParam =  $self->{params}->{AcceleratorPassword};
    $passwordParam->{str} = sprintf($passwordParam->{str_templ}, $username);

    return 1;
}


sub checkCustomPluginsPath{
    my ($self, $customPluginsPath) = @_;
    if (! -e $customPluginsPath){
        $self->appendErrorMessage ("Directory '$customPluginsPath' doesn't exist.");
        return undef;
    }
    if (! -d _ ){
        $self->appendErrorMessage ("'$customPluginsPath' isn't a directory.");
        return undef;
    }
    return 1;
}

#-------------------------------------------------------------------------------
# Adds the user name into the password string.

sub changePasswordStr {
    my ($self, $username) = @_;

    if (defined $username && exists($self->{params}->{Password})) {
        $self->{params}->{Password}->{str} =
            sprintf($self->{params}->{Password}->{str_templ}, $username);
    }
}


#-------------------------------------------------------------------------------
# Checks if the specified SID is valid and is used
#
# Parameter: string $existingSID
#            string $paramName   # e.g. $self->{params}->{SID}->{str}
#            SDB::Install::SAPSystem  $sapSystems    # may be undef
#
# Returns int retCode

sub checkExistingSID {
    my ($self, $existingSID, $paramName, $sapSys) = @_;

    if (!$self->matchesPatternSID($existingSID)) {
        $self->PushError ("Invalid $paramName \"$existingSID\"");
        return 0;
    }

    if (!defined $sapSys) {
         my $systems = $self->getCollectSAPSystems();
         $sapSys     = $systems->{$existingSID};
    }

    if (!defined $sapSys) {
        $self->PushError ("$paramName \"$existingSID\" " . "does not exist");
        return 0;
    }

    if (!$sapSys->hasNewDB){
        $self->PushError ("$paramName \"$existingSID\" " . "has no HDB instance");
        return 0;
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Checks that the specified hostagent password is not empty.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.
#
# 'saphostexec -install -passwd' allows any nonempty password for sapadm
# (although it warns if pw is weak (e.g. short)):

sub checkHostagentPassword {

    my ($self, $value) = @_;

    if (!$value || length ($value) == 0) {
        $self->PushError
            ("$self->{params}->{HostagentPassword}->{str} must not be empty.");
        return 0;
    }

    if (!$self->{isSlave} || !$self->isCheckOnly()) {

        my $userName = $self->getValue('HostagentUserName');
        $userName    = 'sapadm' if (!defined $userName);
        my $user     = new SDB::Install::User($userName);
        if ($user->exists()){
            $user->setMsgLstContext ([$self->getMsgLst ()]);
            my $rc = $user->verifyPassword ($value);
            if (!defined $rc){
                # cannot verify password => skipping check
                return 1;
            }
            if (!$rc) {
                $self->PushError ($self->{params}->{HostagentPassword}->{str}
                              . ': Unknown user password combination');
                return 0;
            }
        }
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Checks that the specified hostagent user name is not empty.
# This user name is used under Windows only.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.

sub checkHostagentUserName {
    my ($self, $userName) = @_;

    if (!$userName || length ($userName) == 0) {
        $self->PushError("$self->{params}->{HostagentUserName}->{str} must not be empty.");
        return 0;
    }

    my $paramType = sapadmUserExists($userName) ? 'passwd' : 'initial_passwd';
    $self->setType('HostagentPassword', $paramType);
    return 1;
}


#-------------------------------------------------------------------------------

sub checkInstanceNumber {

    my ($self, $value) = @_;

    if ($value =~ /\D/){
        $self->PushError("Instance number \'$value\' is not a positive number");
        return 0;
    }

    if ($value > $MAX_INSTANCE_NUMBER){
        $self->PushError("Instance number \'$value\' must not be greater than $MAX_INSTANCE_NUMBER");
        return 0;
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Check if the $instanceNumber is not used yet
#
# Parameters:   string $instanceNumber
#               string $paramDescription
#               string $ignoreOwnSid contains the SID (optional)

sub checkAdditionalInstanceNumber {
    my ($self, $instanceNr, $paramDescription, $ignoreOwnSid) = @_;
    if ($instanceNr < 10){
        $instanceNr = sprintf ("%02d", $instanceNr);
    }

    my $sid = SDB::Install::SAPSystem::ExistsInstanceNumber ($instanceNr);
    my $rc = 1;
    if ($sid && !(defined $ignoreOwnSid && $ignoreOwnSid eq $sid)){
        $self->AddError ("$paramDescription \"$instanceNr\" is already in use by local SAP System $sid");
        $rc = 0;
    }
    if (defined $self->{remoteInstances} && defined $self->{remoteInstances}->{$instanceNr}){
        foreach my $entry (@{$self->{remoteInstances}->{$instanceNr}}){
            if (defined $ignoreOwnSid && $ignoreOwnSid eq $entry->[1]){
                next;
            }
            $self->PushError ("$paramDescription \"$instanceNr\" is already in use by system '$entry->[1]' on host '$entry->[0]' (type = $entry->[2])");
            $rc = 0;
        }
    }
    return $rc;
}

sub checkShell {
    my ($self, $value) = @_;
    my $etcShells = "/etc/shells";
    if (!-x $value){
        $self->PushError ("Invalid shell: Cannot find executable file '$value'");
        return 0;
    }

    return 1 if(-r $etcShells);

    if (!$self->{shells}) {
        if (!open (FD, '/etc/shells')){
            return 1;
        }
        $self->{shells} = [ grep {chomp $_} <FD> ];
        close (FD);
    }
    if (! grep { $_ eq $value } @{$self->{shells}}) {
        $self->PushError ("Invalid login shell '$value': Please see /etc/shells");
        return 0;
    }
    return 1;
}

#-------------------------------------------------------------------------------
# Checks the group ID
#
# Returns int retCode

sub checkGID {

    my ($self, $gid) = @_;

    my $gidParam = $self->getParamGID();
# Must make sure that no auto-vification of the
# GID param key occurs during scenarios that should
# not have it. This would blow up the WebUI
    my $paramName = (exists $self->{params}->{GID}) ? $self->{params}->{GID}->{str} : $gidParam->{str};

    if (!defined $gid || $gid eq ''){
        $self->PushError ("$paramName is empty");
        return 0;
    }
    if ($gid =~ /\D/){
        $self->PushError ("$paramName '$gid' has to be a decimal number");
        return 0;
    }
    if ($gid < 1){
        $self->PushError ("$paramName '$gid' has to be greater than zero");
        return 0;
    }

    my $sapsysGID = SDB::Common::BuiltIn->get_instance()->getgrnam ($gSapsysGroupName);

    if (defined $sapsysGID) {
        if ($sapsysGID != $gid) {
            my $errorMessage = "Group ID '$gid' for '$gSapsysGroupName' is invalid because '$gSapsysGroupName' already exists with ID '$sapsysGID' on the local host";
            $self->PushError($errorMessage);
            return 0;
        }
    }
    else {
        my $groupName = SDB::Common::BuiltIn->get_instance()->getgrgid($gid);

        if (defined $groupName) {
            my $errMsg =
                "$paramName '$gid' is already in use by group '$groupName'";
            my $freeID = $self->getFreeGroupID();
            if (defined $freeID) {
                $errMsg .= " (available ID: $freeID)";
            }
            $self->PushError($errMsg);
            return 0;
        }
    }
    return $self->_checkRemoteGroups($gid);
}

sub _checkRemoteNonSapsysGroups {
    my ($self, $gid, $msgLst) = @_;
    my $remoteGroups = $self->{remoteGids}->{$gid} // [];
    my @nonSapsysRemoteGroups = grep { $_->[1] ne $gSapsysGroupName } @{$remoteGroups};
    return 1 if !@nonSapsysRemoteGroups;

    for my $group (@nonSapsysRemoteGroups) {
        my $groupName = $group->[1];
        my $host = $group->[0];
        $msgLst->addError("Group ID $gid is taken by group '$groupName' on host $host.");
    }
    return 0;
}

sub _checkRemoteGroups {
    my ($self, $gid) = @_;
    my $sapsysids = $self->{sapsysids};
# sapsysids should always have just one entry otherwise
# CollectOtherHostInfos would fail
    if (defined $sapsysids && (scalar(keys %$sapsysids) == 1) && $gid ne (keys %$sapsysids)[0]) {
        my $existingSapsysGid = (keys %$sapsysids)[0];
        my $hostsStr = join(', ', @{(values %$sapsysids)[0]});
        $self->appendErrorMessage("Group ID '$gid' for '$gSapsysGroupName' is invalid because '$gSapsysGroupName' already exists with ID '$existingSapsysGid' on host(s) $hostsStr");
        return 0
    }
# The check for remote non-sapsys groups is necessary here because of the case
# when sapsys exists on none of the hosts, but the parameter has a value, which
# is already taken by another group on some of the remote hosts.
    my $msgLst = SDB::Install::MsgLst->new();
    if (!$self->_checkRemoteNonSapsysGroups($gid, $msgLst)) {
        my $error = "Group ID $gid for '$gSapsysGroupName' is invalid because different group with the same ID exists on some of the remote hosts:";
        $self->appendErrorMessage($error, $msgLst);
        return 0;
    }

    return 1;
}

#-------------------------------------------------------------------------------
# Checks the parameter '--keep_user'

sub checkKeepUser {

    my ($self, $value) = @_;

    if (defined $value && ($value =~ /$bool_true_pattern/i)) {
        $self->setSkip('KeepUserHomeDir');
    }
    return 1;
}

sub getTenantDatabaseOSUsers{
    my ($self) = @_;
    if (!defined $self->{_mdcDatabaseOSUsers}){
        $self->{_mdcDatabaseOSUsers} = [];
        my $instance = $self->getOwnInstance ();
        if ($instance->isIsolatedMultiDb ()){
            $self->getMsgLst()->addProgressMessage ("Checking multidb isolation...");
            my $tenantDatabases = $instance->getTenantDatabases();
            if (defined $tenantDatabases){
                foreach my $tenantDB (@$tenantDatabases){
                    my $user = new SDB::Install::User ($tenantDB->getUsername());
                    if ($user->exists ()){
                        push @{$self->{_mdcDatabaseOSUsers}}, $user;
                    }
                }
            }
        }
    }
    return $self->{_mdcDatabaseOSUsers};
}


our $mcdErrorMessageTemplate = "$gShortProductNameTenant '%s' ERROR: %s";

sub checkLocalTenantDBUsersAndGroups{
    my ($self, $localhostname) = @_;
    my $rc = 1;
    if (defined $self->{_mdcDatabases}){
        my ($database, $container, $username, $groupname, $uid, $gid);
        $self->{_mdcDatabaseOSUsers} = [];
        foreach my $entry (@{$self->{_mdcDatabases}}){
            ($database, $container, $username, $groupname, $uid, $gid) = @$entry;
            require SDB::Install::User;
            my $user = new SDB::Install::User ($username);
            my $user_exists = 1;
            if (!$user->exists ()){
                $self->appendErrorMessage (sprintf ($mcdErrorMessageTemplate, $database, "required operating system user '$username' doesn't exist on host '$localhostname'"));
                $rc = undef;
                $user_exists = 0;
            }
            else{
                push @{$self->{_mdcDatabaseOSUsers}}, $user;
                if($user->id() != $uid){
                    $self->appendErrorMessage (sprintf ($mcdErrorMessageTemplate, $database, "operating system user '$username' has wrong uid '" .
                        $user->id()."' on host '$localhostname', expected '$uid'"));
                    $rc = undef;
                }
            }

            require SDB::Install::Group;
            my $group = new SDB::Install::Group ($groupname);
            if (!$group->exists ()){
                $self->appendErrorMessage (sprintf ($mcdErrorMessageTemplate, $database, "required operating system group '$groupname' doesn't exist on host '$localhostname'"));
                $rc = undef;
            }
            else{
                if($group->id() != $gid){
                    $self->appendErrorMessage (sprintf ($mcdErrorMessageTemplate, $database, "operating system group '$groupname' has wrong gid '" .
                        $group->id()."' on host '$localhostname', expected '$gid'"));
                    $rc = undef;
                }
            }
        }
    }
    return $rc;
}


#-------------------------------------------------------------------------------
# Checks the password of the root user in case of a distributed system.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.

sub checkRootPassword {

    my ($self, $password) = @_;

    my $rc = 1;
    foreach my $remoteHosts (@{$self->getRemoteHostsObjects()}){
        $remoteHosts->setUserPass ($password);
        $remoteHosts->ResetError(); # reset key authorization error
        $remoteHosts->resetMsgLstContext();

        my $paramInfo = "user name = '$self->{params}->{RootUser}->{value}'";

        my $msg = $self->AddMessage("Checking authorization ($paramInfo)");
        if ($remoteHosts->authPassword($self->getValue('InstallSSHKey')) != 0) {
            $self->AddError ("Authorization failed ($paramInfo)",
                             $remoteHosts);
            $rc = 0;
        }
        $self->AddSubMsgLst ($msg, $remoteHosts);
    }
    if ($rc){
    	if ($self->isCollectOtherHostInfosRequired()) {
	        if (!defined $self->CollectOtherHostInfos()){
	                $self->{params}->{RootPassword}->{no_retry} = 1;
	                return undef;
	        }
    	}
    }
    return $rc;
}


#-------------------------------------------------------------------------------
# Checks the root user name in case of a  distributed system.
# Furthermore this method tries to use or create an authorization key
# to access remote hosts.
# An authoization key can only be created if all hosts provide the same
# root user name.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.

sub checkRootUser {

    my ($self, $remoteRootUser) = @_;

    my $rc = $self->initRemoteHosts();

    if ($rc) {
        $rc = $self->tryRemoteKeyAuthorization($remoteRootUser);
    }

    return $rc;
}


sub checkSSOCertificate{
    my ($self, $value) = @_;
    my $param = $self->{params}->{SSOCertificate};
    if (!defined $value || !-f $value) {
        $self->appendErrorMessage ("$param->{str} '$value' has to be an existing file");
        return 0;
    }
    my $msg = $self->getMsgLst ()->addMessage ("Checking $param->{str} '$value'");
    my $instance = $self->getOwnInstance ();
    if (!defined $instance){
        $msg->getSubMsgLst ()->addWarning ("Skipping $param->{str} check, HDB instance is not defined.");
        return 1;
    }
    my $nr = $instance->get_nr ();
    my ($sc, $hc, $rc);

    my $error = 0;
    my $remoteHosts = $instance->get_hosts();
    my $hostAgentUserName;
    if ($self->isUseSAPHostagent ()){
        $hostAgentUserName = $self->getBatchValue('HostagentUserName');
        if (!defined $hostAgentUserName){
            $hostAgentUserName = 'sapadm';
        }
    }
    my $errorTemplate = "$param->{str} '$value' %s check on host '%s' failed";
    foreach my $host (@{$remoteHosts}){
        $sc = new SDB::Install::SAPControl ($host, $nr, undef, undef, 1, $value);
        $sc->setMsgLstContext ([$msg->getSubMsgLst ()]);
        if (!$sc->isServiceRunning()){
            $msg->getSubMsgLst ()->addMessage ("Skipping SAPControl check on host '$host': service not running");
        }
        elsif ($sc->accessCheck () != 0){
            $self->appendErrorMessage (sprintf ($errorTemplate, 'SAPControl', $host), $sc->getErrMsgLst());
            $error = 1;
        }

        if (defined $hostAgentUserName){
            $hc = new SDB::Install::SAPHostControl ($host, $hostAgentUserName, '', 0, $value);
            $hc->setMsgLstContext ([$msg->getSubMsgLst ()]);
            $rc = $hc->ListInstances ();
            if (!defined $rc){
                 $self->appendErrorMessage (sprintf ($errorTemplate, 'SAPHostControl', $host),
                    $hc->getErrMsgLst ());
                    $error = 1;
                    next;
            }
            elsif ((ref($rc) eq 'HASH') && (defined $rc->{'faultstring'})){
                $self->appendErrorMessage (sprintf ($errorTemplate, 'SAPHostControl', $host) . ": $rc->{faultstring}");
                $error = 1;
            }
        }
    }
    $self->setSkip('Password');
    $self->setSkip('srcPasswd');
    $self->setSkip('HostagentPassword');

    if ($error){
        $param->{no_retry} = 1;
        return 0;
    }

    return 1;
}



#-------------------------------------------------------------------------------
# Disables the hostagent password if '--skip_hostagent_password' is set.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.

sub checkSkipHostagentPw {

    my ($self, $value) = @_;

    if (defined $value && ($value =~ /$bool_true_pattern/i)) {
        $self->setSkip('HostagentPassword');
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Checks the user ID

sub checkUID{
    my ($self, $uid, $msglst) = @_;

    if (!defined $msglst){
        $msglst = $self;
    }

# Must make sure that no auto-vification of the
# UID param key occurs during scenarios that should
# not have it. This would blow up the WebUI
    my $uidParam = (exists $self->{params}->{UID}) ? $self->{params}->{UID} : $self->getParamUID();
    my $paramName = $uidParam->{str};
    if ($uid =~ /\D/){
        $msglst->PushError ("$paramName '$uid' has to be a decimal number");
        return 0;
    }
    if ($uid < 1){
        $msglst->PushError ("$paramName '$uid' has to be greater than zero");
        return 0;
    }

    my $rc = 1;

    my $freeID = $self->getFreeUserID($uid);
    my $userName = SDB::Common::BuiltIn->getpwuid($uid);
    my $sidadm = $self->getSysAdminUserName($self->{current_sid});
    $paramName = sprintf($uidParam->{str_templ}, $sidadm);
    if (defined $userName && ($userName ne $sidadm)) {
        $rc &&= 0;
        my $errMsg = "$paramName '$uid' is already in use by user '$userName'";
        $errMsg .= " (available ID: $freeID)" if (defined $freeID);
        $msglst->PushError($errMsg);
    }

    if (defined $self->{remoteUids} && defined $self->{remoteUids}->{$uid}){
        foreach my $entry (@{$self->{remoteUids}->{$uid}}){
            if ($sidadm eq $entry->[1]){
                next;
            }
            $rc &&= 0;
            my $errMsg = "$paramName '$uid' is already in use by user '$entry->[1]' on host '$entry->[0]'";
            $errMsg .= " (available ID: $freeID)" if (defined $freeID);
            $msglst->PushError($errMsg);
        }
    }

    my $remoteSidadms = defined $self->{remoteSidadms} ? $self->{remoteSidadms}->{$sidadm} // [] : [];
    foreach my $entry (@$remoteSidadms) {
        my ($remoteHost, $remoteUid, $remoteGid) = @$entry;
        next if ($remoteUid == $uid);
        $rc &&= 0;
        $msglst->PushError("$paramName on remote host $remoteHost has a different value of '$remoteUid'");
    }

    return $rc;
}

sub checkRemoteExecution {
    my ($self, $value) = @_;
    my $isShaExecution = $value eq 'saphostagent' ? 1 : 0;
    if(!isAdmin() && !$isShaExecution) {
        my $remoteExecutionOpt = $self->getOpt('RemoteExecution');
        $self->getErrMsgLst()->addError("Invalid value '$value' of option '$remoteExecutionOpt' (only 'saphostagent' allowed when tool is started as system administrator)");
        return 0;
    }

    if ($isShaExecution && $self->shouldSkipHostagentCalls()) {
        $self->getErrMsgLst()->addMessage("Remote execution 'saphostagent' is not possible when skipping all $gProductNameSHA calls");
        return 0;
    }
    return 1;
}

sub checkUseHttp{
    my ($self, $value) = @_;
    my $paramSapControlUseHttp = $self->{params}->{SAPControlUseHttp};
    if (defined $value && ($value =~ /$bool_true_pattern/i)) {
        $self->setSkip ('SSOCertificate');
    }
    else{
        $self->setSkip ('SSOCertificate', 0);
    }
    if (defined $paramSapControlUseHttp && $value){
        $self->setValue ('SAPControlUseHttp', 1);
    }
    return 1;
}

#-------------------------------------------------------------------------------
# Tries to use an authorization key to access remote hosts.

sub tryRemoteKeyAuthorization {

    my ($self, $remoteRootUser, $isNotVerbose) = @_;
    my $paramInfo = "user name = '$remoteRootUser'";
    my $msg = $self->AddMessage("Checking key authorization ($paramInfo)");
    my $ignorePubKeyMissing = 1;
    my $success = 1;
    foreach my $remoteHosts (@{$self->getRemoteHostsObjects()}){
        $remoteHosts->setUserName($remoteRootUser,
                                  $self->getValue('InstallSSHKey'));

        if ($remoteHosts->authKey($ignorePubKeyMissing) != 0){

            $self->AddMessage("Key authorization not available ($paramInfo)");
            $success = 0;
        }
        $self->AddSubMsgLst ($msg, $remoteHosts);
    }

    if ($success){
    	if ($self->isCollectOtherHostInfosRequired()) {
	        if (!defined $self->CollectOtherHostInfos($isNotVerbose)){
	            $self->{params}->{RootUser}->{no_retry} = 1;
	            return undef;
	        }
    	}
    }
    else{
        $self->setSkip('RootPassword', 0);
        if ($remoteRootUser ne 'root') {
            $self->{params}->{RootPassword}->{str} =
                        sprintf($self->{params}->{RootPassword}->{str_templ},
                                $remoteRootUser);
        }
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Deletes the empty directories '<SID>/global/hdb/install/install'
# of the installation path under Linux. Ignores any error.
#
# Parameter: $sid string

sub clearInstallationPath {

    my ($self, $sid) = @_;

    if ($isWin) {
        return 1;
    }

    my @dirs = (join($path_separator, $self->getValue('Target'), $sid));

    if ( -d $dirs[0] ) {

        push @dirs, join($path_separator, $dirs[0], 'global' );
        push @dirs, join($path_separator, $dirs[1], 'hdb'    );
        push @dirs, join($path_separator, $dirs[2], 'install');
        push @dirs, join($path_separator, $dirs[3], 'install');

        for (my $i = 4; $i >= 0; $i--) {

            if (rmdir $dirs[$i]) {
                $self->AddMessage("Empty directory '$dirs[$i]' deleted");
            }
            else {
                $self->AddMessage("Cannot delete directory '$dirs[$i]'");
                last;
            }
        }
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Writes a summary of the parameters onto the console and into the log file.

sub displayParameterSummary {
    my ($self) = @_;

    my $headline = 'Summary before execution:';
    my $msg      = $self->getMsgLst()->addMessage($headline);
    print "\n$headline\n";

    $self->{params}->{Target}->{hidden} = 0 if (defined $self->{params}->{Target});
    $self->{params}->{SID}->{hidden}    = 0 if (defined $self->{params}->{SID});

    my %summaryOrderMap;
    my @additionalOrderedList;

    foreach my $paramID (@{$self->getParamIds()}) {
        my $summaryOrder = $self->{params}->{$paramID}->{summaryOrder};
        if (defined $summaryOrder) {
            $summaryOrderMap{$summaryOrder} = $paramID;
        }
        else {
            push @additionalOrderedList, $paramID;
        }
    }
    my $submsglst = $msg->getSubMsgLst();
    foreach my $i (0 .. ((scalar keys %summaryOrderMap) -1) ) {
        $self->displaySummaryItem($submsglst, $summaryOrderMap{$i});
    }

    foreach my $paramID (@additionalOrderedList) {
        $self->displaySummaryItem($submsglst, $paramID);
    }
    print "\n";
}


#-------------------------------------------------------------------------------
# Writes a summary of the parameters onto the console and into the log file.

sub displaySummaryItem {

    my ($self, $submsglst, $paramID) = @_;

    my $result = $self->getSummaryItem($paramID, 0, 1); # 1 - getResultArray

    if (defined $result) {
        foreach my $str (@$result) {
            print "   $str\n";
            $submsglst->addMessage("   $str");
        }
    }
}

#-------------------------------------------------------------------------------
# Enables the saphostagent password if a saphostagent is not installed
# on this host.

sub enableHostagentPasswordIfRequired {
    my ($self) = @_;
    if ($self->getValue('InstallHostagent')
        && !defined getSaphostexecPath()
        && ($isWin || !sapadmUserExists())) {
            $self->setSkip('HostagentUserName', 0);
            $self->setSkip('HostagentPassword', 0);
    }
}


#-------------------------------------------------------------------------------
# Enables the multiple host parameters.
#
# A comment at the beginning of AnyConfig shows depending parameters.

sub enableMultipleHostParameters {

    my ($self, $useSidadmForHostctrl) = @_;
    if ($self->isUseSAPHostagent()) {

        my $remoteHosts = $self->getRemoteHosts();
        if (defined $remoteHosts && $remoteHosts->isHostctrl()
            && $useSidadmForHostctrl) {
            $remoteHosts->useSidadm();
        }
        elsif(!$useSidadmForHostctrl && !$self->getValue ('SSOCertificate')) {
            $self->_enableMultipleHostHostagentPassword();
        }
    }
    else {
        $self->setSkip('RootUser', 0);
        $self->{params}->{RootUser}->{init_with_default} = 1;
        $self->setSkip('InstallSSHKey', 0);

        if ($self->getValue('InstallHostagent')) {
            $self->setSkip('SkipHostagentPw', 0);
        }
    }
    return 1;
}

sub _enableMultipleHostHostagentPassword {
    my ($self) = @_;
    my $remoteHosts = $self->getRemoteHosts();
    $self->setType('HostagentPassword', 'passwd');
    $self->setSkip('HostagentUserName', 0);
    $self->setSkip('HostagentPassword', 0);
    if (defined $remoteHosts && $remoteHosts->isHostctrl()) {
        $remoteHosts->useHostagentUser(); # resets useSidadm
    }
}

#-------------------------------------------------------------------------------
# Sends the specified tool at each remote host of a distributed system.
#
# These parameters are set implicitly:
#    --batch
#    --check_only
#    --install_hostagent (SSH connection only)
#    --read_password_from_stdin=xml
#
# Furthermore ignore and timeout options are set implicitly.
#
# Returns undef in case of an error, but 1 in case of single host system

sub excuteRemoteParallel {

    my ($self,
        $actionProgressive, # string (e.g. 'Renaming' => Renaming host 'lu123456')
        $actionDone,        # string (e.g. 'renamed'  => Host 'lu123456' renamed)
        $remoteTool,        # string (e.g. 'hdbupd')
        $remoteMainClass,   # string (e.g. 'DBUpgradeHost')
                            # (undef if '-main' is not required)
        $hostctrlOperation, # string (e.g. operation 'RegisterHost')
        $passwordKeys,      # string array containing the parameter IDs of passwords      (may be undef)
        $paramIDs,          # string array containing parameter IDs of additional options (may be undef)
        $cmdLineParams,     # string containing additional options for SSH connection only(may be undef)
        $remoteHosts,       # instance of SDB::Install::RemoteHosts/RemoteExcution
                            # (undef if $self->getRemoteHosts should be used)
        $remoteInstconfig,  # used to get valid ignore/timeout options for the remote tool
                            # (e.g. instance of SDB::Install::Configuration::Rename)
                            # (undef if HdbReg or the same class as $self is executed on remote hosts)
        $hostctrlParamMap,  # additional host specific parameters (may be undef)
                            # e.g. $hostctrlParamHostMap->{lu123456}->{ROLES}->'rdsync,ets_worker'
                            #                           ->{lu222444}->{ROLES}->'streaming'
        $templateParamRepl, # replaces '%s' in the command line with host specific values (may be undef)
                            #      $templateReplacements->[ [<replacements of host0>], [<repl of host1>] ]
                            # e.g. $templateReplacements->[ ['rdsync,ets_worker'],     ['streaming']     ]
        $executeOnlyOnHosts,# array containing subset of host names to execute on this hosts only
        $hostCtrlOptionMap, # map of additional hostctrl options with their values
       ) = @_;

    $remoteHosts = $self->getRemoteHosts() if (!defined $remoteHosts);

    if (!defined $remoteHosts || ($remoteHosts->getNumberOfHosts == 0)) {
        return 1;
    }


    #
    # live output per host
    #

    my $progressHandler = $self->getMsgLst ()->getProgressHandler();

    if (defined $progressHandler){
        require SDB::Install::OutputHandler;
        my $outputHandler;
        foreach my $host (@{$remoteHosts->{_hosts}}){
            $outputHandler = new SDB::Install::OutputHandler (
                                $progressHandler,
                                "  $host:  ");
            $remoteHosts->setOutputHandler ($host, $outputHandler);
        }
    }

    my $rc = 0;
    my $progress_message = "$actionProgressive host '%s'...";
    my $done_message     = "Host '%s' $actionDone.";
    my $error_message    = "$actionProgressive host '%s' failed!";

    if (!defined $remoteInstconfig) {
        if (defined $remoteMainClass
            && !$self->isa('SDB::Install::Configuration::HdbReg')
            && (($remoteMainClass eq 'DBUpgradeHost') ||
                ($remoteMainClass eq 'HdbModifyHost') ||
                ($remoteMainClass eq 'HdbRegHost'))) {

            require SDB::Install::Configuration::HdbReg;
            $remoteInstconfig = new SDB::Install::Configuration::HdbReg();
        }
        else {
            $remoteInstconfig = $self;
        }
    }

    my $actionID         = $self->{options}->{action_id};
    my $remoteOptIgnore  = $remoteInstconfig->getOptionIgnore();
    my $remoteOptTimeout = $remoteInstconfig->getOptionTimeout();

    my $cntRemoteHosts = (defined $executeOnlyOnHosts)
                         ? scalar @$executeOnlyOnHosts
                         : $remoteHosts->getNumberOfHosts();
    my $isOneRemoteHost = ($cntRemoteHosts > 1) ? 0 : 1;

    my $msg;
    my $msgLstContext;
    if(!$isOneRemoteHost){
        my $info = "$actionProgressive remote hosts...";
        $msg = $self->AddProgressMessage($info);
        $msgLstContext = [$msg->getSubMsgLst()];
    } else {
        $msgLstContext = $self->getMsgLstContext();
    }
    my $saveCtxt = $remoteHosts->setMsgLstContext ($msgLstContext);

    my $currParamIDs = ['CheckOnly'];
    $currParamIDs    = [@$currParamIDs, @$paramIDs] if (defined $paramIDs);
    if ($remoteHosts->isHostctrl()) {

        my $optionMap = defined($hostCtrlOptionMap) ? $hostCtrlOptionMap : {};
        $optionMap->{SID} = $self->getSID();
        $optionMap->{SAPMNT} = $self->getSAPSystem()->get_target();
        $optionMap->{ACTION_ID} = $actionID if (defined $actionID);

        $rc = $remoteHosts->executeHostctrlParallel($hostctrlOperation,
                                                    $self, # instconfig
                                                    $currParamIDs,
                                                    $passwordKeys,
                                                    $remoteOptIgnore,
                                                    $remoteOptTimeout,
                                                    $progress_message,
                                                    $done_message,
                                                    $error_message,
                                                    $optionMap,
                                                    $hostctrlParamMap,
                                                    $executeOnlyOnHosts);
    }
    else {
        my $xml = $self->getXmlPasswordStream($passwordKeys);
        my $cmd = $self->getRemoteExecutable ($actionProgressive, $remoteTool);

        if (!defined $cmd) {
            return undef;
        }
        if (defined $remoteMainClass) {
            $cmd .= ' -main SDB::Install::App::Console::'
                    . $remoteMainClass . '::main';
        }
        $cmd .= ' --batch';
        $cmd .= " --action_id=$actionID"          if (defined $actionID);
        $cmd .= ' --read_password_from_stdin=xml' if (defined $xml);
        $cmd .= ' ' . $cmdLineParams if (defined $cmdLineParams);
        $cmd .= $self->getIgnoreAndTimeoutOptions($remoteOptIgnore,
                                                  $remoteOptTimeout);

        my $params = $self->{params};
        foreach my $id (@$currParamIDs) {
            my $value = $self->getValue($id);
            if (defined $value) {
                if ($params->{$id}->{type} =~ /bool/) {
                    $value = ($value) ? 'on' : 'off';
                }
                $cmd .= ' ' . $self->getOpt($id) . '=' . $value;
            }
        }

        $rc = $remoteHosts->executeParallel($cmd,
                                            $progress_message,
                                            $done_message,
                                            $error_message,
                                            $xml,
                                            $templateParamRepl,
                                            $executeOnlyOnHosts);
    }

    $msg->endMessage(0) if(!$isOneRemoteHost);
    $remoteHosts->setMsgLstContext($saveCtxt);

    if (!defined $rc || ($rc != 0)) {
        $self->setErrorMessage("$actionProgressive remote hosts failed!",
                               $remoteHosts->getErrMsgLst());
        return undef;
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Sends the specified tool at each remote host of a distributed system.

sub excuteRemoteSerialOrParallel {

    my ($self,
        $actionProgressive, # string (e.g. 'Renaming' => Renaming host 'lu123456')
        $actionDone,        # string (e.g. 'renamed'  => Host 'lu123456' renamed)
        $remoteTool,        # string (e.g. 'hdbupd')
        $remoteMainClass,   # string (e.g. 'DBUpgradeHost')
                            # (undef if '-main' is not required)
        $hostctrlOperation, # string (e.g. operation 'RegisterHost')
        $passwordKeys,      # string array containing the parameter IDs of passwords      (may be undef)
        $paramIDs,          # string array containing parameter IDs of additional options (may be undef)
        $cmdLineParams,     # string containing additional options for SSH connection only(may be undef)
        $remoteHosts,       # instance of SDB::Install::RemoteHosts/RemoteExcution
                            # (undef if $self->getRemoteHosts should be used)
        $remoteInstconfig,  # used to get valid ignore/timeout options for the remote tool
                            # (e.g. instance of SDB::Install::Configuration::Rename)
                            # (undef if HdbReg or the same class as $self is executed on remote hosts)
        $hostctrlParamMap,  # additional host specific parameters (may be undef)
                            # e.g. $hostctrlParamHostMap->{lu123456}->{ROLES}->'rdsync,ets_worker'
                            #                           ->{lu222444}->{ROLES}->'streaming'
        $templateParamRepl, # replaces '%s' in the command line with host specific values (may be undef)
                            #      $templateReplacements->[ [<replacements of host0>], [<repl of host1>] ]
                            # e.g. $templateReplacements->[ ['rdsync,ets_worker'],     ['streaming']     ]
        $executeOnlyOnHosts,# array containing subset of host names to execute on this hosts only
        $isSerialExecution,
       ) = @_;

    $remoteHosts = $self->getRemoteHosts() if (!defined $remoteHosts);

    if (!defined $remoteHosts || ($remoteHosts->getNumberOfHosts == 0)) {
        return 1;
    }

    if (!$isSerialExecution) {
        return $self->excuteRemoteParallel($actionProgressive,
                                           $actionDone,
                                           $remoteTool,
                                           $remoteMainClass,
                                           $hostctrlOperation,
                                           $passwordKeys,
                                           $paramIDs,
                                           $cmdLineParams,
                                           $remoteHosts,
                                           $remoteInstconfig,
                                           $hostctrlParamMap,
                                           $templateParamRepl,
                                           $executeOnlyOnHosts);
    }

    $self->AddMessage('Serial execution of remote hosts');

    my $wantedHosts = (defined $executeOnlyOnHosts)
                      ? $executeOnlyOnHosts
                      : $remoteHosts->getHostNames();

    for my $currHost (@$wantedHosts) {
        if (!$self->excuteRemoteParallel($actionProgressive,
                                         $actionDone,
                                         $remoteTool,
                                         $remoteMainClass,
                                         $hostctrlOperation,
                                         $passwordKeys,
                                         $paramIDs,
                                         $cmdLineParams,
                                         $remoteHosts,
                                         $remoteInstconfig,
                                         $hostctrlParamMap,
                                         $templateParamRepl,
                                         [$currHost])) {
            return undef;
        }
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Returns a hash containing instances of SDB::Install::SAPSystem
# Parmeter:
#           boolean      $no_cache (optional, default is false)
#

sub getCollectSAPSystems {
    return CollectSAPSystems (undef, $_[1]);
}


#-------------------------------------------------------------------------------
# Returns the number of all hosts
# (including additional hosts, without hosts to be removed)

sub getNumberOfDefinedHosts {
    my ($self)   = @_;
    my $remoteHosts = $self->getRemoteHosts();
    return (defined $remoteHosts) ? 1 + $remoteHosts->getNumberOfHosts() : 1;
}


#-------------------------------------------------------------------------------
# Returns a reference to the own instance if the hash 'sapSys' is defined.
#
# Return: SDB::Install::SAPInstance::TrexInstance

sub getOwnInstance {

    my ($self, $no_cache)   = @_;
    my $instance = $self->{ownInstance};

    if (!defined $instance || $no_cache) {

        my $sapSys = $self->getSAPSystem($no_cache);
        $instance  = $sapSys->getNewDBInstances()->[0]
                                    if (defined $sapSys && $sapSys->hasNewDB());

        if (defined $instance) {
            $self->{ownInstance} = $instance;
        }
    }

    return $instance;
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter
# '--ets_administrator_password.

sub getParamAcceleratorPassword {
    my ($self,
        $order,
        $section,
        $constraint, # e.g. 'Valid for ets_worker/ets_standby'
        $typePw,     # default: 'initial_passwd'
        $userParamId
       ) = @_;

    return {'order'             => $order,
            'opt'               => 'ase_user_password',
            'type'              => (defined $typePw) ? $typePw : 'initial_passwd',
            'opt_arg'           => '<password>',
            'section'           => $section,
            'value'             => undef,
            'default'           => undef,
            'str'               => "$gNameAccelerator Administrator Password",
            'str_templ'         => $gNameAccelerator . ' Administrator (%s) Password',
            'desc'              => "$gNameAccelerator administrator password",
            'init_with_default' => 0,
            'log_value'         => sub {defined $_[0] ? '***' : '<not defined>';},
            'mandatory'         => 1,
            'skip'              => 1,
            'constraint'        => $constraint,
            'user_paramid'      => $userParamId
    };
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter '--ets_administrator.

sub getParamAcceleratorUser {
    my ($self,
        $order,
        $section,
        $constraint # e.g. 'Valid for ets_worker/ets_standby'
       ) = @_;

    return {'order'   => $order,
            'opt'     => 'ase_user',
            'opt_arg' => '<name>',
            'type'    => 'string',
            'section' => $section,
            'value'   => undef,
            'default' => 'sa',
            'str'     => "$gNameAccelerator Administrator User",
            'desc'    => "Specifies the administrator user of $gShortProductNameAccelerator",
            'skip'    => 1,
            'init_with_default' => 1,
            'set_interactive'   => 1,
            'mandatory' => 1,
            'constraint'        => $constraint,
            'hostctrl_opt'      => 'ASE_USER',
    };
}

#-------------------------------------------------------------------------------
# Returns parameter '--force' of hdbremovehost.

sub getParamForce {

    my ($self,
        $order,
        $section,
        $constraint,
        $hidden,     # default: false
       ) = @_;

    my $isHidden = (defined $hidden) ? $hidden : 0;

    return {'order'             => $order++,
            'opt'               => 'force',
            'hostctrl_opt'      => 'FORCE',
            'type'              => 'boolean',
            'section'           => $section,
            'value'             => undef,
            'default'           => 0,
            'str'               => 'Forces removal even if there are steps failing',
            'init_with_default' => 1,
            'set_interactive'   => 0,
            'hidden'            => $isHidden,
            'constraint'        => $constraint,
    }
}

#-------------------------------------------------------------------------------
# TEMPORARY FIX for knowing if hdbaddhost is called when system is in
# compatiblity mode. Should use SHA conf or some main.. dunno

sub getParamAddHostCalledInCompatiblityMode {
    my ($self, $order, $section) = @_;
    return {'order'             => $order++,
            'opt'               => 'addhost_compatiblity_mode',
            'hostctrl_opt'      => 'ADDHOST_COMP_MODE',
            'type'              => 'boolean',
            'section'           => $section,
            'value'             => undef,
            'default'           => 0,
            'str'               => 'States that hdbaddhost is called in compatiblity mode',
            'init_with_default' => 1,
            'set_interactive'   => 0,
            'hidden'            => 1,
            'hostctrl_opt'      => 'ACM',
    }
}

#-------------------------------------------------------------------------------
sub getParamAcceleratorDataPath{
    my ($self, $order, $section, $skip, $withDesc, $initDefault) = @_;

    my $desc = ($withDesc)
               ? "Required for the $gProductNameAccelerator (new installation)"
               : undef;

    return {
        'order'      => $order,
        'opt'        => 'ase_datapath',
        'alias_opts' => ['ets_datapath'],
        'opt_arg'    => '<path>',
        'type'       => 'path',
        'section'    => $section,
        'value'      => undef,
        'default'    => $isWin ? '$Drive\usr\sap\data_ase\$SID' : '/hana/data_ase/$SID',
        'str'        => "Location of Data Volumes of the $gShortProductNameAccelerator",
        'desc'       => $desc,
        'skip'       => defined $skip ? $skip : 0,
        'mandatory'  => 0,
        'init_with_default' => defined $initDefault ? $initDefault : 0,
        'set_interactive'   => 1,
    };
}

#-------------------------------------------------------------------------------
sub getParamAcceleratorLogPath{
    my ($self, $order, $section, $skip, $withDesc, $initDefault) = @_;

    my $desc = ($withDesc)
               ? "Required for the $gProductNameAccelerator (new installation or update)"
               : undef;

    return {
        'order'      => $order,
        'opt'        => 'ase_logpath',
        'alias_opts' => ['ets_logpath'],
        'opt_arg'    => '<path>',
        'type'       => 'path',
        'section'    => $section,
        'value'      => undef,
        'default'    => $isWin ? '$Drive\usr\sap\log_ase\$SID' : '/hana/log_ase/$SID',
        'str'        => "Location of Log Volumes of the $gShortProductNameAccelerator",
        'desc'       => $desc,
        'skip'       => defined $skip ? $skip : 0,
        'mandatory'  => 0,
        'init_with_default' => defined $initDefault ? $initDefault : 0,
        'set_interactive'   => 1,
    };
}


#-------------------------------------------------------------------------------
# The roles 'xs_worker'/'xs_standby' are added automatically to each
# 'worker'/'standby'.

sub getParamAutoAddXS2Roles {

    my ($self, $order, $section, $constraint, $wantedDefault) = @_;

    my $default = (defined $wantedDefault) ? $wantedDefault : 1;

    return {'order'             => $order,
            'opt'               => 'autoadd_xs_roles',
            'hostctrl_opt'      => 'AUTOADD_XS_ROLES',
            'type'              => 'boolean',
            'section'           => $section,
            'value'             => undef,
            'str'               => 'Automatic adding of xs roles?',
            'desc'              => 'Adds xs roles automatically',
            'default'           => $default,
            'init_with_default' => 1,
            'set_interactive'   => 0,
            'constraint'        => $constraint,
    };
}


#-------------------------------------------------------------------------------

sub getParamAutoStart{
    my ($self, $order, $section, $skip, $constraint) = @_;
    my $str = 'Restart system after machine reboot';

    return {
        'order'                     => $order,
        'opt'                       => 'autostart',
        'type'                      => 'boolean',
        'section'                   => $section,
        'value'                     => undef,
        'default'                   => 0,
        'str'                       => $str.'?',
        'ui_str'                    => $str,
        'desc'                      => "Restarts $gProductNameSystem after machine reboot",
        'console_omit_word_Enter'   => 1,
        'init_with_default'         => 1,
        'set_interactive'           => 1,
        'skip'                      => (defined $skip) ? $skip : 0,
        'constraint'                => $constraint,
    };
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of a parameter describing a base path.
#
# Parameter: int    $order
#            string $section
#            string $pathName    # e.g. 'datapath'
#            string $description # e.g. 'Location of Data Volumes'
#            string $constraint  # e.g. 'register only')

sub getParamBasePath {

    my ($self, $order, $section, $pathName, $description, $constraint) = @_;

    my %paramBasePath = (
            'order'             => $order,
            'opt'               => $pathName,
            'type'              => 'path',
            'opt_arg'           => '<path>',
            'section'           => $section,
            'value'             => undef,
            'default'           => undef,
            'str'               => $description,
            'init_with_default' => 0,
            'set_interactive'   => 1,
            'skip'              => 1,
            'constraint'        => $constraint,
            );

    return \%paramBasePath;
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'change_initial_ssfs_key'.
#
# Parameter: int    $order
#            string $section

sub getParamChangeInitialSSFSKey {

    my ($self, $order, $section) = @_;

    my %paramChangeInitialSSFSKey = (
            'order'   => $order,
            'opt'     => 'change_initial_ssfs_key',
            'type'    => 'boolean',
            'section' => $section,
            'default' => 1,
            'str'     => "Change the initial SSFS key",
            'set_interactive'   => 0,
            'init_with_default' => 1,
            );

    return \%paramChangeInitialSSFSKey;
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'check_only'.
#
# Parameter: int    $order
#            string $section

sub getParamCheckOnly {

    my ($self, $order, $section, $action) = @_;

    my $currAction = (defined $action) ? $action : 'change';

    my %paramCheckOnly = (
            'order'   => $order,
            'opt'     => 'check_only',
            'type'    => 'boolean',
            'section' => $section,
            'default' => 0,
            'str'     => "Execute checks, do not $currAction $gProductNameSystem",
            'hostctrl_opt'      => 'CHECK_ONLY',
            'set_interactive'   => 0,
            'init_with_default' => 1,
            );

    return \%paramCheckOnly;
}


sub getParamCreateInitialTenant{
    my ($self,
        $order,
        $section,
        $default
        ) = @_;

    return {'order'   => $order,
            'opt'     => 'create_initial_tenant',
            'type'    => 'boolean',
            'section' => $section,
            'value'   => undef,
            'default' => defined $default ? $default : 1,
            'str'     => 'Create initial tenant database',
            'desc'    => 'Creates the initial tenant database',
            'init_with_default' => 1,
            'set_interactive'   => 0,
            'hidden'            => 0
        };
}

sub getParamCustomPluginsPath {
    my ($self, $order, $section) = @_;
    return {'order' => $order,
            'opt' => 'custom_path_for_plugins',
            'type' => 'path',
            'section' => $section,
            'value' => undef,
            'default' => undef,
            'str' => 'Custom Installation Path for HANA Server Plugins',
            'init_with_default' => 0,
            'set_interactive' => 0,
            'hidden' => 1
        };
}

#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'domain'.

sub getParamDomain {
    my ($self, $order, $section) = @_;

    my $str = 'Domain of System Administrator';

    return {
        'order'           => $order,
        'opt'             => 'domain',
        'type'            => 'string',
        'section'         => $section,
        'value'           => undef,
        'default'         => hostname(),
        'str'             => $str,
        'str_templ'       => $str . ' (%s)',
        'set_interactive' => 1,
    };
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the Windows parameter 'Drive'.

sub getParamDrive{
    my ($self, $order, $section) = @_;
    return {
        'order'             => $order,
        'opt'               => 'drive',
        'type'              => 'string',
        'section'           => $section,
        'value'             => undef,
        'default'           => $gDriveWin,
        'str'               => 'Windows Drive Letter',
        'set_interactive'   => 0,
        'init_with_default' => 1,
    };
}

sub getParamEsDataPath{
    my ($self, $order, $section, $skip, $withDesc, $initDefault) = @_;

    my $desc = ($withDesc)
               ? "Required for $gProductNameEs (new installation)"
               : undef;

    return {
        'order'   => $order,
        'opt'     => 'es_datapath',
        'alias_opts' => ['es_data_path'],
        'opt_arg' => '<path>',
        'type'    => 'path',
        'section' => $section,
        'value'   => undef,
        'default' => $isWin ? '$Drive\usr\sap\data_es\$SID' : '/hana/data_es/$SID',
        'str'     => "Location of $gShortProductNameEs Data Volumes",
        'desc'    => $desc,
        'skip'    => defined $skip ? $skip : 0,
        'init_with_default' => defined $initDefault ? $initDefault : 0,
        'set_interactive'   => 1,
        'mandatory'         => 0,
    };
}

sub getParamEsLogPath{
    my ($self, $order, $section, $skip, $withDesc, $initDefault) = @_;

    my $desc = ($withDesc)
               ? "Required for $gProductNameEs (new installation) or if $gShortProductNameEs is running (update)"
               : undef;

    return {
        'order'   => $order,
        'opt'     => 'es_logpath',
        'alias_opts' => ['es_log_path'],
        'opt_arg' => '<path>',
        'type'    => 'path',
        'section' => $section,
        'value'   => undef,
        'default' => $isWin ? '$Drive\usr\sap\log_es\$SID' : '/hana/log_es/$SID',
        'str'     => "Location of $gShortProductNameEs Log Volumes",
        'desc'    => $desc,
        'skip'    => defined $skip ? $skip : 0,
        'init_with_default' => defined $initDefault ? $initDefault : 0,
        'set_interactive'   => 1,
        'mandatory'         => 0,
    };
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'groupid'.
#
# Parameter: int    $order
#            string $section
#            int    $defaultGID
#            string $str
#            bool   $setInteractive
#            string $constraint (e.g. 'register only')

sub getParamGID {

    my ($self, $order, $section, $defaultGID, $setInteractive,$constraint) = @_;

    my %paramGID = (
            'order'                     => $order,
            'opt'                       => 'groupid',
            'short_opt'                 => 'G',
            'type'                      => 'number',
            'section'                   => $section,
            'value'                     => undef,
            'default'                   => $defaultGID,
            'str'                       => "ID of User Group ($gSapsysGroupName)",
            'set_interactive'           => $setInteractive,
            'may_be_interactive'        => 1,
            'init_with_default'         => 1,
            'constraint'                => $constraint,
            'leaveEmptyInConfigDump'    => 1
            );

    return \%paramGID;
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'home'.
#
# Parameter: int    $order
#            string $section
#            string $defaultPath
#            string $constraint (e.g. 'register only')

sub getParamHomeDir {

    my ($self, $order, $section, $defaultPath, $constraint) = @_;

    my %paramHomeDir = (
            'order'             => $order,
            'opt'               => 'home',
            'type'              => 'path',
            'section'           => $section,
            'value'             => undef,
            'default'           => $defaultPath,
            'str'               => 'System Administrator Home Directory',
            'str_templ'         => 'System Administrator (%s) Home Directory',
            'desc'              => 'Home directory of the system administrator',
            'set_interactive'   => 1,
            'init_with_default' => 1,
            'constraint'        => $constraint,
            );

    return \%paramHomeDir;
}

#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'sapadm_password'.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.
#
# Parameter: int    $order
#            string $section
#            string $constraint (e.g. 'register only')

sub getParamHostagentPassword {

    my ($self, $order, $section, $constraint) = @_;

    return {'order'   => $order,
            'opt'     => 'sapadm_password',
            'type'    => 'initial_passwd',
            'section' => $section,
            'value'   => undef,
            'default' => undef,
            'str'     => 'SAP Host Agent User (sapadm) Password',
            'str_templ'=>'SAP Host Agent User (%s) Password',
            'desc'    => 'SAP Host Agent User password',
            'skip'    => 1,
            'mandatory'         => 1,
            'init_with_default' => 0,
            'set_interactive'   => 1,
            'constraint'        => $constraint,
            'retry_count'       => 3,
    };
}

#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'sapadm_username'.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.

sub getParamHostagentUserName {

    my ($self,
        $order,
        $section,
        $constraint, # e.g. 'register only'
        $skip,       # default: true
        $hidden      # default: true
        ) = @_;

    my $isSkip   = (defined $skip  ) ? $skip   : 1;
    my $isHidden = (defined $hidden) ? $hidden : 1;

    return {'order'   => $order,
            'opt'     => 'sapadm_user',
            'type'    => 'string',
            'section' => $section,
            'value'   => undef,
            'default' => 'sapadm',
            'str'     => 'SAP Host Agent User Name',
            'desc'    => 'Name of the user which runs the SAP Host Agent',
            'skip'              => $isSkip,
            'init_with_default' => 1,
            'set_interactive'   => 1,
            'hidden'            => $isHidden,
            'constraint'        => $constraint,
    };
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'hostname'.
#
sub getParamHostName{
    my ($self, $order, $section) = @_;
    return {
        'order'                     => $order,
        'opt'                       => 'hostname',
        'short_opt'                 => 'H',
        'opt_arg'                   => '<name>',
        'type'                      => 'string',
        'section'                   => $section,
        'value'                     => undef,
        'default'                   => lc (hostname ()),
        'desc'                      => 'Host Name of this Location',
        'str'                       => 'Local Host Name',
        'init_with_default'         => 1,
        'set_interactive'           => 1,
        'leaveEmptyInConfigDump'    => 0,
        'persStep'                  => $STEP_CREATE_INSTANCE
    };
}


#-------------------------------------------------------------------------------
sub getParamSSOCertificate{
    my ($self, $order, $section) = @_;
    return {
        'order'             => $order,
        'opt'               => 'sso_cert',
        'hostctrl_opt'      => 'CSSO',
        'type'              => 'string',
        'section'           => $section,
        'value'             => undef,
        'default'           => undef,
        'desc'              => 'SSO certificate file to authenticate SAPControl and SAPHostAgent admin user',
        'str'               => 'SSO certificate',
        'init_with_default' => 0,
        'set_interactive'   => 0,
        'hidden'            => 0,
    };
}

#-------------------------------------------------------------------------------
sub getParamStripSymbols{
    my ($self, $order, $section, $hidden) = @_;
    my $isHidden = (defined $hidden) ? $hidden : 1;  # default: hidden parameter

    return {
        'order'             => $order,
        'opt'               => 'strip_symbols',
        'hostctrl_opt'      => 'STRIP_SYMBOLS',
        'type'              => 'boolean',
        'section'           => $section,
        'value'             => undef,
        'default'           => 0,
        'str'               => 'Removes debug symbols from executables and libraries',
        'init_with_default' => 1,
        'set_interactive'   => 0,
        'hidden'            => $isHidden,
    };
}

#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'SystemIsOffline'.

sub getParamSystemIsOffline {
    my ($self, $order, $section) = @_;

    return {'order'             => $order,
            'opt'               => 'system_is_offline',
            'type'              => 'boolean',
            'section'           => $section,
            'value'             => undef,
            'default'           => 0,
            'str'               => "$gProductNameSystem is already offline",
            'desc'              => "Confirms, that $gProductNameSystem is already offline",
            'init_with_default' => 1,
            'set_interactive'   => 0,
            'skip'              => 0
    }
}

#-------------------------------------------------------------------------------
sub getParamImportContent{
    my ($self, $order, $section) = @_;

    return {
        'order'             => $order,
        'opt'               => 'import_content',
        'hostctrl_opt'      => 'IMPORT_CONTENT',
        'type'              => 'boolean',
        'section'           => $section,
        'value'             => undef,
        'default'           => 1,
        'str'               => 'Import delivery units',
        'init_with_default' => 1,
        'set_interactive'   => 0,
    };
}


#-------------------------------------------------------------------------------
sub getParamImportContentXS2 {
    my ($self, $order, $section, $constraint) = @_;

    return {
        'order'             => $order,
        'opt'               => 'import_xs_content',
        'alias_opts'        => ['hdbmodify_import_content'],
        'hostctrl_opt'      => 'IMPORT_XS_CONTENT',
        'type'              => 'boolean',
        'section'           => $section,
        'value'             => undef,
        'default'           => 1,
        'str'               => "Import initial content of $gShortProductNameXS2",
        'desc'              => "Imports initial content of $gShortProductNameXS2 during installation of xs worker",
        'init_with_default' => 1,
        'set_interactive'   => 0,
        'constraint'        => $constraint,
    };
}

#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'install_hostagent'.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.
#
# Parameter: int    $order
#            string $section
#            string $constraint (e.g. 'register only')

sub getParamInstallHostagent {

    my ($self, $order, $section, $constraint, $defaultValue) = @_;

    my $default = (defined $defaultValue) ? $defaultValue : 0;

    return {'order'   => $order,
            'opt'     => 'install_hostagent',
            'type'    => 'boolean',
            'section' => $section,
            'value'   => undef,
            'default' => $default,
            'str'     => 'Enable the installation or upgrade of the SAP Host Agent',
            'desc'    => 'Enables the installation or upgrade of the SAP Host Agent',
            'init_with_default' => 1,
            'set_interactive'   => 0,
            'constraint'        => $constraint,
            'hostctrl_opt'      => 'INSTALL_HOSTAGENT',
    };
}

#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'InstallSSHKey'.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.

sub getParamInstallSSHKey {
    my ($self, $order, $section) = @_;
    return {
        'order'                     => $order,
        'opt'                       => 'install_ssh_key',
        'type'                      => 'boolean',
        'section'                   => $section,
        'value'                     => undef,
        'default'                   => 1,
        'str'                       => 'Install SSH Key',
        'desc'                      => 'Installs SSH key to access remote hosts',
        'skip'                      => 1,
        'init_with_default'         => 1,
        'hidden'                    => 0,
        'set_interactive'           => 0,
    };
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'number'.
#
# Parameter: int    $order
#            string $section
#            string $str
#            bool   $hidden

sub getParamInstanceNumber {

    my ($self, $order, $section, $str, $hidden) = @_;

    return {'order'             => $order,
            'opt'               => 'number',
            'short_opt'         => 'n',
            'alias_opts'        => ['instance'],
            'hostctrl_opt'      => 'NUMBER',
            'type'              => 'number',
            'section'           => $section,
            'value'             => undef,
            'default'           => '00',
            'str'               => $str,
            'mandatory'         => 0,
            'init_with_default' => 1,
            'may_be_interactive'=> 1,
            'hidden'            => $hidden,
    };
}


#-------------------------------------------------------------------------------
sub getParamISCMode {
    my ($self, $order, $section) = @_;

    return {
        'order'           => $order,
        'opt'             => 'isc_mode',
        'hostctrl_opt'    => 'ISC_MODE',
        'type'            => 'string',
        'opt_arg'         => 'standard|ssl',
        'valid_values'    => ['standard', 'ssl'],
        'section'         => $section,
        'value'           => undef,
        'default'         => undef,
        'str'             => 'Inter Service Communication Mode',
        'set_interactive' => 0,
    };
}


#-------------------------------------------------------------------------------
sub getParamJavaVm{
    my ($self, $order, $section, $constraint) = @_;
    return {
        'order'             => $order,
        'opt'               => 'vm',
        'opt_arg'           => '<path>',
        'type'              => 'path',
        'section'           => $section,
        'value'             => undef,
        'default'           => '',
        'str'               => 'Java Runtime',
        'constraint'        => $constraint,
        'init_with_default' => 1,
        'mandatory'         => 1,
        'set_interactive'   => 0
    };
}

#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'KeepUser'
# without the item 'desc'.
#
# Parameter: int    $order
#            string $section
#            string $constraint (e.g. 'unregister only')

sub getParamKeepUser {

    my ($self, $order, $section, $constraint, $userAttr) = @_;

    my $attr = (defined $userAttr) ? "$userAttr " : '';
    my $str  = "Keep $attr".'System Administrator User';
    my $desc = 'Does not remove the ' . lc($attr) . 'System Administrator User';

    return {'order'             => $order,
            'opt'               => 'keep_user',
            'alias_opts'        => ['keepuser'],
            'short_opt'         => 'k',
            'hostctrl_opt'      => 'KEEP_USER',
            'type'              => 'boolean',
            'section'           => $section,
            'value'             => undef,
            'str'               => $str,
            'str_templ'         => $str . ' (%s)',
            'desc'              => $desc,
            'default'           => 0,
            'init_with_default' => 1,
            'set_interactive'   => 0,
            'constraint'        => $constraint,
            'console_omit_word_Enter' => 1,
    };
}

#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'KeepXsUser'
#
# Parameter: int    $order
#            string $section

sub getParamKeepXsUsers {
    my ($self, $order, $section, $constraint) = @_;

    return {
        'order'             => $order,
        'opt'               => 'keep_xs_os_users',
        'hostctrl_opt'      => 'KXU',
        'type'              => 'boolean',
        'section'           => $section,
        'value'             => undef,
        'str'               => 'Do not Remove XS Advanced OS Users',
        'desc'              => 'Do not remove XS Advanced OS users',
        'default'           => 0,
        'init_with_default' => 1,
        'set_interactive'   => 0,
        'constraint'        => $constraint,
        'console_omit_word_Enter' => 1,
    };
}

#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'KeepXsUser'
#
# Parameter: int    $order
#            string $section

sub getParamSkipModifySudoers {
    my ($self, $order, $section) = @_;

    return {
        'order'             => $order,
        'opt'               => 'skip_modify_sudoers',
        'short_opt'         => 'M',
        'hostctrl_opt'      => 'SUD',
        'type'              => 'boolean',
        'section'           => $section,
        'value'             => undef,
        'str'               => 'Do not Modify \'/etc/sudoers\' File',
        'desc'              => 'Do not modify \'/etc/sudoers\' file',
        'default'           => 0,
        'init_with_default' => 1,
        'set_interactive'   => 0,
        'console_omit_word_Enter' => 1,
    };
}

#-------------------------------------------------------------------------------
# Returns Windows only parameter '--skip_vcredist'

sub getParamSkipVCRedist {

    my ($self,
        $order,
        $section,
       ) = @_;

    return {'order'             => $order++,
            'opt'               => 'skip_vcredist',
            'type'              => 'boolean',
            'section'           => $section,
            'value'             => undef,
            'default'           => 0,
            'str'               => 'Skip installation of MS Visual Studio C/C++ Runtime libraries',
            'init_with_default' => 1,
            'set_interactive'   => 0,
            'hidden'            => 0
    }
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'KeepUserHomeDir'.
#
# Parameter: int    $order
#            string $section
#            string $constraint (e.g. 'unregister only')

sub getParamKeepUserHomeDir {

    my ($self, $order, $section, $constraint, $userAttr) = @_;

    my $attr = (defined $userAttr) ? "$userAttr " : '';
    my $str  = "Keep Home Directory of $attr".'System Administrator';
    my $desc = 'Does not delete the home directory of the ' . lc($attr)
             . 'System Administrator';

    return {'order'             => $order,
            'opt'               => 'keep_user_home_dir',
            'hostctrl_opt'      => 'KEEP_USER_HOME_DIR',
            'type'              => 'boolean',
            'section'           => $section,
            'value'             => undef,
            'str'               => $str,
            'str_templ'         => $str . ' (%s)',
            'desc'              => $desc,
            'default'           => 0,
            'init_with_default' => 1,
            'set_interactive'   => 0,
            'constraint'        => $constraint,
            'console_omit_word_Enter' => 1,
    };
}


sub getParamMachineUtilization{
    my ($self, $order, $section) = @_;
    return {
        'order'             => $order,
        'opt'               => 'machine_utilization',
        'type'              => 'number',
        'section'           => $section,
        'value'             => undef,
        'default'           => 100,
        'str'               => 'Percentage of machine resources required for this system',
        'init_with_default' => 0,
        'set_interactive'   => 0,
        'hidden'            => 1
    };
}

sub getParamNoStart{
    my ($self, $order, $section, $str, $desc, $hidden, $constraint) = @_;
    return {
        'order'                     => $order,
        'opt'                       => 'nostart',
        'hostctrl_opt'              => 'NOSTART',
        'type'                      => 'boolean',
        'section'                   => $section,
        'value'                     => undef,
        'default'                   => 0,
        'str'                       => $str,
        'desc'                      => $desc,
        'console_omit_word_Enter'   => 1,
        'init_with_default'         => 1,
        'set_interactive'           => 0,
        'hidden'                    => $hidden,
        'constraint'                => $constraint,
    };
}

#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'password' (system
# administrator password) without the items 'alias_opts', 'desc' and 'str'.
#
# Parameter: int    $order
#            string $section

sub getParamPassword {

    my ($self, $order, $section, $constraint) = @_;

    my %paramPassword = (
            'order'     => $order,
            'opt'       => 'password',
            'short_opt' => 'p',
            'type'      => 'passwd',
            'opt_arg'   => '<password>',
            'section'   => $section,
            'value'     => undef,
            'default'   => undef,
            'str'       => 'System Administrator Password',
            'str_templ' => 'System Administrator (%s) Password',
            'log_value' => sub {'***';},
            'mandatory' => 1,
            'constraint'=> $constraint,
            'retry_count'=> 3,
            );

    return \%paramPassword;
}


sub getParamPhase {

    my ($self, $order, $section, $valid_values) = @_;

    my %paramPhase = (
           'order' => $order,
           'opt' => 'phase',
           'type' => 'string',
           'section' => $section,
           'value' => undef,
           'default' => undef,
           'valid_values' => $valid_values,
           'str'  => 'For phased installation or update, used internally by hdblcm',
           'desc' => 'For phased installation or update, used internally by hdblcm',
           'init_with_default' => 1,
           'hidden' => 1,
           'set_interactive' => 0,
            );
    return \%paramPhase;
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'PATH'
#
# Parameter: int    $order
#            string $section
#            string $default

sub getParamPATH {
    my ($self, $order, $section, $default) = @_;

    my %paramPATH = (
        'order' => $order,
        'opt' => 'path',
        'short_opt' => 'p',
        'type' => 'path',
        'section' => $section,
        'value' => undef,
        'default' => $default,
        'str' => 'Installation Path',
        'init_with_default' => 1,
        'set_interactive' => 1,
        'mandatory' => 1
    );
    return \%paramPATH;
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter '--remote_execution'.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.

sub getParamRemoteExecution {

    my ($self,
        $order,
        $section,
        $skip  # default: false
       ) = @_;

    my $isSkip = (defined $skip) ? $skip : 0;

    return {
        'order'             => $order,
        'opt'               => 'remote_execution',
        'type'              => 'string',
        'section'           => $section,
        'opt_arg_switch'    => 'saphostagent',
        'value'             => undef,
        'default'           => ($isWin || !isAdmin()) ? 'saphostagent' : 'ssh',
        'valid_values'      => ['ssh', 'saphostagent'],
        'str'               => 'Remote Execution',
        'desc'              => 'Specifies the connectivity method for multiple host operations',
        'skip'              => $isSkip,
        'init_with_default' => 1,
        'set_interactive'   => 0,
    };
}

#-------------------------------------------------------------------------------
# Returns a 'retry' parameter.
#
# According to the order of this 'retry' parameter, the specified other
# parameter is performed again if the value of the other parameter is not
# already set.
#
# Skip and hidden is set to prevent the handling of this parameter by any
# subroutines except SDB::Install::Configuration::CheckParams
# and SDB::Install::App::Console::ConfigureInteractive

sub getParamRetryParam {

    my ($self, $order, $otherParamID) = @_;

    return {'order'       => $order,
            'retry_param' => $otherParamID,
            'hidden'      => 1, # prevents any output
            'skip'        => 1, # prevents handling of this parameter by other routines
    };
}

#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'root_password'.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.
#
# Parameter: int    $order
#            string $section

sub getParamRootPassword {

    my ($self, $order, $section,$userParamId) = @_;

    return {'order'   => $order,
            'opt'     => 'root_password',
            'type'    => 'passwd',
            'section' => $section,
            'value'   => undef,
            'default' => undef,
            'str'     => 'Root User Password For Remote Hosts',
            'str_templ'=>'Root User (%s) Password For Remote Hosts',
            'desc'    => 'Root user password to access remote hosts without SSH key',
            'skip'    => 1,
            'init_with_default' => 0,
            'set_interactive'   => 1,
            'mandatory'         => 1,
            'retry_count'       => 3,
            'user_paramid'      => $userParamId
    };
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'root_user'.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.
#
# Parameter: int    $order
#            string $section

sub getParamRootUser {

    my ($self, $order, $section) = @_;

    return {'order'   => $order,
            'opt'     => 'root_user',
            'opt_arg' => '<name>',
            'type'    => 'string',
            'section' => $section,
            'value'   => undef,
            'default' => 'root',
            'str'     => 'Root User Name For Remote Hosts',
            'desc'    => 'Specifies the root user of remote hosts',
            'skip'    => 1,
            'init_with_default' => 0,
            'help_with_default' => 1,
            'set_interactive'   => 1,
            'may_be_interactive'=> 1,
    };
}


#-------------------------------------------------------------------------------
# Returns the entries of the hidden parameter '--skip_hostagent_password'.
#
# If SAPHostagents are already installed at each host
# and the parameter '--install_hostagent=on' is set,
# the SAPHostagents are updated without using the hostagent password.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.
#
# hdblcm checks the local SAPHostagent and the remote SAPHostagents.
# If all SAPHostagents are already installed (i.e. hostagent password is not
# needed), hdblcm passes this information via the provisional parameter
# '--skip_hostagent_password' to those underlying tools that do not call
# CollectOtherHostInfos to detect the existence of remote SAPHostagents.

sub getParamSkipHostagentPw {

    my ($self,
        $order,
        $section,
        $constraint,
        $skip  # default: true
       ) = @_;

    my $isSkip = (defined $skip) ? $skip : 1;

    return {
        'order'           => $order,
        'opt'             => 'skip_hostagent_password',
        'type'            => 'boolean',
        'section'         => $section,
        'value'           => undef,
        'desc'            => 'Does not request a password of SAP Host Agent',
        'skip'            => $isSkip,
        'set_interactive' => 0,
        'hidden'          => 1,
        'constraint'      => $constraint,
    };
}

sub getParamCollectUpdateHostCredentials {
    my ($self, $order, $section, $constraint) = @_;

    return {
        'order'           => $order,
        'opt'             => 'collect_update_host_credentials',
        'type'            => 'boolean',
        'section'         => $section,
        'default'         => 0,
        'value'           => undef,
        'desc'            => 'Collects System User Credentials for Update Local Host',
        'skip'            => 0,
        'set_interactive' => 0,
        'hidden'          => 0,
        'constraint'      => $constraint,
        'init_with_default' => 1,
        'mandatory'       => 1,
    };
}

#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'system_user'.

sub getParamSystemUser {

    my ($self,
        $order,
        $section,
        $skip,      # default: false
        $constraint # e.g. 'Valid with role ets_worker/ets_standby'
       ) = @_;

    my $isSkip = (defined $skip) ? $skip : 0;

    return {'order'   => $order,
            'opt'     => 'system_user',
            'opt_arg' => '<name>',
            'type'    => 'string',
            'section' => $section,
            'value'   => undef,
            'default' => 'SYSTEM',
            'str'     => 'Database User Name',
            'desc'    => 'Specifies the system user of the database',
            'skip'    => $isSkip,
            'init_with_default' => 1,
            'help_with_default' => 1,
            'set_interactive'   => 1,
            'mandatory' => 1,
            'constraint'        => $constraint,
            'hostctrl_opt'      => 'SYSTEM_USER',
    };
}

#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter '--scope'.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.
#
# Parameter: int    $order
#            string $section
#            string $strParam   (optional)
#            string $constraint (e.g. 'register only')

sub getParamScope {

    my ($self, $order, $section, $descParam, $constraint) = @_;

    my $desc = (defined $descParam)
              ? $descParam
              : "Execution on $gProductNameSystem (all hosts) "
                . 'or manually execution at local instance';

    return {
        'order'             => $order,
        'opt'               => 'scope',
        'type'              => 'string',
        'opt_arg_switch'    => 'system',
        'opt_arg'           => 'instance|system',
        'valid_values'      => ['instance', 'system'],
        'section'           => $section,
        'value'             => undef,
        'default'           => 'instance',
        'str'               => 'Execution Scope',
        'desc'              => $desc,
        'skip'              => 1,
        'init_with_default' => 0,
        'init_batch_with_default' => 1,
        'help_with_default' => 1,
        'set_interactive'   => 0,
        'constraint'        => $constraint,
    };
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the hidden parameter
# 'ScopeInteractive' that is only used to ask interactively for scope
# 'system' (true) or 'instance' (false).
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.
#
# Parameter: int    $order
#            string $section
#            string $strParam   (optional)
#            string $constraint (e.g. 'register only')

sub getParamScopeInteractive {

    my ($self, $order, $section, $strParam, $constraint) = @_;

    my $str = (defined $strParam) ? $strParam
                                  : "Automatic execution on remote hosts?";

    return {'order'   => $order,
            'opt'     => 'scopeInteractive',
            'type'    => 'boolean',
            'section' => $section,
            'value'   => undef,
            'default' => 1,
            'str'     => $str,
            'console_omit_word_Enter' => 1,
            'skip'                    => 1,
            'init_with_default'       => 0,
            'set_interactive'         => 1,
            'hidden'                  => 1,
            'constraint'              => $constraint,
    };
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'shell'.
#
# Parameter: int    $order
#            string $section
#            string $constraint (e.g. 'register only')

sub getParamShell {

    my ($self, $order, $section, $constraint) = @_;

    my %paramShell = (
            'order'             => $order,
            'opt'               => 'shell',
            'type'              => 'path',
            'section'           => $section,
            'value'             => undef,
            'default'           => '/bin/sh',
            'str'               => 'System Administrator Login Shell',
            'str_templ'         => 'System Administrator (%s) Login Shell',
            'set_interactive'   => 1,
            'init_with_default' => 1,
            'constraint'        => $constraint,
            );

    return \%paramShell;
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'show_pending'.
#
# Parameter: int    $order
#            string $section
#            string $strParam   (optional)
#            string $constraint (e.g. 'register only')

sub getParamShowPending {

    my ($self, $order, $section, $strParam, $constraint) = @_;

    my $str = (defined $strParam) ? $strParam : 'Shows outstanding hosts';

    my %paramShowPending = (
            'order'           => $order,
            'opt'             => 'show_pending',
            'type'            => 'boolean',
            'section'         => $section,
            'value'           => undef,
            'str'             => $str,
            'set_interactive' => 0,
            'constraint'      => $constraint,
            );

    return \%paramShowPending;
}

#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'sid'.
#
# Parameter: int    $order
#            string $section

sub getParamSID {

    my ($self, $order, $section, $skipInitDefault) = @_;

    my %paramSID = (
            'order'     => $order,
            'opt'       => 'sid',
            'short_opt' => 's',
            'type'      => 'string',
            'section'   => $section,
            'value'     => undef,
            'default'   => undef,
            'str'       => "$gProductNameSystem ID",
            'opt_arg'   => '<SID>',
            'mandatory' => 1,
            'may_be_interactive'=> 1,
            );

    if (!$skipInitDefault) {
        $paramSID{init_with_default} = 1;
    }

    return \%paramSID;
}


sub getParamSourceRepositoryUrl{
    my ($self, $order,$section, $constraint) = @_;
    return {
        'order'             => $order,
        'opt'               => 'repository',
        'opt_arg'           => '<path>',
        'short_opt'         => 'r',
        'type'              => 'string',
        'section'           => $section,
        'value'             => undef,
        'str'               => "Source path of $gProductNameStudio repository",
        'constraint'        => $constraint,
        'init_with_default' => 1,
        'mandatory'         => 1,
        'set_interactive'   => 0
    };
}



sub getParamSQLSysPasswd{
    my ($self,
        $order,
        $section,
        $type,
        $skip, # default: false
        $username,
        $constraint, # e.g. 'Valid for add host with ets_worker/ets_standby'
        $userParamId
       ) = @_;

    my $isSkip = (defined $skip) ? $skip : 0;

    $type = $type // 'passwd';

    my $param = {
        'order'             => $order,
        'opt'               => 'system_user_password',
        'type'              => $type,
        'opt_arg'           => '<database_password>',
        'section'           => $section,
        'value'             => undef,
        'default'           => undef,
        'str'               => 'Database User Password',
        'str_templ'         => 'Database User (%s) Password',
        'init_with_default' => 0,
        'log_value'         => sub {defined $_[0] ? '***' : '<not defined>';},
        'mandatory'         => 1,
        'skip'              => $isSkip,
        'constraint'        => $constraint,
        'retry_count'       => 3,
        'user_paramid'      => $userParamId
    };
    if (defined $username) {
        $param->{str} = sprintf($param->{str_templ}, $username);
    }
    return $param
}

sub getParamSystemUsage{
    my ($self, $order, $section, $constraint) = @_;
    return {
        'order'                     => $order,
        'opt'                       => 'system_usage',
        'type'                      => 'string',
        'valid_values'              => \@systemUsageValidValues,
        'ui_values'                 => \@systemUsageUIValues,
        'section'                   => $section,
        'value'                     => undef,
        'default'                   => 'custom',
        'str'                       => 'System Usage',
        'interactive_index_selection' => 1,
        #'console_omit_word_Enter'   => 1,
        'init_with_default'         => 1,
        'constraint'                => $constraint,
        'hidden'                    => 0,
        'set_interactive'           => 1
    };
}



#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter
# 'sapmnt' (installation path / target).
#
# Parameter: int    $order
#            string $section

sub getParamTarget {

    my ($self, $order, $section) = @_;

    my %paramTarget = (
            'order'             => $order,
            'opt'               => 'sapmnt',
            'alias_opts'        => ['target'],
            'type'              => 'path',
            'opt_arg'           => '<installation_path>',
            'section'           => $section,
            'value'             => undef,
            'default'           => undef,
            'str'               => 'Installation Path',
            'hidden'            => 0,
            'set_interactive'   => 1,
            'init_with_default' => 1,
            );

    return \%paramTarget;
}


sub getParamTargetRepositoryPath{
    my ($self, $order, $section, $constraint) = @_;
    return {
        'order'             => $order,
        'opt'               => 'copy_repository',
        'opt_arg'           => '<target_path>',
        'type'              => 'path',
        'section'           => $section,
        'value'             => undef,
        'str'               => "Target path to which $gProductNameStudio repository should be copied",
        'constraint'        => $constraint,
        'set_interactive'   => 0,
        'exit_if_faulty'    => 1
    };
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'userid'.
#
# Parameter: int    $order
#            string $section
#            string $constraint (e.g. 'register only')

sub getParamUID {

    my ($self, $order, $section, $constraint) = @_;

    my %paramUID = (
            'order'                     => $order,
            'opt'                       => 'userid',
            'short_opt'                 => 'U',
            'type'                      => 'number',
            'section'                   => $section,
            'value'                     => undef,
            'default'                   => undef,
            'str'                       => 'System Administrator User ID',
            'str_templ'                 => 'System Administrator (%s) User ID',
            'set_interactive'           => 1,
            'init_with_default'         => 1,
            'constraint'                => $constraint,
            'leaveEmptyInConfigDump'    => 1,
            'mandatory'                 => 1,
            );

    return \%paramUID;
}


#-------------------------------------------------------------------------------
# Parameter '--use_http'
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.

sub getParamUseHttp {

    my ($self,
        $order,
        $section,
        $hidden, # default: true
        $skip    # default: false
       ) = @_;

    my $isHidden = (defined $hidden) ? $hidden : 1;
    my $isSkip   = (defined $skip  ) ? $skip   : 0;

    return {'order'           => $order,
            'opt'             => 'use_http',
            'type'            => 'boolean',
            'section'         => $section,
            'value'           => undef,
            'default'         => 0,
            'str'             => 'Use http instead of https',
            'desc'            => 'Uses http instead of https',
            'set_interactive' => 0,
            'hidden'          => $isHidden,
            'skip'            => $isSkip,
            'console_omit_word_Enter' => 1,
    };
}

#-------------------------------------------------------------------------------
# Parameter '--sapcontrol_use_http'
#
# It's used to support upgrades from revisions < 90
# This parameter is switched internally to on, if the start revision < 90.
#
sub getParamSAPControlUseHttp {

    my ($self,
        $order,
        $section,
        $hidden, # default: true
        $skip    # default: true
       ) = @_;

    my $isHidden = (defined $hidden) ? $hidden : 1;
    my $isSkip   = (defined $skip  ) ? $skip   : 1;

    return {'order'           => $order,
            'opt'             => 'sapcontrol_use_http',
            'type'            => 'boolean',
            'section'         => $section,
            'value'           => undef,
            'default'         => 0,
            'str'             => 'Use http instead of https for SAPControl web service',
            'desc'            => 'Uses http instead of https for SAPControl web service',
            'set_interactive' => 0,
            'hidden'          => $isHidden,
            'skip'            => $isSkip,
            'console_omit_word_Enter' => 1,
            'hostctrl_opt'    => 'SAPCONTROL_USE_HTTP'
    };
}

#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'secure_store', which
# is used in hdbinst/hdbupd (and also hdblcm)

sub getParamSecureStore {
    my ($self, $order, $section) = @_;
    return {
        'order'             => $order,
        'section'           => $section,
        'type'              => 'string',
        'valid_values'      => [$gSecureStoreTypeSSFS, $gSecureStoreTypeLSS],
        'default'           => $gSecureStoreTypeSSFS,
        'opt'               => 'secure_store',
        'set_interactive'   => 0,
        'init_with_default' => 1,
        'str'               => "$gProductNameEngine secure store",
        'desc'              => "Specifies whether $gProductNameEngine should use standard SSFS secure store or $gProductNameLSS.",
        'skip'              => 0,
        'hidden'            => 0,
    };
}

sub getParamSkipHostagentCalls {
  my ($self, $order, $section, $constraint) = @_;
  my $string = "Skip all $gProductNameSHA calls";

  return {
    'order'                   => $order,
    'opt'                     => 'skip_hostagent_calls',
    'type'                    => 'boolean',
    'section'                 => $section,
    'value'                   => undef,
    'desc'                    => $string,
    'str'                     => $string,
    'default'                 => 0,
    'init_with_default'       => 1,
    'console_omit_word_Enter' => 1,
    'set_interactive'         => 0,
    'constraint'              => $constraint,
    'skip'                    => 0,
    'mandatory'               => 1,
  };
}

#-------------------------------------------------------------------------------
# If the parameter 'root_user' was checked before this method
# returns SDB::Install::RemoteHosts in case of a distributed system.

sub getRemoteHosts {
    return $_[0]->{_remoteHosts};
}

#-------------------------------------------------------------------------------
# Deletes the hash that refers to the instance of RemoteHosts.

sub clearRemoteHosts {
    delete $_[0]->{_remoteHosts};
}

#-------------------------------------------------------------------------------
# Directly sets the {_remoteHosts} value
sub setRemoteHosts {
    return $_[0]->{_remoteHosts} = $_[1];
}


#-------------------------------------------------------------------------------
# Creates the class RemoteHosts or RemoteHostctrlHosts with the existing
# remote hostnames.
# Establishes connections to the existing remote hosts in case of SSH.
# If the host map parameter is used (e.g. hdbgreg, hdbrename) the class is
# created with the new hostnames.

sub initRemoteHosts {
    my ($self) = @_;
    if (!defined $self->{_remoteHosts}){

        my $instance = $self->getOwnInstance();
        if (!defined $instance || !$instance->exists_remote_host()) {
            return 1;
        }

        my @newRemoteHosts;
        my $oldNewHostMap = (exists $self->{params}->{HostMap})
                            ? $self->{params}->{HostMap}->{value}
                            : undef;

        if (defined $oldNewHostMap) {

            foreach my $currHost (@{$instance->get_allhosts()}) {

                my $lcCurrHost = lc($currHost);
                my $newHost    = $oldNewHostMap->{$lcCurrHost};
                $lcCurrHost    = lc($newHost) if (defined $newHost);

                if (!$instance->isHostLocal($lcCurrHost)) {
                    push @newRemoteHosts, $lcCurrHost;
                }
            }
        }
        else {
            @newRemoteHosts = @{$instance->get_hosts()};
        }

        if (!@newRemoteHosts) {
            return 1;
        }

        eval{
            require SDB::Install::RemoteHosts;
            if ($self->isUseSAPHostagent()) {
                require SDB::Install::RemoteHostctrlHosts;
            }
        };

        if ($@){
            $self->setErrorMessage ("Cannot establish remote execution: $@");
            return undef;
        }

        $self->{_remoteHosts} =
                    ($self->isUseSAPHostagent())
                    ? new SDB::Install::RemoteHostctrlHosts(@newRemoteHosts)
                    : new SDB::Install::RemoteHosts(@newRemoteHosts);

        $self->{_remoteHosts}->SetProgressHandler($self->GetProgressHandler());

        if (!$self->{_remoteHosts}->isHostctrl()) {
            my $rc = $self->{_remoteHosts}->connect();
            if ($rc != 0){
                $self->AddError(undef, $self->{_remoteHosts});
                $self->setRemoteHosts(undef) if ($rc == 2); # loading libssh package failed
                return 0;
            }
        }
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Creates the class RemoteHosts and establishes connections to the existing remote hosts.

sub establishRemoteHostConnections {
    return $_[0]->initRemoteHosts();
}


#-------------------------------------------------------------------------------
# Returns SDB::Install::SAPSystem
# and creates the hash 'sapSys' if not exist before.

sub getSAPSystem {
    my ($self, $no_cache) = @_;

    my $sapSys = $self->{sapSys};
    if (!defined $sapSys || $no_cache) {

         my $systems = $self->getCollectSAPSystems($no_cache);

         if (defined $systems) {
            my $sid         = $self->getSID();
            $sapSys         = $systems->{$sid} if (defined $sid);
            $self->{sapSys} = $sapSys          if (defined $sapSys);
         }
    }

    return $sapSys;
}

#-------------------------------------------------------------------------------
# Returns the SAP System ID.
# In case of rename the current valid SID (source or target) is returned.

sub getSID {
    $_[0]->getValue('SID');
}


#-------------------------------------------------------------------------------
# Returns a string array containing valid arguments of the option '--timeout'.

sub getTimeoutValues{
    return [qw (start_service stop_service start_instance stop_instance)];
}


#-------------------------------------------------------------------------------
# Returns a reference to an array containing user specific parmeter IDs.

sub getUserParams {

    my @userParams = $isWin ? qw () : qw (HomeDir Shell UID GID);
    return  \@userParams;
}


#-------------------------------------------------------------------------------
# Initializes any derived configuration class
#
# Returns int retCode

sub InitDefaults{
    my ($self) = @_;

    if ($isWin && exists $self->{params}->{Drive}){
        my $drive = getSAPDrive ($self->getMsgLst ());
        if ($drive) {
            $self->{params}->{Drive}->{value} = uc ($drive);
        }
    }

    #
    # find out if the installer program is part of an installation
    #

    my $sapSys = $self->getInstaller ()->getInstallation
        ($self->getMsgLst(),
         !$self->{isSlave} && $self->isa('SDB::Install::Configuration::Rename'));

    if (defined $sapSys && $sapSys->isa('SDB::Install::SAPSystem')){

        $sapSys->SetProgressHandler($self->GetProgressHandler());

        my $setDefaultTarget = 0;

        if ($sapSys->isPreInitWithTargetSID()) {
            $self->{sapSys}   = $sapSys;
            $setDefaultTarget = 1;
            if (exists $self->{params}->{SID}){
                $self->setDefault ('SID', $sapSys->getTargetSID(), 1);
            }
        }
        elsif ($sapSys->hasNewDB) {
            $self->{sapSys}   = $sapSys;
            $setDefaultTarget = 1;
            if (exists $self->{params}->{SID}){
                $self->setDefault ('SID', $sapSys->get_sid(), 1);
                $self->{params}->{SID}->{set_interactive} = 0;
            }
            if( exists $self->{params}->{'ChangeSystemPasswd'}){
                $self->setDatabaseToSystemUserCredentialsParameter($self->{params}->{'ChangeSystemPasswd'});
            }
            if( exists $self->{params}->{'SystemUser'}){
                $self->setDatabaseToSystemUserCredentialsParameter($self->{params}->{'SystemUser'});
            }
            if( exists $self->{params}->{'SQLSysPasswd'}){
                $self->setDatabaseToSystemUserCredentialsParameter($self->{params}->{'SQLSysPasswd'});
            }
            if( exists $self->{params}->{'SrcSQLSysPasswd'}){
                $self->setDatabaseToSystemUserCredentialsParameter($self->{params}->{'SrcSQLSysPasswd'});
            }
            if( exists $self->{params}->{'TrgSQLSysPasswd'}){
                $self->setDatabaseToSystemUserCredentialsParameter($self->{params}->{'TrgSQLSysPasswd'});
            }
            if( exists $self->{params}->{'TenantUser'}){
                $self->setDatabaseToTenantUserCredentialsParameter($self->{params}->{'TenantUser'});
            }
            if( exists $self->{params}->{'SQLTenantUserPassword'}){
                $self->setDatabaseToTenantUserCredentialsParameter($self->{params}->{'SQLTenantUserPassword'});
            }
        }

        if ($setDefaultTarget && exists $self->{params}->{Target}){
            $self->setDefault ('Target', $sapSys->get_target () , 1);
            $self->{params}->{Target}->{set_interactive} = 0;
       }
    }
    return 1;
}

sub getInstaller
{
    return new SDB::Install::Installer ();
}


#-------------------------------------------------------------------------------
# Returns 1 if authentication failed on CollectOtherHostInfos.

sub isAuthenticationFailedOnCollectOtherHostInfos {
	my ($self) = @_;

	if (! exists $self->{_isAuthenticationFailedOnCollectOtherHostInfos}) {
		return 0;
	}

	return $self->{_isAuthenticationFailedOnCollectOtherHostInfos};
}

#-------------------------------------------------------------------------------
# Returns 1 if checks have to be performed only.

sub isCheckOnly {
    my ($self) = @_;
    $self->getValue('CheckOnly');
}


#-------------------------------------------------------------------------------
# Returns 1 if CollectOtherHostInfos() should be executed.

sub isCollectOtherHostInfosRequired {
	return 1;
}


#-------------------------------------------------------------------------------
# Returns 1 if the specified host name can be found with nslookup.
#
# Parameter: string $hostname
#
# Returns int retCode

sub isHostAccessible {

    my ($self, $hostname) = @_;

    my $errlst   = new SDB::Install::MsgLst();
    my $nslookup = nslookup($hostname, $errlst);

    if (!defined $nslookup) {
        $self->appendErrorMessage
                   ("Error calling nslookup", $errlst);
    }
    elsif (!defined $nslookup->{address}) {
        $self->appendErrorMessage ("Cannot resolve host name '$hostname'"
                          . ($nslookup->{msg} ? ": $nslookup->{msg}" : ''));
        return undef;
    }

    return $nslookup;
}


sub verifyNewHostName{
    my ($self, $newHost, $isRemoteHost) = @_;

    if ($newHost eq "") {
        $self->appendErrorMessage ("Host name or ipv4 address is empty");
        return 0;
    }

    my $rc = 1;

    if (length ($newHost) > 64){
        $self->appendErrorMessage ("Host name '$newHost' has more than 64 characters.");
        $self->appendErrorMessage ("HANA doesn't support host names > 64 characters.");
        $rc = 0;
    }

    if ($newHost !~ /$hostname_regex|$ipv4_regex/){
        $self->appendErrorMessage ("'$newHost' is no valid host name or ipv4 address");
        return 0;
    }

    if ($newHost =~ /^\d+$/){
        $self->appendErrorMessage ("Cannot use '$newHost' as a host name. Numeric host names are not supported");
        return 0;
    }

    my $nslookup = $self->isHostAccessible ($newHost);

    if (!defined $nslookup) {
        return 0;
    }

    if (!$isRemoteHost){
        my $errlst = new SDB::Install::MsgLst ();
        my $isLocalIp = is_local_address ($nslookup->{address}, $errlst);
        if (!defined $isLocalIp){
            $self->getMsgLst()->addWarning ("Error checking ip address is local => Skipping host name check", $errlst);
            return $rc;
        }

        if (!$isLocalIp) {
             $self->appendErrorMessage ("IP address '$nslookup->{address}' (host '$nslookup->{name}') is not assigned to a local network interface");
             return 0;
        }
        return $rc;
    }

    return $rc;
}


#-------------------------------------------------------------------------------
# Returns 1 if the scope parameter requires remote execution

sub isRemoteScope {
    my ($self, $scope) = @_;
    $scope = $self->getValue('Scope') if (!defined $scope);

    return (defined $scope
            && (($scope eq 'system') || ($scope eq 'all_instances'))) ? 1 : 0;
}

#-------------------------------------------------------------------------------
# Returns 1 if 'saphostagent' is set for the parameter '--remote_execution'.

sub isUseSAPHostagent {
    my ($self) = @_;
    my $value = $self->getValue('RemoteExecution');
    return defined $value && ($value eq 'saphostagent');
}

#-------------------------------------------------------------------------------
# Returns 1 if the HANA Secure store should be initialized/migrated to LSS

sub shallUseLSS {
    my ($self) = @_;
    my $secureStoreType = $self->hasValue('SecureStore') ? $self->getValue('SecureStore') : undef;
    return $secureStoreType && ($secureStoreType eq $gSecureStoreTypeLSS);
}

#-------------------------------------------------------------------------------
# Returns 1 if scenario should use sidadm user for execution of SHA operations
# with SDB::Install::RemoteHostctrlHosts on remote hosts.

sub isUseSidadmUserForRemoteHostctrlHosts {
	my ($self) = @_;

	if (not defined $self->{_isUseSidadmUserForRemoteHostctrlHosts}) {
		return undef;
	}
	return $self->{_isUseSidadmUserForRemoteHostctrlHosts};
}


sub setUseSidadmUserForRemoteHostctrlHosts {
	$_[0]->{_isUseSidadmUserForRemoteHostctrlHosts} = $_[1];
}


#-------------------------------------------------------------------------------
# Returns 1 if the parameter 'show_pending' is set.

sub isShowPending {
    my ($self) = @_;
    $self->getValue('ShowPending');
}


#-------------------------------------------------------------------------------
# Deletes the hashes concerning SAPSystems.

sub resetSAPSystemHashes {

    my ($self) = @_;

    delete $self->{sapSys};
    CleanupSAPSystemCache();
}


#-------------------------------------------------------------------------------
# Enables the parameters '--scope' and '--remote_execution'
# in case of a distributed system.
#
# A comment at the beginning of AnyConfig shows the correct order of
# depending parameters.

sub tryEnableScopeParams {

    my ($self) = @_;

    my $instance = ($self->{isSlave}) ? undef # local work only
                                      : $self->getOwnInstance();

    if (defined $instance && $instance->exists_remote_host()) {

        $self->setSkip('Scope',            0);
        $self->setSkip('ScopeInteractive', 0);

        if (defined $self->getBatchValue('RootUser')
            && !defined $self->getBatchValue('Scope')) {
            $self->setScope('system');
        }
    }
    else {
        $self->{params}->{RemoteExecution}->{hidden} = 1; # hide summary output
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Sets the hostagent password
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.

sub setHostagentPassword {

    my ($self, $value) = @_;

    my $rc = $self->checkHostagentPassword($value);
    if (!$rc) {
        return $rc;
    }

    $self->{params}->{HostagentPassword}->{value} = $value;

    if ($self->isUseSAPHostagent() && $self->isCollectOtherHostInfosRequired()) {
        if (!defined $self->CollectOtherHostInfos()) {
            $self->{params}->{HostagentPassword}->{no_retry} = 1;
            return undef;
        }
    }

    return 1;
}

#-------------------------------------------------------------------------------
# Sets the boolean parameter 'show_pending'.

sub setShowPending {

    my ($self, $value) = @_;

    if (defined $value && ($value =~ /$bool_true_pattern/i)) {

        $self->{params}->{ShowPending}->{value} = 1;

        foreach my $id (keys (%{$self->{params}})) {

            if (($id eq 'SID') || ($id eq 'Target')) {

                if (defined $self->getDefault($id)) {
                    $self->{params}->{$id}->{set_interactive}   = 0;
                    $self->{params}->{$id}->{init_with_default} = 1;
                }
            }
            else {
                $self->setSkip($id);
            }
        }
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Sets the value 'ssh' or 'saphostagent' for the parameter '--remote_execution'.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.

sub setRemoteExecution {
    my ($self, $value) = @_;
    if (!$self->checkRemoteExecution($value)) {
        return 0;
    }

    $self->handleSidadmExecution();
    my $lcValue = (defined $value)
                  ? lc($value)
                  : $self->getDefault('RemoteExecution');

    $self->{params}->{RemoteExecution}->{value} = $lcValue;
    return 1;
}


#-------------------------------------------------------------------------------
# Sets the scope, i.e. working on localhost only or on all hosts
# of a distributed system.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.

sub setScope {

    my ($self, $value) = @_;

    if (!defined $value) {
        return 1;
    }

    my $rc = $self->checkValue('Scope', $value);
    if (!$rc) {
        return $rc;
    }

    $self->{params}->{Scope}->{value} = lc($value);

    $self->setSkip('ScopeInteractive');

    if ($self->isRemoteScope(lc($value))) {

        if ($self->isUseSAPHostagent() && !$self->initRemoteHosts()) {
            return undef;
        }
        $self->enableMultipleHostParameters($self->isUseSidadmUserForRemoteHostctrlHosts());
    }
    else {
        $self->{params}->{RemoteExecution}->{hidden} = 1; # hide summary output

        if ($self->getValue('InstallHostagent')) {
            $self->setSkip('SkipHostagentPw', 0);
        }
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Sets the scope 'system' if the boolean value true is specified.
# The member 'value' of 'ScopeInteractive' remains undef.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.

sub setScopeInteractive {

    my ($self, $value) = @_;

    if (defined $value && ($value =~ /$bool_true_pattern/i)) {
        $self->setScope('system');
    }
    else {
        $self->setScope('instance');
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Sets the hash 'summaryOrder' for each parameter contained in the specified
# parameter ID list.
# The subroutine 'displayParameterSummary' displays the parameters in the
# specified order. Parameters that are not contained in $orderedParamIDs
# are displayed behind the specified parameters.

sub setSummaryOrder {

    my ($self, $orderedParamIDs) = @_;

    my $currOrder = 0;

    foreach my $paramID (@$orderedParamIDs) {
        if (exists $self->{params}->{$paramID}) {
            $self->{params}->{$paramID}->{summaryOrder} = int($currOrder);
            $currOrder++;
        }
    }
}


#-------------------------------------------------------------------------------
# Sets the $self->{_isAuthenticationFailedOnCollectOtherHostInfos} flag

sub setIsAuthenticationFailedOnCollectOtherHostInfos {
	my ($self, $isFailed) = @_;
	$self->{_isAuthenticationFailedOnCollectOtherHostInfos} = $isFailed;
}


#-------------------------------------------------------------------------------
# Checks the specified instance number and assigns it with 2 digits
# to the specific parameter value.
#
# Returns int retCode

sub setInstanceNumber {

    my ($self, $value) = @_;

    if (!$self->checkInstanceNumber($value)) {
        return 0;
    }

    my $strValue = sprintf ('%02d', $value);

    $self->{params}->{InstanceNumber}->{value} = $strValue;

    return 1;
}


#-------------------------------------------------------------------------------
# Shows the configured SID, instance number and host names.
#
# Parameter:  string $destinName   # e.g. 'Source', 'Target'
#
# Returns int retCode

sub showSystemProperties {

    my ($self, $destinName) = @_;
    return if $self->{systemPropertiesShown};
    my $msgLst       = new SDB::Install::MsgLst ();
    my $databaseInfo = $self->{sapSys}->asString ($msgLst, undef, 1, 1); # multiple lines with product names
    my $header       = 'System Properties:';
    $header          = "$destinName $header" if (defined $destinName);

    $self->AddProgressMessage('');
    $self->AddProgressMessage($header, $msgLst);

    if (defined $self->GetProgressHandler()) {
        $self->GetProgressHandler()->SetProgress($databaseInfo);
    }
    $self->{systemPropertiesShown} = 1;
}

#-------------------------------------------------------------------------------
sub getInternalDeviceIP{
    return $_[0]->{internalLocalIp};
}

#-------------------------------------------------------------------------------
sub getInternalNetworkIP{
    return $_[0]->{internalNetworkPrefix};
}

#-------------------------------------------------------------------------------
sub getSysInfo{
    my ($self) = @_;
    if (!defined $self->{sysinfo}){
        require SDB::Install::SysInfo;
        $self->{sysinfo} = new SDB::Install::SysInfo ();
    }
    return $self->{sysinfo};
}

sub getPhysicalMem{
	my $mi = GetMachInfo();
	my $mem = undef;
	if (defined $mi) {
		# calculate memory in bytes: [0] is the mumber, [1] is the scaling factor (bytes, KB, etc.)
		$mem = $mi->{PhysicalMemoryTotal}->[0] * $mi->{PhysicalMemoryTotal}->[1];
	}
	return $mem;
}


sub getRemoteInstallerExecutableFromKit{
    my ($self) = @_;
    if (!defined $self->{_remoteInstallerFromKit}){
        my $remotePath = $self->_getRemotePathForInstaller();
        my $remoteInstaller =
            $self->{_remoteInstallerFromKit} =
            $self->copyInstallerToOtherHosts ($remotePath);
        if (!defined $remoteInstaller){
            $self->setErrorMessage ('Cannot distribute installer files', $self->getErrMsgLst ());
            return undef;
        }
        #
        # check whether the shared fs is really shared
        #
        my $statErrors;
        foreach my $remoteHosts (@{$self->getRemoteHostsObjectsForCollectHostInfo ()}){
            $statErrors = [];
            my $rc = $remoteHosts->existsFile ($self->{_remoteInstallerFromKit},0, $statErrors);
            if (defined($rc) && !$rc){
                my $checkmnt = $self->getValue ('CheckMnt');
                my $target = $self->getValue ('Target');
                $self->AddError (undef, $remoteHosts);
                my (@perm_denied_hosts, @enoent_hosts);
                foreach my $hostError (@$statErrors){
                    if ($hostError->[0] == 3){
                        push @perm_denied_hosts, $hostError->[2];
                    }
                    if ($hostError->[0] == 2){
                        push @enoent_hosts, $hostError->[2];
                    }
                }
                if (@perm_denied_hosts){
                    $self->appendErrorMessage ('User \''.$remoteHosts->getUserName."' doesn't have the permissions to access '$self->{_remoteInstallerFromKit}' on host(s): :" .
                        join (', ', @perm_denied_hosts));
                }
                if (!@enoent_hosts){
                    # skip error message
                    return undef;
                }
                if (!$checkmnt){
                    my $paramName = $self->{params}->{Target}->{str};
                    $self->PushError ("File system '$target' has to be shared across all hosts.");
                    $self->PushError ("If you have shared a directory other than the $paramName '$target', specify it with the --checkmnt option.");
                }
                else{
                    $self->PushError ("File system '$checkmnt' has to be shared across all hosts.");
                }
                return undef;
            }
        }
    }
    return $self->{_remoteInstallerFromKit};
}

sub removeRemoteTmpInstaller{
    my ($self) = @_;
    if ($self->{_localInstallerDir}){
        my $msg = $self->getMsgLst ()->addMessage ("Removing shared temporary installer copy");
        deltree ($self->{_localInstallerDir}, $msg->getSubMsgLst ());
    }
    elsif($self->{_remoteInstallerDir}){
        my $msg = $self->getMsgLst ()->addMessage ("Removing temporary installer copies");
        foreach my $remoteHosts (@{$self->getRemoteHostsObjectsForCollectHostInfo ()}){
            $remoteHosts->setMsgLstContext ([$msg->getSubMsgLst ()]);
            $remoteHosts->deltree ($self->{_localInstallerDir});
        }
        delete $self->{_remoteInstallerDir};
    }
    undef $self->{_remoteInstallerFromKit};
    return 1;
}

sub _getCheckmnt {
    my ( $self ) = @_;
    my $checkmnt = $self->getValue ('CheckMnt');

    if (!defined $checkmnt){
         $checkmnt = $self->getValue ('Target');
    }
    return $checkmnt;
}

sub _getRemotePathForInstaller {
    my ( $self ) = @_;

    my $sid = $self->getSID();
    if (isSidadmin($sid)) {
       return $self->getValue ('Target') . $path_separator . $sid . $path_separator . $remoteTmpDir;
    }

    my $checkmnt = $self->_getCheckmnt();
    if (defined $checkmnt){
       return $checkmnt . $path_separator . $remoteTmpDir;
    } else {
       return '/var/tmp/' . $remoteTmpDir;
    }

}

sub copyInstallerToOtherHosts{
    my ($self, $remotePath) = @_;
    my $installer = new SDB::Install::Installer ();
    my $checkmnt = $self->_getCheckmnt();
    my @installer_files =
        @SDB::Install::Configuration::EffectiveModulesRemoteCheck::effective_modules;
    my $installerExe = $self->getInstallerExe ();
    push @installer_files, $installerExe;

    my $installerDir = $installer->getInstallerDir ();

    if (defined $checkmnt){
        #
        # use shared filesystem /sapmnt to copy installer once
        #
        my $cfg = {'mode' => 0700};
        if (!defined makedir ($remotePath, $cfg)){
            $self->AddError ("Cannot create directory '$remotePath'", $cfg);
            return undef;
        }
        $self->{_localInstallerDir} = $remotePath;

        foreach my $file (@installer_files){
            $cfg = {'binmode' => 1, 'nochown' => 1, 'createdir' => 1};
            if (!defined copy_file ($installerDir . $path_separator . $file, $remotePath . $path_separator . $file, $cfg)){
                $self->AddError ("Cannot copy installer", $cfg);
                return undef;
            }
        }
        # copy signature manifest and filelist for hostagent
        $cfg = {'binmode' => 1, 'nochown' => 1};
        my $sigManifestSource = determineSignedManifestPath($installerDir);

        if (-f $sigManifestSource) {
            if (!defined copy_file ($sigManifestSource, $remotePath, $cfg)){
                $self->AddMessage ("Cannot copy signature manifest", $cfg);
            } else {
                my $rTgt = $remotePath.$path_separator.$gSignatureManifestName;
                if(! chmod (0644, $rTgt)) {
                    $self->AddMessage("cannot chmod file '$rTgt': $!");
                }
            }
        }
        my $filelistSource = $installerDir.$path_separator.'filelist.hdbinst_remote_check';
        if (-f $filelistSource) {
            if (!defined copy_file ($filelistSource, $remotePath, $cfg)){
                $self->AddMessage ("Cannot copy signature file list", $cfg);
            }
            else {
                my $rTgt = $remotePath.$path_separator.'filelist.hdbinst_remote_check';
                if(! chmod (0644, $rTgt)) {
                    $self->AddMessage("cannot chmod file '$rTgt': $!");
                }
            }
        }
        $filelistSource = $installerDir.$path_separator.'filelist.hdblcm_remote_check';
        if (-f $filelistSource) {
            if (!defined copy_file ($filelistSource, $remotePath, $cfg)){
                $self->AddMessage ("Cannot copy signature file list", $cfg);
            }
            else {
                my $rTgt = $remotePath.$path_separator.'filelist.hdblcm_remote_check';
                if(! chmod (0644, $rTgt)) {
                    $self->AddMessage("cannot chmod file '$rTgt': $!");
                }
            }
        }
        my $remote_runtimedir = $remotePath.$path_separator.'instruntime';
        if(! chmod (0755, $remote_runtimedir)) {
            $self->AddMessage ("cannot chmod dir '$remote_runtimedir': $!");
        }
        my $dirWalker = SDB::Install::DirectoryWalker->new(undef, undef, undef, undef, undef, undef,
            sub{   # action callback
                my $isDir = $_[6];
                if($isDir) {
                    my $tgt = $_[3].$path_separator.$_[4];
                    if(! chmod (0755, $tgt)) {
                        $_[1]->AddError ("cannot chmod dir '$tgt': $!");
                        return undef;
                    }
                }
                return 1;
            },
            undef, undef, 1, 1, 1 );
        my $rc = $dirWalker->findAndProcess($remote_runtimedir);
        if(not defined $rc) {
            $self->AddMessage ($dirWalker->getErrorString());
        }
    }
    else{
        #
        # copy installer to each hosts /var/tmp filesystem
        #
        $self->{_remoteInstallerDir} = $remotePath;
        foreach my $remoteHosts (@{$self->getRemoteHostsObjectsForCollectHostInfo ()}){
            if ($remoteHosts->mkdir ($remotePath, 0700) != 0){
                $self->PushError (undef, $remoteHosts);
                return undef;
            }
            if ($remoteHosts->copyTree ($installerDir, $remotePath, \@installer_files) != 0){
                $self->PushError (undef, $remoteHosts);
                return undef;
            }
        }
    }
    return $remotePath . $path_separator . $installerExe;
}


#-------------------------------------------------------------------------------
sub getRemoteInstallerExecutable{
    my ($self, $actionProgressive, $remoteTool) = @_;

    return $self->getRemoteExecutable($actionProgressive, $remoteTool);
}


#-------------------------------------------------------------------------------
sub getRemoteExecutable{
    my ($self, $actionProgressive, $remoteTool) = @_;

    if (!defined $remoteTool){
        $remoteTool = 'hdbreg';
    }

    my $sapSys = $self->getCurrentSAPSystem();

    if (!defined $sapSys) {
        my $sid = $self->getCurrentSID();
        $sid    = 'undefined' if (!defined $sid);
        $self->setErrorMessage("Invalid $gProductNameSystem ID '$sid'"
                             . " ($actionProgressive remote hosts failed)");
        return undef;
    }
    return join ($path_separator,
                 $sapSys->get_globalTrexInstallDir(), 'bin', $remoteTool);
}

sub getCurrentSAPSystem {
    my ($self) = @_;
    return $self->getSAPSystem();
}

sub getCurrentSID {
    my ($self) = @_;
    return $self->getSID();
}


#-------------------------------------------------------------------------------
# CollectOtherHostInfos
#
# A comment at the beginning of AnyConfig shows the names and the order of
# of the subroutines calling collect host info.
#
# Executes CollectHostInfo on remote hosts in parallel and creates these hash maps:
# (If the collection of an xml tag is skipped, the corresponding hash map
#  is not reset, i.e. the hash map from a previous collection is still available.)
#
# If xml tag 'groups' is wanted:
# (A subset is created if the xml tag 'sapsysid' is wanted or
#                                        the parameter $wantedGID is specified.)
#                      groupID       host        name
# $self->{remoteGids}->{79} ->[0]->['lu123456', 'sapsys']
#                           ->[1]->['lu222222', 'sapsys']
#
# If sapsys is found on a remote host, other groups of that host
# (except '--groupid') are omitted.
#
#
# If xml tag 'SAPSystems' is wanted:
#                           instNr         host        SID  instType  landscapeID
# $self->{remoteInstances}->{'01'}->[0]->['lu123456', 'DB1', 'HDB',  '51ddf0d6-83bd...']
#                                   [1]->['lu222222', 'DB1', 'HDB',  '51ddf0d6-83bd...']
#                           {'05'}->[0]->['lu123456', 'XX5', 'other']
#                           {'06'}->[0]->['lu222222', 'DB6', 'HDB',  '51d3f1ef-84b5...']
#                           {'07'}->[0]->['lu123456', 'DB7', 'HDB',  '51d52dce-8eb9...']
#                                   [1]->['lu222222', 'DB7', 'HDB',  '51d52dce-8eb9...']
#
#
# If xml tag 'networkAddresses' is wanted:
#                       host          IPaddr array
# $self->{remoteIPs}->{'lu123456'}->['127.0.0.1/8', '100.100.100.150/24']
#                     {'lu222222'}->['127.0.0.1/8', '100.100.100.151/24']
#
#
# If xml tag 'saphostagentVersion' is wanted:
#                                 host         version
# $self->{remoteSaphostagents}->{'lu123456'}->['7.20.162']
#                               {'lu222222'}->['7.20.150']
#
#
# If xml tag 'user' is enabled:
# (A subset may be created if the parameter $wantedSidadm or $wantedUID is specified.)
#                           user             host       userID  groupID
# $self->{remoteSidadms}->{'db1adm'}->[0]->['lu123456',  1010,  79]
#                                     [1]->['lu222222',  1010,  79]
#                         {'db6adm'}->[0]->['lu222222',  1003,  79]
#                         {'db7adm'}->[0]->['lu123456',  1002,  79]
#                                     [1]->['lu222222',  1002,  79]
#
#
# If xml tag 'SAPSystems' is enabled:
#                          instNr   host array
# $self->{remoteSystems}->{'DB1'}->['lu123456', 'lu222222']
#                         {'DB6'}->['lu222222']
#                         {'DB7'}->['lu123456', 'lu222222']
#                         {'XX5'}->['lu123456']
#
#
# If xml tag 'user' is enabled:
# (A subset may be created if the parameter $wantedSidadm or $wantedUID is specified.)
#                      userID         host        user     gid
# $self->{remoteUids}->{0}   ->[0]->['lu123456', 'root',    0]
#                      {1002}->[0]->['lu123456', 'db7adm', 79]
#                              [1]->['lu222222', 'db7adm', 79]
#                      {1003}->[0]->['lu222222', 'db6adm', 79]
#                      {1005}->[0]->['lu123456', 'xx5adm', 79]
#                      {1010}->[0]->['lu123456', 'db1adm', 79]
#                              [1]->['lu222222', 'db1adm', 79]
#
#
# If parameter $wantedUsers is set:
# $self->{remoteHostUsersAndGroups}->{lu123456}->{users} ->{ds1adm}->[1001, 79]
#                                                          {ds2adm}->[1002, 79]
#                                                {groups}->{sapsys}->[79,rudi,alf]
#                                                          {ds1grp}->[1000, ds1adm]
#                                    {lu222222}->{users} ->{ds3adm}->[1003, 79]
#                                                          {ds2adm}->[1002, 79]
#                                                {groups}->{sapsys}->[79, hans, uschi]
#                                                          {ds3grp}->[1000, ds3adm]
#
# If xml tag 'sapsysid' is enabled:
#                      ID    host array
# $self->{sapsysids}->{79}->['lu123456', 'lu222222']
# (This subroutine returns undef if different group IDs for sapsys are found!)
#
#
# If xml tag 'certificateHostname' is enabled:
#                                host           external_hostname
# $self->{CertificatesHostmap}->{host5598}->'host5598'
# 								{host5599}->'host5599'
# 								{host5600}->'host5600'
#
#
# If $self->getOwnInstance()->getDataPath(1) is defined:
#                                     host          path exists
# $self->{dataPathExistenceHostmap}->{host5598}->{'exists'} -> 1,
#                                                {'sidadmCanAccess'} -> 1
#                                                {'isSharedVolume'}  -> 0
#                                                {'msgLst'}   -> SDB::Instalal::MsgLst
#
#
# If $self->getOwnInstance()->getLogPath(1) is defined:
#                                    host          path exists
# $self->{logPathExistenceHostmap}->{host5598}->{'exists'} -> 1,
#                                               {'sidadmCanAccess'} -> 1
#                                               {'isSharedVolume'}  -> 1
#                                               {'msgLst'}   -> SDB::Instalal::MsgLst
#
#
# If either 'LssInstPath' parameter has value or $lssInstance->getLssSharedPath():
#                                        'LssInstPath' is shared
# $self->{lssPathExistenceHostmap}->{host5598}-> 1
#                                   {host5600}-> 0

sub CollectOtherHostInfos{
    my ($self,
        $isNotVerbose,
        $skipCollectAll, # bool value: does not collect any main xml tag,
                         # except $wantedSidadm, $wantedUID/GID, --dataPath, --logPath
                         # [cannot be applied together with $collectOnlyMap]
                         #
        $skipCollectMap, # does not collect main xml tags that are contained in
                         # this hash map, e.g. {'groups' => 1, 'users' => 1)
                         # [cannot be applied together with $skipCollectAll and $collectOnlyMap]
                         #
        $collectOnlyMap, # collect main xml tags that are contained in
                         # this hash map, e.g. {'groups' => 1, 'users' => 1)
                         #
        $wantedSidadm,   # if xml tag 'users' is skipped, the properties of this sidadm are collected
        $wantedSidcrypt,
        $wantedUID,      # if xml tag 'users' is skipped, the properties of this user ID are collected
        $wantedGID,      # if xml tag 'groups' is skipped, the group name is collected
        $wantedLSSGroupID,
        $wantedUsers,    # if xml tag 'groups' is skipped, the user names are collected
        $wantedGroups,   # if xml tag 'groups' is skipped, the group names are collected
        $wantedDataPath,
        $wantedLogPath,
       ) = @_;

    my $executable = $self->getRemoteInstallerExecutable
                                                 ('Collecting information of ');
    if (!defined $executable){
        return undef;
    }
    my $msg = $self->getMsgLst ()->addMessage ("Collecting host informations");
    $self->{host_data} = {};
    my $success = 1;

    my $progress_message = "Collecting information from host '%s'...";
    my $done_message     = "Information collected from host '%s'.";
    my $error_message    = "Collecting information from host '%s' failed!";
    my $rc               = 0;
    my $skipCollect      = undef;
    my $collectOnly      = undef;
    my $dataPath         = $wantedDataPath;
    my $logPath          = $wantedLogPath;
    my $wantedSapadm     = undef;
    my $lssPath          = $self->getLssPath();
    my $chkSharedDataPathFile = undef;
    my $chkSharedLogPathFile = undef;
    my $chkSharedDataSubvolumePathFile = undef;
    my $chkSharedLogSubvolumePathFile = undef;
    my $chkSharedLssPathFile = undef;

    if ((!$skipCollectAll
        && (!defined $skipCollectMap || !$skipCollectMap->{saphostagentVersion})
        && (!defined $collectOnlyMap ||  $collectOnlyMap->{saphostagentVersion}))) {
        $wantedSapadm = $self->getBatchValue('HostagentUserName');
        $wantedSapadm = 'sapadm' if (!defined $wantedSapadm);
    }

	my $ownInstance = $self->getOwnInstance();
	if (defined $ownInstance) {
		$dataPath  = $ownInstance->getDataPath(1);
		$logPath   = $ownInstance->getLogPath(1);
    }

    if(defined $dataPath && File::stat::stat($dataPath)){
        $chkSharedDataPathFile = $self->createTempCheckShared($dataPath);

        require SDB::Install::Configuration::NewServerConfig;
        my $subvolumePath = File::Spec->catfile($dataPath, $SDB::Install::Configuration::NewServerConfig::volumeSubDirPerHost);
        if(File::stat::stat($subvolumePath)){
            $chkSharedDataSubvolumePathFile = $self->createTempCheckShared($subvolumePath);
        }
    }
    if(defined $logPath && File::stat::stat($logPath)){
        $chkSharedLogPathFile = $self->createTempCheckShared($logPath);

        require SDB::Install::Configuration::NewServerConfig;
        my $subvolumePath = File::Spec->catfile($logPath, $SDB::Install::Configuration::NewServerConfig::volumeSubDirPerHost);
        if(File::stat::stat($subvolumePath)){
            $chkSharedLogSubvolumePathFile = $self->createTempCheckShared($logPath);
        }
    }
    if(defined $lssPath && File::stat::stat($lssPath)){
        $chkSharedLssPathFile = $self->createTempCheckShared($lssPath);
    }

	if (defined $collectOnlyMap) {
        foreach my $tag (keys %$collectOnlyMap) {
            if ($tag) {
                if (defined $collectOnly) {
                    $collectOnly .= ",$tag";
                }
                else {
                    $collectOnly = $tag;
                }
            }
        }
    }
    elsif ($skipCollectAll) {
        $skipCollect = 'all';
    }
    elsif (defined $skipCollectMap) {
        foreach my $tag (keys %$skipCollectMap) {
            if ($tag) {
                if (defined $skipCollect) {
                    $skipCollect .= ",$tag";
                }
                else {
                    $skipCollect = $tag;
                }
            }
        }
    }

	$self->setIsAuthenticationFailedOnCollectOtherHostInfos(0);

    my $installedServerManifest = (defined $ownInstance) ? $ownInstance->getManifest() : undef;
    my $targetServerManifest = (defined $self->{kit}) ? $self->{kit}->getManifest() : undef;
    my $manifestToUse = $installedServerManifest || $targetServerManifest;
    my $compilerVersion = $manifestToUse->getValue('compiler-version');

    foreach my $remoteHosts (@{$self->getRemoteHostsObjectsForCollectHostInfo()}){
        $remoteHosts->resetMsgLstContext ();

        my $saveCntxt = $remoteHosts->setMsgLstContext ([$msg->getSubMsgLst()]);
        if ($remoteHosts->isHostctrl()) {

            my ($operation, $optionMap);
# 1 stands for 'Get the SID value from the file system itself'
# This is done in case of sidmountpoint register rename, because the SID value on the fs
# might differ from the current SID of the SAP Hana System itself
# (see implementation of getSID in LCM::Configuration::RenameSystemConfiguration)
            my $sid = ($self->isPendingInstallation()) ? undef : $self->getSID(1);

            if (defined $self->{_remoteInstallerFromKit}
                         && ($executable eq $self->{_remoteInstallerFromKit})) {

                my $target   = $self->getValue('Target');
                my $checkMnt = $self->getValue('CheckMnt');
                $operation   = $self->getInstallerCollectOtherHostInfoOperation();
                $optionMap   = {
                    'SAPMNT'       => $target,
                    'HOSTNAME'     => hostname(),
                    'INSTALLER_ID' => $$,
                    'SID' => $sid,
                };

                if (defined $checkMnt) {
                    my ($sapmnt, $sid) = ($checkMnt =~ /(.*)\/+([^\/]+)\/*$/);
                    if (!defined $sid || !$self->matchesPatternSID ($sid) ||
                        $sapmnt ne $target){
                        $self->PushError ('Value of parameter '
                            . $self->getOpt('CheckMnt')
                            . "=$checkMnt' not supported for SAPHostControl"
                            . " ('<Installation Path>/<SID>' is required)");
                        return undef;
                    }
                    $optionMap->{SAPMNT} = $sapmnt;
                    $optionMap->{SID} = $sid;
                }
            }
            else {
                $operation = $gOperationCollectHostInfo;
                $optionMap = {
					'SAPMNT' => $self->getSAPSystem()->get_target(),
					'SID'    => $sid
                };
            }

			$optionMap->{'DATA_PATH'} = $dataPath if defined $dataPath;
			$optionMap->{'LOG_PATH'} = $logPath if defined $logPath;
			$optionMap->{'COLLECT_ONLY'} = $collectOnly  if (defined $collectOnly);
			$optionMap->{'SKIP_COLLECT'} = $skipCollect  if (defined $skipCollect);
			$optionMap->{'SAPADM'}       = $wantedSapadm if (defined $wantedSapadm);
			$optionMap->{'SIDADM'}       = $wantedSidadm if (defined $wantedSidadm);
			$optionMap->{'USERID'}       = $wantedUID    if (defined $wantedUID);
			$optionMap->{'GROUPID'}      = $wantedGID    if (defined $wantedGID);
			$optionMap->{'USERNAMES'}    = $wantedUsers  if (defined $wantedUsers);
			$optionMap->{'GROUPNAMES'}   = $wantedGroups if (defined $wantedGroups);
            $optionMap->{'COMPILER_VERSION'} = $compilerVersion if (defined $compilerVersion);
            $optionMap->{'CHKSHARED_DATAFILE'} = $chkSharedDataPathFile if (defined $chkSharedDataPathFile);
            $optionMap->{'CHKSHARED_LOGFILE'} = $chkSharedLogPathFile if (defined $chkSharedLogPathFile);
            $optionMap->{'CHKSHARED_SUBVOLDATA'} = $chkSharedDataSubvolumePathFile if(defined $chkSharedDataSubvolumePathFile);
            $optionMap->{'CHKSHARED_SUBVOLLOG'} = $chkSharedLogSubvolumePathFile if(defined $chkSharedLogSubvolumePathFile);
            $optionMap->{'CHKSHARED_LSSPATH'} = $chkSharedLssPathFile if(defined $chkSharedLssPathFile);

            $rc = $remoteHosts->executeHostctrlParallel($operation,
                                                        $self, # instconfig
                                                        undef, # param IDs
                                                        undef, # password IDs
                                                        undef, # remote ignore
                                                        undef, # remote timeout
                                                        $progress_message,
                                                        $done_message,
                                                        $error_message,
                                                        $optionMap,
                                                        undef, # host option map
                                                        undef, # only on hosts
                                                        undef, # do not fail
                                                        $isNotVerbose,
                                                        1); # isCollectHostInfo
        }
        else {
			my $cmd = " -main SDB::Install::App::Console::CollectHostInfo::main";

			$cmd .= " --dataPath=$dataPath"        if (defined $dataPath);
			$cmd .= " --logPath=$logPath"          if (defined $logPath);
			$cmd .= " --collect_only=$collectOnly" if (defined $collectOnly);
			$cmd .= " --skip_collect=$skipCollect" if (defined $skipCollect);
			$cmd .= " --sapadm=$wantedSapadm"      if (defined $wantedSapadm);
			$cmd .= " --sidadm=$wantedSidadm"      if (defined $wantedSidadm);
			$cmd .= " --sidcrypt=$wantedSidcrypt"  if (defined $wantedSidcrypt);
			$cmd .= " --userid=$wantedUID"         if (defined $wantedUID);
			$cmd .= " --groupid=$wantedGID"        if (defined $wantedGID);
			$cmd .= " --usernames=$wantedUsers"    if (defined $wantedUsers);
			$cmd .= " --groupnames=$wantedGroups"  if (defined $wantedGroups);
			$cmd .= " --sidcrypt_groupid=$wantedLSSGroupID"  if (defined $wantedLSSGroupID);
            $cmd .= " --compiler_version='$compilerVersion'" if (defined $compilerVersion);
            $cmd .= " --chkshared_datavolume_file='$chkSharedDataPathFile'" if (defined $chkSharedDataPathFile);
            $cmd .= " --chkshared_logvolume_file='$chkSharedLogPathFile'" if (defined $chkSharedLogPathFile);
            $cmd .= " --chkshared_datasubvolume_file='$chkSharedDataSubvolumePathFile'" if (defined $chkSharedDataSubvolumePathFile);
            $cmd .= " --chkshared_logsubvolume_file='$chkSharedLogSubvolumePathFile'" if (defined $chkSharedLogSubvolumePathFile);
            $cmd .= " --chkshared_lsspath_file='$chkSharedLssPathFile'" if (defined $chkSharedLssPathFile);

            $rc = $remoteHosts->executeParallel ($executable . $cmd,
                       $progress_message,
                       $done_message,
                       $error_message,
                       undef,          # password input stream
                       undef,          # template values
                       undef,          # only on hosts
                       undef,          # do not fail
                       $isNotVerbose,
                       1);             # isCollectHostInfo
        }
        if ($rc != 0){
            $self->PushError ("Cannot collect host infos", $remoteHosts);
            $remoteHosts->setMsgLstContext ($saveCntxt);
            $success = undef;

            # Authentication failure, Invalid Credentials
            if (($rc && 2) > 0) {
            	$self->setIsAuthenticationFailedOnCollectOtherHostInfos(1);
            }

            next;
        }
        my $outputs = $remoteHosts->getOutputBuffer ();
        my $host;
        my $parser;
        foreach my $i (0..(scalar (@$outputs) - 1)){
            $host = $remoteHosts->getHostNameByIndex ($i);
            $parser = new SDB::Install::XMLParser ();
            eval {
                # In some cases CollectHostInfo contains redundant text before xml and parsing fails
                $parser->parsestring ( substr( $outputs->[$i], index( $outputs->[$i], '<?xml version="1.0" encoding="UTF-8"?>' ) ) );
            };
            if ($@){
                my $parserErr  = $@;
                my $hostOutput = $outputs->[$i];
                my $errlst    = new SDB::Install::MsgLst();

                if (!($parserErr =~ s/^\r\n//)){
                    $parserErr =~ s/^\n//
                }

                if (!defined $hostOutput || (length($hostOutput) == 0)) {
                    $errlst->addError('CollectHostInfo does not return any information');
                }
                else {
                    if (length($hostOutput) > 10000) {
                        $hostOutput = substr($hostOutput, 0, 10000) . '...';
                    }
                    $errlst->addError ("Output of CollectHostInfo: $hostOutput");
                }
                $errlst->addError("Error message from XML parser: $parserErr");
                $self->PushError  ("Cannot parse output of CollectHostInfo of host '$host'", $errlst);
                $remoteHosts->setMsgLstContext ($saveCntxt);
                $success = undef;
                next;
            }
            $self->{host_data}->{$host} = $parser;
        }
        $remoteHosts->freeOutputBuffer ();
        $remoteHosts->setMsgLstContext ($saveCntxt);
    }

    $self->removeRemoteTmpInstaller ();
    foreach ($chkSharedDataPathFile, $chkSharedDataSubvolumePathFile, $chkSharedLogPathFile, $chkSharedLogSubvolumePathFile, $chkSharedLssPathFile){
        $self->removeTempCheckShared($_);
    }

    if (!$success) {
        return undef;
    }

    my $anyHostagentMissing    = 0;
    my $checkHostagentMissing  = 0;
    $self->{remoteUserIsAdmin} = {};

    if (!defined $self->{CertificatesHostmap}
        || (!$skipCollectAll
            && (!defined $skipCollectMap || !$skipCollectMap->{certificateHostname})
            && (!defined $collectOnlyMap ||  $collectOnlyMap->{certificateHostname}))) {
        $self->{CertificatesHostmap} = {};
    }

    if (!defined $self->{remoteGids}
        || (!$skipCollectAll
            && (!defined $skipCollectMap || !$skipCollectMap->{groups})
            && (!defined $collectOnlyMap ||  $collectOnlyMap->{groups}))) {
        $self->{remoteGids} = {};
    }

    if (!defined $self->{remoteIPs}
        || (!$skipCollectAll
            && (!defined $skipCollectMap || !$skipCollectMap->{networkAddresses})
            && (!defined $collectOnlyMap ||  $collectOnlyMap->{networkAddresses}))) {
        $self->{remoteIPs} = {};
    }

    if (!defined $self->{remoteSaphostagents} || defined $wantedSapadm) {
        $self->{remoteSaphostagents} = {};
        $checkHostagentMissing       = 1 if (defined $wantedSapadm);
    }

    if (!$skipCollectAll
        && (!defined $skipCollectMap || !$skipCollectMap->{sapsysid})
        && (!defined $collectOnlyMap ||  $collectOnlyMap->{sapsysid})) {

        $self->{sapsysids} = {};

        if (defined $self->{localSapsysGID}) {
            $self->{sapsysids}->{$self->{localSapsysGID}->[0]} =
                                               [ $self->{localSapsysGID}->[1] ];
        }
    }

    $self->{remoteInstances} = {} if (!defined $self->{remoteInstances});
    $self->{remoteSystems}   = {} if (!defined $self->{remoteSystems});

    if (!$skipCollectAll
        && (!defined $skipCollectMap || !$skipCollectMap->{SAPSystems})
        && (!defined $collectOnlyMap ||  $collectOnlyMap->{SAPSystems})) {

         $self->{remoteInstances} = {};
         $self->{remoteSystems}   = {};
    }

    $self->{remoteSidadms}  = {} if (!defined $self->{remoteSidadms});
    $self->{remoteUids}     = {} if (!defined $self->{remoteUids});

    if (!$skipCollectAll
        && (!defined $skipCollectMap || !$skipCollectMap->{users})
        && (!defined $collectOnlyMap ||  $collectOnlyMap->{users})) {
        $self->{remoteSidadms}  = {};
        $self->{remoteUids}     = {};
    }

    my ($data, $id, $gid, $group, $name, $user, $sapsysid, $system, $instance, $landscape,
        $currentUser, $reservedInstanceNrs, $instanceNr, $reservedInstanceNr, $i);

    my $hasSystemCheckErrors = 0;
    foreach my $host (keys (%{$self->{host_data}})){
        $data = $self->{host_data}->{$host};

        $currentUser = $data->getElementByTagName ('currentUser');
        if (defined $currentUser && defined $currentUser->{attr} &&
            defined $currentUser->{attr}->{isAdmin}){

            $self->{remoteUserIsAdmin}->{$host} = $currentUser->{attr}->{isAdmin};
        }

        $group = $data->getElementByTagName ('group');
        while ($group){
            $name = $group->{content};
            $gid  = $group->{attr}->{id};
            if ($wantedGroups){
                my $usersAndGroups = $self->{remoteUsersAndGroups};
                if (!defined $usersAndGroups){
                    $usersAndGroups = $self->{remoteUsersAndGroups} = {};
                }
                my $usersAndGroupsHost = $self->{remoteUsersAndGroups}->{$host};
                if (!defined $usersAndGroupsHost){
                    $usersAndGroupsHost =
                    $self->{remoteUsersAndGroups}->{$host} = {};
                }
                my $usersAndGroupsHostGroups = $usersAndGroupsHost->{groups};
                if (!defined $usersAndGroupsHostGroups){
                    $usersAndGroupsHostGroups = $usersAndGroupsHost->{groups} = {};
                }
                $usersAndGroupsHostGroups->{$name} = [$gid];
            }
            pushItem($self->{remoteGids}, $gid, [$host, $name]);
            $group = $group->{neighbor};
        }

        $user = $data->getElementByTagName ('user');
        while ($user){
            $name = $user->{content};
            $id   = $user->{attr}->{id};
            $gid  = $user->{attr}->{gid};
            pushItem($self->{remoteUids}, $id, [$host, $name, $gid]);
            # save all collected sidadm contigurations
            # to check it as soon as the SID is known
            if ($name =~ /^[a-z][a-z\d]{2}adm$/){
                pushItem($self->{remoteSidadms}, $name, [$host, $id, $gid]);
            }
            if ($wantedUsers){
                my $usersAndGroups = $self->{remoteUsersAndGroups};
                if (!defined $usersAndGroups){
                    $usersAndGroups = $self->{remoteUsersAndGroups} = {};
                }
                my $usersAndGroupsHost = $self->{remoteUsersAndGroups}->{$host};
                if (!defined $usersAndGroupsHost){
                    $usersAndGroupsHost =
                    $self->{remoteUsersAndGroups}->{$host} = {};
                }
                my $usersAndGroupsHostUsers = $usersAndGroupsHost->{users};
                if (!defined $usersAndGroupsHostUsers){
                    $usersAndGroupsHostUsers = $usersAndGroupsHost->{users} = {};
                }
                $usersAndGroupsHostUsers->{$name} = [$id, $gid];
            }
            $user = $user->{neighbor};
        }

        $sapsysid = $data->getElementByTagName ('sapsysid');
        if (defined $sapsysid && defined $sapsysid->{content}){
            pushItem($self->{sapsysids},  $sapsysid->{content}, $host);
            pushItem($self->{remoteGids}, $sapsysid->{content},
                                                    [$host, $gSapsysGroupName]);
        }

        $system = $data->getElementByTagName ('System');
        while ($system){
            my $currSID     = $system->{attr}->{sid};
            my $landscapeID = undef;
            pushItem($self->{remoteSystems}, $currSID, $host);
            $landscape = $data->getElementByTagName ('Landscape', $system);
            if (defined $landscape) {
                $landscapeID = $landscape->{attr}->{id};
            }
            $instance = $data->getElementByTagName ('Instance', $system);
            while ($instance){
                my $instanceInfo =
                    (defined $landscapeID)
                    ? [$host, $currSID, $instance->{attr}->{type}, $landscapeID]
                    : [$host, $currSID, $instance->{attr}->{type}];
                $instanceNr = $instance->{content};
                pushItem($self->{remoteInstances}, $instanceNr, $instanceInfo);
                $reservedInstanceNrs = $instance->{attr}->{reserved_instance_numbers};
                if (defined $reservedInstanceNrs && $reservedInstanceNrs > 0){
                    foreach $i (1..$reservedInstanceNrs){
                        $reservedInstanceNr = sprintf ('%02d', $instanceNr + $i);
                        pushItem($self->{remoteInstances}, $reservedInstanceNr,
                                                                 $instanceInfo);
                    }
                }
                $instance = $instance->{neighbor};
            }
            $system = $system->{neighbor};
        }

        my $ip = $data->getElementByTagName ('ip');
        while ($ip){
            pushItem($self->{remoteIPs}, $host, $ip->{content});
            $ip = $ip->{neighbor};
        }

        my $hostagentVers = $data->getElementByTagName ('saphostagentVersion');
        if (defined $hostagentVers && defined $hostagentVers->{content}){
            $self->{remoteSaphostagents}->{$host}->{version} = $hostagentVers->{content};
        }

        my $hostagentSslConfig = $data->getElementByTagName('saphostagentSslConfig');
        if (defined $hostagentSslConfig && defined $hostagentSslConfig->{content}) {
            $self->{remoteSaphostagents}->{$host}->{sslConfig} = $hostagentSslConfig->{content};
        }

        elsif ($checkHostagentMissing) {
            if ($isWin) {
                $anyHostagentMissing = 1;
            }
            else {
                my $sapadmList        = $self->{remoteSidadms}->{$wantedSapadm};
                my $hostWithoutSapadm = 1;
                if (defined $sapadmList) {
                    foreach my $arrayHostUidGid (@$sapadmList) {
                        if ($arrayHostUidGid->[0] eq $host) {
                            $hostWithoutSapadm = 0;
                            last;
                        }
                    }
                }
                if ($hostWithoutSapadm) {
                    $anyHostagentMissing = 1;
                }
            }
        }

        my $certificateHostname = $data->getElementByTagName ('certificateHostname');
        if (defined $certificateHostname && defined $certificateHostname->{content}){
        	$self->{CertificatesHostmap}->{$host} = $certificateHostname->{content};
        }

        $self->addPathExistanceHostEntry('dataPathExistenceHostmap', $host, 'dataPath');
        $self->addPathExistanceHostEntry('logPathExistenceHostmap', $host, 'logPath');

        my $systemCheckErrors = $data->getElementByTagName('systemCheckErrors');
        if (defined $systemCheckErrors && defined $systemCheckErrors->{content}) {
            my $errorMessages = $systemCheckErrors->{content};
            if ($self->getIgnore('check_platform')){
                my $infoMsg = $self->getMsgLst->addWarning("Ignoring remote host '$host' system check errors due to command line switch '--ignore=check_platform'");
                foreach (split(/\n/, $errorMessages)) {
                    $infoMsg->getSubMsgLst->addWarning($_);
                }
            } else {
                my $errMsg = $self->getErrMsgLst->addError("Encountered erros during system requirements check on host '$host':");
                foreach (split(/\n/, $errorMessages)) {
                    $errMsg->getSubMsgLst->addError($_);
                }
                $hasSystemCheckErrors = 1;
            }
        }

        my $lssPathElement = $data->getElementByTagName ('lssPath');
        if (defined $lssPathElement && defined $lssPathElement->{attr}){
            $self->{'lssPathExistenceHostmap'}->{$host} = $lssPathElement->{attr}->{isShared};
        }

    }

    if ($hasSystemCheckErrors) {
        return undef;
    }

    delete $self->{host_data};

    if (!$self->_validateRemoteSapsysIDs()) {
        return undef;
    }

    if (!$self->checkRemoteRootUserPrivileges){
        return undef;
    }

    if ($anyHostagentMissing && $self->getValue('InstallHostagent')) {
        if (exists $self->{params}->{HostagentUserName}) {
            $self->setSkip('HostagentUserName', 0);
        }
        elsif (sapadmUserExists()) {
            $self->{params}->{HostagentPassword}->{type} = 'passwd';
        }
        $self->setSkip('HostagentPassword', 0);
    }

    return 1;
}

sub createTempCheckShared {
    my ($self, $parentDir) = @_;
    my $filename = sha256_hex(hostname().time());
    my $tempFile = File::Spec->catfile($parentDir, $filename);
    if( ! IO::File->new(">$tempFile") ){
        $self->getMsgLst()->addWarning("Cannot create temporary file '$tempFile': $!");
        $tempFile = undef;
    }
    return $tempFile;
}

sub checkLocalHost {
    my ($self, $sid, $sapSys) = @_;
    my $instance = $sapSys->getNewDBInstances()->[0];

    if (!defined $instance || $instance->hasLocalHost()) {
        return 1;
    }
    my $errorMsgLst = new SDB::Install::MsgLst();
    my $detectedHosts = $instance->get_hosts();
    if (defined $detectedHosts) {
        my $rc = $self->_addSystemHostsInfo($detectedHosts, $errorMsgLst);
        $rc &= $self->_addLocalInterfacesInfo($errorMsgLst);
        $self->appendErrorMessage("The current host is not part of the $gProductNameSystem '$sid'.", $errorMsgLst);
        $self->appendErrorMessage("Neither of the $gProductNameSystem hosts' IP addresses belongs to a local network interface.") if ($rc);
    }
    return 0;
}

sub _addSystemHostsInfo {
    my ($self, $hosts, $msgLst) = @_;
    $msgLst = (defined $msgLst) ? $msgLst : $self->getMsgLst();
    my $infoCollected = 0;
    my $msg = $msgLst->addMessage("Collecting IP addresses of the HANA hosts");
    my $nslookup;
    foreach my $host (@$hosts) {
        $nslookup = nslookup($host, $msgLst);
        if (defined $nslookup) {
            $msg->getSubMsgLst()->addMessage("Host name '" . $host . "' resolves to address: " . $nslookup->{address});
            $infoCollected = 1;
        }
    }
    return $infoCollected;
}

sub _addLocalInterfacesInfo {
    my ($self, $msgLst) = @_;
    $msgLst = (defined $msgLst) ? $msgLst : $self->getMsgLst();
    my $msg = $msgLst->addMessage("Collecting IP addresses of local network interfaces");
    my $interfaces = SDB::Install::SysInfo::interfaceInfo ($msgLst);
    if (!defined $interfaces) {
        return 0;
    }
    my $ipaddr;
    foreach my $ipaddrlist (values %$interfaces){
        foreach $ipaddr (@$ipaddrlist){
            $msg->getSubMsgLst()->addMessage($ipaddr->{type}.' '.$ipaddr->{addr}) if (!is_loopback_address($ipaddr->{addr}));
        }
    }
    return 1;
}

#-------------------------------------------------------------------------------
# Adds a hash key with an array item:
#
#   new hash entry with the specified item as first array element:
#      $hash->{$key}->[0]->$item
#
#   appends the item to an existing array:
#      $hash->{$key}->[0]-><existingArrayItem0>
#                     [1]-><existingArrayItem1>
#                     [2]->$arrayItem

sub pushItem {
    my ($hash, $key, $arrayItem) = @_;

    if (defined $hash->{$key}) {
        push @{$hash->{$key}}, $arrayItem;
    }
    else {
        $hash->{$key} = [$arrayItem];
    }
}


#-------------------------------------------------------------------------------
# Returns the first free group ID of a single-host or multiple-host system
# Undef is returned if the password database should not be scanned
# or a free ID is not found.

sub getFreeGroupID {
    my ($self, $wantedGID, $reservedGIDs) = @_;

    return undef if ($self->getIgnore('scan_password_database'));

    if (defined $wantedGID && $self->isGroupIDFree($wantedGID)) {
        return $wantedGID;
    }
    for my $gid ($gSapsysGroupIdDefault .. $gMaxSapsysGroupIDDefault) {
        return $gid if ($self->isGroupIDFree($gid, $reservedGIDs));
    }
    return undef;
}

#-------------------------------------------------------------------------------
# Checks that a GID is free across all system hosts

sub isGroupIDFree {
    my ($self, $gid, $reservedGIDs) = @_;
    $reservedGIDs = $reservedGIDs || [];
    my $isReserved = grep { $_ == $gid } @{$reservedGIDs};
    my $remoteGids = $self->{remoteGids};
    if ((!defined $remoteGids || !defined $remoteGids->{$gid}) && !SDB::Common::BuiltIn->get_instance()->getgrgid ($gid) && !$isReserved) {
        return 1;
    }
    return undef;
}


sub getFreeInstanceNumber{
    my ($self,$isTopDown, $perferredNumber) =  @_;
    if (!defined $isTopDown){
        $isTopDown = 0;
    }
    my $saveContext = $self->getMsgLstContext ();
    my $lable = 'Instance Number';
    $self->resetMsgLstContext ();
    if (defined $perferredNumber){
        if (!$self->checkInstanceNumber ($perferredNumber, $lable)){
            $saveContext->[0]->addWarning ("Preferred $lable '$perferredNumber' is invalid.");
            $perferredNumber = undef;
        }
        elsif ($self->checkAdditionalInstanceNumber ($perferredNumber, $lable)){
            $self->setMsgLstContext ($saveContext);
            return sprintf ("%02d", $perferredNumber);
        }
        else{
            $saveContext->[0]->addMessage ("Preferred $lable '$perferredNumber' is already in use.");
        }
    }

    my @numberList = (0..$MAX_INSTANCE_NUMBER);
    if ($isTopDown){
        @numberList = reverse @numberList;
    }
    foreach my $nr (@numberList){
        if ($self->checkAdditionalInstanceNumber ($nr)){
            $self->setMsgLstContext ($saveContext);
            return sprintf ("%02d", $nr);
        }
    }
    $self->setMsgLstContext ($saveContext);
    return undef;
}

sub getFreeInstanceNumbers{
    my ($self) =  @_;
    my $saveContext = $self->getMsgLstContext ();
    my $lable = 'Instance Number';
    $self->resetMsgLstContext ();
    my @freeInstanceNumbers = ();
    foreach my $nr (0..$MAX_INSTANCE_NUMBER){
        if ($self->checkAdditionalInstanceNumber ($nr)){
            push @freeInstanceNumbers, sprintf ('%02d', $nr);
        }
    }
    $self->setMsgLstContext ($saveContext);
    return \@freeInstanceNumbers;
}

sub getTargetLssSid {
    my ($self) = @_;
    return $self->getValue('newSID') // $self->getValue('SID');
}

#-------------------------------------------------------------------------------
# Returns the first free user ID of a single-host or multiple-host system
# Undef is returned if a local sidadm exists or the password database
# should not be scanned or a free ID is not found.

sub getFreeUserID {
    my ($self, $preferredUID, $reservedUIDs) = @_;

    $reservedUIDs = $reservedUIDs || [];

    return undef if(defined($self->{existingSidAdm}));
    return undef if($self->getIgnore('scan_password_database'));
    return $self->getFreeOsUserID($preferredUID, $reservedUIDs);
}

sub getFreeOsUserID {
    my ($self, $preferredUID, $reservedUIDs) = @_;
    $reservedUIDs = $reservedUIDs || [];
    return $preferredUID if ((defined $preferredUID) && ($self->_isFreeUserId($preferredUID, $reservedUIDs)));
    for my $uid (1000 .. 29999) {
        return $uid if ($self->_isFreeUserId($uid, $reservedUIDs));
    }
    return undef;
}

sub _isFreeUserId {
    my ($self, $uid, $reservedUIDs) = @_;
    my $remoteUids = $self->{remoteUids} || {};
    my $isReserved = grep { $_ == $uid } @{$reservedUIDs};
    return 0 if($isReserved);
    return 0 if(defined($remoteUids->{$uid}));
    return 0 if(getpwuid($uid));
    return 1;
}

sub _validateRemoteSapsysIDs {
    my ($self) = @_;
    my $sapsysIDs = [ keys(%{$self->{sapsysids}}) ];
    my $sapsysIDsCount = scalar(@{$sapsysIDs});
    return 1 if !$sapsysIDsCount; # 0 remote sapsys groups;

    if ($sapsysIDsCount > 1) {
        my $sublst = new SDB::Install::MsgLst();
        foreach my $gid (@{$sapsysIDs}) {
            my $hosts = join ',', @{$self->{sapsysids}->{$gid}};
            $sublst->AddMessage ("Group '$gSapsysGroupName' has id $gid on the following hosts: $hosts.");
        }
        $self->appendErrorMessage ("There are several ids for group '$gSapsysGroupName', group id has to be the same on all hosts", $sublst);
        return undef;
    }

    my $sapsysGID = $sapsysIDs->[0];
    my $msgLst = SDB::Install::MsgLst->new();
    if (!$self->_checkRemoteNonSapsysGroups($sapsysGID, $msgLst)) {
        my $hosts = $self->{sapsysids}->{$sapsysGID};
        my $hostsString = join ',', @{$hosts};
        my $error = "Group '$gSapsysGroupName' exists on host(s) $hostsString with id $sapsysGID, which is taken by another group on some of the other hosts:";
        $self->appendErrorMessage($error, $msgLst);
        return undef;
    }

    return 1;
}

sub checkRemoteRootUserPrivileges{
    my ($self) = @_;
    my $userName = $self->getBatchValue ('RootUser');
    my $remoteUserIsAdmin = $self->{remoteUserIsAdmin};
    if (!defined $remoteUserIsAdmin){
        # skip due to missing information
        return 1;
    }
    my $hostUserIsAdmin;
    my @failed_hosts;
    foreach my $host (sort keys (%$remoteUserIsAdmin)){
        $hostUserIsAdmin = $remoteUserIsAdmin->{$host};
        if (!defined $hostUserIsAdmin){
            # skip host due to missing information
            next;
        }
        if (!$hostUserIsAdmin){
            push @failed_hosts, $host;
        }
    }
    if (@failed_hosts){
        $self->setErrorMessage ("User '$userName' has no admin privileges on host(s): " .
            join (', ', @failed_hosts));
        return 0;
    }
    return 1;
}


sub createSQLConnection {
    my ($self, $user, $password, $dbName, $service) = @_;
    my $instance = $self->getOwnInstance();
    $dbName //= $instance->getSQLDatabaseName(defined $service && $service eq 'IndexServer');
    my $connectionProvider = new SDB::Install::Sql::SqlConnectionProvider();
    my $sql = $connectionProvider->getSqlConnection(
                $dbName,
                $instance,
                $user,
                $password);
    return (defined $sql, $connectionProvider->getSqlObj());
}

sub checkDatabaseUserPassword {
    my (
        $self,
        $user,
        $password,
        $paramId,
        $outSqlConnection,
        $failIfNoConnection,
        $dbName,
        $service,
    ) = @_;
    $self->clearParameterWarnings($paramId);

    if ($password eq ''){
        $self->AddError ("Database user password is empty");
        return 0;
    }

    $self->{params}->{$paramId}->{connected} = 0;

    my ($rc, $sql) = $self->createSQLConnection($user, $password, $dbName, $service);
    my $msglst = new SDB::Install::MsgLst();
    my $license_check_rc = undef;
    my $instance = $self->getOwnInstance();
    if(!$rc) {
        # connect failed
        if ($sql->getConnErrorCode() == 19) {
            $license_check_rc = 0;
            $self->AddError("The system has no valid license.");
        } elsif ($sql->isInvalidCredentials()) {
            $self->AddError("Invalid database username or password.");
            return 0;
        } elsif($sql->isECONNREFUSED ()) {
            my $specificWarning = $sql->GetErrorString();
            chomp($specificWarning) if ($specificWarning);
            my $host = $instance->getSqlHost();
            if($failIfNoConnection){
                my $message = $self->getErrMsgLst()->addError("Database is not running on host '$host'");
                if($specificWarning){
                    $message->getSubMsgLst()->addError($specificWarning);
                }
                return undef;
            }

            my $msg = $self->AddWarning("Database is not running on host '$host'.");
            $msg->getSubMsgLst()->AddWarning($specificWarning) if ($specificWarning);
            my $systemWarning = "Cannot verify database user ($user) password in advance: no connection available.";
            my $licenseWarning = "Cannot perform license check: no connection available";
            $self->AddWarning($systemWarning);
            $self->AddWarning($licenseWarning);
            $self->addParameterWarning($paramId, $systemWarning);
            $self->addParameterWarning($paramId, $licenseWarning);
            return 1;
        }
        else{
            $self->setErrorMessage ('Cannot establish database connection', $sql->getErrMsgLst());
            $self->{params}->{$paramId}->{no_retry} = 1;
            return 0;
        }
    } else {
        # connect successful
        $self->AddMessage("$user user password verified.");
        $self->{params}->{$paramId}->{connected} = 1;
        # check whether the system is locked down
        if (!defined $license_check_rc) {
            $license_check_rc = $instance->checkLicense ($sql,$msglst);
        }
        if ($license_check_rc){
            $self->AddMessage ("License check successfully performed.");
        }
        else{
            $self->AddError (undef, $msglst);
        }
    }
    if (!$license_check_rc) {
        if ($self->getIgnore ('check_license')){
            $self->AddMessage ("Ignoring error due to command line switch '--ignore=check_license'");
            $license_check_rc = 1;
        }
    }
    if (!$license_check_rc){
        $self->{params}->{$paramId}->{no_retry} = 1;
    }
    if ($license_check_rc && defined $outSqlConnection){
        $$outSqlConnection = $sql;
    }
    return $license_check_rc;

}

sub checkSQLSysPasswd {
    my (
        $self,
        $value,
        $msglst_arg,
        $paramId,
        $outSqlConnection,
        $failIfNoConnection,
        $systemUser,
        $service
    ) = @_;
    my $sqlSysPasswdParamId = $paramId // 'SQLSysPasswd';


    my $user = defined $systemUser ? $systemUser : $self->{params}->{SystemUser}->{value};
    if (!$user) {
        $user = $self->getDefault('SystemUser');
    }
    if (!$user) {
        $user = 'SYSTEM';
    }

    return $self->checkDatabaseUserPassword($user, $value, $sqlSysPasswdParamId, $outSqlConnection, $failIfNoConnection, undef, $service);
}


sub complyDefaultSqlPasswordPolicy{
    my ($self, $sqlPassword, $msglst, $keyName) = @_;
    my $rc = 1;
    if (!$sqlPassword || length ($sqlPassword) < 8){
        $msglst->AddMessage ("$keyName must contain at least 8 characters");
        $rc = 0;
    }

    if ($sqlPassword !~ /\d/){
        $msglst->AddMessage ("$keyName must contain at least one digit");
        $rc = 0;
    }

    if ($sqlPassword !~ /[A-Z]/){
        $msglst->AddMessage ("$keyName must contain at least one upper-case character");
        $rc = 0;
    }

    if ($sqlPassword !~ /[a-z]/){
        $msglst->AddMessage ("$keyName must contain at least one lower-case character");
        $rc = 0;
    }
    return $rc;
}

sub complySqlPasswordPolicy {
    my ($self, $sqlPassword, $isSystemDb, $msglst, $paramId) = @_;

    my $keyName = $self->{params}->{$paramId}->{str};
    my $rcOfflineCheck = $self->complySqlPasswordPolicyOffline($sqlPassword, $isSystemDb, $msglst, $keyName);
    if (!$rcOfflineCheck) {
        return undef;
    }

    my $user = $self->{params}->{SystemUser}->{value};
    if (!$user) {
        $user = $self->getDefault('SystemUser');
    }
    $user //= 'SYSTEM';
    my $password = $self->getValue('SQLSysPasswd') // $self->getValue('SrcSQLSysPasswd');
    my ($rc, $sql) = $self->createSQLConnection($user, $password, undef, $isSystemDb ? 'SystemServer' : '');

    if(!$rc) {
        my $warning = "$keyName cannot be completely checked for compliance with password policy because connection to database is not avalaible.";
        $self->AddWarning ($warning);
        $self->addParameterWarning($paramId, $warning);
        return 1;
    }

    return $self->complySqlPasswordPolicyOnline($sql, $sqlPassword, $msglst);
}

sub complySqlPasswordPolicyOnline{
    my ($self, $sql, $sqlPassword, $msglst) = @_;

    my $stmnt = "CALL SYS.IS_VALID_PASSWORD(?,?,?)";
    if (!defined $sql->prepare ($stmnt)){
        $msglst->AddMessage ("Prepare '$stmnt' failed", $sql->getErrMsgLst ());
        return undef;
    }

    $sql->bindParam (1, $sqlPassword);

    my $errorCode;
    if (!defined $sql->bindParam (2, \$errorCode, 5)) {
        $msglst->AddMessage ('Cannot bind param error_code', $sql->getErrMsgLst());
        return undef;
    };

    my $errorMessage;
    if (!defined $sql->bindParam (3, \$errorMessage, 128)) {
        $msglst->AddMessage ('Cannot bind param error_message', $sql->getErrMsgLst());
        return undef;
    };

    if (!defined $sql->execute()){
        $msglst->AddMessage ('Cannot execute IS_VALID_PASSWORD procedure', $sql->getErrMsgLst());
        return undef;
    }
    if ($errorCode != 0) {
        $msglst->AddMessage ($errorMessage);
        return undef;
    }
    return 1;
}

sub complySqlPasswordPolicyOffline{
    my ($self, $sqlPassword, $inSystemDb, $msglst) = @_;

    my ($pattern, $minLength) = $self->_getPasswordPolicyData($inSystemDb, $msglst);
    my $passwordPolicy = new SDB::Install::PasswordPolicy ($pattern, $minLength);
    if (!$passwordPolicy->isCompliant($sqlPassword)) {
        $msglst->AddError($passwordPolicy->getMessage());
        return 0;
    }

    return 1;
}

sub _getPasswordPolicyData {
    my ($self, $inSystemDb, $msglst) = @_;

    my $pattern;
    my $minLength;

    my $instance = $self->getOwnInstance();
    if ($inSystemDb) {
        my $iniFile;
        my $layered_cfg = $instance->getLayeredConfig();
        if ($instance->isMultiDb()) {
            $iniFile = $layered_cfg->getIniFile ('nameserver.ini');
            if (!defined $iniFile){
                $msglst->AddError ('Cannot get nameserver.ini', $layered_cfg);
                return undef;
            }
        } else {
            $iniFile = $layered_cfg->getIniFile ('indexserver.ini');
            if (!defined $iniFile){
                $msglst->AddError ('Cannot get indexserver.ini', $layered_cfg);
                return undef;
            }
        }
        if (!defined $iniFile->readValues ()){
            $msglst->addError("Cannot read ini file", $iniFile->getErrMsgLst());
            return undef;
        }
        $pattern = $iniFile->getValue('password policy', 'password_layout');
        $minLength = $iniFile->getValue('password policy', 'minimal_password_length');
    } else {
        my $indexserver_ini = File::Spec->catfile($instance->get_globalHdbCustomConfigDir(), "DB_".$instance->get_sid(), 'indexserver.ini');
        my $data = readini ($indexserver_ini, $msglst);
        if (!defined $data){
            $msglst->AddError ('Cannot read default tenant indexserver.ini');
            return undef;
        }
        $pattern = $data->{'password policy'}->{'password_layout'};
        $minLength = $data->{'password policy'}->{'minimal_password_length'};
    }

    return ($pattern, $minLength);
}


#-------------------------------------------------------------------------------
# Checks the parameter '--system_user' and sets the username to SQLSysPasswd

sub checkSystemUser {
    my ($self, $username) = @_;

    if (length($username) == 0) {
        $self->PushError ("System User value is empty");
        return 0;
    }
    my $isCreatingNewSystemUser = 0;
    if (exists($self->{params}->{SrcSQLSysPasswd})) {
        $isCreatingNewSystemUser = 1;
    }
    $self->setUsernameToSQLSysPasswd($username, $isCreatingNewSystemUser);

    return 1;
}


#-------------------------------------------------------------------------------
# Sets the username to SQLSysPasswd

sub setUsernameToSQLSysPasswd {
    my ($self, $username, $isCreatingNewSystemUser) = @_;

    if ($isCreatingNewSystemUser) {
        $self->setUsernameToSQLUserPassword('SrcSQLSysPasswd',$username);
        $self->setUsernameToSQLUserPassword('TrgSQLSysPasswd',$username);
    } else {
        $self->setUsernameToSQLUserPassword('SQLSysPasswd',$username);
    }
}


#-------------------------------------------------------------------------------
# Returns file name.
# File contains the last step of an aborted installation/update.

sub getPersFilenameInstUpd {

    my ($self,
        $nameExtension, # e.g. 'hdbinstall'
        $isInstallation
       ) = @_;

    my $sid = $self->{current_sid};
    if (!defined $sid){
        $self->AddError ('SID is unknown');
        return undef;
    }

    my $persName  = "$sid.$nameExtension";
    my $filename = undef;

    if ($isWin){
        my $msglst = new SDB::Install::MsgLst ();
        my $appData = getAllUsersAppDataPath ($msglst);
        if (!$appData){
            $self->AddError ('Cannot get Common AppData path', $msglst);
            return undef;
        }
        my $path = join($path_separator, $appData, '.hdb');
        if (!-e $path){
            my $cfg = {'mode' => 0775};
            if (!defined makedir ($path, $cfg)){
                $self->AddError("Cannot create directory $path", $cfg);
                return undef;
            }
        }
        $filename = join($path_separator, $path, $persName);
    }
    else{
        my $instPath   = undef;
        if (defined $self->{sapSys}
            && ($self->{sapSys}->hasNewDB() || $isInstallation)) {
            $instPath = $self->{sapSys}->get_globalSidDir();
        }
        if (!defined $instPath) {
            $instPath = $self->getValue('Target');

            if (!defined $instPath){
                return undef;
            }
            $filename =
               join($path_separator, $instPath, $sid, $persName);
        }
        else{
            $filename =
               join($path_separator, $instPath, $persName);
        }
    }
    return $filename;
}


#-------------------------------------------------------------------------------
# returns old status file name (/usr/sap/<SID>.<nameExtension>)
# if this file doesn't exist, then undef is returned
#

sub getOldLocalPersFilenameInstUpd {
    my ($self,
        $nameExtension # e.g. 'hdbinstall'
       ) = @_;

    if ($isWin){
        return undef;
    }

    my $sid = $self->{current_sid};
    if (!defined $sid){
        $self->AddError ('SID is unknown');
        return undef;
    }

    my $persName  = "$sid.$nameExtension";
    my $filename = undef;
    my $statusFileName = "/usr/sap/$sid.$nameExtension";

    if (!-f $statusFileName){
        return undef;
    }

    return $statusFileName;
}


#-------------------------------------------------------------------------------
# returns old status file name (compatiblity for versions < rev.74)
# is overridden in NewServerConfig and Updrade class
#

sub getOldLocalPersFilename{
    return undef;
}

#-------------------------------------------------------------------------------
# check if there is an old local status file for server
# installation or update
# returns $pers hash referece, if it's valid
#
sub validateOldLocalPersistencyFile{

    my ($self) = @_;

    my $statusFileName = $self->getOldLocalPersFilename ();

    if (!defined $statusFileName){
        return undef;
    }

    my $pers = $self->pers_load ($statusFileName);

    my $msglst = $self->getMsgLst ();

    my $kitversion = $pers->{_kitversion};
    my $persVersion = new SDB::Install::Version ((split ('\.', $pers->{_kitversion}))[0,1,2]);
    if ($persVersion->isNewerThan (new SDB::Install::Version (1, 0, 73))){
        $msglst->addMessage ("Ignoring local pers file, because its version > 1.00.73 ($pers->{_kitversion})");
        return undef;
    }

    my $sapSys = $self->{sapSys};
    if (!defined $sapSys && defined $self->{current_sid}){
        my $systems = $self->getCollectSAPSystems();
        if (defined $systems){
            $sapSys = $systems->{$self->{current_sid}};
        }
    }

    if (!defined $sapSys){
        return $pers;
    }

    my $manifest = $sapSys->getManifest ();

    if (!defined $manifest){
        return $pers;
    }

    my $currentVersion = $manifest->getVersionObject ();
    $persVersion = new SDB::Install::Version (split ('\.', $pers->{_kitversion}));
    if ($currentVersion->isNewerThan ($persVersion)){
        $msglst->addMessage (sprintf (
            "Ignoring local pers file (version = %s), because the current version (%s) is already newer",
            $pers->{_kitversion}, $manifest->getVersion ()));
        return undef;
    }
    $self->{_pers_date} = (stat ($statusFileName)) [9];
    return $pers;
}



#-------------------------------------------------------------------------------
# Returns the specified command appended by ignore and timeout options.

sub getIgnoreAndTimeoutOptions {

    my ($self,
        $optIgnore,  # SDB::Install::Configuration::OptionIgnore
        $optTimeout, # SDB::Install::Configuration::OptionTimeout
       ) = @_;

    my $result = '';
    my $handlerIgnore = $self->getOptionIgnore();

    if (defined $optIgnore && defined $handlerIgnore) {
        my $ignoreArgs = $handlerIgnore->getSetKeysValidFor($optIgnore);
        my $ignoreStr = $handlerIgnore->getOptionStringFromArgList($ignoreArgs);

        if (defined $ignoreStr) {
            $result .= ' ' . $ignoreStr;
        }
    }

    if ($self->{options}->{ignore_unknown_option}) {
        $result .= ' --ignore_unknown_option';
    }

    my $currentOptionTimeout = $self->getOptionTimeout();

    if (defined $optTimeout && defined $currentOptionTimeout) {
        my $timeoutArgs = $currentOptionTimeout->getSetKeysValidFor($optTimeout);
        my $timeoutStr = $currentOptionTimeout->getOptionStringFromArgList($timeoutArgs);

        if (defined $timeoutStr) {
            $result .= ' ' . $timeoutStr;
        }
    }

    return $result;
}

#Override
sub getSystemHosts{
    my $self = shift;
    my $instance = $self->getOwnInstance();
    if (!defined $instance){
        return undef;
    }
    return $instance->get_allhosts();
}

sub isDistributedSystem {
    my $systemHosts = $_[0]->getSystemHosts();
    if (!defined $systemHosts){
        return undef;
    }
    return (scalar @$systemHosts) > 1;
}

#----------------------------------------------------------------------------
#
# Gets remote hosts with available connection or returns undef.
# Will reset the remote hosts list of "this".
#
sub getRefreshedRemoteHosts {
    my ( $self, $doNotundefCurrentHosts, $isNotVerbose ) = @_;

    unless($doNotundefCurrentHosts) {
        $self->{_remoteHosts} = undef;
    }
    $self->getMsgLst ()->addMessage ( 'Refreshing remote hosts connections...' );
    return undef unless $self->initRemoteHosts();    # refresh remote hosts list

    if($self->{_remoteHosts}->isHostctrl()){
        if ($self->isCollectOtherHostInfosRequired()) {
                return undef unless $self->CollectOtherHostInfos ($isNotVerbose);
        }
        return $self->{_remoteHosts};
    }

    my $rootUser = $self->getValue ( 'RootUser' ) ? $self->getValue ( 'RootUser' ) : 'root';
    $self->{_remoteHosts}->resetMsgLstContext ();

    $self->tryRemoteKeyAuthorization ( $rootUser, $isNotVerbose );

    if (! $self->{_remoteHosts}->areHostsAuthenticated() ) {
        my $rootPasswd = $self->getValue ( 'RootPassword' );
        $self->{_remoteHosts}->setUserPass        ( $rootPasswd );
        $self->{_remoteHosts}->ResetError         (); # reset key authorization error
        $self->{_remoteHosts}->resetMsgLstContext ();
        return undef if $self->{_remoteHosts}->authPassword
                                             ($self->getValue('InstallSSHKey'));
        if ($self->isCollectOtherHostInfosRequired()) {
            return undef unless $self->CollectOtherHostInfos ($isNotVerbose);
        }
    }
    return $self->{_remoteHosts};
}

#----------------------------------------------------------------------------
# Will ask for root credentials if needed.
#
# A comment at the beginning of AnyConfig shows the order of depending
# parameters.

sub ensureRootCredentialsIfNeeded {
    my $self = shift();

    return 0 unless $self->isDistributedSystem ();
    $self->AddMessage ( "Checking if root credentials are needed..." );
    if (!$self->initRemoteHosts()) { # populate remote hosts table
        return undef;
    }
    my $rootUser = $self->getValue('RootUser');

    if (!defined $rootUser) {
        $self->{params}->{RootUser}->{skip}            = 0;
        $self->{params}->{RootUser}->{set_interactive} = 1;
        return 1;
    }

    $self->tryRemoteKeyAuthorization($rootUser);    # check if root credentials will be needed

    if ( defined $self->{params}->{RootPassword}->{skip}
        && $self->{params}->{RootPassword}->{skip} == 0 ) {
        $self->AddMessage ( "root credentials are needed" );

        # if the check with root as user name failed, enable root user
        if(defined $self->{params}->{RootUser})
        {
          $self->{params}->{RootUser}->{skip}            = 0;
          $self->{params}->{RootUser}->{set_interactive} = 1;
        }
    }
    return 1;
}

sub canUpgradeComponent{
    my ($self, $srcManifest, $targetManifest, $componentName, $resultMsgLst) = @_;
    my $srcVersion = $srcManifest->getVersionObject ();
    my $targetVersion = $targetManifest->getVersionObject ();
    my $componentString = ($componentName ? ' for ' . $componentName : '');
    my $msg = $self->getMsgLst ()->addMessage ("Performing upgrade version check$componentString...");
    my $saveContext = $self->setMsgLstContext ([$msg->getSubMsgLst ()]);
    $self->getMsgLst ()->addMessage ("Comparing versions");
    my $cloudEditionUpdateCheck = new SDB::Install::CloudEditionUpdateCheck($srcManifest, $targetManifest);
    if($cloudEditionUpdateCheck->isCloudEditionVersionCheckRequired()){
        $cloudEditionUpdateCheck->setMsgLstContext($self->getMsgLstContext());
        if (!$cloudEditionUpdateCheck->canUpdate()){
            if (defined $resultMsgLst) {
                $resultMsgLst->addMessage($cloudEditionUpdateCheck->getErrorString());
            }
            $self->setMsgLstContext ($saveContext);
            return 0;
        }
    }
    elsif ($srcVersion->isNewerThan ($targetVersion)){
        my $msg = $self->getMsgLst ()->addMessage ("Newer version (".$srcManifest->getVersion ().") is already installed.");
        $self->getErrMsgLst()->initMsgLst ();
        my $componentString = ($componentName ? ' of ' . $componentName : '');
        my $errMsg = "Newer version$componentString (".$srcManifest->getVersion ().") is already installed.";
        if (defined $resultMsgLst) {
            $resultMsgLst->addMessage($errMsg);
        }
        $self->getErrMsgLst()->addError($errMsg);
        $self->setMsgLstContext ($saveContext);
        return 0;
    }
    $msg = $self->getMsgLst()->addMessage ("Checking further upgrade limitations");
    $targetManifest->setMsgLstContext ([$msg->getSubMsgLst]);
    if (!$targetManifest->checkSPSUpgradeLimitation ($srcManifest)){
        my $errMsgLst = $targetManifest->getErrMsgLst();
        $self->setErrorMessage (undef, $errMsgLst);
        if (defined $resultMsgLst) {
            $resultMsgLst->appendMsgLst($errMsgLst);
        }
        $self->setMsgLstContext ($saveContext);
        return 0;
    }
    $self->getMsgLst ()->addMessage ('Upgrade from ' . $srcManifest->getVersion () .
        ' to ' . $targetManifest->getVersion (). ' is allowed.');
    $self->setMsgLstContext ($saveContext);
    return 1;
}

sub getRemoteHostsObjectsForCollectHostInfo{
    return $_[0]->getRemoteHostsObjects ();
}

sub getRemoteHostsObjects{
    if (!defined $_[0]->{_remoteHosts}){
        return [];
    }
    return [$_[0]->{_remoteHosts}];
}


#-------------------------------------------------------------------------------
# Checks the password of a system administrator user

sub verifySidAdmPassword {

    my ($self,
        $user,     # SBD::Install:User
        $password,
        $str       # e.g. $self->{params}->{Password}->{str}
       ) = @_;

    my $rc = (defined $user) ? $user->verifyPassword ($password)
                             : undef;

    if (!defined $rc) {
        $self->AddWarning ("Cannot check password of $str:", $user);
    }
    elsif (!$rc) {
        $self->PushError ("$str: Unknown user password combination");
        return 0;
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Returns SDB::Install::Configuration::AddHostsParser
# providing the host attributes given by '--addhosts'

sub getAddHostsParser{
    return $_[0]->{_addHostsParser};
}


#-------------------------------------------------------------------------------
# Returns SDB::Install::RemoteHosts
# containing the additional hosts given by '--addhosts'

sub getAdditionalRemoteHosts{
    return $_[0]->{_additionalRemoteHosts};
}


sub getValidJavaVersions{
    return [qw (1.6 1.7 1.8)];
}

sub getIs64BitJavaRequired{
    return $installerIs64bit;
}


sub _getVersionRegexp{
    my ($javaVersions) = @_;
    # create regexp
    my $rJavaVersions = '^';
    foreach my $v (@$javaVersions) {
        if (!($rJavaVersions eq '^')) {
            $rJavaVersions .= '|^';
        }
        $rJavaVersions .= quotemeta($v) . '\.';
    }
    return $rJavaVersions;
}


sub findJava{
    my ($self) = @_;

    require SDB::Install::Java;
    my $javaVersions = $self->getValidJavaVersions ();
    if (!defined $javaVersions){
        return undef;
    }
    my $sJavaVersions = join "/", @$javaVersions;
    my $is64bit = $self->getIs64BitJavaRequired ();
    my $msg = $self->getMsgLst ()->addMessage ("Looking for Java installations");
    my $java = new SDB::Install::Java;
    $java->setMsgLstContext ([$msg->getSubMsgLst]);
    my $javas = $java->GetJavas();

    if (!defined $java || !@$javas){
        $self->setErrorMessage ("Java not found");
        $self->{java_not_found_error} = "No Java installation found (version $sJavaVersions, " . ($is64bit ? '64bit' : '32bit') . ' is required.)';
        $self->setErrorMessage ($self->{java_not_found_error}, $java);
        return undef;
    }

    my $rJavaVersions = _getVersionRegexp ($javaVersions);

    my $found = 0;
    my $foundJava;
    foreach my $i (0.. (scalar (@$javas) - 1)){
        if (!($javas->[$i]->[3] xor $is64bit) && ($javas->[$i]->[2] =~ /$rJavaVersions/)){
            $found = 1;
            $foundJava = $javas->[$i]->[0];
            last;
        }
    }

    if (!$found){
        $self->{java_not_found_error} = "No suitable Java installation found (version $sJavaVersions, " . ($installerIs64bit ? '64bit' : '32bit') . ' is required)';
        if (!defined $self->{javafinder_msglist}){
            my $msglst = new SDB::Install::MsgLst();
            $msglst->appendMsgLst ($java->getMsgLst ());
            $self->{javafinder_msglist} = $msglst;
        }
        $self->setErrorMessage ($self->{java_not_found_error}, $java->getMsgLst ());
        return undef;
    }
    return $foundJava;
}


sub checkJavaVm{
    my ($self, $value) = @_;
    my $is32on64 = 0;
    if ($isWin && !$installerIs64bit){
        my $sysinfo = new SDB::Install::SysInfo ();
        if ($sysinfo->{architecture} eq 'AMD64'){
            $is32on64 = 1;
        }
    }

    if (!$value){
        my $msglst = new SDB::Install::MsgLst ();
        if ($is32on64) {
            $self->appendErrorMessage ("You are installing " . $self->getProductName() . " " .
                ($installerIs64bit ? '64bit' : '32bit') . ".");
        }
        $self->appendErrorMessage ($self->{java_not_found_error}, $self->{javafinder_msglist});
        return 0;
    }

    if (!-x $value){
        $self->appendErrorMessage ("Cannot access Java executable '$value': $!");
        return 0;
    }

    my $errlst = new SDB::Install::MsgLst ();

    require SDB::Install::Java;
    my $rc = SDB::Install::Java::_get_java_version ($value, $errlst);

    if (!defined $rc){
        $self->appendErrorMessage ("Cannot determine java version", $errlst);
        return 0;
    }

    my ($version,$is64bit) = @$rc;
    my $javaVersions = $self->getValidJavaVersions ();
    my $sJavaVersions = join "/", @$javaVersions;
    my $rJavaVersions = _getVersionRegexp ($javaVersions);

    if ($version !~ /$rJavaVersions/){
        $self->appendErrorMessage ("Wrong Java version $version: Version $sJavaVersions is required.");
        return 0;
    }

    if ($is64bit xor $installerIs64bit){
        $self->appendErrorMessage ("You are installing " . $self->getProductName() . " " .
            ($installerIs64bit ? '64bit' : '32bit') . ",");
        $self->appendErrorMessage ("therefore " . ($installerIs64bit ? '64bit' : '32bit') . ' Java is required.');
        return 0;
    }
    return 1;
}

#-------------------------------------------------------------------------------
# Returns the path with the remote tool.
# '<sapmnt>/<SID>/hdblcm/hdblcm' is returned if the parameter $remoteTool
# is ommitted.

sub getResidentInstallerExecutable{
    my ($self, $remoteTool) = @_;

    if (!defined $remoteTool){
        $remoteTool = 'hdblcm';
    }

    my $sapSys = $self->getCurrentSAPSystem();

     if (!defined $sapSys) {
         my $sid = $self->getCurrentSID();
         $sid    = 'undefined' if (!defined $sid);
         $self->setErrorMessage("Invalid $gProductNameSystem ID '$sid'"
                              . " (finding resident installer on remote hosts failed)");
         return undef;
    }
    if ($isWin){
        return join ($path_separator,
                  $sapSys->getUsrSapSid (), 'hdblcm', $remoteTool . '.exe');
    }
     return join ($path_separator,
                  $sapSys->get_globalSidDir (), 'hdblcm', $remoteTool);
}

#-------------------------------------------------------------------------------
sub getInstallerCollectOtherHostInfoOperation{
	return $gOperationCollectInstallerInfo;
}


#-------------------------------------------------------------------------------
# Creates a config file containing all current parameters
# and tries to copy the config file to the trace directory.

sub dumpConfigFile {
    my ($self,
        $programName,
        $traceDirHostname,  # if undef, local hostname of TrexInstance is used
       ) = @_;

    my $extension     = ($self->isCheckOnly()) ? '.check.cfg' : '.cfg';
    my ($rc, $source) = $self->SUPER::dumpConfigFile($programName, $extension);
    return $rc if (!$rc);

    my $actionID = $self->{options}->{action_id};
    my $sapSys   = $self->getSAPSystem();
    my $instance = $self->getOwnInstance();
    my $traceDir = (defined $instance)
                   ? $instance->get_hostNameTraceDir($traceDirHostname)
                   : undef;

    if (!defined $actionID || !defined $sapSys ||
        !defined $traceDir || ! -d $traceDir) {
        return 1; # ignore not existing
    }

    my $destin       = $traceDir . $path_separator;
    my $oldFilenames = getFilesTimestampPID($traceDir, $programName, $extension);

    foreach my $currOldCfgFile (@$oldFilenames) {
        unlink $destin . $currOldCfgFile;
    }

    if (!$self->{options}->{is_action_master} || ($actionID !~ /^$programName/)){
        $destin .= $programName . '_';
    }

    $destin        .= $actionID . $extension;
    my $copyCfg     = {};
    $copyCfg->{uid} = $sapSys->getUID();
    $copyCfg->{gid} = $sapSys->getGID();

    if (!copy_file($source, $destin, $copyCfg)) {
        # ignore copy error
        $self->AddMessage("Cannot copy file '$source' to '$destin'", $copyCfg);
    }
    else {
        $source .= '.xml';
        $destin .= '.xml';

        if (-f $source && !copy_file($source, $destin, $copyCfg)) {
            # ignore copy error
            $self->AddMessage("Cannot copy file '$source' to '$destin'", $copyCfg);
        }
    }

    return 1;
}

#-------------------------------------------------------------------------------
# This subroutine may be overriden.

sub isOutstandingAddHostStep {
    return 1;
}


#-------------------------------------------------------------------------------
# Changes the inter service communication mode.
# Returns 1     - if the ini file is changed or has to be changed
#         0     - if the ini file is up to date
#         undef - in case of an error

sub handleGlobalIniISCMode {
    my ($self,
        $wantedISCMode,
        $skipChanging, # 1 - checks the entry only
        $instance,
       ) = @_;

    if (!defined $wantedISCMode) {
        return 0; # nothing to do
    }

    if (!defined $instance) {
        $instance = $self->getOwnInstance();
    }

    if (!defined $instance) {
        $self->setErrorMessage("Change isc_mode: instance does not exist");
        return undef;
    }
    $instance->setMsgLstContext ($self->getMsgLstContext ());
    my $globalIni = $instance->getGlobalIni($self, 1); # 1=nocache
    if (!defined $globalIni) {
        return undef;
    }

    $globalIni->setMsgLstContext ([$self->getMsgLst()]);
    my $iniSection  = 'communication';
    my $iniKey      = 'ssl';
    my $currMode    = $globalIni->getValue($iniSection, $iniKey);

    my $decodedBoolValue  = 0;
    my $currModeSystemPki = 0;
    if (defined $currMode && ($currMode =~ /^systempki$/i)){
        $currModeSystemPki = 1;
    }
    else{
        $decodedBoolValue = $self->assignBoolValue ("global.ini/$iniSection/$iniKey", $currMode);
        if (!defined $decodedBoolValue){
            return undef;
        }
    }

    if (!$currModeSystemPki && !$decodedBoolValue) {
        if ($wantedISCMode eq 'standard') {
             return 0; # nothing to do
        }
    }
    elsif (($wantedISCMode eq 'ssl')) {
        return 0; # nothing to do
    }

    if (!$skipChanging) {
        my $value = ($wantedISCMode eq 'ssl') ? 'systempki' : 'off';
        $globalIni->setValue(CFG_LAYER_SYSTEM, $iniSection, $iniKey, $value);
        my $rc = $globalIni->write();
        if (!$rc) {
            $self->setErrorMessage("Cannot set global ini value $iniKey=$value",
                                   $globalIni->getErrMsgLst());
            return undef;
        }
    }
    return 1;
}

#-------------------------------------------------------------------------------
# Stops the system.

sub stopSystemForModify {

    my ($self,
        $util,          # SDB::Install::SAPSystemUtilities
        $sapSys,        # SDB::Install::SAPSystem
        $instance,      # SDB::Install::SAPInstance::TrexInstance
        $localHostOnly, # optional boolean
        $skipSapstartsrv,
       ) = @_;

    $sapSys->setMsgLstContext([$self->getMsgLst()]);
    my $rc = $util->stopSystem($sapSys,
                               $instance,
                               $self->getValue('Password'),
                               $self->getTimeout ('stop_service'),
                               $self->getTimeout ('stop_instance'),
                               $self->isUseHttps(),
                               $self->getValue ('SSOCertificate'),
                               $localHostOnly,
                               undef, # ignore error when stopping sapstartsrv
                               $skipSapstartsrv);

    if (!$rc){
        $self->setErrorMessage('Cannot stop system', $sapSys->getErrMsgLst());
        return undef;
    }
    return 1;
}

sub willInstallOrUpdateSHA {
    return 0;
}

sub skipOrgManagerCredentialsIfNecessary {
    my ($self, $addRolesOrHostsCsv) = @_;
    my $noStart = $self->hasValue('NoStart') ? $self->getValue('NoStart') : $self->getDefault('NoStart');
    my $importXs2Content = $self->hasValue('ImportContentXS2') ? $self->getValue('ImportContentXS2') : $self->getDefault('ImportContentXS2');

    if ((defined $importXs2Content && $importXs2Content =~ /$bool_false_pattern/) || $noStart) {
        $self->skipOrgManagerCredentials();
        return 1;
    }

    my $hasXsWorker = ($addRolesOrHostsCsv =~ /$gHostRoleXS2Worker/) ? 1 : 0;
    my $isUsingAutoAssign = ($self->getValue('AutoAddXS2Roles')=~ /$bool_true_pattern/) ? 1 : 0;
    if (!$hasXsWorker && !$isUsingAutoAssign) {
        $self->skipOrgManagerCredentials();
        return 1;
    }
    return 0;
}

sub unskipLoadInitialContentParamsIfNecessary {
    my ($self, $addRolesOrHostsCsv) = @_;

    return if $self->skipOrgManagerCredentialsIfNecessary($addRolesOrHostsCsv);

    my $instance = $self->getOwnInstance();
    if (! defined $instance->getXsControllerHostname()) {
        $self->unskipLoadInitialContentParams();
    } else {
        $self->skipOrgManagerCredentials();
    }
}

sub unskipLoadInitialContentParams {
    my ($self) = @_;
    $self->setSkip('SystemUser', 0);
    $self->setSkip('SQLSysPasswd', 0);
    $self->setSkip('OrgManagerUser', 0);
    $self->setSkip('OrgManagerPassword', 0);
    if($self->isSystemInCompatibilityMode()){
        $self->setSkip('TenantUser', 0);
        $self->setSkip('SQLTenantUserPassword', 0);
    }
}

sub skipOrgManagerCredentials {
    my ($self) = @_;
    $self->setSkip('OrgManagerUser', 1);
    $self->setSkip('OrgManagerPassword', 1);
}

sub shouldSkipHostagentCalls {
    my ($self) = @_;
    return 0 if (!exists $self->{params}->{SkipHostagentCalls});
    my $isSkipped = $self->isSkipped('SkipHostagentCalls');
    my $paramValue = $self->getValue('SkipHostagentCalls');
    return (!$isSkipped && $paramValue) ? 1 : 0;
}

sub getParamAutoInitializeServices{
    my ($self, $order, $section, $skip, $constraint) = @_;
    return {
        'order'                     => $order,
        'opt'                       => 'auto_initialize_services',
        'type'                      => 'boolean',
        'section'                   => $section,
        'value'                     => undef,
        'default'                   => 1,
        'str'                       => 'Auto Initialize Services',
        'desc'                      => "Enables/Disables the default service provisioning for HANA family components in multitenant database containers",
        'init_with_default'         => 1,
        'set_interactive'           => 0,
        'skip'                      => (defined $skip) ? $skip : 0,
        'constraint'                => $constraint,
        'hostctrl_opt'              => 'AIS',
    };
}

sub getParamTenantUser{
    my ($self, $order, $section, $skip, $constraint) = @_;
    return {
        'order'                     => $order,
        'opt'                       => 'tenantdb_user',
        'type'                      => 'string',
        'section'                   => $section,
        'value'                     => undef,
        'str'                       => 'Tenant Database User Name',
        'str_templ'                 => 'Tenant Database (%s) User Name',
        'desc'                      => "Specifies an user of the tenant database with priviliges to initialize services of the HANA Family components",
        'init_with_default'         => 1,
        'set_interactive'           => 1,
        'skip'                      => (defined $skip) ? $skip : 0,
        'mandatory'                 => 1,
        'constraint'                => $constraint,
        'help_with_default'         => 1,
        'default'                   => 'SYSTEM',
        'hostctrl_opt'              => 'TENANT_USER',
    };
}

sub getParamSQLTenantUserPassword{
    my ($self, $order, $section, $skip, $constraint, $type, $userParamId) = @_;
       my $param = {
        'order'             => $order,
        'opt'               => 'tenantdb_user_password',
        'type'              => !defined $type ? 'passwd' : $type,
        'section'           => $section,
        'value'             => undef,
        'default'           => undef,
        'str'               => 'Tenant Database User Password',
        'desc'              => 'Tenant Database User Password',
        'str_templ'         => 'Tenant Database User (%s) Password',
        'init_with_default' => 0,
        'log_value'         => sub {defined $_[0] ? '***' : '<not defined>';},
        'mandatory'         => 1,
        'skip'              => (defined $skip) ? $skip : 0,
        'constraint'        => $constraint,
        'retry_count'       => 3,
        'optional_password' => 1,
        'user_paramid'      => $userParamId
    };
    return $param
}

sub setUsernameToSQLUserPassword {
    my ($self,$paramId,$username) = @_;
    my $params = $self->{params};
    my $str_value = sprintf($self->{params}->{$paramId}->{str_templ}, $username);
    $self->{params}->{$paramId}->{str} = $str_value;
}

sub isAutoInitializeServicesRequired {
     my ($self)                = @_;
     my $isMDC                 = $self->getOwnInstance()->isMultiDb();
     my $isInCompatibilityMode = SDB::Install::Configuration::AnyConfig::isSystemInCompatibilityMode($self);
     return
          $isMDC
       && $isInCompatibilityMode;
}

sub checkSQLTenantUserPassword{
    my ($self, $value, $msglst_arg, $dbName) = @_;
    my $tenantUser = $self->getValue('TenantUser') // $self->getDefault('TenantUser');
    return $self->checkDatabaseUserPassword( $tenantUser, $value, 'SQLTenantUserPassword', undef, undef, $dbName, "IndexServer");
}

sub isSystemInCompatibilityMode{
    my ($self, $instanceDuringAddLocalHost) = @_;
    if (exists $self->{isInCompatibilityMode}) {
        return $self->{isInCompatibilityMode};
    }
    my $instance = $self->getOwnInstance();
    $instance = $instance // $instanceDuringAddLocalHost;
    if(!$instance->isMultiDb()){
        $self->{isInCompatibilityMode} = 0 ;
        return 0;
    }
    my $global_ini = $instance->getGlobalIni();
    my $saveContext = $global_ini->setMsgLstContext ([$self->getMsgLst ()]);
    if (defined $global_ini){
         my $value = ( $global_ini->existsValue('multidb', 'singletenant') ? $global_ini->getValue ('multidb', 'singletenant') : 'no' );
         $self->{isInCompatibilityMode} = ($value =~ /$bool_true_pattern/i) ? 1 : 0;
    } else {
       my $hdbnsutil = SDB::Install::SAPInstance::NameserverUtility->new($instance);
       $self->{isInCompatibilityMode} = $hdbnsutil->isSystemInCompatibilityMode($self->getMsgLst());
    }
    $global_ini->setMsgLstContext ($saveContext);
    return $self->{isInCompatibilityMode};
}

# used mostly by subclasses in their 'shouldWarnIfCalledStandalone()' method,
# when a warning should be emitted if the resp. low-level executable is not called from hdblcm:
sub _calledStandaloneCriterion{
    my ($self) = @_;
    my $instLogDir = $self->{'options'}->{'instlog_dir'};
    if(defined $instLogDir && ($instLogDir =~ /hdblcm/)) {
        return 0;
    }
    my $actionMaster = $self->{'options'}->{'is_action_master'};
    if(!$actionMaster) {
        return 0;
    }
    if($self->isCalledBySHA()) {
        return 0;
    }
    return 1;
}

sub shouldWarnIfCalledStandalone{
    return 0;
}

sub getExeName{
    return undef;
}

# used by subclasses in their 'getResidentHdblcmPath()' method,
sub _constructResidentHdblcmPath{
    my ($self) = @_;
    my $retval = undef;
    my $instance = $self->getOwnInstance(); # a big fat and ugly TrexInstance object
    if(defined $instance) {
        $retval = $instance->get_globalSidDir().$path_separator."hdblcm".$path_separator."hdblcm";
    }
    return $retval;
}

sub getResidentHdblcmPath{
    return undef;
}

sub setDatabaseToSystemUserCredentialsParameter {
    my ($self,$systemUserCredentialsParam) = @_;
    my $instance = $self->getOwnInstance();
    if(defined $instance && $instance->isMultiDb()){
        $systemUserCredentialsParam->{str} =~ s/(?<!System\s)Database/System Database/g;
        $systemUserCredentialsParam->{str_templ} =~ s/(?<!System\s)Database/System Database/g;
    }
}

sub setDatabaseToTenantUserCredentialsParameter {
    my ($self,$tenantUserCredentialsParam,$tenant) = @_;
    my $tenantStr = $tenant;
    if(!defined $tenantStr && $self->isSystemInCompatibilityMode()){
        $tenantStr=$self->getSID() // (defined $self->getSAPSystem() ? $self->getSAPSystem()->get_sid() : undef);
    }
    return if (!defined $tenantStr || length($tenantStr) == 0);
    $tenantUserCredentialsParam->{str} = $tenantUserCredentialsParam->{str} =~ s/Tenant Database ('\w{3}')?\s?User/Tenant Database '$tenantStr' User/r;
    $tenantUserCredentialsParam->{str_templ} = $tenantUserCredentialsParam->{str_templ} =~ s/(Tenant\sDatabase)\s(('\w{3}')|(\(%s\)))?\s?User/Tenant Database '$tenantStr' User/r;
}

sub setCollectUpdateHostCredentials {
    my ($self, $value) = @_;
    $self->{params}->{CollectUpdateHostCredentials}->{value} = $value;
    if (!$value) {
        $self->setSkip('SystemUser');
        $self->setSkip('SQLSysPasswd');
    }
    return 1;
}

sub getAutoInitializeServicesAction {
    return "adding";
}

sub getTenantCredentialsConstraint{
    my $self = shift;
    my $action = $self->getAutoInitializeServicesAction();
    my $validAutoInitializeServicesRolesComponentMap = GetAutoInitializeServicesRolesComponentMap();
    my $tenantCredentialsRoles = join(',',keys(%$validAutoInitializeServicesRolesComponentMap));
    my $tenantCredentialsConstraint = "Valid for $action additional $tenantCredentialsRoles roles or hosts only";
    return $tenantCredentialsConstraint;
}

sub isCalledBySHA {
    my ($self) = @_;
    return $self->{options}->{called_by_hostagent};
}

sub getParamKeepLSSUser{
    my ($self, $order, $section, $constraint) = @_;

    my $str  = "Keep $gShortProductNameLSS User";
    my $desc = "Does not remove the $gShortProductNameLSS User";

    return {'order'             => $order,
            'opt'               => 'keep_lss_user',
            'alias_opts'        => ['keeplssuser'],
            'hostctrl_opt'      => 'KEEP_LSS_USER',
            'type'              => 'boolean',
            'section'           => $section,
            'value'             => undef,
            'str'               => $str,
            'str_templ'         => $str . ' (%s)',
            'desc'              => $desc,
            'default'           => 0,
            'init_with_default' => 1,
            'set_interactive'   => 0,
            'constraint'        => $constraint,
            'console_omit_word_Enter' => 1,
            'skip'              => 0,
    };
}

sub getParamKeepLSSUserGroup{
    my ($self, $order, $section, $constraint) = @_;

    my $str  = "Keep $gShortProductNameLSS User Group";
    my $desc = "Does not remove the $gShortProductNameLSS User Group";

    return {'order'             => $order,
            'opt'               => 'keep_lss_user_group',
            'alias_opts'        => ['keeplssusergrp'],
            'hostctrl_opt'      => 'KEEP_LSS_USER_GRP',
            'type'              => 'boolean',
            'section'           => $section,
            'value'             => undef,
            'str'               => $str,
            'str_templ'         => $str . ' (%s)',
            'desc'              => $desc,
            'default'           => 0,
            'init_with_default' => 1,
            'set_interactive'   => 0,
            'constraint'        => $constraint,
            'console_omit_word_Enter' => 1,
            'skip'              => 0,
    };
}

sub getParamLSSPassword {
    my ($self, $order, $section, $skip) = @_;

    return {
        'order'             => $order,
        'opt'               => 'lss_user_password',
        'type'              => 'passwd',
        'section'           => $section,
        'value'             => undef,
        'default'           => undef,
        'set_interactive'   => 1,
        'str'               => "$gShortProductNameLSS User Password",
        'str_templ'         => "$gShortProductNameLSS User (%s) Password",
        'desc'              => "$gShortProductNameLSS User Password",
        'skip'              => $skip // 1,
        'mandatory'         => 1,
        'init_with_default' => 0,
    };
}

sub getParamVolumeEncryption {
    my ($self, $order, $section) = @_;
    my $desc = "Enable data and log volume encryption";
    return {
            'order'   => $order,
            'opt'     => 'volume_encryption',
            'type'    => 'boolean',
            'section' => $section,
            'default' => 0,
            'str'     => "Do you want to enable data and log volume encryption?",
            'ui_str'  => $desc,
            'desc'    => $desc,
            'set_interactive'   => 1,
            'init_with_default' => 1,
            'console_omit_word_Enter' => 1,
            };
}

sub skipLSSUserParams{
    my ($self, $skip) = @_;
    $self->setSkip('KeepLSSUser', $skip // 1 );
    $self->setSkip('KeepLSSUserGroup', $skip // 1);

    my $keepUser = $self->{params}->{'KeepLSSUser'};
    my $keepUserGroup = $self->{params}->{'KeepLSSUserGroup'};
    my $sidcrypt = getSidcryptName($self->getSID());
    my $sidcryptUser = SDB::Install::LSS::LssSidcryptUser->new($sidcrypt);
    $keepUser->{str} = sprintf ($keepUser->{str_templ}, $sidcrypt );
    $keepUserGroup->{str} = sprintf ($keepUserGroup->{str_templ}, $sidcryptUser->group());
}

sub getExistingUserId {
    my ($self, $userName) = @_;
    my $user = new SDB::Install::User($userName);
    return $user->id() if($user->exists());

    my $remoteUIDs = $self->{remoteUids} // {};
    return $self->getRemoteId($remoteUIDs, $userName);
}

sub getExistingGroupId {
    my ($self, $groupName) = @_;
    my $group = new SDB::Install::Group($groupName);
    return $group->id() if($group->exists());

    my $remoteGIDs = $self->{remoteGids} // {};
    return $self->getRemoteId($remoteGIDs, $groupName);
}

sub getRemoteId {
    my ($self, $remoteIds, $name) = @_;
    for my $id (keys(%{$remoteIds})){
        my $data = $remoteIds->{$id};
        for my $entry (@{$data}){
            return $id if($name eq $entry->[1]);
        }
    }
    return undef;
}

sub getDbHostsWithSidcryptUser {
    my ($self) = @_;
    my $localHost = $self->getLocalHanaHost();
    my $instance = $self->getOwnInstance();
    my $sidcryptUsername = getSidcryptName($self->getValue('SID'));
    my $sidcryptUser = SDB::Install::LSS::LssSidcryptUser->new($sidcryptUsername);
    my $dbHosts = $self->getAllDbHosts($instance);

    my @hostArraysWithSidcryptUser = map {
        $_->[0]->[1] eq $sidcryptUsername ? @$_ : ();
    } values(%{$self->{remoteUids} // {}});

    my @hostsWithSidcryptUser = map { $_->[0] } @hostArraysWithSidcryptUser;
    my @dbHostsWithSidcryptUser = grep { $_ ~~ $dbHosts } @hostsWithSidcryptUser;
    unshift(@dbHostsWithSidcryptUser, $localHost) if($sidcryptUser->exists() && $localHost ~~ $dbHosts);

    return \@dbHostsWithSidcryptUser;
}

sub getLssPath {
    my($self) = shift;
    my $ownInstance = $self->getOwnInstance();
    return undef if (!defined $ownInstance);
    my $lssInstance = $ownInstance->getLssInstance();
    return defined $lssInstance ? $lssInstance->getLssSidDir() : undef;
}

sub removeTempCheckShared {
    my ( $self, $chkSharedFile) = @_;
    if( $chkSharedFile && File::stat::stat($chkSharedFile)){
        my $builtin = SDB::Common::BuiltIn->get_instance();
        if(!$builtin->unlink($chkSharedFile)){
            $self->getMsgLst()->addWarning("Cannot delete temporary file '$chkSharedFile': $!");
        }
    }
}

sub setUsesNonSharedStorage {
    my($self, $value) = @_;
    $self->{'usesNonSharedStorage'} = $value;
}

sub usesNonSharedStorage {
    my ($self) = @_;
    return 1 if($self->{'usesNonSharedStorage'});
    my $instance = $self->getOwnInstance();
    return ($instance && $instance->usesStorageApi()) ? 1 : 0;
}

sub addPathExistanceHostEntry {
    my ($self, $pathExistanceKey, $host, $xmlPathTag) = @_;
    my $data = $self->{host_data}->{$host};
    my $xmlPathElement = $data->getElementByTagName ($xmlPathTag);
    return if(!$xmlPathElement || !$xmlPathElement->{attr});
    $self->{$pathExistanceKey}->{$host}->{'exists'} = $xmlPathElement->{attr}->{isExists};
    $self->{$pathExistanceKey}->{$host}->{'sidadmCanAccess'}  = $xmlPathElement->{attr}->{'sidadmCanAccess'};
    $self->{$pathExistanceKey}->{$host}->{'isSharedVolume'}  = $xmlPathElement->{attr}->{'isSharedVolume'};
    if(!$self->{$pathExistanceKey}->{$host}->{'isSharedVolume'}){
        $self->setUsesNonSharedStorage(1);
    }
    $self->{$pathExistanceKey}->{$host}->{'isSharedSubvolume'}  = $xmlPathElement->{attr}->{'isSharedSubvolume'};
    my $msgLstElement =  $data->getElementByTagName ('MSGL', $xmlPathElement);
    if(defined  $msgLstElement){
        my $subMsgLst = SDB::Install::MsgLst->new();
        my $msgEl = $data->getElementByTagName('MSG',$msgLstElement);
        while($msgEl){
            my $text = $msgEl->{attr}->{'_TEXT'};
            my $type = $msgEl->{attr}->{'_TYPE'};
            $subMsgLst->addMessage($text,undef,$type);
            $msgEl = $msgEl->{neighbor};
        }
        $self->{$pathExistanceKey}->{$host}->{'msgLst'} = $subMsgLst;
    }
}

sub handleSidadmExecution {
    my ($self) = @_;
    return 1 if(isAdmin());

    $self->setHidden('RemoteExecution', 1);
    for my $paramId ('InstallSSHKey', 'RootUser', 'RootPassword'){
        $self->setSkip($paramId, 1);
    }
    return 1;
}

1;

