package SDB::Install::Configuration::AnyMultiHostConfig;

use strict;

use File::Spec;
use File::stat qw(stat);
use File::Basename qw(dirname);
use LCM::FileUtils;
use SDB::Install::Configuration::AnyConfig;
use SDB::Install::Globals  qw ($gHostRoleAcceleratorStandby
                               $gHostRoleAcceleratorWorker
                               $gHostRoleWorker
                               $gHostRoleStandby
                               $gHostRoleXS2Worker
                               $gHostRoleXS2Standby
                               $gHostRoleComputeServer
                               $gNameAccelerator
                               $gOperationAddHost
                               $gProductNameSystem
                               $gProductNameSHA
                               $gShortProductNameAccelerator
                               $gShortProductNameXS2
                               $gSapsysGroupName
                               $gShortProductNameLSS
                               GetHostRoleProperties
                               $gXSASystemUserParamId $gXSASystemUserPwdParamId $gXSAOrgManagerUserParamId $gXSAOrgManagerPwdParamId $gXSAProdSpaceDBSystemUserParamId $gXSAProdSpaceDBSystemUserPwdParamId
                               $gSkipSslCiphersuitesCheck
                               );
use SDB::Install::IniFile;
use SDB::Install::LayeredConfig qw (CFG_LAYER_SYSTEM);
use SDB::Install::RemoteHostctrlHosts;
use SDB::Install::RemoteHosts;
use SDB::Install::System qw (changeOwn
                             isAdmin
                             is_local_address
                             nslookup
                             $hostname_regex
                             $ipv4_regex);
use SDB::Install::SysVars   qw($isWin $path_separator);
use SDB::Install::System::IPAddr;
use SDB::Install::SAPProfiles qw ($cipherSuitesId $recommendedCipherSuites);
use SDB::Install::Saphostagent qw (getActiveSaphostexecDir);
use SAPDB::Install::Hostname qw(hostname);
use SDB::Common::Utils qw(getSidcryptName);
use experimental qw (smartmatch);

our @ISA = qw (SDB::Install::Configuration::AnyConfig);

our @EXPORT = qw (@listenInterfaceValidValues
                  @listenInterfaceUIValues
                  $hostRoleDefaultValue
                  $validHostRoles
                  getSortedHostRoleNames);

our @listenInterfaceMultiHost   = qw (global internal);
our @listenInterfaceUIMultiHost = ('The HANA services will listen on all network interfaces',
                                   'The HANA services will only listen on a specific network interface');

our @listenInterfaceValidValues = (@listenInterfaceMultiHost, 'local');
our @listenInterfaceUIValues    = (@listenInterfaceUIMultiHost,
                                   'The HANA services will only listen on loopback interface (127.0.0.1)');

our $validHostRoles       = GetHostRoleProperties();
our $hostRoleDefaultValue = $gHostRoleWorker;


#-------------------------------------------------------------------------------
# Constructor

sub new{
    my $self = shift->SUPER::new (@_);

    return $self;
}

sub InitDefaults{
	my $self = shift;

	my $returnCode = $self->SUPER::InitDefaults(@_);
	return undef if !$returnCode;

    if (!$self->isa('SDB::Install::Configuration::HdbModify') &&
        !$self->isa('SDB::Install::Configuration::HdbReg')    &&
        !$self->isa('SDB::Install::Configuration::Rename')) {
        $self->initRecommendationTexts();
    }
	return 1;
}

sub initRecommendationTexts {
	my ($self) = @_;
	my $hasParameterInternalNetwork = exists($self->{params}->{InternalNetwork});
	my $hasParameterListenInterface = exists($self->{params}->{ListenInterface});
	my $isMultiHostSystem = ($self->isDistributedSystem() || $self->getValue('AddHosts'));

	if($hasParameterInternalNetwork){
		$self->{params}->{InternalNetwork}->{recommendation} = 'It is recommended to select network address, which is part of an internal network';
	}
	if($hasParameterListenInterface){
		my $systemType = $isMultiHostSystem ? 'multiple-host' : 'single-host';
		my $recommendedInterface = $isMultiHostSystem ? 'internal' : 'local';
		$self->{params}->{ListenInterface}->{recommendation} = sprintf("The recommended value for %s system is '%s'", $systemType, $recommendedInterface);
	}
}

#-------------------------------------------------------------------------------
# Returns a sorted array of valid host role names

sub getSortedHostRoleNames {

    my ($withoutColumnStoreRoles,
        $columnStoreRolesOnly,
        $forRemove,
       ) = @_;

    my $roleInfos;

    if ($withoutColumnStoreRoles || $columnStoreRolesOnly) {
        $roleInfos = {};
        foreach my $role (keys %$validHostRoles) {
            if ($validHostRoles->{$role}->{isColumnStore}) {
                next if ($withoutColumnStoreRoles);
            }
            else {
                next if ($columnStoreRolesOnly);
            }
            next if ($forRemove && !$validHostRoles->{$role}->{removable});
            $roleInfos->{$role} = $validHostRoles->{$role};
        }
    }
    else {
        $roleInfos = $validHostRoles;
    }

    my @filteredRoles = grep { $_ ne $gHostRoleComputeServer } keys(%$roleInfos);
    return [sort {$roleInfos->{$a}->{order} <=> $roleInfos->{$b}->{order}} @filteredRoles];
}


#-------------------------------------------------------------------------------
# Checks the parameter 'internal_network'.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.

sub checkInternalNetwork {
	my ($self, $newInternalNetwork) = @_;
	if($self->getValue('ListenInterface') eq 'global' && $newInternalNetwork eq 'none') {
		return 1;
	}

	my $prefix = $self->getIPPrefix ($newInternalNetwork);
	if (!defined $prefix){
		$self->setErrorMessage ($self->getParamName('InternalNetwork') . " is not defined");
		return 0;
	}
	if ($prefix->iptype() eq 'LOOPBACK'){
		$self->setErrorMessage ("Loopback ip address '".$prefix->print."' is not allowed");
		return 0;
	}
	delete $self->{internalLocalIp};
	delete $self->{internalNetworkPrefix};
	my $msglst = $self->getMsgLst ();
	my $rc = 1;
	my $ipList = $self->getLocalIPs ();

	my $internalIp = $prefix->findSubnetDeviceIps ($ipList)->[0];
	if (defined $internalIp){
		$self->{internalLocalIp} = $internalIp->short ();
		$self->{internalNetworkPrefix} = $prefix->print ();
	}
	else{
		$self->setErrorMessage ("No matching network interface on local host found");
		my $msglstSubnets = new SDB::Install::MsgLst ();
		foreach my $ip (@$ipList){
			if ($ip->version != 4){
				next;
			}
			if ($ip->iptype eq 'LOOPBACK'){
				next;
			}
			$msglstSubnets->addMessage ($ip->getPrefix()->print());
		}
		$self->appendErrorMessage ("Available networks:", $msglstSubnets);
		$rc = 0;
	}

	if (defined $self->{remoteIPs}){
		foreach my $host (keys %{$self->{remoteIPs}}){
			$ipList = $self->getRemoteIPs ($host);
			$internalIp = $prefix->findSubnetDeviceIps ($ipList)->[0];
			if (! defined $internalIp){
				$self->appendErrorMessage ("No matching network interface found on host $host");
				my $msglstSubnets = new SDB::Install::MsgLst ();
				foreach my $ip (@$ipList){
					if ($ip->version != 4){
						next;
					}
					if ($ip->iptype eq 'LOOPBACK'){
						next;
					}
					$msglstSubnets->addMessage ($ip->getPrefix()->print());
				}
				$self->appendErrorMessage ("Available networks on $host:", $msglstSubnets);
				$rc = 0;
			}
		}
	}
	return $rc;
}

sub requiresChangeOfInternalNetwork {
	return 1;
}

#-------------------------------------------------------------------------------
# Checks the option 'listen_interface'.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.
#
# initTopologyArgs =
#      ['--hostnameResolution=internal', "--internalAddress=$internalLocalIp"];
#
# Returns int retCode

sub checkListenInterface {
	my ($self, $newInterfaceType) = @_;
	my $currentInterfaceType = $self->getListenInterfaceType();

	if (!defined $newInterfaceType) {
		return 1;
	}

  my $existsValidInternalNetwork = (scalar(@{$self->getValidValues('InternalNetwork')})) ? 1 : 0;
  if ($newInterfaceType eq 'local' || ! $existsValidInternalNetwork) {
    $self->setSkip('InternalNetwork');
  }

  if ($newInterfaceType eq 'local') {
    if($currentInterfaceType eq 'local') {
      return 1;
    }

    my $cntHosts = $self->getNumberOfDefinedHosts();
    if ($cntHosts > 1) {
      $self->PushError("Cannot change listen interface to 'local' ($cntHosts hosts are defined)");
      return 0;
    }
	}
	else {
		$self->_setMandatoryInternalNetworkIfPossible();
	}

	$self->updateInternalNetworkValidValues($newInterfaceType);

	return 1;
}

sub updateInternalNetworkValidValues {
	my ($self, $interfaceType) = @_;
	my $internalNetworkValidValues = $self->{params}->{InternalNetwork}->{valid_values};

	return if (!defined($internalNetworkValidValues) || !scalar(@$internalNetworkValidValues));

	if ($internalNetworkValidValues->[0] eq 'none') {
		if ($interfaceType ne 'global') {
			shift @{$internalNetworkValidValues};
		}
	} else {
		if ($interfaceType eq 'global') {
			unshift @{$internalNetworkValidValues}, 'none';
		}
	}

	my $isBatchMode = 0;
	require LCM::App::ApplicationContext;
	my $app = LCM::App::ApplicationContext::getInstance()->getApplication();
	if (defined $app && !$app->isGUIApplication()){
		$isBatchMode = $app->isBatchMode();
	}
	my $default = undef;
	if ($interfaceType eq 'global' ) {
		$default = 'none';
	} elsif (scalar(@$internalNetworkValidValues)) {
		$default = $internalNetworkValidValues->[0];
	}
	my $currentInternalNetwork = $self->getOwnInstance()->getInternalNetworkPrefix();
	if (defined($currentInternalNetwork) && $currentInternalNetwork ~~ @$internalNetworkValidValues) {
		$default = $currentInternalNetwork;
	}
	$self->setDefault('InternalNetwork', $default);
}


#-------------------------------------------------------------------------------
# Checks the parameter 'internal_network' for underlying LCM tools only.

sub checkInternalNetworkOnly {

    my ($self, $value) = @_;

    my $prefix = $self->getIPPrefix ($value);
    if (!defined $prefix){
        return 0;
    }
    if ($prefix->iptype() eq 'LOOPBACK'){
        $self->setErrorMessage ("Loopback ip address '".$prefix->print."' is not allowed");
        return 0;
    }
    delete $self->{internalLocalIp};
    delete $self->{internalNetworkPrefix};
    my $msglst = $self->getMsgLst ();
    my $rc = 1;
    my $ipList = $self->getLocalIPs ();

    my $internalIp = $prefix->findSubnetDeviceIps ($ipList)->[0];
    if (defined $internalIp){
        $self->{internalLocalIp} = $internalIp->short ();
        $self->{internalNetworkPrefix} = $prefix->print ();
    }
    else{
        $self->setErrorMessage ("No matching network interface on local host found");
        my $msglstSubnets = new SDB::Install::MsgLst ();
        foreach my $ip (@$ipList){
            if ($ip->version != 4){
                next;
            }
            if ($ip->iptype eq 'LOOPBACK'){
                next;
            }
            $msglstSubnets->addMessage ($ip->getPrefix()->print());
        }
        $self->appendErrorMessage ("Available networks:", $msglstSubnets);
        $rc = 0;
    }

    if (defined $self->{remoteIPs}){
        foreach my $host (keys %{$self->{remoteIPs}}){
            $ipList = $self->getRemoteIPs ($host);
            $internalIp = $prefix->findSubnetDeviceIps ($ipList)->[0];
            if (! defined $internalIp){
                $self->appendErrorMessage ("No matching network interface found on host $host");
                my $msglstSubnets = new SDB::Install::MsgLst ();
                foreach my $ip (@$ipList){
                    if ($ip->version != 4){
                        next;
                    }
                    if ($ip->iptype eq 'LOOPBACK'){
                        next;
                    }
                    $msglstSubnets->addMessage ($ip->getPrefix()->print());
                }
                $self->appendErrorMessage ("Available networks on $host:", $msglstSubnets);
                $rc = 0 ;
            }
        }
    }
    return $rc;
}


#-------------------------------------------------------------------------------
# Checks the option 'listen_interface' for underlying LCM tools only.

sub checkListenInterfaceOnly {

    my ($self, $wantedInterface) = @_;

    if (!defined $wantedInterface) {
        return 1;
    }

    if ($wantedInterface eq 'internal') {
        $self->_setMandatoryInternalNetworkIfPossible();
    }
    if ($wantedInterface eq 'local') {
        $self->setSkip('InternalNetwork');
        my $cntHosts = $self->getNumberOfDefinedHosts();
        if ($cntHosts > 1) {
            $self->PushError("Cannot change listen interface to 'local'"
                             . " ($cntHosts hosts are defined)");
            return 0;
        }
    }
    if ($wantedInterface eq 'global') {
        # internal_network will be set explicitly or should not be defined in case of 'none'
        delete $self->{internalLocalIp};
        delete $self->{internalNetworkPrefix};
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Collects information of remote hosts and checks SID, instance number,
# and user/group ID. These values have to belong to this SAP HANA system
# or have to be free.
#
# User/group IDs are not scanned if a specific ID is passed via the
# command line ('--userid', '--groupid') or the scanning of the password
# database should be suppressed ('--ignore=scan_password_database')

sub CollectOtherHostInfos {

    my ($self,
        $isNotVerbose,
        $wantedSidadm,   # if xml tag 'users' is skipped, the properties of this sidadm are collected
                         #
        $skipCollectAll, # bool value: does not collect any main xml tag,
                         # except $wantedSidadm, --dataPath, --logPath, --userid, --groupid
                         # [cannot be applied together with $skipCollectMap and $collectOnlyMap]
                         #
        $skipCollectMap, # does not collect main xml tags that are contained in
                         # this hash map, e.g. {'groups' => 1, 'users' => 1)
                         # [cannot be applied together with $collectOnlyMap and $skipCollectAll]
                         #
        $collectOnlyMap, # collect main xml tags that are contained in
                         # this hash map, e.g. {'groups' => 1, 'users' => 1)
        $wantedDataPath,
        $wantedLogPath
       ) = @_;

    my $currSkipMap = $skipCollectMap;
    my $wantedUID   = $self->getBatchValue('UID');
    my $wantedGID   = $self->getBatchValue('GID');
    my $wantedLSSGroupID = $self->getBatchValue('LSSGroupID');
    my $sid = $self->getBatchValue('SID');
    my $wantedSidcrypt = (defined $sid) ? getSidcryptName($sid) : undef;
    my $wantedUsers  = undef;
    my $wantedGroups = undef;

    my $hdbInstance = $self->getOwnInstance();
    my $tenantDatabases;
    my $lssInstance;
    if (defined $hdbInstance){
        $tenantDatabases = $hdbInstance->getTenantDatabases();
        $lssInstance = $hdbInstance->getLssInstance();
    }

    if (defined $tenantDatabases && @$tenantDatabases){
        my @users;
        my @groups;
        foreach my $entry (@$tenantDatabases){
            push @users,  $entry->getUsername();
            push @groups, $entry->getGroupname();
        }
        if (@users){
            $wantedUsers  = join (',', @users);
            $wantedGroups = join (',', @groups);
        }
    }

    if(defined $lssInstance){
        $wantedLSSGroupID = $lssInstance->getUserConfig($self->getSAPSystem())->getGID();
        my $sidCryptUser = $lssInstance->getLssCryptUser();
        if($sidCryptUser->exists()){
            $wantedUsers  .= join (',', $sidCryptUser->getname());
            $wantedGroups .= join(',', $sidCryptUser->group());
        }
    }

    if (!$skipCollectAll && !defined $collectOnlyMap && !defined $currSkipMap) {

        my $doNotScan = $self->getIgnore('scan_password_database');

        if ($doNotScan || defined $wantedUID || !exists $self->{params}->{UID}){
            $currSkipMap->{users} = 1;
        }

        if ($doNotScan || (defined $wantedGID && defined $wantedLSSGroupID) ||
            (!exists $self->{params}->{GID}  && !exists $self->{params}->{LSSGroupID})){
            $currSkipMap->{groups} = 1;
        }
    }

    return $self->SUPER::CollectOtherHostInfos($isNotVerbose,
                                               $skipCollectAll,
                                               $currSkipMap,
                                               $collectOnlyMap,
                                               $wantedSidadm,
                                               $wantedSidcrypt,
                                               $wantedUID,
                                               $wantedGID,
                                               $wantedLSSGroupID,
                                               $wantedUsers,
                                               $wantedGroups,
                                               $wantedDataPath,
                                               $wantedLogPath);
}

#-------------------------------------------------------------------------------
# Returns 1 if any extended storage, accelerator or streaming host role exists

sub existsNonColumnStoreHostRole {

    my ($self, $instance) = @_;

    $instance    = $self->getOwnInstance()       if (!defined $instance);
    my $roleInfo = $instance->getHostRolesInfo() if (defined $instance);

    if (defined $roleInfo) {

        foreach my $currHost (keys %$roleInfo) {
            my $roles = $roleInfo->{$currHost};

            if (defined $roles) {
                foreach my $currRole (split(/\s+/, $roles)) {
                    my $validRole = $validHostRoles->{$currRole};
                    if (defined $validRole && ($validRole->{isExtendedStorage} ||
                                               $validRole->{isAccelerator} ||
                                               $validRole->{isStreaming})) {
                        return 1;
                    }
                }
            }
        }
    }
    return 0;
}



#-------------------------------------------------------------------------------
# CollectOtherHostsInfo must be called before this method !

sub fillInternalNetworkAndListenInterfaceValidValues{
	my ($self) = @_;
	my $networkAddresses = $self->_getAllValidNetworkAddresses();
	$self->{params}->{InternalNetwork}->{valid_values} = $networkAddresses;
# handle situation when no valid networks are detected
	if (! scalar(@$networkAddresses)) {
		$self->_setListenInterfaceValidValues();
		my $recommendation = "No common internal networks were found between hosts. Omitting value 'internal'.";
		$self->{params}->{ListenInterface}->{recommendation} = $recommendation;

		$recommendation = "No common internal networks found.";
		$self->{params}->{InternalNetwork}->{recommendation} = $recommendation;
		$self->setSkip('InternalNetwork', 1);
		return;
	}

	my $listenInterfaceValue = $self->getValue("ListenInterface") || $self->getDefault("ListenInterface");
	return if $listenInterfaceValue eq '';
	$self->setSkip('ListenInterface',0);
	$self->_setMandatoryInternalNetworkIfPossible() if ($listenInterfaceValue ne 'local');
	$self->_setListenInterfaceValidValues();
	$self->initRecommendationTexts();
	if(!$self->getDefault('InternalNetwork')) {
		$self->setDefault('InternalNetwork', $networkAddresses->[0]) if $networkAddresses->[0];
	}
	$self->updateInternalNetworkValidValues($self->getListenInterfaceType());
}

#-------------------------------------------------------------------------------
# CollectOtherHostsInfo must be called before this method !

sub _getAllValidNetworkAddresses{
	my ($self) = @_;
  my @localHostIPs = @{$self->getLocalIPs()}; #filters out loopback
  @localHostIPs = grep {$_->version == 4} @localHostIPs;

  my %validLocalAddresses = ();
  for (@localHostIPs) {
    my $subnet = $_->getPrefix()->print();
    $validLocalAddresses{$subnet}++;
  }
 # In case of WebUI execution with single-host system
 # {remoteIPs} is defined but empty, hence the additional check
  my $numberOfRemoteHosts = (defined $self->{remoteIPs}) ? scalar(keys(%{$self->{remoteIPs}})) : 0;
  return [keys(%validLocalAddresses)] if (! $numberOfRemoteHosts);

  my %timesSubnetIsEncountered = ();
  for my $host (keys(%{$self->{remoteIPs}})) {
    my @hostIPs = @{$self->getRemoteIPs($host)}; #filters out loopback
    @hostIPs = grep {$_->version == 4} @hostIPs;
    for (@hostIPs) {
      my $subnet = $_->getPrefix()->print();
      (exists $timesSubnetIsEncountered{$subnet}) ? $timesSubnetIsEncountered{$subnet}++ : ($timesSubnetIsEncountered{$subnet} = 1);
    }
  }
  my @sharedAddresses = ();
  for my $localAddress (keys(%validLocalAddresses)) {
    push (@sharedAddresses, $localAddress) if ($timesSubnetIsEncountered{$localAddress} == $numberOfRemoteHosts);
  }
  return \@sharedAddresses;
}

sub _setListenInterfaceValidValues {
  my ($self) = @_;
  my $validInternalNetworkValues = $self->{params}->{InternalNetwork}->{valid_values};
  my $listenInterface = $self->{params}->{ListenInterface};
  if (! scalar(@$validInternalNetworkValues)) {
    $listenInterface->{valid_values} = ["global"];
    $self->setDefault("ListenInterface", "global");
  }
  else {
    my $isMultiHostSystem = ($self->isDistributedSystem() || $self->getValue('AddHosts')) ? 1 : 0;
    if ($isMultiHostSystem) {
      $self->restrictListenInterfaceMultiHost();
    }
    else {
      $self->restrictListenInterfaceSingleHost();
    }
  }
}

#-------------------------------------------------------------------------------
# Returns the name of an addhost status file.

sub getAddHostPersFile {
    my ($self, $instance, $hostname) = @_;
    return join ($path_separator,
                 $instance->get_hostNameDir($hostname),
                 'hdbaddhost');
}


#-------------------------------------------------------------------------------
# Returns the pending step if an add host operation is pending at the specified
# host, otherwise undef.

sub getAddHostPendingStep {
    my ($self, $instance, $hostname) = @_;

    my $pendingStep = undef;
    my $persfile    = $self->getAddHostPersFile($instance, $hostname);

    if ($self->pers_exists($persfile)) {
        my $persdata = $self->pers_load($persfile);
        if (defined $persdata->{step} && !defined $persdata->{InstallDate}) {
            $pendingStep = $persdata->{step};
        }
    }
    return $pendingStep;
}

#-------------------------------------------------------------------------------
#
# Get subnet ip (prefix) of an ip address
#
# Parameter:  string                   $ipString       # CIDR notation
#
# Returns SDB::Install::System::IPAddr $prefix

sub getIPPrefix{
    my ($self, $ipSting) = @_;
    my $ip = new SDB::Install::System::IPAddr ($ipSting);
    if (!defined $ip){
        $self->setErrorMessage (Net::IP::Error());
        return undef;
    }
    if (!defined $ip->{prefixLen}){
        $self->setErrorMessage ("Invalid ip address prefix notation (CIDR): Prefix length is unknown");
        return undef;
    }
    my $prefix = $ip->getPrefix ();
    if (!defined $prefix){
        $self->setErrorMessage (Net::IP::Error());
        return undef;
    }
    return $prefix;
}


#-------------------------------------------------------------------------------
# Returns the listen interface type

sub getListenInterfaceType {
    my ($self) = @_;
    my $interfaceType = $self->getOwnInstance()->getTrexNetListenInterface();
    return (defined $interfaceType) ? substr($interfaceType, 1) : 'local';
}


#-------------------------------------------------------------------------------
# If the parameter '--add_roles' is specified with the local host name,
# this local host name is returned. Otherwise undef is returned.

sub getLocalHostOfAddRemoveRoles {
    return $_[0]->{_localHostOfAddRemoveRoles};
}


#-------------------------------------------------------------------------------
#
# get local device ip addresses
#
# Parameter: none
#
# Returns array of SDB::Install::System::IPAddr (array ref)

sub getLocalIPs{
    my ($self) = @_;
    my $sysinfo = $self->getSysInfo ();
    my $networkInterfaces = $sysinfo->interfaceInfo ();
    my $msglst = $self->getMsgLst ();
    my $found = 0;
    my @ipList;
    my ($localIp, $ip);
    foreach my $ips (values %$networkInterfaces){
        foreach $ip (@$ips){
            $localIp = new SDB::Install::System::IPAddr ($ip->{addr}.'/'.$ip->{netbits});
            if (!defined $localIp){
                $msglst->addWarning ("Failed to parse local ip address '$ip->{addr}': " . Net::IP::Error());
                next;
            }
            if ($localIp->iptype () eq 'LOOPBACK'){
                next;
            }
            push @ipList, $localIp;
        }
    }
    return \@ipList;
}


#-------------------------------------------------------------------------------
# Returns a description of valid host roles.
#
# Example:
#    "role: 'worker'   - host is used for database processing [default]\n"
#  . "      'standby'  - host is idle and available for HA failover\n"

sub getHostRoleDescription {

    my ($self,
        $withoutColumnStoreRoles,
        $columnStoreRolesOnly,
        $forRemove,
       ) = @_;

    my $desc       = 'Role: ';
    my $indent     = (' ' x length($desc));
    my $maxRoleLen = 0;
    my $isFirst    = 1;

    foreach my $currRole (keys %$validHostRoles) {

        if ($validHostRoles->{$currRole}->{isColumnStore}) {
            next if ($withoutColumnStoreRoles);
        }
        else {
            next if ($columnStoreRolesOnly);
        }

        next if ($forRemove && !$validHostRoles->{$currRole}->{removable});

        my $len     = length($currRole);
        $maxRoleLen = $len if ($len > $maxRoleLen);
    }

    foreach my $currRole (@{getSortedHostRoleNames($withoutColumnStoreRoles,
                                                   $columnStoreRolesOnly,
                                                   $forRemove)}) {

        if ($isFirst) {
            $isFirst = 0;
        }
        else {
            $desc .= $indent;
        }
        $desc   .= "'$currRole'";
        my $diff = $maxRoleLen - length($currRole);
        $desc   .= (' ' x $diff) if ($diff > 0);
        $desc   .= ' - ';
        $desc   .= $self->convertDescIntoLowerCase
                                         ($validHostRoles->{$currRole}->{desc});
        $desc   .= ' [default]' if ($currRole eq $hostRoleDefaultValue);
        $desc   .= "\n";
    }
    return $desc;
}

#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter '--addhosts.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.

sub getParamAddHosts{
    my ($self, $order, $section, $withHostRoleDesc, $constraint) = @_;

    my $param = {
        'order'             => $order,
        'opt'               => 'addhosts',
        'opt_arg'           => '<host1>[,<host2>]...',
        'type'              => 'string',
        'section'           => $section,
        'value'             => undef,
        'default'           => undef,
        'str'               => 'Additional Hosts',
        'desc'              => "Specifies additional hosts for $gProductNameSystem",
        'constraint'        => $constraint,
        'init_with_default' => 0,
        'set_interactive'   => 0,
        'interactive_str' => "Comma Separated Host Name",
    };

    if ($withHostRoleDesc) {
        $param->{additional_desc} =
              "Syntax of a host entry:\n"
            . "<host_name>:role=<role1>[:role=<role2>]...[:group=<name>][:storage_partition=<number>][:workergroup=<workergroup1>[:workergroup=<workergroup2>]...]\n"
            . $self->getHostRoleDescription();
    }

    return $param;
}

#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'InternalNetwork'.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.

sub getParamInternalNetwork {
    my ($self,
        $order,
        $section,
        $setInteractive, # default false
        $constraint,     # e.g. 'register only'
        $indexSelection, # default false
        $skip,           # default false
        $acceptNone,     # accept an address prefix or the string 'none'
       ) = @_;

    my $isInteractive = (defined $setInteractive) ? $setInteractive : 0;
    my $isIdxSelect  = (defined $indexSelection) ? $indexSelection : 0;
    my $isSkip       = (defined $skip)           ? $skip           : 0;
    my $optArg       = '<address>';
    $optArg         .= '|none' if ($acceptNone);

    return {'order'   => $order,
            'opt'     => 'internal_network',
            'type'    => 'string',
            'section' => $section,
            'value'   => undef,
            'default' => undef,
            'str'     => 'Internal Network Address',
            'desc'    => 'Internal subnet address in prefix notation (CIDR), e.g. 192.168.1.0/24',
            'opt_arg' => $optArg,
            'init_with_default'           => 0,
            'set_interactive'             => $isInteractive,
            'skip'                        => $isSkip,
            'constraint'                  => $constraint,
            'interactive_index_selection' => $isIdxSelect,
    };
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'listeninterface'.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.
#
# Possible values are:
#  - '.global':   all interfaces
#  - '.internal': all interfaces that are listed in [internal_hostname_resolution]
#  - '.local':    only local interfaces (e. g. 127.0.0.1)

sub getParamListenInterface{
    my ($self, $order, $section, $standardLayout, $constraint) = @_;

    my $param = {
        'order' => $order,
        'opt'               => 'listen_interface',
        'type'              => 'string',
        'section'           => $section,
        'value'             => undef,
        'default'           => undef,
        'valid_values'      => \@listenInterfaceValidValues,
        'ui_values'         => \@listenInterfaceUIValues,
        'str'               => 'Listen Interface',
        'desc'              => 'Change communication settings',
        'init_with_default' => 0,
        'set_interactive'   => 1,
        'constraint'        => $constraint,
        'interactive_index_selection' => 1,
    };

    if (!$standardLayout) {
        $param->{str} = 'Inter-Service Communication';
        $param->{console_text} = "Select Inter-Service Communication\n";
    }

    return $param;
}

sub getParamInternalHostnameResolution{
    my ($self, $order, $section) = @_;

    my $param = {
        'order'             => $order,
        'opt'               => 'internal_hostname_resolution',
        'type'              => 'string',
        'section'           => $section,
        'opt_arg'           => '<ip_address>=<hostname>[,<ip_address>=<hostname>]...',
        'value'             => undef,
        'default'           => undef,
        'str'               => 'Internal Hostname Resolution',
        'set_interactive'   => 0,
        'hidden'            => 1,
    };

    return $param;
}

#-------------------------------------------------------------------------------
# If the parameter '--add_roles' or '--remove_roles' is specified
# with remote host names, an instance of SDB::Install::RemoteHosts
# is returned containing these hosts.
# Otherwise undef is returned.

sub getRemoteHostsOfAddRemoveRoles {
    return $_[0]->{_remoteHostsOfAddRemoveRoles};
}


#-------------------------------------------------------------------------------
#
# get remote host device ip addresses
# collectHostInfo() required
#
# Parameter: string $host
#
# Returns array of SDB::Install::System::IPAddr (array ref)

sub getRemoteIPs{
    my ($self,$host) = @_;
    my $msglst = $self->getMsgLst ();
    my $found = 0;
    my @ipList;
    if (!defined $self->{remoteIPs} || !defined $self->{remoteIPs}->{$host}){
        return [];
    }
    my $remoteIp;
    foreach my $ip (@{$self->{remoteIPs}->{$host}}){
        $remoteIp = new SDB::Install::System::IPAddr ($ip);
        if (!defined $remoteIp){
            $msglst->addWarning ("Failed to parse remote ip address '$ip' ($host): " . Net::IP::Error());
            next;
        }
        if ($remoteIp->iptype () eq 'LOOPBACK'){
            next;
        }
        push @ipList, $remoteIp;
    }
    return \@ipList;
}


#-------------------------------------------------------------------------------
# If the parameter '--removehosts' is specified with the local host name,
# this local host name is returned. Otherwise undef is returned.

sub getRemovingLocalHost {
    return $_[0]->{_removingLocalHost};
}


#-------------------------------------------------------------------------------
# If the parameter '--removehosts' is specified with remote host names,
# an instance of SDB::Install::RemoteHosts is returned containing these hosts.
# Otherwise undef is returned.

sub getRemovingRemoteHosts{
    return $_[0]->{_removingRemoteHosts};
}


#-------------------------------------------------------------------------------
# Checks whether --system_is_offline can be used

sub checkSystemIsOffline{
    my ($self, $value) = @_;
    my $param = $self->{params}->{SystemIsOffline};
    if ($value){
        if (!isAdmin()){
             $self->appendErrorMessage ("--$param->{opt} requires admin privileges, please restart as root user!");
            return 0;
        }
        if ($self->isRemoteScope()){
            $self->appendErrorMessage ("--$param->{opt} requires scope=instance");
            return 0;
        }
        if (isAdmin()){
            $self->setSkip('Password');
        }
        $self->setSkip('srcPasswd');
    }
    return 1;
}


sub checkWorkerGroup {
    my ($self, $workerGroup) = @_;

    if($workerGroup =~ /^(master|all|slave)/){
        $self->getErrMsgLst()->addError("Value '$workerGroup' is invalid. Name cannot start with reserved words 'master', 'all' or 'slave'");
        return undef;
    }
    if($workerGroup !~ /^[a-z0-9_]{1,16}$/){
        $self->getErrMsgLst()->addError("Value '$workerGroup' is invalid. Name must consist of lower case letters, '_' and numbers (max length 16 symbols)");
        return undef;
    }
    return 1;
}

sub isHostKnownAsAlias{
    my ($self, $host, $existingRemoteHosts, $existingRemoteHostsByIP) = @_;
    my $nslookup;
    $existingRemoteHostsByIP //= {};
    if (!keys %$existingRemoteHostsByIP){
        foreach my $remoteHost (@$existingRemoteHosts){
            $nslookup = $self->isHostAccessible($remoteHost);
            if (!defined $nslookup || !$nslookup->{address}){
                next;
            }
            $existingRemoteHostsByIP->{$nslookup->{address}} = $remoteHost;
        }
    }
    $nslookup = $self->isHostAccessible($host);
    if (!defined $nslookup || !$nslookup->{address}){
        return undef;
    }
    if (defined $existingRemoteHostsByIP->{$nslookup->{address}}){
        $self->appendErrorMessage("Host '$host' is already known as '$existingRemoteHostsByIP->{$nslookup->{address}}'.");
        return 1;
    }
    $existingRemoteHostsByIP->{$nslookup->{address}} = $host;
    return 0;
}


#-------------------------------------------------------------------------------
# Checks the option 'addhosts', 'add_roles', 'removehosts' or 'remove_roles'.
#
# Additional hosts:
#   - 'getAddHostsParser'        returns SDB::Install::Configuration::AddHostsParser
#   - 'getAdditionalRemoteHosts' returns SDB::Install::RemoteHosts
#
# Hosts to be removed:
#   - 'getRemovingRemoteHosts'   returns SDB::Install::RemoteHosts
#   - 'getRemovingLocalHost'     returns the local hostname
#
# Additional roles:
#   - 'getAddHostsParser'              returns SDB::Install::Configuration::AddHostsParser
#   - 'getRemoteHostsOfAddRemoveRoles' returns SDB::Install::RemoteHosts
#   - 'getLocalHostOfAddRemoveRoles'   returns the local hostname
#
# Roles to be removed:
#   - 'getAddHostsParser'              returns SDB::Install::Configuration::AddHostsParser
#   - 'getRemoteHostsOfAddRemoveRoles' returns SDB::Install::RemoteHosts
#   - 'getLocalHostOfAddRemoveRoles'   returns the local hostname

sub handleAddOrRemoveHosts {
    my ($self,
        $specifiedHosts,      # e.g. --addhosts=<specifiedHostList>
        $paramID,             # valid values: 'AddHosts', 'RemoveHosts', 'AddRoles', 'RemoveRoles'
        $doNotFailOnLocalHost, # 1 - the list may contain a local host
        $doNotUseDefaultRoleValue # host parser will not give the default role value if no roles are given
       ) = @_;

    eval{
        require SDB::Install::RemoteHosts;
        if ($self->isUseSAPHostagent()) {
            require SDB::Install::RemoteHostctrlHosts;
        }
    };

    if ($@){
        my $errmsg = new SDB::Install::MsgLst();
        $errmsg->addError ($@);
        $self->appendErrorMessage ('Cannot establish remote execution',$errmsg);
        return undef;
    }

    my $activeRoles         = {};
    my $instance            = $self->getOwnInstance();
    my $existingRemoteHosts = undef;
    my $isAutoAddXS2        = 0;

    if (defined $instance) {  # instance is undef in case of installation

        $existingRemoteHosts = $instance->get_hosts();
        $existingRemoteHosts = undef if (!@$existingRemoteHosts);

        if (($paramID eq 'AddRoles')    ||
            ($paramID eq 'RemoveRoles') ||
            ($paramID eq 'RemoveHosts')) {

            $activeRoles = $instance->getActiveHostRoles();
            if (!defined $activeRoles) {
                $self->setErrorMessage(undef, $instance->getErrMsgLst());
                return undef;
            }
        }

        if ((($paramID eq 'AddHosts') || ($paramID eq 'AddRoles'))
            && $self->getValue('AutoAddXS2Roles')
            && $instance->existsXS2Dir()) {
            $isAutoAddXS2 = 1;
        }
    }

    require SDB::Install::Configuration::AddHostsParser;

    my $parser = new SDB::Install::Configuration::AddHostsParser ();
    if (!defined $parser->parse ($specifiedHosts, $paramID, $isAutoAddXS2,$doNotUseDefaultRoleValue)){
        $self->PushError (undef, $parser);
        return 0;
    }

    my $rc                  = 1;
    my @allRoles            = ();
    my @detectedRemoteHosts = ();
    my $detectedLocalHost   = undef;
    my %storage_partitions  = ('1' => 'Master host');
    my $existingRemoteHostsByIP = {};
    foreach my $host (@{$parser->getHosts()}) {

        if ($paramID eq 'AddHosts') {
            my $storage_partition = $parser->getStoragePartition($host);
            if (defined $storage_partition) {
                if (exists $storage_partitions{$storage_partition}) {
                    $self->PushError ("Storage partition '$storage_partition' of host"
                        . " '$host' is already assigned to host '$storage_partitions{$storage_partition}'");
                    $rc = 0;
                }
                else {
                    $storage_partitions{$storage_partition} = $host;
                }
            }
        }

        if (($paramID eq 'AddHosts') || ($paramID eq 'AddRoles')) {
            my $roles = $parser->getRoles($host);
            push @allRoles, @$roles if(defined $roles);
            for my $role(@{$roles}){
            	$rc = 0 if (!$self->checkRole($role,$host));
                if (($paramID eq 'AddRoles') && defined $activeRoles) {
                    my $roleHosts = $activeRoles->{$role};
                    if (defined $roleHosts && ($host ~~ @$roleHosts)) {
                        $self->PushError ("Role '$role' already exists on host '$host'");
                        $rc = 0;
                    }
                }
            }
            my $isAddingRoles = ($paramID eq 'AddRoles');
            my $aExistingRoles = defined($instance) && $isAddingRoles ? $instance->getHostRolesByIniFile($host) : undef;
            if (!$self->areNewRolesCompatibleWithExistingRoles($host, $roles, $aExistingRoles)) {
                return 0;
            }
        }
        if ($paramID eq 'AddHosts') {
            my $workerGroups = $parser->getWorkerGroups($host) || [];
            for my $group (@{$workerGroups}){
                $rc = 0 if(!$self->checkWorkerGroup($group));
            }
        }
        if (($paramID eq 'RemoveRoles') && defined $activeRoles) {
            my $roles = $parser->getRoles($host);
            for my $role (@$roles){
                my $roleHosts = $activeRoles->{$role};
                if (!defined $roleHosts || !($host ~~ @$roleHosts)) {
                    $self->PushError ("Role '$role' is not installed on host '$host'");
                    $rc = 0;
                }
            }
        }

        if ($host !~ /$hostname_regex/ and $host !~ /$ipv4_regex/){
            $self->PushError ("\'$host' is no valid host name");
            $rc =  0;
        }

        my $isLocal = 0;
        if (!defined $detectedLocalHost && (lc(SAPDB::Install::Hostname::hostname()) eq $host)) {
            $isLocal           = 1;
            $detectedLocalHost = $host;
        }
        else {
            my $errlst = new SDB::Install::MsgLst ();
            my $address;
            if ($host =~ /$ipv4_regex/){
                $address = $host;
            }
            else{
                my $nslookup = SDB::Install::System::nslookup ($host, $errlst);
                if (!defined $nslookup){
                    $self->setErrorMessage ("Error calling nslookup for host '$host'", $errlst);
                    $rc = 0;
                    next;
                }
                if (defined $nslookup->{address}){
                    $address = $nslookup->{address};
                }
                else{
                    my $info = (defined $nslookup->{msg}) ? ": $nslookup->{msg}"
                                                          : '';
                    $self->PushError ("Cannot resolve host name '$host'$info");
                    $rc = 0;
                    next;
                }
            }

            if (defined $address){
                $isLocal = SDB::Install::System::is_local_address ($address, $errlst);
                if (!defined $isLocal){
                    $self->PushError (undef, $errlst);
                    $rc = undef;
                }
                elsif ($isLocal) {
                    $detectedLocalHost = $host;
                }
            }
        }

        if ($isLocal){
            if($doNotFailOnLocalHost) {
                $self->AddMessage ("Hostname '$host' specifies the local machine.");
                next;
            }
            $self->PushError ("Hostname '$host' specifies the local machine.");
            $rc = 0;
        }
        else {
            push @detectedRemoteHosts, $host;
        }
    }

	if ($rc && (($paramID eq 'AddHosts') || ($paramID eq 'AddRoles'))) {
		$rc = 0 if (!$self->checkNewRolesCompatibility(\@allRoles));
	}

    if ($rc && @detectedRemoteHosts) {
        $existingRemoteHosts //= [];
        foreach my $host (@detectedRemoteHosts) {
            if ($host ~~ @$existingRemoteHosts) {
                if (($paramID eq 'AddHosts')
                    &&
                    (!defined $instance ||
                     !defined $self->getAddHostPendingStep($instance, $host))) {

                    $self->PushError
                        ("Host '$host' is already part of the $gProductNameSystem.");
                    $rc = 0;
                }
            }
            elsif (($paramID eq 'AddRoles')    ||
                   ($paramID eq 'RemoveRoles') ||
                   ($paramID eq 'RemoveHosts')) {
                $self->PushError
                    ("Host '$host' is not part of the $gProductNameSystem.");
                $rc = 0;
            }
            elsif ($paramID eq 'AddHosts'){
                if ($self->isHostKnownAsAlias($host,$existingRemoteHosts,$existingRemoteHostsByIP)){
                    $rc = 0;
                }
            }
        }
    }

    if ($rc && @detectedRemoteHosts) {

        my $addOrRemove = ($self->isUseSAPHostagent())
                          ? new SDB::Install::RemoteHostctrlHosts(@detectedRemoteHosts)
                          : new SDB::Install::RemoteHosts        (@detectedRemoteHosts);

        $addOrRemove->SetProgressHandler($self->GetProgressHandler());

        if ($paramID eq 'RemoveHosts') {
            $self->{_removingRemoteHosts} = $addOrRemove;
        }
        elsif (($paramID eq 'AddRoles') || ($paramID eq 'RemoveRoles')) {
            $self->{_remoteHostsOfAddRemoveRoles} = $addOrRemove;
        }
        else {
            # establish SSH connection to new hosts
            if (!$addOrRemove->isHostctrl() && ($addOrRemove->connect() != 0)) {
                $self->PushError (undef, $addOrRemove);
                $rc = 0;
            }
            $self->{_additionalRemoteHosts} = $addOrRemove;
        }
    }

    if ($rc && defined $detectedLocalHost) {
        if (($paramID eq 'AddRoles') || ($paramID eq 'RemoveRoles')) {
            $self->{_localHostOfAddRemoveRoles} = $detectedLocalHost;
        }
        elsif ($paramID eq 'RemoveHosts') {
            $self->{_removingLocalHost} = $detectedLocalHost;
        }
    }

    if ($rc && (defined $detectedLocalHost || @detectedRemoteHosts)
        &&
        (($paramID eq 'AddHosts') ||
         ($paramID eq 'AddRoles') ||
         ($paramID eq 'RemoveRoles'))) {
        $self->{_addHostsParser} = $parser;
    }

    if ($rc && @detectedRemoteHosts) {
        my $useSidadmCredentials = (($paramID eq 'AddRoles') || ($paramID eq 'RemoveRoles')) ? 1 : undef;
        $self->enableMultipleHostParameters($useSidadmCredentials);
    }

    if ($rc && (defined $detectedLocalHost || @detectedRemoteHosts)) {

        my $acceleratorFound = 0;

        if ($paramID eq 'RemoveHosts') {
            my $aseWorkerHosts = $activeRoles->{$gHostRoleAcceleratorWorker} || [];
            my $aseStandbyHosts = $activeRoles->{$gHostRoleAcceleratorStandby} || [];
            my @dbHosts = ( @{$activeRoles->{$gHostRoleWorker} // []}, @{$activeRoles->{$gHostRoleStandby} // []});
            my $isRemovingDbHost = 0;
            my @aseHosts = ( @{$aseWorkerHosts}, @{$aseStandbyHosts} );

            for my $host (@{$parser->getHosts()}){
                if($host ~~ @aseHosts){
                    $acceleratorFound = 1;
                }
                if($host ~~ @dbHosts){
                    $isRemovingDbHost = 1;
                }
            }

            my $shouldSkipLssParams = !defined $instance->getLssInstance() || !$isRemovingDbHost;
            $self->skipLSSUserParams($shouldSkipLssParams);

        }
        elsif ($parser->isAcceleratorFound()) {
            $acceleratorFound = 1;
            if (($paramID ne 'RemoveRoles')
                && exists $self->{params}->{AcceleratorPassword}) {
                $self->{params}->{AcceleratorPassword}->{type}='initial_passwd';
            }
        }

        if ($acceleratorFound) {
            $self->setSkip('SystemUser',          0);
            $self->setSkip('SQLSysPasswd',        0);
            $self->setSkip('AcceleratorUser',     0);
            $self->setSkip('AcceleratorPassword', 0);
        }
    }

    return $rc;
}


#-------------------------------------------------------------------------------
# This method initializes the default value of 'ListenInterface'
# and sets 'InternalNetwork' to mandatory if necessary.

sub handleInitListenInterfaceParam {
	my ($self) = @_;

	my $interfaceType = $self->getListenInterfaceType();
	$self->setDefault('ListenInterface', $interfaceType);
	if ($interfaceType eq 'local') {
		$self->setSkip('InternalNetwork');
	} else {
		$self->_setMandatoryInternalNetworkIfPossible();
	}
	return 1;
}


#-------------------------------------------------------------------------------
# Returns undef instead of a status file name.

sub pers_filename{
    return undef;
}


#-------------------------------------------------------------------------------
# Sets valid values of listen interface in case of multiple host system.

sub restrictListenInterfaceMultiHost {
    my ($self) = @_;
    my $param  = $self->{params}->{ListenInterface};
    $param->{valid_values} = \@listenInterfaceMultiHost;
    $param->{ui_values}    = \@listenInterfaceUIMultiHost;
    if ($self->getListenInterfaceType() eq 'local') {
        $self->setDefault('ListenInterface', 'global');
        $self->{params}->{'ListenInterface'}->{init_with_default} = 1;
        $self->setDefault('InternalNetwork', 'none');
        $self->updateInternalNetworkValidValues('global');
    }
}

sub restrictListenInterfaceSingleHost {
  my ($self) = @_;
  my $param  = $self->{params}->{ListenInterface};
  $param->{valid_values} = \@listenInterfaceValidValues;
  $param->{ui_values} = \@listenInterfaceUIValues;
}

#-------------------------------------------------------------------------------
# Sends 'hdbaddhost' commands to remote hosts specified by '--addhosts'.

sub sendAddHostCommands {

    my ($self, $msglst, $sapSys, $instance) = @_;

    my $addhostsParser = $self->getAddHostsParser();

    if (!defined $addhostsParser){
        return 1;  # add hosts not specified;
    }

    my @templateReplacements;
    my $hostMapOptionMap;
    my $rHosts = $self->getAdditionalRemoteHosts();

    my $progressHandler = $msglst->GetProgressHandler();
    my $additionalArgs;
    foreach my $hostname (@{$rHosts->getHostNames()}) {

        my $roles = $addhostsParser->getRoles($hostname);
        $additionalArgs = '';
        if (!defined $roles) {
            $msglst->AddError("Host role not defined for host '$hostname'");
            return undef;
        }
        my $roleList = join (',', @$roles);

        my @currTemplateRepl = ($hostname, $roleList);
        $hostMapOptionMap->{$hostname}->{HOSTNAME} = $hostname;
        $hostMapOptionMap->{$hostname}->{ROLES}    = $roleList;

        my $group = $addhostsParser->getGroup($hostname);
        if (defined $group) {
            $additionalArgs .= " --group=$group";
            $hostMapOptionMap->{$hostname}->{GROUP} = $group;
        }

        my $storagePartition = $addhostsParser->getStoragePartition($hostname);
        if (defined $storagePartition) {
            $additionalArgs .= sprintf(' --storage_partition=%d', $storagePartition);
            $hostMapOptionMap->{$hostname}->{'STORAGE_PARTITION'} =
                                                              $storagePartition;
        }

        my $workerGroups = $addhostsParser->getWorkerGroups($hostname);
        if (defined($workerGroups) && scalar(@{$workerGroups})) {
            my $workerGroupCSV = join(',', @{$workerGroups});
            $additionalArgs .= " --workergroup='$workerGroupCSV'";
            $hostMapOptionMap->{$hostname}->{'WG'} = $workerGroupCSV;
        }

        push @currTemplateRepl, $additionalArgs;

        push @templateReplacements, \@currTemplateRepl;

        if (defined $progressHandler){
            require SDB::Install::OutputHandler::AddHostOutHndlr;
            my $outputHandler = new SDB::Install::OutputHandler::AddHostOutHndlr (
                    $progressHandler, "  $hostname:  ");
            $rHosts->setOutputHandler ($hostname, $outputHandler);
        }
        if (defined $instance) {
            $instance->setNewRemoteHost($hostname);
        }
    }

    my $msg;
    my $numHosts = $rHosts->getNumberOfHosts();

    if ($numHosts == 1) {
        $msg = $msglst->AddProgressMessage ('Adding additional host...');
    }
    else {
        $msg = $msglst->AddProgressMessage
                (sprintf ('Adding %d additional hosts in parallel', $numHosts));
    }
    $rHosts->setMsgLstContext ([$msg->getSubMsgLst ()]);
    $rHosts->resetError();
    require SDB::Install::Configuration::AddHost;

    my $rc               = 0;
    my $addhostTool      = new SDB::Install::Configuration::AddHost();
    my $remoteOptIgnore  = $addhostTool->getOptionIgnore();
    my $remoteOptTimeout = $addhostTool->getOptionTimeout();
    my $progress_message = "Adding host '%s'...";
    my $done_message     = "Host '%s' added.";
    my $error_message    = "Adding host '%s' failed!";
    my $passwordKeys     = ['Password', 'SQLSysPasswd', 'SQLTenantUserPassword', 'AcceleratorPassword', 'OrgManagerPassword', 'LSSPassword'];
    my $actionID         = $self->{options}->{action_id};
    my $importContentXS2 = $self->getValue('ImportContentXS2');
    my $landscapeID      = (defined $instance) ? $instance->getLandscapeID()
                                               : undef;
    my $usesNonSharedStorage = $self->usesNonSharedStorage();

    if ($rHosts->isHostctrl) {

        my $paramIDs  = ['NoStart', 'SystemUser', 'TenantUser', 'AcceleratorUser', 'InstallHostagent', 'SkipModifySudoers', 'OrgManagerUser', 'AutoInitializeServices'];
        my $optionMap = {'SAPMNT' => $sapSys->get_target(),
                         'SID'    => $self->getSID()};
        $optionMap->{'LANDSCAPE_ID'} = $landscapeID if (defined $landscapeID);
        $optionMap->{'ACTION_ID'}    = $actionID    if (defined $actionID);
        if (defined $importContentXS2) {
            my $key            = $self->getHostctrlOpt('ImportContentXS2');
            $optionMap->{$key} = ($importContentXS2) ? 'on' : 'off';
        }
        if ($self->isSystemInCompatibilityMode($instance)) {
            $optionMap->{'ACM'} = 'on';
        }
        $optionMap->{'UNSV'} = 'true' if ($usesNonSharedStorage);

        $rc = $rHosts->executeHostctrlParallel($gOperationAddHost,
                                               $self,
                                               $paramIDs,
                                               $passwordKeys,
                                               $remoteOptIgnore,
                                               $remoteOptTimeout,
                                               $progress_message,
                                               $done_message,
                                               $error_message,
                                               $optionMap,
                                               $hostMapOptionMap);
    }
    else {
        # SSH connection
        my $hdbaddhost = join ($path_separator,
                               $sapSys->get_globalTrexInstallDir(),
                               'bin',
                               'hdbaddhost');

        my $acceleratorUser = $self->getValue('AcceleratorUser');
        my $systemUser      = $self->getValue('SystemUser');
        my $tenantUser      = $self->getValue('TenantUser');
        my $orgManagerUser  = $self->getValue('OrgManagerUser');
        my $autoInitalizeServices = $self->getValue('AutoInitializeServices');

        my $cmd = "\"$hdbaddhost\" -b "
                . '--hostname=%s --roles=%s%s';

        $cmd .= ' --install_hostagent' if ($self->getValue('InstallHostagent'));
        $cmd .= ' --nostart'           if ($self->getValue('NoStart'));
        $cmd .= " --landscape_id=$landscapeID" if (defined $landscapeID);
        $cmd .= " --action_id=$actionID"       if (defined $actionID);

        if (!$self->isSkipped('UseHttp') && $self->getValue('UseHttp')) {
            $cmd .= sprintf(' %s', $self->getOpt('UseHttp'));
        }

        if (defined $autoInitalizeServices) {
            $cmd .= ' ' . $self->getOpt('AutoInitializeServices') . '=' . $autoInitalizeServices;
        }

        if (defined $systemUser) {
            $cmd .= ' ' . $self->getOpt('SystemUser') . '=' . $systemUser;
        }

        if (defined $tenantUser) {
            $cmd .= ' ' . $self->getOpt('TenantUser') . '=' . $tenantUser;
        }

        if (defined $orgManagerUser) {
            $cmd .= ' ' . $self->getOpt('OrgManagerUser') . '=' . $orgManagerUser;
        }

        if (defined $acceleratorUser) {
            $cmd .= ' ' . $self->getOpt('AcceleratorUser')
                 .  '='   . $acceleratorUser;
        }

        if (defined $importContentXS2) {
            $cmd .= ' ' . $self->getOpt('ImportContentXS2')
                 .  '=' . ($importContentXS2 ? 'on' : 'off');
        }
        if ($self->hasValue('SkipModifySudoers')) {
            my $value = $self->getValue('SkipModifySudoers');
            $cmd .= ' ' . $self->getOpt('SkipModifySudoers') . '=' . ($value ? 'on' : 'off');
        }
        if ($self->isSystemInCompatibilityMode($instance)) {
            $cmd .= ' --addhost_compatiblity_mode=1';
        }

        $cmd .= " --use_non_shared_storage=true" if ($usesNonSharedStorage);

        $cmd .= $self->getIgnoreAndTimeoutOptions($remoteOptIgnore,
                                                  $remoteOptTimeout);

        push @$passwordKeys, 'HostagentPassword';
        my $xml = $self->getXmlPasswordStream($passwordKeys);
        $cmd .= ' --read_password_from_stdin=xml ' if (defined $xml);

        if (defined $ENV{ASAN_OPTIONS})
        {
            # ASan install, pass on environment variables to remote installer
            $cmd = "ASAN_OPTIONS=$ENV{ASAN_OPTIONS} bash -c '$cmd'";
        }

        $rc = $rHosts->executeParallel($cmd,
                                       $progress_message,
                                       $done_message,
                                       $error_message,
                                       $xml,
                                       \@templateReplacements);
    }

    if ($rc != 0) {
        $msglst->AddError ('Creating a multiple host system failed', $rHosts);
        return undef;
    }

    $self->resetSAPSystemHashes();
    delete $self->{ownInstance};
    $instance = $self->getOwnInstance();

    $rc = 1; # Perl return code

    if (!defined $instance) {
        $msg->getSubMsgLst->addMessage
                      ('Added hosts cannot be checked (instance is undefined)');
    }
    else {
        foreach my $hostName (sort @{$rHosts->getHostNames()}) {

            my $persFilename = $self->getAddHostPersFile($instance, $hostName);
            my $addhostData  = $self->pers_load($persFilename);

            if (defined $addhostData && defined $addhostData->{InstallDate}) {

                my $instDate = $addhostData->{InstallDate};
                $instDate    = substr($instDate, 0, 10) . ' '  # YYYY-MM-DD
                             . substr($instDate, 11, 8);       # hh:mm:ss

                $msg->getSubMsgLst->addMessage("Host '$hostName' ("
                                               . $addhostData->{HostRoles}
                                               . ") installed at $instDate");
            }
            else {
                $msglst->AddError('Creating a multiple host system failed: '
                                  . "Host '$hostName' not added");
                $rc = undef;
            }
        }
    }

    return $rc;
}


#-------------------------------------------------------------------------------
# Sets the parameter '--addhosts'.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.

sub setAddHosts {

    my ($self, $value) = @_;

    my $rc = $self->checkAddHosts($value);

    if ($rc) {
        $self->{params}->{AddHosts}->{value} = $value;
    }
    return $rc;
}


#-------------------------------------------------------------------------------
# Sets 'InternalNetwork' to mandatory if necessary.

sub _setMandatoryInternalNetworkIfPossible {
    my ($self) = @_;

    my $param           = $self->{params}->{InternalNetwork};

# The InternalNetwork paramter has valid_values only
# in the context of hdblcm. hdbmodify doesn't specify this property.
# If hdbmodify is ever changed to have it then this condition
# should be updated.
    my $hasValidValues = (exists $param->{valid_values} && scalar @{$param->{valid_values}}) ? 1 : 0;
    return 1 if (!$hasValidValues);

    my $internalNetwork = $self->getOwnInstance()->getInternalNetworkPrefix();
    my $listenInterface = $self->getValue('ListenInterface');
    $param->{mandatory} = (defined $listenInterface && $listenInterface eq 'internal');
    $self->setDefault('InternalNetwork', $internalNetwork) if defined $internalNetwork;
    $param->{set_interactive} = 1;
    $self->setSkip('InternalNetwork', 0);

    return 1;
}


sub validateListenInterface {
	my ($self, $listenInterface, $internalNetworkPrefix, $errMsg1, $errMsg1Append1, $errMsg1Append2, $errMsg2) = @_;

	if (!defined $listenInterface) {
		return undef;
	}

	if ($listenInterface eq '.local'){
		$self->setErrorMessage ($errMsg1) if defined $errMsg1;
		$self->appendErrorMessage ($errMsg1Append1) if defined $errMsg1Append1;
		$self->appendErrorMessage ($errMsg1Append2) if defined $errMsg1Append2;
		return 0;
	}

	if (defined $internalNetworkPrefix){
		my $prefix = $self->getIPPrefix ($internalNetworkPrefix);
		if (!defined $prefix){
			return 0;
		}
		my $ipList = $self->getLocalIPs ();
		my $internalIp = $prefix->findSubnetDeviceIps ($ipList)->[0];
		if (defined $internalIp) {
			$self->{internalLocalIp} = $internalIp->ip ();
		} else {
			$self->setErrorMessage ("No matching local network interface found (prefix = ".$prefix->print.")");
			return 0;
		}
	} elsif ($listenInterface eq '.internal') {
		$self->setErrorMessage ($errMsg2) if defined $errMsg2;
		return 0;
	}

	return 1;
}

sub validateSidBasedOnRemoteSidadms {
	my ($self, $value) = @_;
	my ($remoteHost, $remoteUid, $remoteGid);
	my $user = new SDB::Install::NewDBUser ($value);
	my $sidadm = $user->getSidAdmName();
	$self->{username} = $sidadm;
	my (%rUids,%rGids);
    my $localUserUid = $user->uid();
    my $localUserGid = $user->userGid();

# Check local sidadm
  if (!$isWin && $user->exists()){
    my $errlst = new SDB::Install::MsgLst();
    if (!$self->checkUID ($localUserUid, $errlst)){
      $self->_PushError ("Error checking uid of already existing sidadm user '$sidadm'", $errlst);
      return 0;
    }
    if (! $user->gid() || $localUserGid != $user->gid()) { # $user->gid() returns sapsys gid
      $self->_PushError ("Existing user '$sidadm' on the local host does not have '$gSapsysGroupName' configured as primary group.");
      return 0;
    }
  }

  if (! defined $self->{remoteSidadms} || ! length(keys(%{$self->{remoteSidadms}}))){
    return 1;
  }

# Check remote sidadms
	if (!$isWin && $user->exists()){
		$rUids{$localUserUid} = ['local host'];
		$rGids{$localUserGid} = ['local host'];

		for my $remoteSidAdm ( keys %{$self->{remoteSidadms}} ) {
			($remoteHost, $remoteUid, $remoteGid) = @{@{ $self->{remoteSidadms}->{$remoteSidAdm}}[0]};
			if ( $remoteSidAdm ne $sidadm && $remoteUid == $user->uid() ) {
				$self->_PushError ("Existing user '$sidadm' id '$remoteUid' is assigned to different user ('$remoteSidAdm') on host '$remoteHost'");
				return 0;
			}
		}
	}

	my $sapSys = $self->getSAPSystem();
	my $userConfig = defined($sapSys) ? $sapSys->getUserConfig() : undef;
	if (exists($self->{sapsysids}) && defined($userConfig)) {
		my $expectedGID = $userConfig->{sapsys_groupid};

		for my $remoteGID (keys(%{$self->{sapsysids}})){
			next if($remoteGID == $expectedGID);

			my @remoteHosts = @{$self->{sapsysids}->{$remoteGID}};
			my $remoteHostsString = (scalar(@remoteHosts) > 1 ? 'hosts ' : 'host ') .  join(', ', @remoteHosts);
			$self->_PushError("Existing group '$gSapsysGroupName' has wrong GID on remote $remoteHostsString (expected '$expectedGID', found '$remoteGID')");
			return 0;
		}
	}
	if (exists($self->{remoteSidadms}->{$sidadm})) {			# if-statment to prevent auto-vivification
		for my $remoteHostInformation (@{$self->{remoteSidadms}->{$sidadm}}){
				($remoteHost, $remoteUid, $remoteGid) = @$remoteHostInformation;
				if (!exists($self->{sapsysids}->{$remoteGid})) {
					$self->_PushError("Existing user '$sidadm' on remote host '$remoteHost' does not have '$gSapsysGroupName' configured as primary group.");
					return 0;
				}
                if (!$self->checkUidMatchesExistingSidadmUid($remoteUid)) {
                    my $validUid = $userConfig->{sidadm_id};
                    $self->_PushError("Existing user '$sidadm' with uid '$remoteUid' on remote host '$remoteHost' " .
                                      "does not match the uid '$validUid' of the $gProductNameSystem '$sidadm' user.");
                    return 0;
                }
				(defined $rUids{$remoteUid}) ? (push @{$rUids{$remoteUid}}, $remoteHost) : ($rUids{$remoteUid} = [$remoteHost]);
				(defined $rGids{$remoteGid}) ? (push @{$rGids{$remoteGid}}, $remoteHost) : ($rGids{$remoteGid} = [$remoteHost]);
			}
	}
# If no sidadm user exists on local host check if the
# system's sidadm UID is not taken on the remote hosts
    if (exists($self->{remoteUids})) {
        my $sidadmUid = $self->getSystemSidadmUid();
        my $remoteUsersWithSidadm = defined ($sidadmUid) && defined ($self->{remoteUids}->{$sidadmUid})
                                    ? $self->{remoteUids}->{$sidadmUid}
                                    : [];
        for my $remoteHostEntry (@$remoteUsersWithSidadm) {
            my $remoteHost = $remoteHostEntry->[0];
            my $remoteUserWithSidadmUid = $remoteHostEntry->[1];
            if ($remoteUserWithSidadmUid ne $sidadm) {
                $self->_PushError("The id '$sidadmUid' of the system's '$sidadm' user is assigned to different user ('$remoteUserWithSidadmUid') on host '$remoteHost'");
                return 0;
            }
        }
    }

	if (scalar (keys (%rUids)) > 1){
		$self->_pushUserPropertyMismatchError(\%rUids, 1); #isCheckingUid=1
    return 0;
	}

	if (scalar (keys (%rGids)) > 1){
		$self->_pushUserPropertyMismatchError(\%rGids);
		return 0;
	}

	if (scalar (keys (%rUids))){
		my $uid = (keys (%rUids))[0];
		my $errlst = new SDB::Install::MsgLst ();
		if (!$self->setValue ('UID',$uid, $errlst)){
			$self->_PushError ("User '$sidadm' already exists on host(s) ".join (', ', @{$rUids{$uid}}). " with uid '$uid'");
			$self->_PushError ("Cannot create user '$sidadm' with uid '$uid' on local host", $errlst);
			return 0;
		}
	}

	return 1;
}

sub checkUidMatchesExistingSidadmUid {
    my ($self, $uid) = @_;
    my $userCfg = (defined $self->{sapSys}) ? $self->{sapSys}->getUserConfig() : undef;
    return 1 if (!defined $userCfg);
    return $uid eq $userCfg->{sidadm_id};
}

# Copied from LCM::Utils::CommonUtils so that
# SDB code wouldn't rely on LCM code
sub getSystemSidadmUid {
    my ($self) = @_;
    my $sapSystem = $self->getSAPSystem();
    my $userConfigurationFile = (defined $sapSystem) ? $sapSystem->getUserConfig() : undef;
    if (!defined $userConfigurationFile){
        return undef;
    }
    return $userConfigurationFile->{sidadm_id};
}

sub _pushUserPropertyMismatchError {
  my ($self, $propertyHash, $isCheckingUid) = @_;
  my $sidadm = $self->{username};
  my $idPropertyString = ($isCheckingUid) ? "UID" : "'sapsys' GID";

  my $simplerError = $self->_getSimplerErrorMessage($propertyHash, $isCheckingUid);
  my $errorMessage = $simplerError || "Existing user '$sidadm' has different ${idPropertyString}s on several hosts.";
  $self->_PushError ("$errorMessage The $idPropertyString must be the same on all hosts.");
  if (!$simplerError || scalar(keys(%{$propertyHash})) > 2) {
    for my $id (keys (%{$propertyHash})){
      $self->_PushError ("$idPropertyString is '$id' on host(s) ".join (', ', @{$propertyHash->{$id}}) . ".");
    }
  }
}

# Returns a simpler error message in the case
# where only one host's sidadm user properties
# differ from the other on which the user exists.
# Otherwise it returnds 'undef'.
# It can check for both UID and GID differences.
sub _getSimplerErrorMessage {
	my ($self, $propertyHash, $isCheckingUid) = @_;

  my $userCfg = (defined $self->{sapSys}) ? $self->{sapSys}->getUserConfig()
                                          : undef;
  my $numberOfDetectedIds = scalar(keys(%{$propertyHash}));
  my $sidadm = $self->{username};
  my $idPropertyString = ($isCheckingUid) ? "UID" : "'sapsys' GID";

  if (!defined $userCfg && $numberOfDetectedIds == 2){
    my $idMatchingOnlyOneHost = undef;
    my $idMatchingGroupOfHosts = undef;
    for my $id (keys (%{$propertyHash})){
      $idMatchingOnlyOneHost = $id if (scalar(@{$propertyHash->{$id}}) == 1);
      $idMatchingGroupOfHosts = $id if (scalar(@{$propertyHash->{$id}}) > 1);
    }
    if (defined $idMatchingOnlyOneHost && $idMatchingGroupOfHosts) {
      my @singleHost = @{$propertyHash->{$idMatchingOnlyOneHost}};
      my @groupOfHosts = @{$propertyHash->{$idMatchingGroupOfHosts}};
      my $err = "Existing user '$sidadm' on host @singleHost has a different " .
                "$idPropertyString ($idMatchingOnlyOneHost) compared to " .
                "the $idPropertyString ($idMatchingGroupOfHosts) on " .
                "all other hosts.";
      return $err;
    }
  }

  if (defined $userCfg) {
    my $sapsysGid = $userCfg->{sapsys_groupid};
    my $sidadmUid = $userCfg->{sidadm_id};
    my $comparedProperty = ($isCheckingUid) ? $sidadmUid : $sapsysGid;
    my $hostsWithCorrectUserProperty = $propertyHash->{$comparedProperty};
    my $hostsWithWrongUserProperty = [];
    return undef if (!$hostsWithCorrectUserProperty || !scalar(@$hostsWithCorrectUserProperty));

    my $lastMismatchingId = undef;
    for my $id (keys(%{$propertyHash})) {
      next if ($id == $sapsysGid || $id == $sidadmUid);
      $lastMismatchingId = $id;
      push (@$hostsWithWrongUserProperty, @{$propertyHash->{$id}});
    }
    my $showSingleMismatchingId = '';
    $showSingleMismatchingId = " ($lastMismatchingId)" if ($numberOfDetectedIds == 2);
    my $err = "Existing user '$sidadm' on host(s) " . join(', ', @$hostsWithWrongUserProperty) .
              " has a different $idPropertyString$showSingleMismatchingId compared to the $idPropertyString ($comparedProperty)" .
              " on the $gProductNameSystem hosts.";
    return $err;
  }

	return undef;
}

# Avoids having "host local host" in error messages
# Can't use the hostname() method instead because
# of possible virtual names of the HANA host
sub _PushError {
  my ($self, $errorString, $errorSublist) = @_;
  if ($errorString =~ s/host local host/the local host/g) {
    $self->PushError($errorString, $errorSublist);
  }
  elsif ($errorString =~ s/host\(s\) local host/the local host/g) {
    $self->PushError($errorString, $errorSublist);
  }
  elsif ($errorString =~ s/local host,//g) {
    chop($errorString) if (substr($errorString, -1) eq '.');
    $errorString .= " and on the local host.";
    $self->PushError($errorString, $errorSublist);
  } else {
    $self->PushError($errorString, $errorSublist);
  }
}

#-------------------------------------------------------------------------------
# If a sapsys group ID is colelcted or exists, this group id is assigned
# to the value of the parameter 'GID', otherwise a default value is assigned.

sub tryInitGroupAndUserID {

    # no UID or GID handling required on Windows
    return 1 if ($isWin);

    my ($self) = @_;

    my $sapsysids = $self->{sapsysids};
    if (getgrnam($gSapsysGroupName)) {
        my $localSapsysGid = getgrnam($gSapsysGroupName);
        $self->setDefault('GID', $localSapsysGid) if ($self->checkGID($localSapsysGid));
    } elsif (defined $self->{sapsysids} && (scalar(keys %$sapsysids) == 1)) {
        my $sapsysGid = (keys %$sapsysids)[0];
        $self->setDefault('GID', $sapsysGid) if ($self->checkGID($sapsysGid));
    }

    if (exists $self->{params}->{GID} && !defined $self->getBatchValue('GID')) {


        if (defined $sapsysids && (scalar (keys %$sapsysids) == 1)) {
            $self->{params}->{GID}->{value} = (keys %$sapsysids)[0];
        }
        else {
            # usercfg_sapsys_groupid is set by tryHandleExistingGroupAndUser
            # in case of register/rename
            $self->setDefault('GID', $self->getFreeGroupID($self->{usercfg_sapsys_groupid}, $self->getIDsNotApplicableForSapsysGroup()));
        }
    }

    if (exists $self->{params}->{UID} && !defined $self->getBatchValue('UID')){

        if (defined $self->{existingSidAdm}) {
            my $existingUID = $self->{existingSidAdm}->{uid};
            my $userName    = getpwuid($existingUID);
            if (defined $userName) {
                $self->PushError($self->{params}->{UID}->{str}
                    . " '$existingUID' is already in use by user '$userName'");
                return undef;
            }
            $self->{params}->{UID}->{value} = $self->{existingSidAdm}->{uid};
        }
        else {
            # usercfg_sidadm_id is set by tryHandleExistingGroupAndUser
            # in case of register/rename
            $self->setDefault('UID', $self->getFreeUserID($self->{usercfg_sidadm_id}, $self->getIDsNotApplicableForSidadm()));
        }
    }
    return 1;
}

sub getIDsNotApplicableForSidadm {
    my ($self) = @_;

    my @reservedIDs = ();
    my $lssUserIDBatchValue = $self->getBatchValue('LSSUserID');
    push @reservedIDs, $lssUserIDBatchValue if(defined $lssUserIDBatchValue);
    return \@reservedIDs;
}

sub getIDsNotApplicableForSapsysGroup {
    my ($self) = @_;

    my @reservedIDs = ();
    my $lssGroupIDBatchValue = $self->getBatchValue('LSSGroupID');
    push @reservedIDs, $lssGroupIDBatchValue if(defined $lssGroupIDBatchValue);
    return \@reservedIDs;
}

#-------------------------------------------------------------------------------
# Tries to create the data and log directories of the accelerator
# (with the parent directories) and changes the owner of the created data and
# log directories to sidadm.

sub tryCreateAcceleratorPaths {

    my ($self, $sapsys, $instance, $msglst) = @_;

    my $pathArray = [$instance->getAcceleratorDataPath(1),
                     $instance->getAcceleratorLogPath (1)];

    return $self->tryCreatePathsOfSpecialHosts($sapsys,
                                               $instance,
                                               $pathArray,
                                               $msglst);
}


#-------------------------------------------------------------------------------
# Tries to create the data and log directories of extended storage (with the
# parent directories) and changes the owner of the created data and log
# directories to sidadm.

sub tryCreateExtendedStoragePaths {

    my ($self, $sapsys, $instance, $msglst) = @_;

    my $pathArray = [$instance->getEsDataPath(1),
                     $instance->getEsLogPath (1)];

    return $self->tryCreatePathsOfSpecialHosts($sapsys,
                                               $instance,
                                               $pathArray,
                                               $msglst);
}


#-------------------------------------------------------------------------------
# Tries to create the specified directories (with the parent directories, if not
# exist before) and changes the owner of the created directories (without parent
# directories) to sidadm.

sub tryCreatePathsOfSpecialHosts {

    my ($self, $sapsys, $instance, $pathArray, $msglst) = @_;

    if (!defined $msglst){
        $msglst = $self;
    }

    for my $currDir (@$pathArray) {

        my $errlst = new SDB::Install::MsgLst();

        if (defined $currDir && !-d $currDir) {

            # creates the parent directories and the current directory
            # (e.g. '/hana', '/hana/data_es', '/hana/data_es/<SID>')

            my $msg = $msglst->AddMessage("Creating directory '$currDir'");
            # uid/gid is undef to avoid chown of parent directories
            if (!LCM::FileUtils::createDirectory($currDir,
                                                 0755,
                                                 undef, # uid
                                                 undef, # gid
                                                 1,     # isRecursive
                                                 $errlst)) {
                my $errorMessage = $errlst->getMsgLstString();
                chomp($$errorMessage);
                $msglst->getMsgLst()->AddWarning($$errorMessage);
            }
        }

        if (defined $currDir && -d $currDir) {

            my $parentDir = dirname($currDir);
            my $msg = $msglst->AddMessage("Changing owner of '$parentDir'");
            if (defined changeOwn(undef,           # old uid filter
                                   -1,
                                   undef,           # old gid filter
                                   $sapsys->getGID,
                                   $parentDir,
                                   $errlst,
                                   $msg->getSubMsgLst(),
                                   SDB::Install::SAPSystem::SKIP_CHOWN_DIR())) {
                $msg = $msglst->AddMessage("Changing permissions of '$parentDir'");
                if (!defined LCM::FileUtils::changeFilePermissions($parentDir, 01775, $errlst)) {
                    $msglst->AddWarning("Cannot chmod '$parentDir'", $errlst);
                }
            } else {
                $msglst->AddWarning("Cannot chown '$parentDir'", $errlst);
            }

            # changes the owner of the current directory to sidadm
            # (e.g. '/hana/data_es/<SID>')

            $msg = $msglst->AddMessage("Changing owner of '$currDir'");
            if (!defined changeOwn(undef,           # old uid filter
                                   $sapsys->getUID,
                                   undef,           # old gid filter
                                   $sapsys->getGID,
                                   $currDir,
                                   $errlst,
                                   $msg->getSubMsgLst(),
                                   SDB::Install::SAPSystem::SKIP_CHOWN_DIR())) {
                $msglst->AddError("Cannot chown '$currDir'", $errlst);
                return undef;
            }
        }
    }
    return 1;
}


#-------------------------------------------------------------------------------
# this subroutine may be overridden
sub checkNewRolesCompatibility {
    return 1;
}


#-------------------------------------------------------------------------------
# this subroutine may be overridden
sub checkRole {
    my ($self, $hostRole) = @_;
    if(! $self->isSpecialComponentInstalled($hostRole)) {
        $self->appendErrorMessage("Role '$hostRole' is invalid: required component is not installed");
        return 0;
    }
    if($hostRole eq $gHostRoleXS2Worker || $hostRole eq $gHostRoleXS2Standby){
        return $self->_checkXSSpaceIsolationParameters();
    }

    if ($hostRole eq $gHostRoleStandby && !$self->isBasePathShared()) {
        $self->getErrMsgLst()->addError("Role '$gHostRoleStandby' is not supported when using non-shared data/log volumes");
        return 0;
    }
    return 1;
}

sub isBasePathShared {
    my ($self) = @_;
    my $instance = $self->getOwnInstance();
    return (defined $instance) ? $instance->isBasePathShared() : $self->isBasePathSharedFromCustomGlobalIni() // 1;
}

sub isBasePathSharedFromCustomGlobalIni {
    my $self = shift;
    my $customCfgDir = $self->getBatchValue('CustomConfigDir');
    if (defined $customCfgDir && $self->checkCustomConfigDir($customCfgDir)) {
        my $filepath = File::Spec->catfile($customCfgDir, 'global.ini');
        my $iniFile = SDB::Install::IniFile->new($filepath);
        return $iniFile->getBoolValue('persistence', 'basepath_shared');
    }
    return undef;
}

sub _checkXSSpaceIsolationParameters {
    my ($self) = @_;
    my $userConfig = $self->getSAPSystem()->getUserConfig();
    my $keys = ['xs_space_user_sap', 'xs_space_user_prod'];


    for my $key (@{$keys}){
        my ($userName, $uid) = ($userConfig->{$key}, $userConfig->{"${key}_id"});
        my $errorMessagePrefix = "Checking XS Advanced OS user '$userName' failed";

        next if(!defined($userName) || !defined($uid));

        my $user = new SDB::Install::User($userName);
        my $existingUser = getpwuid($uid);
        if(defined($existingUser) && $existingUser ne $userName){
            $self->getErrMsgLst()->AddError("$errorMessagePrefix. User ID '$uid' is already in use by user '$existingUser'");
            return undef;
        }

        next if(!$user->exists());

        if($user->id() ne $uid){
            my $actualUid = $user->id();
            $self->getErrMsgLst()->AddError("$errorMessagePrefix. Existing user has wrong ID '$actualUid' (expected '$uid')");
            return undef;
        }
        if($key eq 'xs_space_user_prod' && getgrgid($user->gid()) eq 'sapsys'){
            $self->getErrMsgLst()->AddError("$errorMessagePrefix. Existing user must not belong to group 'sapsys'");
            return undef;
        }
    }
    return 1;
}

#-------------------------------------------------------------------------------
# Checks if the host role is a special role and returns 0 if
# the compontent of the special role is not installed.

sub isSpecialComponentInstalled {

    my ($self, $hostRole) = @_;

    if ($validHostRoles->{$hostRole}->{isColumnStore} || $validHostRoles->{$hostRole}->{isDbRole}) {
        return 1;
    }

    my $hdbInstance  = $self->getOwnInstance();
    return 0 if (!defined $hdbInstance);

    my $componentDir = undef;
    if ($validHostRoles->{$hostRole}->{isStreaming}) {
        $componentDir = $hdbInstance->getStreamingLcmDir();
    }
    elsif ($validHostRoles->{$hostRole}->{isExtendedStorage}) {
        $componentDir = $hdbInstance->getExtendedStorageLcmDir();
    }
    elsif ($validHostRoles->{$hostRole}->{isAccelerator}) {
        $componentDir = $hdbInstance->getAcceleratorLcmDir();
    }
    elsif ($validHostRoles->{$hostRole}->{isXS2}) {
        $componentDir = $hdbInstance->getXS2Dir();
    }
    return (defined $componentDir && -d $componentDir) ? 1 : 0;
}


sub areNewRolesCompatibleWithExistingRoles {
	my ($self, $host, $aRoles, $aExistingHostRoles) = @_;

	my %hNewWorkerAndStandbyRoles = %{$self->_getWorkerAndStandbyRolesFromRolesArray($aRoles)};
	my %hExistingWorkerAndStandbyRoles = %{$self->_getWorkerAndStandbyRolesFromRolesArray($aExistingHostRoles)};

	my $numberOfNewStandbyRoles = scalar @{$hNewWorkerAndStandbyRoles{'standby'}};
	my $numberOfNewWorkerRoles = scalar @{$hNewWorkerAndStandbyRoles{'worker'}};

	my $numberOfExistingStandbyRoles = scalar @{$hExistingWorkerAndStandbyRoles{'standby'}};
	my $numberOfExistingWorkerRoles = scalar @{$hExistingWorkerAndStandbyRoles{'worker'}};

	if ( $numberOfNewStandbyRoles > 0 && $numberOfNewWorkerRoles > 0 ) {
		$self->getErrMsgLst()->addError( $self->_createErrorMessage($hNewWorkerAndStandbyRoles{'standby'}, $hNewWorkerAndStandbyRoles{'worker'}) );
		return 0;
	}

	if ( $numberOfNewStandbyRoles > 0 && $numberOfExistingWorkerRoles > 0 ) {
		$self->getErrMsgLst()->addError( $self->_createErrorMessage($hNewWorkerAndStandbyRoles{'standby'}, $hExistingWorkerAndStandbyRoles{'worker'}) );
		return 0;
	}

	if ( $numberOfNewWorkerRoles > 0 && $numberOfExistingStandbyRoles > 0 ) {
		$self->getErrMsgLst()->addError( $self->_createErrorMessage($hNewWorkerAndStandbyRoles{'worker'}, $hExistingWorkerAndStandbyRoles{'standby'}) );
		return 0;
	}

	return 1;
}

sub _getWorkerAndStandbyRolesFromRolesArray {
	my ($self, $aRoles) = @_;
	my %hWorkerAndStandbyHosts = ('standby'=>[], 'worker'=>[]);
	for my $role (@$aRoles) {
		next if !exists(${$validHostRoles}{$role});
		if ( $validHostRoles->{$role}->{isStandby} ) {
			push @{$hWorkerAndStandbyHosts{'standby'}}, $role;
		} else{
			push @{$hWorkerAndStandbyHosts{'worker'}}, $role;
		}
	}
	return \%hWorkerAndStandbyHosts;
}

sub _createErrorMessage {
	my ($self, $firstArray, $secondArray) = @_;
	my @firstArrayRolesString = map {$validHostRoles->{$_}->{str}} @$firstArray;
	my $firstArrayRoles = join(',', @firstArrayRolesString);

	my @secondArrayRolesString = map {$validHostRoles->{$_}->{str}} @$secondArray;
	my $secondArrayRoles = join(',', @secondArrayRolesString);

	my $firstArrayTypeMessage = scalar @$firstArray > 1 ? "roles '%s' are" : "role '%s' is";
	my $secondArrayTypeMessage = scalar @$secondArray > 1 ? "roles '%s'" : "role '%s'";
	return "New " . sprintf($firstArrayTypeMessage, $firstArrayRoles) . " not compatible with " . sprintf($secondArrayTypeMessage, $secondArrayRoles);
}

#-------------------------------------------------------------------------------

sub loadInitialContentXS2 {

    my ($self) = @_;

    my $instance              = $self->getOwnInstance();
    my $command = File::Spec->catfile($instance->getXS2Dir(), 'installation-scripts', 'installation', 'load-initial-content');
    $command .= '.exe' if ($isWin);
    if (!defined File::stat::stat($command)) {
        $self->setErrorMessage("'$command' not found");
        return undef;
    }
    my $actionMsg  = "Loading initial content of $gShortProductNameXS2";
    my $msg = $self->getMsgLst()->addProgressMessage($actionMsg);

    my $args        = $self->_buildLoadInitialContentArgs();
    my $inputBuffer = $self->_buildLoadInitialContentInputBuffer();

    require SDB::Install::OutputHandler::XS2LoadContentOutHndler;
    if (!$instance->runUtilityInEnv($command,
                                    $args,
                                    $msg->getSubMsgLst(),
                                    undef, # ref_output_buffer
                                    $inputBuffer, # ref_input_buffer
                                    undef, # additionalEnvironment
                                    new SDB::Install::OutputHandler::XS2LoadContentOutHndler(undef, '  '))) {
        $self->setErrorMessage("$actionMsg failed", $msg->getSubMsgLst());
        return undef;
    }
    return 1;
}

sub _buildLoadInitialContentArgs {
  my ($self) = @_;
  my $systemUser = $self->getValue('SystemUser') || 'SYSTEM';
  my $orgManagerUser = $self->getValue('OrgManagerUser');

  my @args = ();
  push(@args, "$gXSASystemUserParamId=$systemUser");
  push(@args, "$gXSASystemUserPwdParamId=__STDIN__");
  push(@args, "$gXSAOrgManagerUserParamId=$orgManagerUser");
  push(@args, "$gXSAOrgManagerPwdParamId=__STDIN__");
  if($self->isSystemInCompatibilityMode()){
      my $tenantUser =  $self->getValue('TenantUser') || 'SYSTEM';
      push(@args, "$gXSAProdSpaceDBSystemUserParamId=$tenantUser");
      push(@args, "$gXSAProdSpaceDBSystemUserPwdParamId=__STDIN__");
  }

  return \@args;
}

sub _buildLoadInitialContentInputBuffer {
  my ($self) = @_;
  my $systemUserPasswd = $self->getValue('SQLSysPasswd');
  my $orgManagerPasswd = $self->getValue('OrgManagerPassword');
  my $passwords = "$gXSASystemUserPwdParamId=$systemUserPasswd\n$gXSAOrgManagerPwdParamId=$orgManagerPasswd";
  if($self->isSystemInCompatibilityMode()){
      my $tenantUserPasswd = $self->getValue('SQLTenantUserPassword');
      $passwords=$passwords."\n$gXSAProdSpaceDBSystemUserPwdParamId=$tenantUserPasswd";
  }
  return [$passwords];
}


#-------------------------------------------------------------------------------
# Returns the host where the xs2 controller service is defined

sub getXS2ControllerHost {

    my ($self, $instance, $activeRoles) = @_;
    my $controllerHost = undef;

    if (defined $activeRoles) {

        my $XS2workerHosts = $activeRoles->{$gHostRoleXS2Worker};

        if (defined $XS2workerHosts) {
            foreach my $currHost (@$XS2workerHosts) {
                if ($instance->hasXS2Controller($currHost)) {
                    $controllerHost = $currHost;
                    last;
                }
            }
        }
    }
    return $controllerHost;
}

sub shouldUnskipLSSPassword {
    my ($self) = @_;
    my $instance = $self->getOwnInstance();
    return 0 if(!$self->isLSSPasswordRequired($instance));

    my $dbHosts = $self->getAllDbHosts($instance);
    my $dbHostsWithSidcryptUser = $self->getDbHostsWithSidcryptUser();
    return @$dbHostsWithSidcryptUser != @$dbHosts;
}

sub getAllDbHosts {
    my ($self, $instance) = @_;
    my $dbHostsWhichArePartOfTheSystem = defined($instance) ? $instance->getHostsByRole($gHostRoleWorker, $gHostRoleStandby) : [];
    $dbHostsWhichArePartOfTheSystem //= [];
    my $addHostsParser = $self->getAddHostsParser();
    my $dbHostsToBeAdded = defined($addHostsParser) ? $addHostsParser->getHanaHosts() : [];

    my $localHostsParser = $self->getLocalHostParser();
    my $localdbHost = defined($localHostsParser) ? $localHostsParser->getLocalHanaHost() : undef;
    if(defined $localdbHost){
      my $shouldAddLocalHost = !(grep {$_ eq $localdbHost} @{$dbHostsToBeAdded});
      push(@$dbHostsToBeAdded, $localdbHost) if $shouldAddLocalHost;
    }
    return [@$dbHostsWhichArePartOfTheSystem, @$dbHostsToBeAdded];
}

sub setLocalHostParser {
    my ($self, $localHostParser) = @_;
    $self->{localHostParser} = $localHostParser;
}

sub getLocalHostParser {
    my ($self) = @_;
    return $self->{localHostParser};
}

sub isLSSPasswordRequired {
    my ($self, $instance) = @_;
    return defined($instance) && defined($instance->getLssInstance());
}

sub setLSSPasswordType {
    return 1;
}

sub addLSSUserString {
    my ($self, $paramIDs) = @_;
    my $username = getSidcryptName($self->getTargetLssSid());
    foreach my $paramID (@$paramIDs) {
        my $parameter = $self->{params}->{$paramID};
        if(defined($parameter) && defined($parameter->{str_templ})){
            $parameter->{str} = sprintf($parameter->{str_templ}, $username);
        }
    }
}

sub checkLSSPassword {
    my ($self, $password) = @_;
    my $str = $self->getString('LSSPassword');

    if($self->getType('LSSPassword') eq 'initial_passwd') {
        if(!$password || length ($password) < 8) {
            $self->appendErrorMessage("$str must contain at least 8 characters");
            return 0;
        }
    } else {
        my $sidcryptUser = SDB::Install::LSS::LssSidcryptUser->new(getSidcryptName($self->getValue('SID')));
        $sidcryptUser->setMsgLstContext([$self->getMsgLst(), $self->getErrMsgLst()]);
        return $sidcryptUser->verifyPassword($password, $self);
    }

    return 1;
}

sub getParamTargetLSSPassword {
    my ($self, $order, $section, $skip) = @_;

    return {
        'order'             => $order,
        'opt'               => 'target_lss_user_password',
        'type'              => 'initial_passwd',
        'section'           => $section,
        'value'             => undef,
        'default'           => undef,
        'set_interactive'   => 1,
        'str'               => "Target $gShortProductNameLSS User Password",
        'str_templ'         => "Target $gShortProductNameLSS User (%s) Password",
        'desc'              => "Target $gShortProductNameLSS User Password",
        'skip'              => $skip // 1,
        'mandatory'         => 1,
        'init_with_default' => 0,
    };
}

sub getParamUseNonSharedStorage  {
    my ($self, $order, $section) = @_;

    return {
        'order'             => $order++,
        'opt'               => 'use_non_shared_storage',
        'opt_arg'           => '<number>',
        'type'              => 'boolean',
        'section'           => $section,
        'default'           => 0,
        'str'               => 'Uses non shared storage',
        'init_with_default' => 1,
        'set_interactive'   => 0,
        'hidden'            => 1,
        'skip'              => 0,
        'hostctrl_opt'      => 'UNSV'
    };
}

sub checkNonSharedStoragePath {
    my($self, $collectedHostInfoVolumeMap, $path, $hosts ) = @_;

    my @hostsWithWrongPermissions = grep { !$collectedHostInfoVolumeMap->{$_}->{'sidadmCanAccess'} } @$hosts;
    for my $host (@hostsWithWrongPermissions){
        my $msgLst = $collectedHostInfoVolumeMap->{$host}->{'msgLst'};
        $self->appendErrorMessage ( sprintf("Missing permissions for '%s' on %s. %s", $self->getSysAdminUserName($self->getSID()), $host, ${$msgLst->getMsgLstString()}));
        return 0;
    }

    return 1 if($self->usesStorageApi());

    my $instance = $self->getOwnInstance();
    if (defined $instance) {
        return 1 if !$instance->isBasePathShared();
    }

    my @hostsWithNonSharedSubvolumes = grep {!$collectedHostInfoVolumeMap->{$_}->{'isSharedSubvolume'}} @$hosts;
    if(@hostsWithNonSharedSubvolumes) {
        my $subvolumePath = File::Spec->catfile($path, $SDB::Install::Configuration::NewServerConfig::volumeSubDirPerHost);
        $self->setErrorMessage(sprintf("Directory '$subvolumePath' is not shared on %s", join(',', @hostsWithNonSharedSubvolumes)));
        return 0;
    }

    return 1;
}

sub isNetworkReconfigRestartRequired {
    my ($self, $targetNetworkAddress) = @_;
    my $instance = $self->getOwnInstance();
    my $existingNetPrefix = $instance->getInternalNetworkPrefix();
# In case of the WebUI this check is performed only during parameter
# validation where the actual parameter {value} is not set. That is why
# the optional 'targetNetworkAddress' argument is supported when
# this method is called from the checkInternalNetwork method.
    my $internalNetworkValue = $targetNetworkAddress || $self->getValue('InternalNetwork');
    my $targetNetPrefix = ($internalNetworkValue && $internalNetworkValue ne 'none')
                ? $internalNetworkValue
                : undef;
    my $isKeepingNoInternalNetwork = (!defined $targetNetPrefix && !defined $existingNetPrefix) ? 1 : 0;
    my $isChangingToNewInternalNetwork = (defined $targetNetPrefix
                                        &&
                                        (!defined $existingNetPrefix ||
                                         ($targetNetPrefix ne $existingNetPrefix))) ? 1 : 0;

    my $isResettingInternalNetwork = (!defined $targetNetPrefix && defined $existingNetPrefix) ? 1 : 0;

    return !$isKeepingNoInternalNetwork && ($isChangingToNewInternalNetwork || $isResettingInternalNetwork);
}

sub checkSslCiphersConfig {
    my ($self) = @_;
    return 1 if $gSkipSslCiphersuitesCheck;
    my @affectedHosts = ();

    my $profilePath = File::Spec->catfile(getActiveSaphostexecDir(), 'host_profile');
    if(!$self->_checkLocalHostSslCiphersConfig($profilePath)) {
        my $instance = $self->getOwnInstance();
        my $hostname = defined($instance) ? $instance->get_host() : $self->getValue('HostName');
        push (@affectedHosts, $hostname);
    }
    foreach my $hostname (keys %{$self->{remoteSaphostagents}}) {
        my $sslConfig = $self->{remoteSaphostagents}->{$hostname}->{sslConfig};
        if (!defined $sslConfig || $sslConfig ne $recommendedCipherSuites) {
            push (@affectedHosts, $hostname);
        }
    }
    if (!scalar(@affectedHosts)) {
        return 1;
    }
    my $affectedHostsStr = join(', ', @affectedHosts);
    my $msgText = "The $gProductNameSHA is not configured to use strong ciphers on hosts: $affectedHostsStr.";
    $msgText .= " They can be enabled by setting the parameter '$cipherSuitesId = $recommendedCipherSuites' in the $gProductNameSHA profile ($profilePath).";
    $self->setSslCiphersWarningMsg($msgText);
    return 1;
}

sub _checkLocalHostSslCiphersConfig {
    my ($self, $profilePath) = @_;
    my $shaHelper = LCM::SapHostAgentFunctions->new();

    if (!$shaHelper->isHostagentInstalled()) {
        return 1;
    }
    my $profile = SDB::Install::IniParser::SAPProfile->new($profilePath);
    if (!$profile->read($self->getMsgLst())) {
        $self->addWarning("Could not read '$profilePath' on local host");
        return 0;
    }
    my $value = $profile->getValue($cipherSuitesId, 0);
    if (!defined $value || $value ne $recommendedCipherSuites) {
        return 0;
    }
    return 1;
}

sub getSslCiphersWarningMsg {
    my ($self) = @_;
    return $self->{_sslCiphersWarningMsg};
}

sub setSslCiphersWarningMsg {
    my ($self, $text) = @_;
    my $msg = $self->getSslCiphersWarningMsg();
    if (!defined $msg) {
        $self->{_sslCiphersWarningMsg} = $self->addWarning($text);
        return 1;
    }
    my $previousWarning = $msg->getRawText();
    if ($previousWarning eq $text) {
        return 1;
    }
    my $warnings = $self->{warnings};
    for my $i (0..@$warnings) {
        if ($warnings->[$i] eq $previousWarning) {
            splice (@$warnings, $i, 1, ($text));
            last;
        }
    }
    $msg->setMsgText($text);
    return 1;
}

1;
