package SDB::Install::Configuration::LSSConfiguration;
use parent qw(SDB::Install::Configuration);
use strict;
use warnings;
use SDB::Common::Utils qw(getSidcryptName);
use SDB::Install::LSS::LssSidcryptUser;
use SDB::Install::Configuration::AnyMultiHostConfig;
use SDB::Install::Configuration::AnyConfig;
use SDB::Install::Globals qw(
    $gShortProductNameLSS
    $gLSSParametersRenameConstraint
    $gProductNameEngine
);

sub defineLSSRenameParams {
    my ($self) = @_;
    my $order = scalar keys %{ $self->{params} };
    my $section = 'lss';
    my $lssParams = {
        'LSSPassword'    => SDB::Install::Configuration::AnyMultiHostConfig::getParamTargetLSSPassword(undef, $order++, $section),
        'LSSUserID'      => $self->getParamLSSUserID($order++, $section, 1, $gLSSParametersRenameConstraint),
        'LSSGroupID'     => $self->getParamLSSGroupID($order++, $section, 1, $gLSSParametersRenameConstraint),
        'LSSUserHomeDir' => $self->getParamLSSUserHomeDir($order++, $section, '/usr/sap/$SID/lss/home', 1, $gLSSParametersRenameConstraint),
        'LSSUserShell'   => $self->getParamLSSUserShell($order++, $section, 1, $gLSSParametersRenameConstraint),
        'ChangeLSSBackupPassword' => $self->getParamChangeLSSBackupPassword($order++, $section, $gLSSParametersRenameConstraint),
        'LSSBackupPassword' => $self->getParamSourceLSSBackupPassword($order++, $section),
        'TargetLSSBackupPassword' => $self->getParamTargetLSSBackupPassword($order++, $section),
    };
    map {$self->{params}->{$_} = $lssParams->{$_}} keys %$lssParams;
    return 1;
}

sub defineLSSAddHostParams {
    my ($self) = @_;
    my $order = scalar keys %{ $self->{params} };
    my $section = 'lss';
    my $lssParams = {
        'LSSUserID'         => $self->getParamLSSUserID($order++, $section, 0),
        'LSSGroupID'        => $self->getParamLSSGroupID($order++, $section, 0),
        'LSSUserHomeDir'    => $self->getParamLSSUserHomeDir($order++, $section, '/usr/sap/$SID/lss/home', 0),
        'LSSUserShell'      => $self->getParamLSSUserShell($order++, $section, 0),
        'LSSPassword'       => SDB::Install::Configuration::AnyConfig::getParamLSSPassword(undef, $order++, $section, 0),
        'RestorePSE'        => $self->getParamRestorePSE($order++, $section),
        'LSSBackupPassword' => $self->getParamLSSBackupPassword($order++, $section),
    };
    map {$self->{params}->{$_} = $lssParams->{$_}} keys %$lssParams;
    $self->setType('LSSBackupPassword', 'passwd');
    $self->addParameterListener('RestorePSE', sub {
        my ($value) = @_;
        $self->setSkip('LSSBackupPassword', !$value);
        return 1;
    });
    $self->addParameterListener('LSSUserShell', sub {
        my ($value) = @_;
        my $installer = SDB::Install::LSS::Installer->new();
        my $sid = $installer->getSid();
        my $sidCrypt = getSidcryptName($sid);
        my $cryptUser = SDB::Install::LSS::LssSidcryptUser->new($sidCrypt);
        $self->setSkip('LSSPassword', $cryptUser->exists());
        return 1;
    });

    return 1;
}

sub defineCheckInstallationPathParams {
    my $self = shift;
    my $order = 0;
    my $section = 'lss';
    $self->{params} = {
        'RemoteExecution'  => SDB::Install::Configuration::AnyConfig::getParamRemoteExecution   ($self,$order++, $section),
        'SID'              => SDB::Install::Configuration::AnyConfig::getParamSID($self,$order++, $section),
        'Path'             => SDB::Install::Configuration::AnyConfig::getParamBasePath($self,$order++, $section ,'path','Path to be checked on remote hosts'),
        'HostNames'        => {
                                'order'             => $order ++,
                                'opt'               => 'hostnames',
                                'type'              => 'csv',
                                'section'           => $section,
                                'value'             => undef,
                                'str'               => 'Host Names',
                                'desc'              => 'Remote host to check the installation path',
                                'set_interactive'   => 1,
                                'mandatory'         => 1
                                },
        'RootUser'         => SDB::Install::Configuration::AnyConfig::getParamRootUser     ($self,$order++, $section),
        'RootPassword'     => SDB::Install::Configuration::AnyConfig::getParamRootPassword ($self,$order++, $section),
        'Password'         => SDB::Install::Configuration::AnyConfig::getParamPassword ($self,$order++, $section),
    };
    $self->setMandatory('RootPassword', 0);
    $self->setMandatory('Password', 0);
    $self->setMandatory('RootUser', 0);
    $self->setSkip('Path', 0);
    $self->addParameterListener('RemoteExecution', sub {
        my ($value ) = @_;
        if($value && $value eq 'saphostagent'){
            $self->setSkip('Password',0);
        } else{
            $self->setSkip('RootUser',0);
            $self->setSkip('RootPassword',0);
            $self->setSkip('Password',1);
        }
        return 1;
    });
}

sub getParamLSSInstPath {
    my ($self, $order, $section) = @_;
    return {
        'order'             => $order,
        'opt'               => 'lss_inst_path',
        'type'              => 'path',
        'section'           => $section,
        'value'             => undef,
        'default'           => '/lss/shared',
        'set_interactive'   => 1,
        'str'               => "Installation Path for $gShortProductNameLSS",
        'desc'              => "Installation Path for $gShortProductNameLSS",
        'skip'              => 1,
        'mandatory'         => 1,
        'init_with_default' => 1,
    };
}

sub getParamLSSUserID {
    my ($self, $order, $section, $skip, $constraint) = @_;

    return {
        'order'             => $order,
        'opt'               => 'lss_userid',
        'type'              => 'number',
        'section'           => $section,
        'value'             => undef,
        'default'           => undef,
        'set_interactive'   => 1,
        'str'               => "$gShortProductNameLSS User ID",
        'str_templ'         => "$gShortProductNameLSS User (%s) ID",
        'desc'              => "$gShortProductNameLSS User ID",
        'skip'              => $skip // 1,
        'mandatory'         => 1,
        'init_with_default' => 1,
        'constraint'        => $constraint,
    };
}

sub getParamLSSUserHomeDir {
    my ($self, $order, $section, $default, $skip, $constraint) = @_;

    return {
        'order'             => $order,
        'opt'               => 'lss_user_home',
        'type'              => 'path',
        'section'           => $section,
        'value'             => undef,
        'default'           => $default,
        'set_interactive'   => 1,
        'str'               => "$gShortProductNameLSS User Home Directory",
        'desc'              => "$gShortProductNameLSS User Home Directory",
        'skip'              => $skip // 1,
        'mandatory'         => 1,
        'init_with_default' => 1,
        'constraint'        => $constraint,
    };
}

sub getParamLSSUserShell {
    my ($self, $order, $section, $skip, $constraint) = @_;

    return {
        'order'             => $order,
        'opt'               => 'lss_user_shell',
        'type'              => 'path',
        'section'           => $section,
        'value'             => undef,
        'default'           => '/bin/sh',
        'set_interactive'   => 1,
        'str'               => "$gShortProductNameLSS User Login Shell",
        'desc'              => "$gShortProductNameLSS User Login Shell",
        'skip'              => $skip // 1,
        'mandatory'         => 1,
        'init_with_default' => 1,
        'constraint'        => $constraint,
    };
}

sub getParamLSSGroupID {
    my ($self, $order, $section, $skip, $constraint) = @_;

    return {
        'order'             => $order,
        'opt'               => 'lss_groupid',
        'type'              => 'number',
        'section'           => $section,
        'value'             => undef,
        'default'           => undef,
        'set_interactive'   => 1,
        'str'               => "$gShortProductNameLSS User Group ID",
        'desc'              => "$gShortProductNameLSS User Group ID",
        'skip'              => $skip // 1,
        'mandatory'         => 1,
        'init_with_default' => 1,
        'constraint'        => $constraint,
    };
}

sub getParamLSSBackupPassword {
    my ($self, $order, $section, $skip) = @_;

    return {
        'order'             => $order,
        'opt'               => 'lss_backup_password',
        'type'              => 'initial_passwd',
        'section'           => $section,
        'value'             => undef,
        'default'           => undef,
        'set_interactive'   => 1,
        'str'               => "$gShortProductNameLSS Auto Backup Password",
        'desc'              => "$gShortProductNameLSS Auto Backup Password",
        'skip'              => $skip // 1,
        'mandatory'         => 1,
        'init_with_default' => 0,
    };
}

sub getParamSourceLSSBackupPassword {
    my ($self, $order, $section, $skip) = @_;

    return {
        'order'             => $order,
        'opt'               => 'source_lss_backup_password',
        'type'              => 'passwd',
        'section'           => $section,
        'value'             => undef,
        'default'           => undef,
        'set_interactive'   => 1,
        'str'               => "Source $gShortProductNameLSS Auto Backup Password",
        'desc'              => "Source $gShortProductNameLSS Auto Backup Password",
        'skip'              => $skip // 1,
        'mandatory'         => 1,
        'init_with_default' => 0,
    };
}

sub getParamTargetLSSBackupPassword {
    my ($self, $order, $section, $skip) = @_;

    return {
        'order'             => $order,
        'opt'               => 'target_lss_backup_password',
        'type'              => 'initial_passwd',
        'section'           => $section,
        'value'             => undef,
        'default'           => undef,
        'set_interactive'   => 1,
        'str'               => "Target $gShortProductNameLSS Auto Backup Password",
        'desc'              => "Target $gShortProductNameLSS Auto Backup Password",
        'skip'              => $skip // 1,
        'mandatory'         => 1,
        'init_with_default' => 0,
    };
}

sub getParamChangeLSSBackupPassword {
    my ($self, $order, $section, $constraint) = @_;

    return {
        'order'     => $order,
        'opt'       => 'change_lss_backup_password',
        'type'      => 'boolean',
        'section'   => $section,
        'value'     => undef,
        'default'   => 0,
        'str'       => "Change the $gShortProductNameLSS Auto Backup password",
        'desc'      => 'Changes the password of the database user',
        'set_interactive'         => 1,
        'init_with_default'       => 1,
        'console_omit_word_Enter' => 1,
        'skip'       => 1,
        'constraint' => $constraint,
    };
}

sub getParamRestorePSE {
    my ($self, $order, $section) = @_;
    return {
        'order'             => $order,
        'section'           => $section,
        'opt'               => 'restore_pse',
        'type'              => 'boolean',
        'default'           => 0,
        'init_with_default' => 1,
        'str'               => 'Restore LSS keys',
        'desc'              => 'Restore LSS keys',
        'skip'              => 0,
    };
}

sub getParamLSSTrustUnsignedServer {
    my ($self, $order, $section) = @_;
    return {
        'order'             => $order,
        'section'           => $section,
        'opt'               => 'lss_trust_unsigned_server',
        'type'              => 'boolean',
        'default'           => 0,
        'init_with_default' => 1,
        'str'               => "Instruct the Local Secure Store (LSS) to trust an unsigned $gProductNameEngine",
        'desc'              => "Instruct the Local Secure Store (LSS) to trust an unsigned $gProductNameEngine",
        'set_interactive'   => 0,
        'skip'              => 0,
    };
}

sub getParamIsRegisterLSS {
    my ($self, $order, $section) = @_;
    return {
        'order'     => $order,
        'opt'       => 'is_register_lss',
        'type'      => 'boolean',
        'section'   => $section,
        'value'     => undef,
        'default'   => 0,
        'str'       => 'Is Register LSS Sceanrio',
        'init_with_default' => 1,
        'skip'      => 0,
        'hidden'    => 1,
    };
}

sub setChangeLSSBackupPassword {
    my ($self, $value) = @_;
    $self->{params}->{ChangeLSSBackupPassword}->{value} = $value;
    $self->setSkip('TargetLSSBackupPassword', !$value);
    return 1;
}

sub checkLSSBackupPassword {
    my ($self, $value) = @_;
    my $string = $self->getString('LSSBackupPassword');
    if ($self->getType('LSSBackupPassword') =~ /initial/) {
        return $self->complyDefaultSqlPasswordPolicy($value, $self->getErrMsgLst(), $string);
    }
    return 1;
}

sub checkTargetLSSBackupPassword {
    my ($self, $value) = @_;
    my $string = $self->getString('TargetLSSBackupPassword');
    return $self->complyDefaultSqlPasswordPolicy($value, $self->getErrMsgLst(), $string);
}

1;