package SDB::Install::Configuration::RemoveHost;

use strict;
use SDB::Install::Configuration::AnyMultiHostConfig;
use SDB::Install::Globals qw ($gHostRoleAcceleratorStandby
                              $gHostRoleAcceleratorWorker
                              $gHostRoleStandby
                              $gHostRoleWorker
                              $gDirNameAccelerator
                              $gDirNameEs
                              $gDirNameStreaming
                              $gXSParametersRemoveHostOrRolesConstraint
                              $gLSSParametersRemoveHostOrRolesConstraint);
use SDB::Install::SysVars qw ($isWin);
use SDB::Install::Configuration::AutoInitFamilyServicesParamHandler;
use SDB::Install::Configuration::TenantUserListener;
use SDB::Install::Configuration::ValueChangeListeners::KeepLSSUserGroupSkipListener;

# use SDB::Install::DebugUtilities qw(dumpThings);

our @ISA = qw (SDB::Install::Configuration::AnyMultiHostConfig);

our $section = 'RemoveHost';


#-------------------------------------------------------------------------------
# Constructor
#
sub new{
	my $self  = shift->SUPER::new (@_);
	my $order = 0;

	my $acceleratorOnly = "Valid for removing $gHostRoleAcceleratorWorker/standby";
    my $tenantCredentialsConstraint = $self->getTenantCredentialsConstraint();
	
	$self->{params} = {
		'Force'               => $self->getParamForce              ($order++, $section),
        'AddHostCalledInCompatiblityMode' => $self->getParamAddHostCalledInCompatiblityMode($order++, $section),
        'AutoInitializeServices' => $self->getParamAutoInitializeServices($order++, $section, undef, $tenantCredentialsConstraint),
        'SID'             => $self->getParamSID            ($order++, $section),
		'KeepUser'        => $self->getParamKeepUser       ($order++, $section),
		(
			!$isWin
			? (
				'KeepUserHomeDir'   => $self->getParamKeepUserHomeDir($order++, $section),
				'SkipModifySudoers' => $self->getParamSkipModifySudoers($order++, $section),
			  )
			: ()
		),
		'KeepXsUsers'         => $self->getParamKeepXsUsers        ($order++, $section, $gXSParametersRemoveHostOrRolesConstraint),
		'SystemUser'          => $self->getParamSystemUser         ($order++, $section, 1, $acceleratorOnly),
		'SQLSysPasswd'        => $self->getParamSQLSysPasswd       ($order++, $section, 'passwd', 1, undef, $acceleratorOnly),
		'AcceleratorUser'     => $self->getParamAcceleratorUser    ($order++, $section, $acceleratorOnly),
		'AcceleratorPassword' => $self->getParamAcceleratorPassword($order++, $section, $acceleratorOnly, 'passwd'),
        'TenantUser'            => $self->getParamTenantUser    ($order++, $section,1,$tenantCredentialsConstraint),
        'SQLTenantUserPassword' => $self->getParamSQLTenantUserPassword($order++,$section,1,$tenantCredentialsConstraint),
        'KeepLSSUser'        => $self->getParamKeepLSSUser ($order++, $section, $gLSSParametersRemoveHostOrRolesConstraint),
        'KeepLSSUserGroup'    => $self->getParamKeepLSSUserGroup  ($order++, $section, $gLSSParametersRemoveHostOrRolesConstraint),
	};
	
	# SID is set by SBD::Install::Installer::getInstallation
	$self->{params}->{SID}->{hidden} = 1;
	
	# hdbremovehost may be called without system/accelerator user
	$self->{params}->{SQLSysPasswd}->{mandatory}        = 0;
	$self->{params}->{AcceleratorPassword}->{mandatory} = 0;

	$self->setSummaryOrder(['SID',
	                        'SystemUser',
	                        'AcceleratorUser',
	                        'KeepUser',
	                        'KeepUserHomeDir',
	                        'Force',
                            'TenantUser',
                            'KeepLSSUser',
                            'KeepLSSUserGroup']);

    $self->addParameterListener('TenantUser',SDB::Install::Configuration::TenantUserListener->new());
    $self->addParameterListener('KeepLSSUser', SDB::Install::Configuration::ValueChangeListeners::KeepLSSUserGroupSkipListener->new());

	return $self;
}


#-------------------------------------------------------------------------------
# Checks the exsistence of the specified SID
#
# Parameter: string $existingSID
#
# Returns int retCode 

sub checkSID{
	my ($self, $existingSID) = @_;

	my $properties = $self->{params}->{SID};
    
    if (!$self->checkExistingSID($existingSID, $properties->{str})) {
        return 0;
    }
    my $hdbInstance = $self->getCollectSAPSystems ()->{$existingSID}->getNewDBInstances()->[0];

    my $host = $hdbInstance->get_host();

    if (!$self->canRemoveHost ($host, $hdbInstance)){
        $self->setNoRetry ('SID', 1);
        return 0;
    }

    my $roles = $hdbInstance->getHostRolesByIniFile($host);

    if (defined $roles) {
        foreach my $currRole (@$roles) {
            if (($currRole eq $gHostRoleAcceleratorWorker) ||
                ($currRole eq $gHostRoleAcceleratorStandby)) {
                $self->setSkip('SystemUser',          0);
                $self->setSkip('SQLSysPasswd',        0);
                $self->setSkip('AcceleratorUser',     0);
                $self->setSkip('AcceleratorPassword', 0);
                last;
            }
        }
    }

    $self->handleCompatiblityMode();

    my $unskipLssParams = defined $self->getOwnInstance()->getLssInstance() ? 0 : 1;
    $self->skipLSSUserParams($unskipLssParams);

	return 1;
}

sub canRemoveHost{
    my ($self, $host, $hdbInstance) = @_;
    $hdbInstance->setMsgLstContext ([$self->getMsgLst (), $self->getErrMsgLst()]);
    my $rc = 1;
    if ($hdbInstance->hostIsActiveMaster ($host)){
        $self->appendErrorMessage ("The host '$host' is the active nameserver master.");
        $self->appendErrorMessage ("Cannot remove the active nameserver master.");
        $rc = 0;
    }

    my $hostRemoveStatus = $hdbInstance->getHostRemoveStatus ($host, $self);
    if (!$hostRemoveStatus){
	    if ($self->getBatchValue ('Force')){
            $self->getMsgLst ()->addMessage ("Ignoring error due to force option.");
        }
        else{
            $rc = 0;
        }
    }
    else{
        $self->getMsgLst ()->addMessage ("Host '$host' is already prepared to be removed.");
    }
    return $rc;
}



#-------------------------------------------------------------------------------
# Returns undef (hdbremovehost does not stop but kills SAP HANA processes)

sub getTimeoutValues{
    return undef;
}

sub handleCompatiblityMode{
    my($self) = @_;
    return if (!$self->isSystemInCompatibilityMode() || $self->isSkipped('AutoInitializeServices') || !$self->getValue('AutoInitializeServices'));
    $self->{_requiresTenantUserCredentials} = 0;
    my $hdbInstance = $self->getOwnInstance();
    my $localHost = $hdbInstance->get_host();
    my $hostRoles = $hdbInstance->getHostRolesByIniFile($localHost);
    if (! grep {$self->_doesRoleRequireAutoInitialize($_)} @$hostRoles) {
        $self->setSkip('AutoInitializeServices', 1);
    }
    return 0;
}

sub _doesRoleRequireAutoInitialize {
    my($self, $role) = @_;
    my $isDbRole = $validHostRoles->{$role}->{isDbRole};
    my $isXs2Role = $validHostRoles->{$role}->{isXS2};
    my $isStandbyRole = $validHostRoles->{$role}->{isStandby};
    my $isExtendedStorage = $validHostRoles->{$role}->{isExtendedStorage};
    if ($isExtendedStorage || $isDbRole || $isXs2Role || $isStandbyRole){
        return 0;
    }
    my $componentDir;
    my $componentDirName;
    my $hdbInstance = $self->getOwnInstance();
    if ($validHostRoles->{$role}->{isStreaming}) {
        $componentDir = $hdbInstance->getStreamingLcmDir();
        $componentDirName = $gDirNameStreaming;
    }
    elsif ($validHostRoles->{$role}->{isExtendedStorage}) {
        $componentDir = $hdbInstance->getExtendedStorageLcmDir();
        $componentDirName = $gDirNameEs;
    }
    elsif ($validHostRoles->{$role}->{isAccelerator} && !$validHostRoles->{$role}->{isStandby}) {
        $componentDir = $hdbInstance->getAcceleratorLcmDir();
        $componentDirName = $gDirNameAccelerator;
    }
    if(defined $componentDir && -d $componentDir){
        my $manifest = SDB::Install::Manifest->new($hdbInstance->getInstalledComponentManifestPath($componentDirName));
        if($manifest->isAutoInitializeServiceSupported()){
            $self->getMsgLst()->addMessage ("Component directory $componentDirName found, which requires auto initializing of services.");
            if(!$validHostRoles->{$role}->{isExtendedStorage}){
                $self->setSkip('TenantUser', 0);
                $self->setSkip('SQLTenantUserPassword', 0);
                $self->{_requiresTenantUserCredentials} = 1;
            }
            return 1;
        }
    }
    return 0;
}

sub isSystemInCompatibilityMode{
    my $self = shift;
    return $self->getValue('AddHostCalledInCompatiblityMode');
}

sub checkRequiresTenantUserCredentials{
    my ($self) = @_;
    return $self->{_requiresTenantUserCredentials};
}

sub shouldWarnIfCalledStandalone{
    my ($self) = @_;
    return $self->_calledStandaloneCriterion();
}

sub getAutoInitializeServicesAction {
    return "removing";
}

sub getExeName{
    return "hdbremovehost";
}

sub getResidentHdblcmPath{
    my ($self) = @_;
    return $self->_constructResidentHdblcmPath();
}

1;
