package SDB::Install::DependencyParser;

use SDB::Install::DebugUtilities qw (dumpThings);

use base SDB::Install::BaseLegacy;
use strict;

sub parse {
	my ($self, $dependenciesArray) = @_;
	$self->{parseTree} = {};

	my ($name, $vendor, $versions, $item, $itemKey, $itemValue);
	my $rc = 1;
	# use '|' as separator for versions because ',' is used elswhere
	my $dependencies = join(',', @$dependenciesArray);
	$dependencies =~ s/([a-z\d]\s*),/$1\|/g;
	foreach my $dependency (split(',', $dependencies)) {
		$dependency =~ s/^\s+//;
		$dependency =~ s/\s+$//;
		foreach $item (split(';', $dependency)) {
			$item =~ s/^\s+//;
			$item =~ s/\s+$//;
			($itemKey, $itemValue) = ($item =~ /(.*)=\s*"(.*)"/);
			$itemKey =~ s/^\s+//;
			$itemKey =~ s/\s+$//;
			if ($itemKey eq 'vendor') {
				$vendor = $itemValue;
			} elsif ($itemKey eq 'name') {
				$name = $itemValue;
			} elsif ($itemKey eq 'version') {
				$versions = $itemValue;
			}
		}

		if (defined $vendor) {
			if (!exists $self->{parseTree}->{$vendor}) {
				$self->{parseTree}->{$vendor} = {};
			}
		}
		else {
			$self->PushError("No vendor defined in '$dependency'.");
			$rc = undef;
		}


		if (defined $name) {
			if (defined $vendor && !exists $self->{parseTree}->{$vendor}->{$name}) {
				$self->{parseTree}->{$vendor}->{$name} = {};
			}
		}
		else {
			$self->PushError("No name defined in '$dependency'.");
			$rc = undef;
		}
		if (defined $versions) {
			if (defined $self->parseVersions ($versions)) {
				if (defined $vendor && defined $name) {
					$self->{parseTree}->{$vendor}->{$name}->{minVersion} = $self->{minVersion};
					$self->{parseTree}->{$vendor}->{$name}->{maxVersion} = $self->{maxVersion};
				}
			}
			else{
				$rc = undef;
			}
		}
	}
	if ($self->errorState) {
		$self->setErrorMessage("Invalid dependency '$dependencies'.", $self->getErrMsgLst ());
	}
	return $rc;
}


sub parseVersions{
	my ($self, $versions) = @_;
	my $result = 1;
	my ($minInclusive, $maxInclusive) = undef;
	$self->{minVersion} = {};
	$self->{maxVersion} = {};

	if ($versions =~ /^\[/){
		$minInclusive = 1;
	}elsif ($versions =~ /^\(/){
		$minInclusive = 0;
	}elsif ($versions !~ /,/) {
		$minInclusive = 1;
	}else{
		$self->PushError ("Wrong version definition, '(' or '[' expected: '$versions'");
		return undef;
	}

	if ($versions =~ /\]$/) {
		$maxInclusive = 1;
	} elsif ($versions =~ /\)$/) {
		$maxInclusive = 0;
	}elsif ($versions !~ /,/) {
		$maxInclusive = 1;
	}else{
		$self->PushError ("Wrong version definition, ')' or ']' expected: '$versions'");
		return undef;
	}

	# strip brackets off $versions
	my ($substr) = ($versions =~ /^[\[\(](.*)[\]\)]$/);
	if (!$substr){
		$self->PushError ("Wrong version definition, no matching brackets: '$versions'");
		return undef;
	}

	# '|' used as separator for versions
	my @versArray = split ('\|',$substr);

	if (@versArray > 2){
		$self->PushError ("Wrong version definition, more than two versions, separated by ',': '$versions'");
		return undef;
	}

	if (!$versArray[0]){
		$self->PushError ("Wrong version definition, no version defined: '$versions'");
		return undef;
	}

	$versArray[0] =~ s/^\s+//;
	$versArray[0] =~ s/\s+$//;
	require SDB::Install::VersionDependency;
	$self->{minVersion} = new SDB::Install::VersionDependency(1, $minInclusive,
																($versArray[0] =~ /^(\d+\.[a-z\d]+)/i),
																($versArray[0] =~ /^\d+\.[a-z\d]+\.([0-9]+)/i),
																($versArray[0] =~ /^\d+\.[a-z\d]+\.[0-9]+\.([0-9]+)/i),
																($versArray[0] =~ /^\d+\.[a-z\d]+\.[0-9]+\.[0-9]+\.([0-9]+)/i));

	if (!defined $self->{minVersion}){
		$self->appendErrorMessage("Wrong version definition, minimum version '$versArray[0]' is wrong");
		return undef;
	}

	if ($versArray[1]){
		$versArray[1] =~ s/^\s+//;
		$versArray[1] =~ s/\s+$//;
		$self->{maxVersion} = new SDB::Install::VersionDependency(0, $maxInclusive,
																	($versArray[1] =~ /^(\d+\.[a-z\d]+)/i),
																	($versArray[1] =~ /^\d+\.[a-z\d]+\.([0-9]+)/i),
																	($versArray[1] =~ /^\d+\.[a-z\d]+\.[0-9]+\.([0-9]+)/i),
																	($versArray[1] =~ /^\d+\.[a-z\d]+\.[0-9]+\.[0-9]+\.([0-9]+)/i));
		if (!defined $self->{maxVersion}){
			$self->appendErrorMessage("Wrong version definition, maximum version '$versArray[1]' is wrong");
			return undef;
		}
	} else {
		$self->{maxVersion} = new SDB::Install::VersionDependency(0, $maxInclusive,
																	($versArray[0] =~ /^(\d+\.[a-z\d]+)/i),
																	($versArray[0] =~ /^\d+\.[a-z\d]+\.([0-9]+)/i),
																	($versArray[0] =~ /^\d+\.[a-z\d]+\.[0-9]+\.([0-9]+)/i),
																	($versArray[0] =~ /^\d+\.[a-z\d]+\.[0-9]+\.[0-9]+\.([0-9]+)/i));
	}
	my $minVersionString = $self->{minVersion}->getVersionString();
	my $maxVersionString = $self->{maxVersion}->getVersionString();
	if ($minVersionString eq $maxVersionString && ($minInclusive == 0 || $maxInclusive == 0)) {
		$self->PushError("Wrong version definition. An exact version must be surrounded by inclusive '[' and ']’ brackets.");
		return undef;
	}
	return $result;
}


sub getMinVersion{
	my ($self, $vendor, $name) = @_;
	if (!defined $vendor || !defined $name || !(defined $self->{parseTree}->{$vendor}) || !(defined $self->{parseTree}->{$vendor}->{$name})) {
		return undef;
	}
	return $self->{parseTree}->{$vendor}->{$name}->{minVersion};

}


sub getMaxVersion{
	my ($self, $vendor, $name) = @_;
	if (!defined $vendor || !defined $name || !(defined $self->{parseTree}->{$vendor}) || !(defined $self->{parseTree}->{$vendor}->{$name})  ) {
		return undef;
	}
	return $self->{parseTree}->{$vendor}->{$name}->{maxVersion};

}

sub getDependencyString{
    my ($self, $vendor, $name, $productName) = @_;
    my $dependencyStr = '';
    my $dependency = $self->{parseTree}->{$vendor}->{$name};
    if (!defined $dependency){
        return '';
    }
    if (!defined $productName){
        $productName = "$vendor $name"
    }
    my $minDependency = $dependency->{minVersion};
    my $maxDependency = $dependency->{maxVersion};
    if (defined $minDependency){
        $dependencyStr = $minDependency->getDependencyVersionString() . ' ';
    }
    $dependencyStr .= $productName;
    if (defined $maxDependency){
        $dependencyStr .= ' ' . $maxDependency->getDependencyVersionString();
    }
    return $dependencyStr;
}

sub getDependenciesVendors{
	my $self = shift;
	my @vendors =  keys %{$self->{parseTree}};
	return \@vendors;
}

sub getDependenciesKeyNamesForVendor{
	my ($self,$vendor) = @_;
	my $vendorHash = $self->{parseTree}->{$vendor};
    if(!defined $vendorHash){
		$self->PushError("No dependencies found for vendor '$vendor'.");
		return undef;
	}
	my @keyNames =  keys %$vendorHash;
	return \@keyNames;
}
1;
