package SDB::Install::FsManager;

use SDB::Install::System qw (makedir isSameFile deltree);
use SDB::Install::SysVars qw ($isWin);
use SAPDB::Install::System::Unix qw(lchown);
use SDB::Install::SysVars qw ($path_separator);
use SDB::Install::Globals qw ($gPlatform);
use Exporter;
use strict;

our @ISA = qw (SDB::Install::BaseLegacy Exporter);
our @EXPORT = qw(
    $ssfsGlobalKeyFileTemplate
    $ssfsGlobalLockFileTemplate
    $ssfsDirectory
    $systemSsfsKeyDirectory
    $systemSsfsDataDirectory
    $systemSsfsKeyDirectory
);

our $ssfsGlobalKeyFileTemplate  = 'SSFS_%s.KEY';
our $ssfsGlobalLockFileTemplate = 'SSFS_%s.LCK';
our $ssfsLocalKeyFileTemplate   = 'SSFS_%s.LKY';
our $ssfsDirectory = join ($path_separator, qw(global hdb security ssfs));
our $systemSsfsKeyDirectory = join ($path_separator, qw(global security rsecssfs key));
our $systemSsfsDataDirectory = join ($path_separator, qw(global security rsecssfs data));


#-------------------------------------------------------------------------------
# Assigns the specified instance number to the hash '_nr'
# and creates the instance directoy names
#     '_instanceDir'       => /usr/sap/<sid>/HDB<nr>
# and '_globalInstanceDir' => e.g. /hana/shared/<sid>/HDB<nr>

sub set_instanceNr{
    my ($self, $nr) = @_;
    $self->{_nr} = $nr;
    $self->{_instanceDir}       = $self->createInstDirName($self->{_usrSapSid});
    $self->{_globalInstanceDir} = $self->createInstDirName($self->{_globalSidDir});
}


#-------------------------------------------------------------------------------
# Returns the directory name '<$parentDir>/HDB<nr>',
# where <nr> is $self->{_nr} with 2 digits.

sub createInstDirName {
    my ($self, $parentDir) = @_;
    return  $parentDir . $path_separator . sprintf ('HDB%02d', $self->{_nr});
}


#-------------------------------------------------------------------------------
sub setUserSettings{
    $_[0]->{_userId} = $_[1];
    $_[0]->{_groupId} = $_[2];
    $_[0]->{_shmGroupId} = $_[3];
}

sub new{
    my ($class,$usrSapSid,$globalSidDir,$userId,$groupId,$shmGroupId) =  @_;
    my $self = $class->SUPER::new ();
    $self->{_usrSapSid} = $usrSapSid;
    $self->{_globalSidDir} = $globalSidDir;
    $self->{_groupId} = $groupId;
    $self->{_userId} = $userId;
    $self->{_shmGroupId} = $shmGroupId;
    return $self;
}

our @global_dirs = (
    ['profile', 0755],
    ['global'],
    ['global/hdb'],
    ['global/security/lib/tools'],
    ['global/security/data'],
    ['global/security/rsecssfs', 0700],
    [$systemSsfsDataDirectory, 0700],
    [$systemSsfsKeyDirectory, 0700],
    ['global/hdb/custom/config'],
    ['global/hdb/custom/config/lexicon'],
    ['global/hdb/custom/python_support'],
    ['global/hdb/checks'],
    ['global/hdb/data'],
    ['global/hdb/doc'],
    ['global/hdb/opt'],
    ['global/hdb/plugins/3rd_party_libs'],
    ['global/hdb/plugins/1st_party_libs'],
    [$ssfsDirectory, 0700],
    ['global/hdb/security/vsi', 0700],
    ['global/hdb/install/backup'],
    ['global/hdb/install/config'],
    ['global/hdb/IM/reference_data']
);


if (!$isWin){
    push @global_dirs, ['${EXEVERPATH}', 0550];
    push @global_dirs , ['global/hdb/install/hwinfo'];
} else {
    push @global_dirs , ['exe/ntamd64/plugins'];
}

our @global_links = (
    ['${EXEVERPATH}/lexicon', '../../../global/hdb/custom/config/lexicon'],
    ["exe/$gPlatform/hdb", '${VERPATH}']
);

our @instance_dirs = (
    'work', $isWin ? ('exe') : ()
);

our @instance_links = (
    ['exe', "../exe/$gPlatform/hdb"],
);

our @host_dirs = qw(
    lock
    log
    tmp
    trace
    sec
    wdisp/admin
    wdisp/cache
    work
);


our $localOSDir = '/var/lib';


# [ basepath(w/o macros), filename_template, ignoreNotExisting]
our $global_filename_templates = [
    [$ssfsDirectory, 'SSFS_<SID>.DAT', 0],
    [$systemSsfsDataDirectory, 'SSFS_<SID>.DAT', 1],
    [$ssfsDirectory, sprintf ($ssfsGlobalKeyFileTemplate, '<SID>'), 1],
    [$systemSsfsKeyDirectory, sprintf ($ssfsGlobalKeyFileTemplate, '<SID>'), 1],
    [$ssfsDirectory, sprintf ($ssfsLocalKeyFileTemplate, '<SID>') , 1]
];

sub expandFilename{
    my (
        $filename,
        $sid
    ) = @_;
    my $macros = { '<SID>' => $sid };
    foreach my $macro (keys (%$macros)) {
        my $replacement = $macros->{$macro};
        $macro = quotemeta ($macro);
        $filename =~ s/$macro/$replacement/g;           
    }
    return $filename;
}

sub renameGlobalFiles {
    my (
        $self,
        $oldSID,
        $newSID
    ) = @_;
    foreach my $templateRef (@$global_filename_templates) {
        my @template = @$templateRef;
        my $basepath = $template[0];
        my $prefix   = $self->{_globalSidDir}.$path_separator.$basepath.$path_separator;
        my $src      = $prefix . expandFilename($template[1], $oldSID);
        my $dst      = $prefix . expandFilename($template[1], $newSID);

        if (-f $dst) {
            next;
        }
        if(-f $src) {
            $self->AddMessage("Renaming '$src' => '$dst'.");
            if(not ($src eq $dst)) {
                if (not CORE::rename($src, $dst)){
                    $self->AddError ("Could not rename '$src' => '$dst': $!");
                    return undef;
                }
            }
        }
        elsif ($template[2]){
            # ignore not existing
            $self->AddMessage("'$src' doesn't exist");
        }
        else {
            $self->AddError ("Could not rename '$src' => '$dst': '$src' not found.");
            return undef;
        }
    }
    return 1;
}
sub checkDir{
    my ($self, $dir, $uid, $gid, $mode, $msglst) = @_;
    if (-d $dir){
        my @statbuf = stat (_);
        my $changed = 0;
        if ($statbuf[4] != $uid || $statbuf[5] != $gid){
            $changed = 1;
            if (!chown ($uid, $gid, $dir)){
                $msglst->AddError ("Cannot change owner of existing directory '$dir': $!");
                return undef;
            }
            $msglst->AddMessage ("Changed owner of existing directory '$dir'.")
                ->getSubMsgLst ()
                ->addMessage ("uid $statbuf[4] => $uid, gid $statbuf[5] => $gid");
        }
        if (($statbuf[2] & $mode) != $mode){
            $changed = 1;
            if (!chmod ($mode, $dir)){
                $msglst->AddError ("Cannot change mode of existing directory '$dir': $!");
                return undef;
            }
            $msglst->AddMessage ("Changed mode of existing directory '$dir'.");
        }
        if (!$changed){
            $msglst->AddMessage ("Existing directory '$dir' is up-to-date.");
        }
    }
    else{
        my $cfg = {'uid' => $uid, 'gid' => $gid, 'mode' => $mode};
        if (!defined makedir ($dir, $cfg)){
            $msglst->AddError ("Cannot create directory '$dir'", $cfg);
            return undef;
        }
        $msglst->AddMessage ("Directory '$dir' created.");
    }
    return 1;
}

sub createLocalIpcDirectory{
    my ($self) = @_;

    my ($sid) = ($self->{_usrSapSid} =~ /[\/]([A-Z][A-Z0-9][A-Z0-9])[\/]*$/);

    if (!$sid){
        $self->AddError ("Cannot find sid in '$self->{_usrSapSid}'");
        return undef;
    }

    my $msg = $self->AddMessage ("Creating local IPC directory");

    my $msglst = $msg->getSubMsgLst ();
    if (!defined $self->checkDir ($localOSDir,0,0,0755, $msglst)){
        $self->AddError (undef, $msglst);
        return undef;
    }

    my $dir = $localOSDir . $path_separator . 'hdb';
    if (!defined $self->checkDir ($dir,0,$self->{_groupId},0775, $msglst)){
        $self->AddError (undef, $msglst);
        return undef;
    }

    $dir .= $path_separator . $sid;
    if (!defined $self->checkDir ($dir,$self->{_userId},$self->{_groupId},0755, $msglst)){
        $self->AddError (undef, $msglst);
        return undef;
    }

    #my $ipcDir = $dir . $path_separator . 'ipc';

    $dir .= $path_separator . 'shmgrp';

    if (-d $dir){
        my $removeMsglst =
            $msglst->addMessage ("Removing content of shmgrp directory")->getSubMsgLst ();
        deltree ($dir, $removeMsglst, $removeMsglst, 1);
    }

    if (!defined $self->checkDir ($dir,$self->{_userId},$self->{_shmGroupId},02770, $msglst)){
        $self->AddError (undef, $msglst);
        return undef;
    }

    #if (-d $ipcDir){
    #    my $removeMsglst = $self->getMsgLst ()
    #        ->addMessage ("Removing old ipc directory")->getSubMsgLst ();
    #    deltree ($ipcDir, $removeMsglst);
    #}
    return 1;
}

sub createInstanceLink{
    my ($self) = @_;
    my $rc = 1;
    my ($dst,$link) =
        ($self->{_globalInstanceDir}, $self->{_instanceDir});
    $self->AddMessage ("Creating SymLink '$link' => '$dst'");
    my $createLink = 1;
        
    if (-l $link){
        if (readlink ($link) eq $dst){
            $self->AddMessage ("Link already exists");
            $createLink = 0;
        }
        else{
            unlink ($link);
        }
    }
    if ($createLink) {
        if (!symlink ($dst,$link)){
            $self->PushError ("Cannot create symlink '$link': $!");
            $rc = undef;
        }
    }
    lchown ($self->{_userId},$self->{_groupId}, $link);
    return $rc;
}




sub createInstanceFS{
    my ($self) = @_;
    
    my $rc = 1;
    my $config = {'uid' => $self->{_userId}, 'gid' => $self->{_groupId},
            'mode' => 0750};


    my $instance_dir = $self->{_instanceDir};

    if (!$isWin){
        if (!isSameFile ($self->{_usrSapSid}, $self->{_globalSidDir})){
            if (!-d $self->{_globalInstanceDir}){
                if (!makedir ($self->{_globalInstanceDir}, $config)){
                    $self->AddError ("Cannot global create instance dir '$self->{_globalInstanceDir}'",$config);
                    return undef;
                }
            }
            if (!defined $self->createInstanceLink()){
                $rc = undef;
            }
        }
    }

    my $lDir;
    foreach my $dir (@instance_dirs){
        if ($isWin){
            $dir =~ s/\//\\/g;
        }
        $lDir = $instance_dir . $path_separator . $dir; 
        if (!-e $lDir){
            $self->AddMessage ("Creating directory '$lDir'");
            if (!defined makedir ($lDir, $config)){
                $self->PushError (undef, $config);
                $rc = undef;
            }
        }   
        
    }

    if ($isWin){
        return $rc;
    }

    foreach my $l (@instance_links){
        my ($link,$dst) = @$l;
        if ($isWin){
            $dst =~ s/\//\\/g;
            $link =~ s/\//\\/g;
        }
        $link = $instance_dir . $path_separator . $link;
    
        $self->AddMessage ("Creating SymLink '$link' => '$dst'");
        
        if (-l $link){
            if (readlink ($link) eq $dst){
                $self->AddMessage ("Link already exists");
                next;
            }
            unlink ($link);
        }
    
    
        if (!symlink ($dst,$link)){
            $self->PushError ("Cannot create symlink '$link': $!");
            $rc = undef;
        }
        lchown ($self->{_userId},$self->{_groupId}, $link);
    }   
    return $rc;
}

sub getInstanceDir {
    my ($self) = @_;
    return $self->{'_instanceDir'};
}

sub createGlobalFS{
    my ($self,$version) = @_;
    my $rc = 1;

	$rc = $self->createGlobalDirs($version);

    if ($isWin){
        return $rc;
    }

	$rc = $self->createGlobalLinks($version);

    return $rc;
}

sub createGlobalDirs{
    my ($self,$version) = @_;
    my $rc = 1;

    my $mode_default = 0750;
    my $config = {'uid' => $self->{_userId}, 'gid' => $self->{_groupId},
            'mode' => $mode_default};

	my ($verpath, $exeverpath);
	if ($version) {
        $verpath = 'HDB_'.$version;
        $exeverpath = join ($path_separator, 'exe', $gPlatform, $verpath);
	}
    my ($lDir,$dir);
    my $msglst = $self->getMsgLst();
    foreach my $entry (@global_dirs){
        $dir = $entry->[0];
        if ($dir =~ /\$/){
            if ($version){
                $dir =~ s/\$\{EXEVERPATH\}/$exeverpath/g;
                $dir =~ s/\$\{VERPATH\}/$verpath/g;
            }
            else{
                next;
            }
        }
        $lDir = $self->{_globalSidDir} . $path_separator . $dir;
        if (!-e $lDir){
            $self->AddMessage ("Creating directory '$lDir'");
            if ($entry->[1]){
                $config->{mode} = $entry->[1];
            }
            else{
                $config->{mode} = $mode_default;
            }
            if (!defined makedir ($lDir, $config)){
                $self->PushError (undef, $config);
                $rc = undef;
            }
        }
        elsif (!$isWin){
            my @statbuf = stat (_);
            if ($statbuf[4] != $self->{_userId} ||
                $statbuf[5] != $self->{_groupId}){
                    $msglst->addMessage ("Changing owner of existing directory '$lDir'");
                    if (!chown ($self->{_userId}, $self->{_groupId}, $lDir)){
                        $self->appendErrorMessage ("Cannot change owner of existing directory '$lDir': $!");
                        $rc = undef;
                    }
            }
            my $mode = defined $entry->[1] ? $entry->[1] : $mode_default;
            if (($statbuf[2] & 07777) != $mode){
                $msglst->addMessage ("Changing mode of existing directory '$lDir'");
                if (!chmod ($mode, $lDir)){
                    $self->appendErrorMessage ("Cannot change mode of existing directory '$lDir': $!");
                    $rc = undef;
                }
            }
        }
    }

    return $rc;
}

sub createGlobalLinks{
    my ($self,$version) = @_;
    my $rc = 1;

    my $verpath = 'HDB_'.$version;
    my $exeverpath = join ($path_separator, 'exe', $gPlatform, $verpath);

    foreach my $l (@global_links){
        my ($link,$dst) = @$l;
        if ($link =~/\$/){
            $link =~ s/\$\{EXEVERPATH\}/$exeverpath/g;
            $link =~ s/\$\{VERPATH\}/$verpath/g;

        }
        if ($dst =~ /\$/){
            $dst =~ s/\$\{EXEVERPATH\}/$exeverpath/g;
            $dst =~ s/\$\{VERPATH\}/$verpath/g;
            $dst =~ s/\$\{SIDDIR\}/$self->{_globalSidDir}/g;

        }
        $link = $self->{_globalSidDir} . $path_separator . $link;
        $self->AddMessage ("Creating SymLink '$link' => '$dst'");
        
        if ($isWin){
            next;
        }
        
        if (-l $link){
            if (readlink ($link) eq $dst){
                $self->AddMessage ("Link already exists");
                next;
            }
            unlink ($link);
        }
        
        if (!symlink ($dst,$link)){
            $self->PushError ("Cannot create symlink '$link': $!");
            $rc = undef;
        }
        lchown ($self->{_userId},$self->{_groupId}, $link);
    }
    if (!$isWin){
        if (!defined $self->createUsrSapSidLinks ()){
            $rc = undef;
        }
    }
    return $rc;
}


#-------------------------------------------------------------------------------
# Creates the link '/usr/sap/<SID>/SYS' with its sublinks.

sub createUsrSapSidLinks{
    my ($self) = @_;

    if (defined $self->{_shmGroupId}){
        if (!$self->createLocalIpcDirectory ()){
            return undef;
        }
    }

    my $rc = 1;
    my $config = {'uid' => $self->{_userId}, 'gid' => $self->{_groupId},
            'mode' => 0750};
    my $sysdir = $self->{_usrSapSid} . '/SYS/exe';
    if (!-e $sysdir){
        $self->AddMessage ("Creating directory '$sysdir'");
        if (!defined makedir ($sysdir, $config)){
            $self->PushError (undef, $config);
            $rc = undef;
        }
    }
    my @links;
    
    if (isSameFile ($self->{_usrSapSid}, $self->{_globalSidDir})){
         @links = (
            ['../profile', $self->{_usrSapSid} . '/SYS/profile'],
            ['../global', $self->{_usrSapSid} . '/SYS/global'],
            ["../../exe/$gPlatform/hdb", $self->{_usrSapSid} . '/SYS/exe/hdb']
            );
    }
    else{
        @links = (
            [$self->{_globalSidDir} . '/profile', $self->{_usrSapSid} . '/SYS/profile'],
            [$self->{_globalSidDir} . '/global', $self->{_usrSapSid} . '/SYS/global'],
            ["$self->{_globalSidDir}/exe/$gPlatform/hdb", $self->{_usrSapSid} . '/SYS/exe/hdb']);
    }
    
    my ($link,$dst);
    foreach my $l (@links){
        ($dst,$link) = @$l;
        $self->AddMessage ("Creating SymLink '$link' => '$dst'");
        if (-l $link){
            if (readlink ($link) eq $dst){
                $self->AddMessage ("Link already exists");
                next;
            }
            unlink ($link);
        }
    
        if (!symlink ($dst,$link)){
            $self->PushError ("Cannot create symlink '$link': $!");
            $rc = undef;
        }
        lchown ($self->{_userId},$self->{_groupId}, $link);
    }
    return $rc;
}


sub migrate{

    my ($self, $version) = @_;

    if (!$self->createGlobalDirs ($version)){
        return undef;
    }
    
    if ($isWin){
        return 1;
    }

    if (!$self->createLocalIpcDirectory ()){
        return undef;
    }

    if (-e $self->{_usrSapSid} . '/SYS/exe/hdb'){
        return 1;
    }
    
    return $self->createUsrSapSidLinks();
}

sub createHostFS{
    my ($self,$host) = @_;
    my $rc = 1;
    my $config = {'uid' => $self->{_userId}, 'gid' => $self->{_groupId},
            'mode' => 0775};

    my $host_dir = ($isWin ? $self->{_instanceDir} :
        $self->{_globalInstanceDir}) . $path_separator . $host;
    my $lDir;
    foreach my $dir (@host_dirs){
        $lDir = $host_dir . $path_separator . $dir;
        if (!-e $lDir){
            $self->AddMessage ("Creating directory '$lDir'");
            if (!defined makedir ($lDir, $config)){
                $self->PushError (undef, $config);
                $rc = undef;
            }
        }   
    }
    return $rc;
}

sub migrateHost;
*migrateHost = \&createHostFS;

#-------------------------------------------------------------------------------
# Renames global files/directories matching SID and instance number.
# Furthermore links within '/usr/sap/<SID>' are renamed or created.

sub renameFiles {
    my ($self,
        $oldUsrSapSid,    $newUsrSapSid,
        $oldGlobalSidDir, $newGlobalSidDir,
        $newNr,
        $oldSid,          $newSid) = @_;

    if (!-d $oldUsrSapSid && !-d $newUsrSapSid) {
        $self->AddError ("Local directory '$oldUsrSapSid' does not exist");
        return undef;
    }

    if (!-d $oldGlobalSidDir && !-d $newGlobalSidDir) {
        $self->AddError ("Installation directory '$oldGlobalSidDir' does not exist");
        return undef;
    }

    my $oldNr = $self->{_nr};

    my $isSameDirUserGlob = !$isWin
                            &&
                            (isSameFile ($oldUsrSapSid, $oldGlobalSidDir) ||
                             isSameFile ($newUsrSapSid, $newGlobalSidDir));

    if ($oldUsrSapSid ne $newUsrSapSid) {
        if (!-d $newUsrSapSid) {
            my $fromTo = "'$oldUsrSapSid' => '$newUsrSapSid'";
            $self->AddMessage ("Renaming $fromTo");
            if (!CORE::rename ($oldUsrSapSid, $newUsrSapSid)){
                $self->AddError ("Cannot rename $fromTo: $!");
                return undef;
            }
        }
        if (!$isWin){
           my $ipcDir = $localOSDir . $path_separator . 'hdb' . $path_separator . $oldSid;
           if (-d $ipcDir){
                deltree ($ipcDir);
           }
        }
        $self->{_usrSapSid} = $newUsrSapSid;
        if (!$isWin && $isSameDirUserGlob){
            $self->{_globalSidDir} = $newUsrSapSid;
            # refesh _globalInstanceDir due to changed _globalSidDir
            $self->set_instanceNr ($self->{_nr});
        }
    }

    if (!$isWin && !$isSameDirUserGlob){
        my $sidChanged        = $newSid ne $oldSid;
        my $nrChanged         = $newNr  ne $oldNr;
        my $originInstanceDir = $self->{_instanceDir}; # /usr/sap/<oldSID>/HDB<oldNR>
        
        if ($sidChanged){

            if (!-d $newGlobalSidDir) {
                my $fromTo = "'$oldGlobalSidDir' => '$newGlobalSidDir'";
                $self->AddMessage ("Renaming $fromTo");
                if (!CORE::rename ($oldGlobalSidDir, $newGlobalSidDir)){
                    $self->AddError ("Cannot rename $fromTo: $!");
                    return undef;
                }
            }
            $self->{_globalSidDir} = $newGlobalSidDir;

            if (!$self->renameGlobalFiles($oldSid, $newSid)) {
                return undef;
            }

            # refesh _globalInstanceDir due to changed _globalSidDir
            $self->set_instanceNr ($oldNr);
            $originInstanceDir = $self->{_instanceDir}; # /usr/sap/<newSID>/HDB<oldNR>
        }

        if ($sidChanged || !-d $newUsrSapSid.$path_separator.'SYS') {
            if (!$self->createUsrSapSidLinks()) {
                return undef;
            }
        }

        if ($nrChanged){
            # /hana/shared/<newSID>/HDB<oldNR> -> /hana/shared/<newSID>/HDB<newNR>

            my $originGlobInstDir = $self->{_globalInstanceDir};
            $self->set_instanceNr ($newNr);
            my $newGlobInstDir = $self->{_globalInstanceDir};

            if (!-d $newGlobInstDir) {
                my $fromTo = "'$originGlobInstDir' => '$newGlobInstDir'";
                $self->AddMessage ("Renaming $fromTo");
                if (!CORE::rename ($originGlobInstDir, $newGlobInstDir)){
                    $self->AddError ("Cannot rename $fromTo: $!");
                    return undef;
                }
            }
        }

        if ($sidChanged || $nrChanged){

            # try delete /usr/sap/<newSID>/HDB<oldNR>
            if (-l $originInstanceDir && !unlink ($originInstanceDir)){
                $self->AddWarning ("Cannot delete symlink '$originInstanceDir': $!");
            }

            # try delete /usr/sap/<newSID>/HDB<newNR>
            if (-l $self->{_instanceDir} && !unlink ($self->{_instanceDir})){
                $self->AddError ("Cannot delete symlink '$self->{_instanceDir}': $!");
                return undef;
            }
        }

        if (!-l $self->{_instanceDir}) {
            if (!defined $self->createInstanceLink()) {
                return undef;
            }
        }
    }
    else{
        # $isWin || $isSameDirUserGlob
        if (!$self->renameGlobalFiles($oldSid, $newSid)) {
            return undef;
        }
        my $originInstanceDir = $self->{_instanceDir};
        $self->set_instanceNr ($newNr);
        if (!-d $self->{_instanceDir}) {
            my $fromTo = "'$originInstanceDir' => '$self->{_instanceDir}'";
            $self->AddMessage ("Renaming $fromTo");
            if (!CORE::rename ($originInstanceDir,$self->{_instanceDir})){
                $self->AddError ("Cannot rename $fromTo: $!");
                return undef;
            }
        }
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Renames local files/directories matching SID and instance number.
#
# Parameters: $usrSapNewSid    string   # /usr/sap/<newSid>
#             $newInstNr       string


sub renameSlaveFiles{
    my ($self, $usrSapNewSid, $newInstNr) = @_;
    
    my $sidChanged = 0;
    if ($usrSapNewSid ne $self->{_usrSapSid}){
        $self->AddMessage ("Renaming '$self->{_usrSapSid}' => '$usrSapNewSid'");
        if (!CORE::rename ($self->{_usrSapSid}, $usrSapNewSid)){
            $self->AddError ("Cannot rename '$self->{_usrSapSid}' => '$usrSapNewSid': $!");
            return undef;
        }
        if (!$isWin){
            my ($oldSid) = ($self->{_usrSapSid} =~ /[\/]([A-Z][A-Z0-9][A-Z0-9])[\/]*$/);
            if ($oldSid){
                my $ipcDir = $localOSDir . $path_separator . 'hdb' . $path_separator . $oldSid;
                if (-d $ipcDir){
                    deltree ($ipcDir);
                }
            }
        }
        $self->{_usrSapSid} = $usrSapNewSid;
        $sidChanged = 1;
    }
    
    my $nrChanged = $newInstNr != $self->{_nr};
    
    # initialize _instanceDir and _globalInstanceDir
    $self->set_instanceNr ($self->{_nr});
    my $originInstanceDir = $self->{_instanceDir}; # /usr/sap/<newSID>/HDB<oldNR>
        
    if ($sidChanged || !-d $usrSapNewSid.$path_separator.'SYS') {
        if (!$self->createUsrSapSidLinks()) {
            return undef;
        }
    }

    if ($nrChanged){
        $self->set_instanceNr ($newInstNr);
    }
    
    if ($sidChanged || $nrChanged){
        if (-l $originInstanceDir){
            if (!unlink ($originInstanceDir)){
                $self->AddWarning ("Cannot delete symlink '$originInstanceDir': $!");
            }
        }
        if (-l $self->{_instanceDir}){
            if (!unlink ($self->{_instanceDir})){
                $self->AddError ("Cannot delete symlink '$self->{_instanceDir}': $!");
                return undef;
            }
        }
    }

    if (!-l $self->{_instanceDir}) {
        if (!defined $self->createInstanceLink()) {
            return undef;
        }
    }

    return 1;
}

1;
