package SDB::Install::Installation::Client;

use SDB::Install::Installation;
use SDB::Install::SysVars qw ($path_separator $isWin);
use SDB::Install::System qw (makedir isSameFile isAdmin normalizePath);
use SAPDB::Install::Hostname;
use SDB::Install::Globals qw ($gProductNameClient);
use SAPDB::Install::System::Unix qw(lchown);
use SDB::Common::BuiltIn;
use File::Spec;
use File::stat;
use strict;

#----------------------------------------------------------------------------

our @ISA = qw (SDB::Install::Installation);

#----------------------------------------------------------------------------

our $regFileName = 'installations.client';

our $hdbclient_link = 'hdbclient';

our $installations;

#----------------------------------------------------------------------------

sub new{
    my (
        undef,
        undef,
        undef,
        $uid,
        $gid,
        $sid
    ) = @_;
    my $self = shift->SUPER::new (@_);
    $self->{'UID'} = $uid;
    $self->{'GID'} = $gid;
    $self->{'SID'} = $sid;
    $self->init ();
    return $self;
}

sub EnumClientInstallations{
    my ($msglst, $nocache) = @_;
    if (defined $installations && !$nocache){
        return $installations;
    }
    my $obj = new __PACKAGE__;
    $installations = $obj->EnumInstallations ();
    return $installations;
}


sub init{
    my ($self) = @_;
    $self->SUPER::init ();
    if (defined $self->{path}){
        $self->{'registrypath'} = $self->{'path'}. $path_separator . 'install';
        $self->{'newInstallation'} = (-d $self->{'registrypath'}) ? 0 : 1;
    }
}

sub isUserMode{
    return !isAdmin();
}


#----------------------------------------------------------------------------

sub getRegFileName {
    return $regFileName;
}

#----------------------------------------------------------------------------

sub asString{
	my ($self) = @_;
	return "$gProductNameClient " . $self->GetVersion () . ' ' . $self->{path};
}

#----------------------------------------------------------------------------

# despite its name, this method is also called in case of a 
# software update:
sub initNewInstallation{
    my (
        $self
    ) = @_;
    my $cfg = {};
    if(!$isWin) {
        if(!defined $self->{'SID'}) {
            my $installations = EnumClientInstallations($self);
            if(defined $installations) {
                foreach my $installationPath (keys %$installations) {
                    $installationPath = normalizePath($installationPath);
                    my $intendedPath = normalizePath($self->{'path'});
                    my @statbuf = stat($intendedPath);
                    if(-d $intendedPath) {
                        if(isSameFile($installationPath, $intendedPath, \@statbuf)) {
                            # when there's no sid given as option, and
                            # when this is a software update (not a fresh installation),
                            # we give ownership of the newly installed files to the owner
                            # of the preexisting installation root:
                            $self->{'UID'} = $statbuf[4];
                            $self->{'GID'} = $statbuf[5];
                            last;
                        }
                    }
                }
            }
        }
        $cfg = {
            'mode' => 0755,
            'uid'  => $self->{'UID'},
            'gid'  => $self->{'GID'}
        };
    }
    if (!-d $self->{'path'}) {
        if (!defined makedir($self->{'path'}, $cfg)) {
            $self->AddError("Could not create installation root directory: $self->{'path'}", $cfg);
            return undef;
        }
    }
    if (!-d $self->{'registrypath'}) {
        if (!defined makedir($self->{'registrypath'}, $cfg)) {
            $self->AddError("Could not create install registry directory: $self->{'registrypath'}", $cfg);
            return undef;
        }
    }
    return 1;
}

#----------------------------------------------------------------------------

sub getUID{
    my (
        $self
    ) = @_;
    return $self->{'UID'};
}

#----------------------------------------------------------------------------

sub getGID{
    my (
        $self
    ) = @_;
    return $self->{'GID'};
}

#----------------------------------------------------------------------------

sub getProgramPath{
	return $_[0]->{path};
}

sub getDataPath;
*getDataPath = \&getProgramPath;

#----------------------------------------------------------------------------

sub getProductName{
    return $gProductNameClient;
}

#----------------------------------------------------------------------------

sub Register{
    my ($self) = @_;
    if ($self->isSapmntInstallation()) {
        $self->AddMessage ("Skipping local client registration of sapmnt installation");
        return undef if (!$self->updateSapmntSymlink());
        return 0;
    }
    return $self->SUPER::Register ();
}

sub updateSapmntSymlink {
    my ($self) = @_;

    my $instconfig = $self->getConfiguration ();
    my $versiondir = $instconfig->getVersionDirName();
    my $link       = $self->getSapmntSymlinkPath();
    my $builtin    = SDB::Common::BuiltIn->get_instance();

    if (File::stat::stat($link)) {
        $builtin->unlink($link);
    }
    $self->AddMessage ("Creating symlink $link' => '$versiondir'");
    if (!$builtin->symlink($versiondir, $link)) {
        $self->AddError ("Cannot create symlink '$link' => '$versiondir': $!");
        return undef;
    }
    if (!$>){
        lchown ($self->getUID(), $self->getGID(), $link);
    }
    if (!defined $self->makeWritableForOwner ($instconfig->getValue ('PATH'))){
        return undef;
    }
    return 1;
}

sub isSapmntInstallation {
    my ($self) = @_;
    my $instconfig = $self->getConfiguration();
    return defined $instconfig && defined $instconfig->getValue('Target') && !$isWin;
}

sub getSapmntInstallationDir {
    my ($self) = @_;
    my $sep = quotemeta ($path_separator);
    my $installationPath = $self->getProgramPath();
    my ($basepath) = $installationPath =~ /(.*)$sep[^$sep]+$/;
    return $basepath;
}

sub getSapmntSymlinkPath {
    my ($self) = @_;
    my $basepath = $self->getSapmntInstallationDir();
    return File::Spec->catfile($basepath, $hdbclient_link);
}

sub makeWritableForOwner{
    my ($self, $path) = @_;
    require SDB::Install::DirectoryWalker;
    my $dirWalker = SDB::Install::DirectoryWalker->new(
            sub{   # actionMatcher callback, see comments in DirectoryWalker.pm for full signature
                my $statbuf = $_[7];
                if (($statbuf->[2] & 0200) == 0){
                    # owner has no write bit
                    return 1;
                }
                return 0;
            },
            undef, undef, undef, undef, undef,
            sub{
                my $statbuf = $_[7];
                my $msglst = $_[1];
                my $base = new SDB::Install::Base();
                $base->setMsgLstContext ([$_[0],$_[1]]);
                my $path = join ($path_separator, $_[3], $_[4]);
                my $newMode = ($statbuf->[2] | 0200) & 07777;
                if (!chmod ($newMode, $path)){
                    $base->appendErrorMessage ("Cannot change mode of '$path': $!");
                    return 0;
                }
                $msglst->addMessage (sprintf ("Set permission mask of file '$path' to 0%o", $newMode));
                return 1;
            },
            undef,undef,
            1,     # no breadth-first traversal, so we do it depth-first.
            1,    # dont collect list
            0     # follow symlinks
    );
    my $msg = $self->getMsgLst ()->addMessage ("Adding write bits for owner...");
    $dirWalker->setMsgLstContext ([$msg->getSubMsgLst()]);
    if (!$dirWalker->findAndProcess ($path)){
        $self->setErrorMessage ("Cannot change mode of files", $dirWalker->getErrMsgLst());
        return undef;
    }
    return 1;
}



#----------------------------------------------------------------------------

sub Unregister{
    my ($self) = @_;
    
    return $self->SUPER::Unregister() if ($isWin);

    my $sep = quotemeta ($path_separator);
    my $basepath = $self->getSapmntInstallationDir();
    my $path = $self->getProgramPath();
    my $versiondir = ($path =~ /.*$sep([^$sep]+)$/);
    my $link       = $self->getSapmntSymlinkPath();

    if (-l $link && isSameFile ($link, $self->{path})){
        if (!opendir (DH, $basepath)){
            $self->AddError ("Cannot open directory '$basepath': $!");
            return undef;
        }
        my $pattern = '^' . quotemeta ($versiondir) . '$';
        my @entries = grep {/^HDB_CLIENT_/ && !/$pattern/} readdir (DH);
        closedir (DH);
        my @statbuf;
        my $fullpath;
        my $result = '';
        my $ctime = 0;
        foreach my $clientdir (@entries){
            $fullpath = $basepath . $path_separator . $clientdir;
             @statbuf = stat ($fullpath);
             if ($statbuf [10] > $ctime){
                $ctime = $statbuf [10];
                $result = $clientdir;
             }
        }

        unlink ($link);
        if ($result){
            $versiondir = $result;
            $self->AddMessage ("Creating symlink $link' => '$versiondir'");
            if (!symlink ($versiondir, $link)){
                $self->AddError ("Cannot create symlink '$link' => '$versiondir': $!");
                return undef;
            }
            if (!$>){
                lchown ($self->getUID(), $self->getGID(), $link);
            }
        }
    }
    return $self->SUPER::Unregister ();
}

#----------------------------------------------------------------------------

1;
