#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#
# Desc: base class of software packages


package SDB::Install::Kit;

use SDB::Install::PackageManager;
use SDB::Install::Package::Installable;
use SDB::Install::Globals qw ($gPhasePrepareOption);
use SDB::Install::SysVars qw ($isWin $isApple $path_separator $installerIs64bit);
use SDB::Install::System qw (copy_file
                             exec_program
                             getFileSystemInfo
                             getMountPoint
                             removeEmptyDirs);
use SDB::Install::Tools qw (divideIntegrally);
use SDB::Install::Installer;
use SDB::Install::Manifest;
use SDB::Install::DebugUtilities;
use File::Basename qw (dirname basename);
use File::Spec;
use File::stat qw(); # Do not import anything explicitly
use SDB::Install::System::Mounts;
use SDB::Install::Version;
use SDB::Install::System qw (listDir isLink);
use Cwd qw (realpath);
use strict;

our @ISA = qw (SDB::Install::PackageManager);

our @EXPORT = qw (ScanFilesForVCRedists InstallVCRedists);

our $index_file_name = '.sdb_pindex';
our $deprecatedPackagesFilename = 'deprecatedPackages.txt';

#----------------------------------------------------------------------------

sub new {
	my $self = shift->SUPER::new ();
	($self->{archive_dir},$self->{sysinfo}) = @_;
	$self->{debugPackagesEnabled} = 1;
	#$self->{packages} = {};
	my $manifest = $self->getManifest();
	if (defined $manifest){
		$self->AddMessage ("Using manifest file '" . $manifest->getFileName () .
		 "'");
		if ($self->isStudio() && !$self->isa('SDB::Install::Kit::DatabaseStudio')) {
			require SDB::Install::Kit::DatabaseStudio;
			bless ($self, 'SDB::Install::Kit::DatabaseStudio');
		} elsif ($self->isClient() && !$self->isa('SDB::Install::Kit::Client')) {
			require SDB::Install::Kit::Client;
			bless ($self, 'SDB::Install::Kit::Client');
		}
	}
	else{
		$self->AddWarning ("No manifest file found");
	}
	return $self;
}

sub shouldVerifyChecksums {
	my ($self, $instconfig) = @_;
	if (!defined($self->{verifyChecksums})) {
		$self->{verifyChecksums} = $instconfig->isCalledBySHA() && !$self->areArchivesOnReadOnlyLocalFs();
	}
	return $self->{verifyChecksums};
}

sub areArchivesOnReadOnlyLocalFs {
	my ($self) = @_;
	my $packagesDir = $self->getArchiveDir();

	my $mounts = SDB::Install::System::Mounts->new();
	my @roMounts = grep {$_->{mnt_type} !~ /^(nfs|cifs|smb)$/} @{$mounts->getMountsWithOption('ro')};
	return 0 if (!@roMounts);

	my $archiveDirContent = listDir($packagesDir);
	my @archives = grep {$_ =~ /\.tgz$/i} @$archiveDirContent;
	my @symlinkArchives = grep { isLink(File::Spec->catfile($packagesDir, $_)) } @archives;
	if (!@symlinkArchives) {
		return $self->_isDirUnderMount($packagesDir, \@roMounts);
	} elsif (@symlinkArchives < @archives) {
		return 0 if (!$self->_isDirUnderMount($packagesDir, \@roMounts));
	}
	for my $symlink (@symlinkArchives) {
		my $readlink = File::Spec->catfile($packagesDir, readlink($symlink));
		my $linkDir = dirname($symlink);
		return 0 if (!$self->_isDirUnderMount($linkDir, \@roMounts));
	}
	return 1;
}

sub _isDirUnderMount {
	my ($self, $dirPath, $mounts) = @_;
	my $dirStat = File::stat::stat($dirPath);
	for my $mount (@$mounts) {
		my $mountStat = File::stat::stat($mount->{mnt_dir});
		return 1 if ($dirStat->dev() == $mountStat->dev());
	}
	return 0;
}

#----------------------------------------------------------------------------

sub FindArchives{
	my ($self, $instconfig) = @_;
	my $stat_archdir = File::stat::stat ($self->{archive_dir});
    $self->{packages} = {};
	unless ($stat_archdir and $stat_archdir->mode() & 040000){
		$self->AddError ("Archive directory \"".$self->{archive_dir}.
				 "\" doesn\'t exist");
		return undef;
	}
	my $stat_index = File::stat::stat ($self->{archive_dir}.'/'.$index_file_name);
	if ($stat_index and $stat_index->ino() & 0100000){
		if ($stat_index->mtime() <  $stat_archdir->mtime()){
			$self->AddWarning ("package index file outdated, ignore it");
		}
		else{
			my $rc = $self->ScanIndex();
			if (defined $rc){
				return $rc;
			}
		}
	}
	return $self->ScanDir ($instconfig);
	
}

#----------------------------------------------------------------------------

sub getArchiveDir{
    return $_[0]->{archive_dir};
}

#----------------------------------------------------------------------------

sub getLicenseFile{
    return '';
}

#----------------------------------------------------------------------------

sub getProductKey{
    return '';
}

#----------------------------------------------------------------------------

sub getManifestDir;
*getManifestDir = \&getArchiveDir;

#----------------------------------------------------------------------------

sub GenPackageList {
	my ($self, $instconfig) = @_;
	my $retval = $self->{'packages'};    
	if (!defined $retval){
		my $retcode = $self->FindArchives ($instconfig);
		if(defined $retcode) {
			$retval = $self->{'packages'};
		}
		else {
			$self->AddError('SDB::Install::Kit::GenPackageList: could not generate package list');
		}
	}
	return $retval;
}

#----------------------------------------------------------------------------

sub ScanIndex{
	my ($self) = @_;
	$self->AddWarning ("found index file, but feature isn\`t implemented yet");
	return undef;
}

#----------------------------------------------------------------------------

sub ScanDir{
	my ($self, $instconfig) = @_;
	my @files;
	unless (opendir (DH, $self->{archive_dir})){
		$self->AddError ("Cannot open archive directory \"".$self->{archive_dir}."\": $!");
		return undef;
	}
	
	@files = readdir (DH);
	closedir (DH);	
	$self->{archive_dir_content} = [@files];
	my @packages = grep {!/^sdbinst\.tgz$|^resources\.tgz$|^wxperl\.tgz$|\.debug\.tgz$/i and /\.tgz$/i} @files;
	
	$self->ResetError ();
	foreach my $package (@packages){
		
		if (exists $Wx::{wxTheApp}){
			Wx::Yield ();
		}
		$self->CheckArchive ($package, $instconfig);
	}
	if ($self->ErrorState){
		return undef;
	}
	return 1;
}

#----------------------------------------------------------------------------

sub determineLSTFilepath {
	my ($self, $packageName, $instconfig) = @_;
	my $archiveDir = $self->{archive_dir};
	my @candidates = ();
	my $subdir = $self->isServer() ? 'server' : $self->isClient() ? 'client' : undef;
	if ($instconfig->isCalledBySHA() && defined($subdir)) {
# The server and client need escalation of privileges, so the LST files should be first
# searched in the installer dir (as it is potentially safe), and after that in the archive dir
		my $installerDir = SDB::Install::Installer->new()->getInstallerDir();
		push @candidates, File::Spec->catfile($installerDir, $subdir, "$packageName.lst");
		push @candidates, File::Spec->catfile($installerDir, $subdir, "$packageName.LST");
	}

	push @candidates, File::Spec->catfile($archiveDir, "$packageName.lst");
	push @candidates, File::Spec->catfile($archiveDir, "$packageName.LST");

	for my $candidate (@candidates) {
		return $candidate if File::stat::stat($candidate);
	}
	return undef;
}

#----------------------------------------------------------------------------

sub AddPackage{
	my ($self,$package,$inst) = @_;
	
	if (defined $package->{tree_node}){
		return 1;
	}
		
	$package->CheckDependenciesByKit ($self);
	
	if (defined $inst){
		$package->CheckDependenciesByInstallation ($inst);
	}
		
	if (!defined $package->{dependencies} || !%{$package->{dependencies}}){
		#package has no depenedencies
		unless (defined $self->{tree}){
			$self->{tree} = {};
		}
		$self->{tree}->{$package->{id}} = {};
		$package->{tree_node} = $self->{tree}->{$package->{id}};
		return 1;	
	}
	
	my $rc = 1;
	
	my @err_list;
	
	foreach my $dependency (keys (%{$package->{dependencies}})){
			
			my $resolved = 0;
			my $parent_node;
			if ($package->{dependencies}->{$dependency}->{resolved_by_kit}){
				if (defined $self->AddPackage($self->{packages}->{$dependency}, $inst)){
					$parent_node = $self->{packages}->{$dependency}->{tree_node};
					$resolved = 1;
				}
			}		
			
			if (!$resolved and $package->{dependencies}->{$dependency}->{resolved_by_installation}){
				unless (defined $self->{tree}){
					$self->{tree} = {};
				}
				$parent_node = $self->{tree};
				$resolved = 1;
										
			}
			
			if ($resolved){
				if (exists $package->{tree_node}){
					$parent_node->{$package->{id}} = $package->{tree_node};
				}
				else{
					$parent_node->{$package->{id}} = {};
					$package->{tree_node} = $parent_node->{$package->{id}};
				}	
			}
			else{
				push @err_list, $self->GenMsg ('ERR','unresolved dependency: '.$package->{dependencies}->{$dependency}->{str});
				$rc = undef;
			}
	}
	$self->{last_error} = \@err_list;
	return $rc;
}

#----------------------------------------------------------------------------

sub SelectPackagesByInstallation{
	my ($self, $installation) = @_;
	
	if ( !defined $installation or !$installation->isa ('SDB::Install::Installation')){
		$self->AddError ('Parameter is no valid installation');
		return 0;
	}
	
	my $rc = 1;
		
	foreach my $id (keys %{$installation->{packages}}){
		if (exists $self->{packages}->{$id}){
			if ($self->{packages}->{$id}->{selected}){
				next;
			}
			$self->SelectPackage ($self->{packages}->{$id});
		}
		else{
			$self->AddWarning ("package $installation->{packages}->{$id}->{data}->{name} is not in installation kit");
		}
	}
	
	return $rc;
}

#----------------------------------------------------------------------------

sub CheckArchive {
	my ($self, $archiveName, $instconfig) = @_;
	my $msg = $self->AddMessage ("Checking archive $archiveName");

	my $lstFilePath = $self->determineLSTFilepath($archiveName, $instconfig);
	my $archivePath = File::Spec->catfile($self->{archive_dir}, $archiveName);
	my $shouldVerifyChecksums = $self->shouldVerifyChecksums($instconfig) ? 1 : 0;
	my $package = SDB::Install::Package::Installable->new($archivePath, $lstFilePath, $shouldVerifyChecksums);

	$package->setMsgLstContext([$msg->getSubMsgLst()], 1);
	unless (defined $package->init ($self->{archive_dir})){
		$self->PushError ("Cannot initialize archive \"$archiveName\"", $package);
		return undef;
	}
	
	my $packdata = $package->{data};
	$package->LogInformations ();

	if (exists $self->{packages}->{$package->{id}}){
		my $pattern = quotemeta ($self->{archive_dir} . '/');
		my $pack = $self->{packages}->{$package->{id}}->{archive};
		$pack =~ s/$pattern//;
		$self->PushError ('Package id \''.$package->{id}. "\' exists more than once in installation kit! ('$archiveName', '$pack')");
		return undef;
	}

	unless (defined $package->IsRunnable ($self->{sysinfo}, $instconfig)){
		$self->PushError ('Checking package ' . $packdata->{name} . ' failed' ,$package);
		return undef;
	}
	$self->{packages}->{$package->{id}} = $package;

	return 1;
}

#----------------------------------------------------------------------------

sub Preinstall{
	my ($self, $installation, $repair_mode, $force_downgrade) = @_;
	
	if (!defined $installation){
		$self->AddError ('Installation is not defined');
		return undef;
	}

	$installation->{kit} = $self;
	$self->{installation} = $installation;
	if ($installation->RegistryFileExists()){
		if (!defined $installation->GenPackageList ()){
			$self->AddError ('Cannot get installed packages', $installation);
			return undef;
		}
	}
	
	my $packages = $self->GetPackages ($installation);	
	
	unless (defined $packages){
		$self->AddError ('Insufficient installation kit', $self);
		return undef;
	}

	my $instconfig = $installation->getConfiguration ();
	if ($isWin && (!defined $instconfig || !$instconfig->getValue ('SkipVCRedist'))){
		$self->{vc_redistributable} = ScanFilesForVCRedists(
			$self->{archive_dir},
			$self->{archive_dir_content},
			$self->{sysinfo},
			$self
		);
	}

	my $progress_handler = $self->GetProgressHandler ();

	if (defined $progress_handler){
		my $package_num = 0;
		my $files = 0;
		foreach my $package (@$packages){
			$package_num++;
			$files += $package->GetNumberOfFiles ();
		}
		$progress_handler->InitProgress ($package_num, 0);
	}

	foreach my $package (@$packages){
		$package->{installation} = $installation;
		my $packdata = $package->{data};
		my $msg = $self->AddProgressMessage ("Preparing package '$packdata->{name}'...");
		$package->setMsgLstContext ([$msg->getSubMsgLst()], 1);
		if (exists $installation->{packages}->{$package->{id}}){
				unless (defined $package->CheckUpdate ($repair_mode, $force_downgrade)){
					$self->AddError ("Cannot update package '$packdata->{name}'", $package);
					return undef;
				}
		}

		$msg = $self->AddMessage ("Checking package '$packdata->{name}'...");
		$package->setMsgLstContext ([$msg->getSubMsgLst()], 1);
		unless (defined $package->Preinstall ()){
			$self->AddError ("Package check of '$packdata->{name}' failed", $package);
			return undef;
		}
	}
	
	$packages = $self->GetPackages ();
	
	foreach my $package (@$packages){
		if ($package->IsUpdate){
			$package->{installed_package}->PreLockedFilesCheck ();
		}
		
	}
	$installation->setMsgLstContext($self->getMsgLstContext ());
	if (!$installation->checkDependenciesAfterUpgrade ($self)){
		return undef;
	}
		
	
	$self->{installation} = $installation;
	
	return 1;
}

#----------------------------------------------------------------------------

sub Install{
	my ($self) = @_;
	my $rc = 1;
	delete $self->{archive_dir_content};
	my $installation = $self->{installation};
	$installation->setMsgLstContext ($self->getMsgLstContext ());
	unless (defined $installation->OpenExclusive ()){
		return undef;
	}

	if ($self->skipInstallationSteps()) {
		$self->AddMessage ("Installation is up-to-date", undef);
		return $rc;
	}
	
    my $deprecatedPackageIds = $self->getDeprecatedPackageIds();
    $installation->RemoveDeprecatedPackages($deprecatedPackageIds);
    
	my $packages = $self->GetPackages ();
	
	my $progress_handler = $self->GetProgressHandler();

	my @names;

	if (defined $progress_handler){
		my $package_count = 0;
		if (defined $self->{vc_redistributable}){
			$package_count = 1;
			@names = ("Installing Microsoft C/C++ Runtime");
		}
		my $label;	
		foreach my $package (@$packages){
            $label = $package->getInstallProgressMessage();
			push @names, $label;
			$package_count ++;
		}
		
		if (defined $self->{append_progress_steps}){
			foreach my $step (@{$self->{append_progress_steps}}){
				push @names, $step;
				$package_count ++;
			}
		}
		
		
		$progress_handler->InitProgress ($package_count, 1,\@names);
	}

	my %dep_errors;


	my $update = 0;

	if (defined $self->{vc_redistributable}){
		my $pp_handler = $self->GetPackageProgressHandler ();
				
		if (defined $pp_handler){
			$pp_handler->InitProgress (4 + scalar @{$self->{vc_redistributable}},1);
		} 
		my $success = 0;
        ($success, $self->{reboot_required}) = InstallVCRedists(
            $self->{vc_redistributable},
            $self,
            $progress_handler,
            $pp_handler
        );
        if(!$success) {
            return undef;
        }
	}

    my $extFileLists = {};
    my $extList;

	foreach my $package (@$packages){
		my $msg;
		
		if (exists $dep_errors{$package->{id}}){
			$self->PushError ("Skipping installation of package ".$package->GetName." due to previous errors (root cause package $dep_errors{$package->{id}})");
			if (defined $progress_handler){
				$progress_handler->StepFinished(1);
			}
			next;
		}
		
		$extList = $package->getExternalFileListName();
		
		if (defined $extList){
		    $extFileLists->{$extList} = 1;
		}
		
		my $packdata = $package->{data};
		my $msgtxt = $package->getInstallProgressMessage();
		if ($package->IsUpdate) {
			require SDB::Install::Configuration::Upgrade;
			if (defined $self->getInstallation() && defined $self->getInstallation()->getConfiguration()) {
				if ($self->getInstallation()->getConfiguration()->{params}->{PrepareUpgrade}->{value}
				|| $self->getInstallation()->getConfiguration()->{params}->{Phase}->{value} eq $gPhasePrepareOption) {
					# preparation only, change output
					$msgtxt =~ s/Updating/Extracting/;
				}
			}
		}
		$msg = $self->AddProgressMessage($msgtxt . '...');
		my $msglst = $msg->getSubMsgLst ();
		$msglst->setProgressHandler ($self->GetPackageProgressHandler ());
		$package->setMsgLstContext ([$msglst]);
		my $error = 0;
		if ($package->IsUpdate){
			$update = 1;
			if (!defined $package->Update ()){
				$self->PushError ('Cannot update package '.$packdata->{name},$package);
				$error = 1;
			}
		}
		else{
			if (!defined $package->Install ()){
				$self->PushError ('Cannot install package '.$packdata->{name},$package);
				$error = 1;
			}
		}

		if (!$self->copyLstFile($package)) {
			$error = 1;
		}
		
		if ($error){
			undef $rc;
			my $deppackages = $self->GetDependentPackages( $package, 1);
			if (defined $deppackages){
				foreach my $deppackage (@$deppackages){
					$dep_errors{$deppackage->{id}} = $package->GetName ();
				}
			}
			if (defined $progress_handler){
				$progress_handler->StepFinished(1);
			}	
		}
		else{
			if (defined $progress_handler){
				$progress_handler->StepFinished();
			}
		}
	}
	
	#
	# extracted:  $self->{archive_dir} => server, client, studio, or packages
	#

	if (!$self->tryCopySwidtagFile($installation)) {
	    return undef;
	}

    $installation->setMsgLstContext ([$self->getMsgLst()]);
    foreach $extList (keys %$extFileLists){
        if (!defined $installation->syncExtFileList ($extList)){
            $self->AddError ("Cannot maintain external file list '$extList'", $installation);
            return undef;
        }
    }
	
	
	if (!defined $installation->Register ($self)){
		$self->AddError ('Cannot register installation', $installation);
		return undef;
	}

	if (0 && $update){

		my $config = {};
		removeEmptyDirs ($installation->GetProgramPath (), $config);
		$self->AddMessage ("Deleting empty program directories after update", $config);

	}

	return $rc;
}

#----------------------------------------------------------------------------

sub copyLstFile {
	my ($self, $package) = @_;
	my $lstFile = $package->getPackageLSTFile();
	my $sourceLstPath = $lstFile ? $lstFile->getOwnPath() : undef;
	if (!$sourceLstPath) {
		$self->AddMessage(sprintf("Skipping copying of .LST file of package '%s' as it does not exist", $package->GetName()));
		return 1;
	}

	my $targetLstPath = File::Spec->catfile($self->getInstallation()->getTargetLstDir(), basename($sourceLstPath));
	my $cfg = {
		createdir	=> 1,
		dir_mode	=> 0755,
		uid			=> $isWin ? undef : $package->getUID(),
		gid			=> $isWin ? undef : $package->getGID(),
	};

	$self->AddMessage("Copying '$sourceLstPath' to '$targetLstPath'");
	if(!copy_file($sourceLstPath, $targetLstPath, $cfg)) {
		$self->AddError("Cannot copy .LST file '$sourceLstPath' to '$targetLstPath'", $cfg);
		return 0;
	}

	return 1;
}

#----------------------------------------------------------------------------

sub tryCopySwidtagFile {

    my ($self, $installation) = @_;

    my $parentArchiveDir = dirname($self->{archive_dir});

    if (!-d $parentArchiveDir) {
        $self->AddError("Cannot copy '.swidtag' file "
                . "(parent directory of '$self->{archive_dir}' not found)");
        return 1;
    }

    if (!opendir (DH, $parentArchiveDir)) {
        $self->AddError("Cannot read archive directory '$parentArchiveDir': $!");
        return undef;
    }

    my @swidtagFiles = grep {/.swidtag$/} readdir (DH);
    closedir (DH);
    my $cntSwidtagFiles = scalar(@swidtagFiles);

    if ($cntSwidtagFiles == 0) {
        $self->AddMessage ("File '*.swidtag' not found under '$parentArchiveDir'");
    }
    elsif ($cntSwidtagFiles > 1) {
        $self->AddMessage("'*.swidtag' files not copied, different files found: '"
                          . join("', '", @swidtagFiles) . "'");
    }
    else {
        my $cfg      = {'binmode' => 1};
        my $swidtag  = $parentArchiveDir . $path_separator . $swidtagFiles[0];
        my $destin   = $installation->getProgramPath();
        my $destFile = $destin . $path_separator . $swidtagFiles[0];

        if (-f $destFile) {
            unlink $destFile;
        }

        if (!copy_file($swidtag, $destin, $cfg)) {
            $self->AddError ("Cannot copy file '$swidtag' to '$destin'", $cfg);
            return undef;
        }

        $self->AddMessage
            ("File '$swidtagFiles[0]' copied from '$parentArchiveDir' to '$destin'");
    }
    return 1;
}


#----------------------------------------------------------------------------

sub getInstallation{
	return $_[0]->{installation};
}

#----------------------------------------------------------------------------


sub Reset ($){
	foreach my $package (values (%{$_[0]->{packages}})){
		$package->Reset ();
	}
	delete $_[0]->{installation};
}

#----------------------------------------------------------------------------

sub EnableDebugPackages ($){
	
	my $new_value =  defined $_[1] ? ($_[1] ? 1 : 0) : 1;  
	
	if ($_[0]->{debugPackagesEnabled} xor $new_value){
		$_[0]->{debugPackagesEnabled} = $new_value;
		foreach my $package (values %{$_[0]->{packages}}){
			if (defined $package->{debugPackage}){
				$package->{debugPackage}->{skip} = 	!$new_value;
			}
		}	
	}
}

#----------------------------------------------------------------------------


sub HaveDebugPackages ($){
	
	foreach my $package (values %{$_[0]->{packages}}){
		if (defined $package->{debugPackage}){
			return 1;
		}
	}
	return 0;
}

#----------------------------------------------------------------------------


sub FreePackages {
	delete $_[0]->{installation};
	$_[0]->SUPER::FreePackages();
	return 1;
}


#----------------------------------------------------------------------------

sub SelectPackage{
	my ($self,$package) = @_;
	
	if (!defined $package){
		return 0;
	}
	
	if(!exists $self->{'packages'}->{$package->{'id'}}){
		$self->AddError ("Package \"$package->{data}->{name}\" not found");
		return 0;
	}
		
	my $required_packages = $self->GetRequiredPackages ($package);
	
	if (! defined $required_packages){
		return 0;
	}
	
	foreach my $required_package (@$required_packages){
		if (!$self->SelectPackage ($required_package)){
			return 0;
		}
	}

	$package->{'selected'} = 1;

	return 1;
}

#----------------------------------------------------------------------------

sub UnselectPackage{
    my ($self,$package) = @_;
    
    if (!defined $package){
        return 0;
    }
    
    if(!exists $self->{'packages'}->{$package->{id}}){
        $self->AddError ("Package \"$package->{data}->{name}\" not found");
        return 0;
    }
    
    if (exists $package->{'tree_node'}){
        foreach my $child_id (keys %{$package->{tree_node}}){
            if (!$self->UnselectPackage ($self->{'packages'}->{$child_id})){
                return 0;
            }
        }
    }

    $package->{'selected'} = 0;

    return 1;
}

#----------------------------------------------------------------------------

sub getDeprecatedPackageIds{
    my ($self) = @_;
    my $absFileName = $self->{'archive_dir'}.'/'.$deprecatedPackagesFilename;
    my @retval;
    if(-f $absFileName) {
        if(!open (FH, $absFileName)) {
            $self->AddError ("Cannot open file $absFileName: $!");
            return undef;
        }
        while(<FH>) {
            chomp;
            my $line = $_;
            $line =~ s/^\s(.*)\s$/$1/; #trim whitespace
            if($line !~ /^#/ and $line !~ /^$/) {
                # not a comment and not empty:
                push @retval, $line;
            }
        }
        close(FH);
    }
    return \@retval;    
    
}

#----------------------------------------------------------------------------

sub setDataFootprints{
    $_[0]->{data_footprints} = $_[1];
}

#----------------------------------------------------------------------------

sub getFootprintsByFilesystem {
    my ($self) = @_;

    #
    # get footprints of installation packages
    #

    my ($packageManagerFootprints, $success, $errmsg) = $self->SUPER::getFootprintsByFilesystem ();

    if (!$success || !defined $self->{data_footprints} || !@{$self->{data_footprints}}){
        return ($packageManagerFootprints, $success, $errmsg);
    }

    foreach my $mnt (values %$packageManagerFootprints){
        $mnt->{stakeholder} = ['program files'];
    }


    #
    # add footprints reserved for data (e.g. db volumes)
    #

    my ($path,$required_space, $paramName);
    my $retval                        = $packageManagerFootprints;
    $success                       = 1;
    $errmsg                        = undef;
    my $mountPoint                    = undef;
    my $deviceId                      = undef;
    my $fileSystemTotalSizeBlocks     = undef;
    my $filesystemAvailableSizeBlocks = undef;
    my $filesystemBlockSizeBytes      = undef;
    my $footprints                    = undef;

    foreach my $item (@{$self->{data_footprints}}){
        ($path,$required_space,$paramName) = @$item;
        # if > 1, the scaling factor modifies the unit of fields
        # 'fileSystemTotalSizeBytes' and 'filesystemAvailableSizeBytes'.
        # e.g. if scalingFactor is 1024 (1024*1024), the unit is kilobytes
        # (megabytes).
        my $scalingFactor                 = undef;
        # in the unix case, we want to cache mount points which we have already found:
        my $footprintIncrement            = undef;
        ($mountPoint, $success, $errmsg) =
            getMountPoint($path, undef);

        if(!$success) {
            return (undef, $success, $errmsg);
        }
        if (defined $retval->{$mountPoint}){
            $footprints = $retval->{$mountPoint};
            $filesystemBlockSizeBytes = $footprints->{'filesystemBlockSizeBytes'};
        }
        else{
            $retval->{$mountPoint} = $footprints = {};
            ($deviceId,
             $fileSystemTotalSizeBlocks,
             undef,
             $filesystemAvailableSizeBlocks,
             $filesystemBlockSizeBytes,
             $scalingFactor,
             $success,
             $errmsg) =
                getFileSystemInfo($mountPoint);
            if(!$success) {
                return (undef, $success, $errmsg);
            }
            $footprints->{'stakeholder'} = [];
            $footprints->{'deviceId'}                                    = $deviceId;
            $footprints->{'filesystemBlockSizeBytes'}                    = $filesystemBlockSizeBytes;
            $footprints->{'fileSystemTotalSizeBlocks'}                   = $fileSystemTotalSizeBlocks;
            $footprints->{'filesystemAvailableSizeBlocks'}               = $filesystemAvailableSizeBlocks;
            $footprints->{'scalingFactor'}                               = $scalingFactor;
            $footprints->{'estimatedNewInstallationFootprintBlocks'}     = 0;
            $footprints->{'estimatedUpdatedInstallationFootprintBlocks'} = 0;
            $footprints->{'actualInstallationFootprintBlocks'}           = 0;
        }
        push @{$footprints->{'stakeholder'}}, $paramName;
        $footprintIncrement = divideIntegrally($required_space, $filesystemBlockSizeBytes) + 1;
        $footprints->{'estimatedNewInstallationFootprintBlocks'}     += $footprintIncrement;
        $footprints->{'estimatedUpdatedInstallationFootprintBlocks'} += 1; ## TODO: implement this, the value may be negative !!!
        $footprints->{'actualInstallationFootprintBlocks'}           += 1; ## TODO: implement this using 'stat' !!!
    }
    return ($retval, $success, $errmsg);
}

#----------------------------------------------------------------------------

sub is64bit {
	return $installerIs64bit;
}

#----------------------------------------------------------------------------

sub canUpdate{
	my ($self, $inst, $instconfig, $errlst) = @_;
	if (!defined $errlst){
		$errlst = new SDB::Install::MsgLst ();
	}
	my $pn = $self->getShortProductName();
	if ($self->is64bit() xor $inst->is64bit()){
		$errlst->AddError ("Cannot update ".($inst->is64bit() ? "64" : "32")." bit $pn with "
		.($self->is64bit() ? "64" : "32")." bit installation package.");
		return 0;
	}

	my $version = new SDB::Install::Version (split ('\.',$self->GetVersion ()));
	my $installed_version = new SDB::Install::Version (split ('\.', $inst->GetVersion()));

	if ($installed_version->isNewerThan ($version)){
		$errlst->AddError ("No update possible: Version " . $self->GetVersion() . " is lower than " . $inst->GetVersion());
		if (defined $instconfig && $instconfig->getIgnore ('check_version')){
			$errlst->AddMessage ("Ignore error due to ignore option 'check_version'");
		}
		else{
			return 0;
		}
	}
	return 1;
}

#----------------------------------------------------------------------------

sub ScanFilesForVCRedists{
    my (
        $archiveDir,
        $filesRef,
        $sysInfo,
        $msgHandler
    ) = @_;
    my $retval = undef;
    my @files = @$filesRef;
    require SAPDB::Install::System::Win32::VersionInfo;

    my $msg = $msgHandler->AddMessage ("Looking for Microsoft C/C++ runtime redistributable");
    my @msvcr_redistributable = grep {/^\w*vcredist_\w*\.exe$|^vcredist\.msi$/} @files;
    if ($sysInfo->{'arch'} eq 'I386'){
        @msvcr_redistributable = grep {!/_x64/i} @msvcr_redistributable
    } 
    if (@msvcr_redistributable){
        require SDB::Install::System::VCRedistCheck;
        my $vcrc = new SDB::Install::System::VCRedistCheck();
        $vcrc->scan ();
        $msgHandler->AddSubMsgLst ($msg, $vcrc);
        my @list;
        my $version;
        foreach my $file (@msvcr_redistributable){
            $vcrc->resetMsgLstContext ();
            if (!$vcrc->checkRedistributable ($archiveDir. '\\' .$file)){
                push @list, $archiveDir. '\\' .$file;
            }
            $msgHandler->AddSubMsgLst ($msg, $vcrc);
        }
        if (@list){
            $retval = \@list;
        }
    }
    else{
        $msgHandler->AddSubMessage ($msg, "No Microsoft C/C++ runtime redistributable ('vcredist') in installation kit");
    }
    return $retval;
}

#----------------------------------------------------------------------------

sub InstallVCRedists{
    my (
        $vcRedistributables,
        $msgHandler,
        $progressHandler,         # optional
        $packageProgressHandler   # optional
    ) = @_;
    my $retval = 0; ## $self->{reboot_required}
    my $prog;
    my @args;
    my $success = 1;
    my $msg = $msgHandler->AddProgressMessage ("Installing Microsoft C/C++ Runtime...");
    foreach my $vc_redistributable (@{$vcRedistributables}){
        if ($vc_redistributable =~ /\.exe/i){
            $prog = $vc_redistributable;
            if($vc_redistributable =~ /(msdev2008|msdev2010|msdev2013|msdev2015)/i) {
                @args = ('/q');
            }
            else {
                @args = ('/Q:a', '/c:msiexec.exe /qn /i vcredist.msi');
            }
        }
        else {
            $prog = 'msiexec.exe';
            @args = ('/i', $vc_redistributable, '/qn');
        }
        my $cfg = {};
        my $rc = exec_program ($prog,\@args,$cfg);
        if(!defined $rc || $rc != 0) {
            if($rc == 3010) {
                $msgHandler->AddMessage("Microsoft C/C++ Runtime installation requires a reboot!");
                $retval = 1;
            }
            elsif($rc == 5100) {
                $msgHandler->AddMessage("Microsoft C/C++ Runtime installed on this machine is already newer.");
            }
            else {
                $msgHandler->PushError("Cannot install Microsoft C/C++ Runtime", $cfg);
                $success = 0;
            }
        }
        $msgHandler->AddSubMsgLst($msg, $cfg);
        if (defined $packageProgressHandler){
            $packageProgressHandler->SetProgress ();
        }
    }
    if (defined $packageProgressHandler){
        foreach (0..3){
            $packageProgressHandler->SetProgress ();
        }
    } 
    if(defined $progressHandler) {
        if (!$success){
            $progressHandler->StepFinished(1);
        }
        else{
            $progressHandler->StepFinished();
        }
    }
    return ($success, $retval);
}

#----------------------------------------------------------------------------

sub isUpdate {

	my ($self) = @_;
	my $pkgs = $self->GetPackages();
	foreach my $pkg (@$pkgs){
		if ($pkg->IsUpdate()) {
			return 1;
		}
	}
	return 0;

}

#----------------------------------------------------------------------------

sub skipInstallationSteps {

	my ($self) = @_;
	my $pkgs = $self->GetPackages();

	if (!defined $pkgs || !@$pkgs){
		return 1;
	} else {
		return 0;
	}

}

#----------------------------------------------------------------------------
#returns the return code of the p2director call - 0:success 13:crash <other>:warning
sub callDirector {
    my ($self,$path,$args,$msglst,$cfg) = @_;
    if (!defined $msglst){
        $msglst = $self;
    }
    if (!defined $cfg){
        $cfg = {};
    }

    my $prog = join ($path_separator, $path, 'director',
        $isWin ? 'eclipse.exe': $isApple ? 'Eclipse.app/Contents/MacOS/eclipse' : 'eclipse');

    if (! -f $prog && $isApple){
        my $fallback = $path . $path_separator .'director/eclipse.app/Contents/MacOS/eclipse';
        if (-f $fallback){
            $prog = $fallback;
        }
    }

    my @args = (
        '-application', 'org.eclipse.equinox.p2.director',
        @$args
    );
    
    my $rc = exec_program ($prog, \@args,$cfg);
    
    if (!defined $rc || $rc != 0){
        $msglst->AddMessage ("Error calling equinox director", $cfg);
        return $rc;
    }
    $msglst->AddMessage (undef,$cfg);
    return 0;
}

#----------------------------------------------------------------------------

sub getHANAFlavour {
    my ($self) = @_;
    
    my $manifest = $self->getManifest();
    return undef unless defined $manifest;
    
    my $path = dirname($manifest->getFileName());
    my $installation = new SDB::Install::Installation ($path);
    return undef unless defined $installation;
    
    return $installation->getHANAFlavour();
}

#----------------------------------------------------------------------------

# currently (2/17) only the generic kit has an event handler:
sub getEventHandler{
    return undef;
}

#----------------------------------------------------------------------------

1;
