package SDB::Install::LSS::LssSidcryptUser;

use strict;
use warnings;

use SDB::Install::System qw(makedir copy_file changeOwn isAdmin);
use parent qw(SDB::Install::User);
use File::stat qw(stat);
use File::Spec;
use SDB::Install::Globals qw($gProductName);
use experimental qw(smartmatch);

sub new{
    return shift->SUPER::new(@_);
}

sub configureHome{
    my ($self, $sid, $installationPath) = @_;

    my $home = $self->home();

    my %copyitems = (
        'CSHRC' => '.cshrc',
        'PROFILE' => '.profile',
        'BASHRC' => '.bashrc',
        'SAPENV.SH' => '.sapenv.sh',
        'SAPENV.CSH' => '.sapenv.csh'
    );

    my $msg = $self->getMsgLst()->addMessage ("Configuring user home '$home'");
    $msg->setEndTag ('Configure Home Dir');
    my $submsglst = $msg->getSubMsgLst ();

    if (!File::stat::stat($home)){
        my $config = {'uid' => $self->id(), 'gid' => $self->gid(), 'mode' => 0750};
        if (!defined SDB::Install::System::makedir ($home,$config)){
            $self->setErrorMessage ("Cannot create home directory $home" , $config);
            $msg->endMessage ();
            return undef;
        }

    }
    my $dst;
    my $srcScriptsDir = File::Spec->catdir($installationPath,'exe','env');
    foreach my $src (keys %copyitems){
        $dst = File::Spec->catfile($home,$copyitems{$src});
        $src = File::Spec->catfile($srcScriptsDir , $src);
        if (File::stat::stat($dst)){
            $submsglst->addMessage ("$dst already exists -> overwriting");
        }

        $submsglst->addMessage ("Copy $src to $dst");

        my $config = {};
        if (!SDB::Install::System::copy_file ($src,$dst, $config)){
            $submsglst->addWarning ("Could not copy $dst", $config);
        }
    }

    my $errlst = new SDB::Install::MsgLst ();
    if (!SDB::Install::System::changeOwn (undef, $self->id(), undef, $self->gid(), $self->home(), $errlst, $msg->getSubMsgLst ())){
        $self->setErrorMessage ("Change owner of home directory '$home' failed", $errlst);
        return undef;
    }
    $msg->endMessage ();
    return 1;
}

# Override
sub verifyPassword {
    my ($self, $password, $instconfig) = @_;

    if($self->_shouldTryAccessCheck($instconfig)){
        return $self->verifyLSSPassword($password, $instconfig);
    }
    return $self->verifyExistingUserPassword($password, $instconfig);
}

sub verifyExistingUserPassword {
     my ($self, $password, $instconfig) = @_;

    return 1 if(!isAdmin() || !$self->exists());

    my $str = $instconfig->getString('LSSPassword');
    my $rc = $self->SUPER::verifyPassword($password, $instconfig);

    if(!defined($rc)) {
        $instconfig->getMsgLst()->addWarning("Failed to check $str");
        $instconfig->getMsgLst()->addMessage("Ignoring error => password validation skipped.");
        return 1;
    }
    elsif(!$rc) {
        $instconfig->appendErrorMessage("Unknown $str combination");
    }
    return $rc;
}

sub _shouldTryAccessCheck {
    my ($self, $instconfig) = @_;
    my $dbHostsWithSidcryptUser = $instconfig->getDbHostsWithSidcryptUser();
    return (@{$dbHostsWithSidcryptUser} > 0);
}

sub verifyLSSPassword {
    my ($self, $password, $instconfig) = @_;
    my $str = $instconfig->getString('LSSPassword');
    my $instance = $instconfig->getOwnInstance();

    if(!defined($instance)) {
        $instconfig->appendErrorMessage("$gProductName instance is not defined");
        return 0;
    }

    my $host = $instconfig->getDbHostsWithSidcryptUser()->[0];
    my $sidcryptUsername = $self->getname();
    my $sapcontrol = SDB::Install::SAPControl->new($host,
                                                   $instance->get_nr(),
                                                   $sidcryptUsername,
                                                   $password,
                                                   $instconfig->isUseHttps());
    if(!$sapcontrol->isServiceRunning()) {
        $instconfig->addParameterWarning('LSSPassword', "Unable to validate $str because the service sapstartsrv is not running.");
        return 1;
    }

    if($sapcontrol->accessCheck($sidcryptUsername, $password) != 0) {
        if($sapcontrol->isInvalidCredentials) {
            $instconfig->appendErrorMessage("Cannot validate $str on host $host: ".$sapcontrol->getFaultstring());
            return 0;
        } elsif($sapcontrol->isPermissionDenied()) {
            # password validation passed but sidcrypt user does not have permissions to execute the function
            return 1;
        }
        return 0;
    }

    return 1;
}

1;