package SDB::Install::NewDBUser;

use strict;

use SAPDB::Install::Hostname;
use SDB::Install::Globals qw ($gSapsysGroupIdDefault $gSapsysGroupName);
use SDB::Install::System qw (makedir copy_file set_dacl changeOwn);
use SDB::Install::SysVars;
use SDB::Install::User;
use SDB::Install::Group;
use SDB::Common::Utils qw(createSysAdminUserName);
use SDB::Install::BaseLegacy;

our @ISA = qw (SDB::Install::BaseLegacy);

sub new{
	my $self = shift->SUPER::new ();

	($self->{sid},
	$self->{domain},
	$self->{integratedSystem}) = @_;
        
	if (!defined $self->{domain}){
		$self->{domain} = hostname ();
	}

	if ($isWin){
		$self->{admuser} = new SDB::Install::User(createSysAdminUserName($self->{sid},
		                                                                 $self->{domain}));
		$self->{serviceuser} = new SDB::Install::User ($self->{domain} . '\SAPService' .
                                        uc ($self->{sid}));
		$self->{hostadmingroup} = new SDB::Install::Group ('BUILTIN\Administrators');
		$self->{hostusergroup} = new SDB::Install::Group ('BUILTIN\Users');
		$self->{admingroup} = new SDB::Install::Group ('SAP_LocalAdmin');
		$self->{localadmingroup} = new SDB::Install::Group ('SAP_' . uc ($self->{sid}) . '_LocalAdmin');
		$self->{globaladmingroup} = new SDB::Install::Group ($self->{domain} . '\SAP_' . uc ($self->{sid}) . '_GlobalAdmin');
	}
	else{
		$self->{admuser} = new SDB::Install::User(createSysAdminUserName($self->{sid}));
		$self->{group} = new SDB::Install::Group ($gSapsysGroupName);
	}

    $self->{shmgroup} = new SDB::Install::Group (lc ($self->{sid}) . 'shm');

	$self->{createdusers} = [];
	$self->{createdgroups} = [];
	return $self;
}

sub existsUx{
	return $_[0]->{admuser}->exists();
}

sub existsWin{
	return  $_[0]->{admuser}->exists () &&  $_[0]->{serviceuser}->exists ();
}


sub notSupported{
    $_[0]->AddError ("Not supported on this platform");
    return undef;
}



#
# uid of <sid>adm
#
sub uidUx{
    return $_[0]->{admuser}->id(); # returns the user id of the system admin user
}

sub uid;
*uid = $isWin ? \&notSupported : \&uidUx;

#
# gid of sapsys
#

sub gidUx{
    return $_[0]->{group}->id(); # returns the group id of sapsys
}

sub gid;
*gid = $isWin ? \&notSupported : \&gidUx;

sub userGidUx {
    return getgrnam($_[0]->{admuser}->{group});
}

sub userGid;
*userGid = $isWin ? \&notSupported : \&userGidUx;

#
# home directory of <sid>adm
#
sub homeUx{
    return $_[0]->{admuser}->home(); # returns the home directory of the system admin user
}


#
# shell of <sid>adm
#
sub shellUx{
    return $_[0]->{admuser}->shell(); # returns the shell of the system admin user
}


sub createUx{
	my ($self, $templateDir, $comment, $id, $password, $sapsysid, $home, $shell, $existingMdcUsers) = @_;

	#if (!defined $templateDir){
	#	die "not yet implemented"; 
	#}
	
	if (!defined $sapsysid){
		$sapsysid = $gSapsysGroupIdDefault;
	}
	
	my $msglst = new SDB::Install::MsgLst();

	if ($self->exists){
		my $msg = $self->AddMessage ("User already exists");
		$home = $self->{admuser}->{home};
		$shell = $self->{admuser}->{shell};
		$msglst->AddMessage ("Id: " . $self->uidUx());
		$msglst->AddMessage ("Home: $home");
		$msglst->AddMessage ("Shell: $shell");
		$self->AddSubMsgLst ($msg, $msglst);
	}
	else{
		if (!defined $shell){
			$shell = '/bin/sh';
		}
	
	}
    my $shmgroupCreated = 0;
    foreach my $group ($self->{group},$self->{shmgroup}){
        if (!$group->exists){
            my $msg = $self->AddMessage ("Creating '$group->{name}' group");
            if (!defined $group->create (undef, $self->{group} == $group ? $sapsysid : undef)){
                $self->AddError ("Cannot create '$group->{name}' group", $group);
                $self->AddSubMsgLst ($msg, $group);
                return undef;
            }
            $shmgroupCreated = 1;
            $self->AddSubMsgLst ($msg, $group);
            push @{$self->{createdgroups}}, $group;
        }
        else{
            $self->AddMessage ("Group '$group->{name}' already exists");
        }
    }



	if ($self->gidUx() != int $sapsysid){
		$self->AddWarning ('Found group id ' . $self->gidUx() .
		    " instead of expected id $sapsysid for group '$gSapsysGroupName'");
	}


    my $user_created = 0;
    my $sidadmName = $self->getSidAdmName ();
	if (!$self->{admuser}->exists ()){
		my $msg = $self->AddMessage ("Creating admin user");
		if (!defined $self->{admuser}->create ($id, $password,
			$comment, $self->{group}->id, $home, $shell,[$self->{shmgroup}->{name}])){
			$self->AddError ("Cannot create '$sidadmName' user", $self->{admuser});
			$self->AddSubMsgLst ($msg, $self->{admuser});
			return undef;
		}
		$self->AddSubMsgLst ($msg, $self->{admuser});
		push @{$self->{createdusers}}, $self->{admuser};
		$user_created = 1;
	}

	if ($self->{admuser}->{group} ne $self->{group}->{name}){
		$self->AddWarning ("User '$sidadmName' doesn\'t have primary group '$self->{group}->{name}'");
	}

    my @addUsersToShmGroup;

    if (!$user_created){
        push @addUsersToShmGroup, $self->{admuser};
    }

    if (defined $existingMdcUsers){
        push @addUsersToShmGroup, @$existingMdcUsers;
    }

    if (@addUsersToShmGroup){
        if (!defined $self->_addUsersToShmGroup (\@addUsersToShmGroup, $shmgroupCreated)){
            return undef;
        }
    }

	if (defined $templateDir){
		$self->configureHome ($templateDir);
	}
	return 1;
}


sub createWin{
	my ($self, $templateDir, $comment, $id, $password, $sapsysid, $home, $shell) = @_;

	if ($self->exists ()){
		my $msg = $self->AddMessage ("Users already exists");
		$self->AddSubMessage ($msg, "Id: $self->{admuser}->{sid}");
		if ($self->{admuser}->{home}){
			$self->AddSubMessage ($msg, "Home: $self->{admuser}->{home}");
		}
		if ($self->{admuser}->{comment}){
			$self->AddSubMessage ($msg, "Comment: $self->{admuser}->{comment}");
		}
		return 1;	
	}	
	
	my $rc;


	#die "not yet implemented";
	
	#
	# create SAPLocalAdmin
	#
	
	if ($self->{admingroup}->exists ()){
		$self->AddMessage ("Group $self->{admingroup}->{name} already exists");
	}
	else{
		$self->AddMessage ("Creating group $self->{admingroup}->{name}");
		$rc = $self->{admingroup}->create ('SAP Local Administration Group');
		if (!$rc){
			$self->AddError ("Cannot create group $self->{admingroup}->{name}", $self->{admingroup});
			return undef;		
		}
	}
	
	#
	# create SAP_SID_LocalAdmin
	#
	
	if ($self->{localadmingroup}->exists()){
		$self->AddMessage ("Group $self->{localadmingroup}->{name} already exists");
	}
	else{
		$self->AddMessage ("Creating group $self->{localadmingroup}->{name}");
		$rc = $self->{localadmingroup}->create ('SAP Local System Administration Group');
		if (!$rc){
			$self->AddError ("Cannot create group $self->{localadmingroup}->{name}", $self->{localadmingroup});
			return undef;		
		}
	}
	
	#
	# create SAP_SID_GlobalAdmin
	#
	
	if ($self->{globaladmingroup}->exists()){
		$self->AddMessage ("Group $self->{globaladmingroup}->{name} already exists");
	}
	else{
		$self->AddMessage ("Creating group $self->{globaladmingroup}->{name}");
		$rc = $self->{globaladmingroup}->create ('SAP Global Administration Group');
		if (!$rc){
			$self->AddError ("Cannot create group $self->{globaladmingroup}->{name}", $self->{globaladmingroup});
			return undef;		
		}
	}


    # create hdbsidshm
    #

    if ($self->{shmgroup}->exists()){
        $self->AddMessage ("Group $self->{shmgroup}->{name} already exists");
    }
    else{
        $self->AddMessage ("Creating group $self->{shmgroup}->{name}");
        $rc = $self->{shmgroup}->create ('SAP HDB System Shared Memory Group');
        if (!$rc){
            $self->AddError ("Cannot create group $self->{shmgroup}->{name}", $self->{globaladmingroup});
            return undef;
        }
    }

	#
	# create SIDadm
	#
	my $sidadmName = $self->getSidAdmName ();
	if($self->{admuser}->exists()){
		$self->AddMessage ("User $sidadmName already exists");
	}
	else{
		my $msg = $self->AddMessage ("Creating user $sidadmName");
		if(!$self->{admuser}->create ($id, $password, $comment)){
			$self->AddError ("Cannot create user $sidadmName", $self->{admuser});
			return undef;
		}
		push @{$self->{createdusers}}, $self->{admuser};
	}
	

	
	foreach my $group (	
						$self->{hostadmingroup},
						$self->{hostusergroup},
						$self->{admingroup},
						$self->{localadmingroup},
						$self->{globaladmingroup},
						$self->{shmgroup}){
		$rc = $self->{admuser}->addgroup ($group->{name}); 
		if (!$rc){
			$self->AddError ("Cannot add '$sidadmName' to group '$group->{name}'", $self->{admuser});
			return undef;
		}
	}
	
	#
	# create SAPServiceSID
	#
	
	if($self->{serviceuser}->exists()){
		$self->AddMessage ("User $self->{serviceuser}->{name} already exists");
	}
	else{
		
		my $msg = $self->AddMessage ("Creating user $self->{serviceuser}->{name}");
		if(!$self->{serviceuser}->create ($id, $password, $comment)){
			$self->AddError ("Cannot create user $self->{serviceuser}->{name}", $self->{serviceuser});
		}
		push @{$self->{createdusers}}, $self->{serviceuser};
	}
	
	
	foreach my $group (	
						$self->{admingroup},
						$self->{localadmingroup},
						$self->{globaladmingroup},
						$self->{shmgroup}){
		$rc = $self->{serviceuser}->addgroup ($group->{name}); 
		if (!$rc){
			$self->AddError ("Cannot add '$self->{serviceuser}->{name}' to group '$group->{name}'", $self->{serviceuser});
			return undef;
		}
	}
	
	my @priv = qw (
		SeRestorePrivilege
		SeNetworkLogonRight
		SeServiceLogonRight
		SeImpersonatePrivilege
		SeIncreaseQuotaPrivilege
		SeShutdownPrivilege
		SeTcbPrivilege
		SeAssignPrimaryTokenPrivilege);
	
	if (!$self->{admuser}->addAccountRights (\@priv)){
		$self->AddError ("Cannot set account privileges of $sidadmName", $self->{admuser});
		return undef;	
	}
	
	@priv = qw (
		SeRestorePrivilege
		SeNetworkLogonRight
		SeServiceLogonRight
		SeImpersonatePrivilege
		SeIncreaseQuotaPrivilege
		SeShutdownPrivilege
		SeDenyRemoteInteractiveLogonRight
		SeDenyInteractiveLogonRight);
	
	if (!$self->{serviceuser}->addAccountRights (\@priv)){
		$self->AddError ("Cannot set account privileges  of $self->{serviceuser}->{name}", $self->{serviceuser});
		return undef;	
	}

	return 1;
}	
	

sub delete{
	my ($self, $force, $keepUserHomeDir, $keepGroup) = @_;
	
	my $rc = 1;
	
	if ($self->{integratedSystem}){
		if (!$isWin && !$keepUserHomeDir) {
			my $home = $self->{admuser}->{home};
			foreach my $file ('hdbenv.sh', 'hdbenv.csh'){
				unlink ("$home/$file");
			}
		}
		
		return 1;
	}
	
	my $msg;	
	my @users;
	my @groups;


	if (!$force){
		@users = @{$self->{createdusers}};
		@groups = @{$self->{createdgroups}};
	}
	else{
		push @users, $self->{admuser};
				
		if ($isWin){
			push @users, $self->{serviceuser};
			@groups = ($self->{localadmingroup}, $self->{globaladmingroup}, $self->{admingroup});
		}
		else{
			@groups = ($self->{group});
		}
		push @groups, $self->{shmgroup};
	}
	
	foreach my $user (@users){
		if (!$user->exists){
			next;
		}
		$msg = $self->AddMessage ("Deleting user $user->{name}");
		if (!$user->delete($keepUserHomeDir)){
			$self->PushError ("Cannot delete user $user->{name}", $user);
		}
		$self->AddSubMsgLst ($msg,$user);	
	}
	
	if (!$keepGroup) {

		foreach my $group (@groups){
			if (!$group->exists){
				next;
			}
			my $members = $group->getMembers (1);

			if (!defined $members || !@$members){
				$msg = $self->AddMessage ("Deleting group $group->{name}");
				$group->delete();
				$self->AddSubMsgLst ($msg,$group);
			}
			else{
				$self->AddMessage ("Do not delete group '$group->{name}': " .
						"There are still group members (" . join (', ', @$members) . ')');
			}
		}
	}	
	
	return $rc;
}		



sub replace_in_file{
	my ($self,$src,$dst,$replacements, $msglst) = @_;
	
	my $no_dst = 0;
	
	if (!defined $dst){
		$dst = $src;
		$no_dst = 1;
	}
	if (!defined $msglst){
		$msglst = $self;
	}
	my $msg = $msglst->AddMessage ("Processing file '$src'");
	if (!open (FH,$src)){
		$msglst->AddError ("Cannot open file '$src': $!");
		return undef;
	}
	my @buf = <FH>;
	close (FH);
	my $i;
	my $changed = 0;
	my $str_found = 0;
	foreach my $str (keys (%$replacements)){
		$str_found = 0;
		foreach $i (0..$#buf){
			if ($buf[$i] =~ s/$str/$replacements->{$str}/g){
				$str_found = 1;
				$changed = 1;
			}
		}
		if ($str_found){
			$msglst->AddSubMessage ($msg, "Replaced '$str' with '$replacements->{$str}'");
		}
	}
	if ($no_dst && !$changed){
		$msglst->AddMessage ("File '$dst' is up to date");
		return 1;
	}
	if (!open (FH, '>' . $dst)){
		$msglst->AddError ("Cannot create file '$dst': $!");
		return undef;
	}
	
	print FH join ('', @buf);
	close (FH);
	if ($changed){
		$msglst->AddMessage ("File '$dst' was updated");
	}
	else{
		$msglst->AddMessage ("File '$src' just copied to '$dst'");
	}
	return 1;
}



sub getUsersUx{
    return [$_[0]->{admuser}];
}

sub getUsersWin{
    return [$_[0]->{admuser},$_[0]->{serviceuser}];
}


sub _addUsersToShmGroup{
    my ($self, $users, $skipCheck) = @_;
    my ($groupname, $group_found);
    if (!defined $skipCheck){
        $skipCheck = 0;
    }
    foreach my $user (@$users){
        $group_found = 0;
        if (!$skipCheck){
            foreach $groupname (@{$user->getGroups}){
                if ($groupname eq $self->{shmgroup}->{name}){
                    $group_found = 1;
                    $self->AddMessage ("User '$user->{name}' is already a member of group '$self->{shmgroup}->{name}'");
                    last;
                }
            }
        }
       if (!$group_found){
            my $msg = $self->AddMessage (" Adding user '$user->{name}' to group '$self->{shmgroup}->{name}'");
            $user->setMsgLstContext ([$msg->getSubMsgLst ()]);
            if (!defined $user->addgroup ($self->{shmgroup}->{name})){
                $self->PushError ("Cannot add '$user->{name}' to group '$self->{shmgroup}->{name}'", $user);
                return undef;
            }
        }
    }
    return 1;
}


sub adaptExistingUser{
    my ($self, $mdcUsers) = @_;
    if (!defined $mdcUsers){
        $mdcUsers = [];
    }
    my $shmgroupCreated = 0;
    if (!$self->{shmgroup}->exists()){
       my $msg = $self->AddMessage ("Creating shm group");
        $self->{shmgroup}->setMsgLstContext ([$msg->getSubMsgLst ()]);
       if (!defined $self->{shmgroup}->create('SAP HDB System Shared Memory Group')){
            $self->AddError ("Cannot create shm group", $self->{shmgroup});
            return undef;
       }
       $shmgroupCreated = 1;
    }
    my @users = (@{$self->getUsers()}, @$mdcUsers);

    if (!defined $self->_addUsersToShmGroup (\@users, $shmgroupCreated)){
        return undef;
    }
    return 1;
}



sub checkExistingUser{
    my ($self) = @_;
    my $rc = 1;
    my $msg = $self->AddMessage ("Checking shm group");
    if ($self->{shmgroup}->exists()){
        $self->AddSubMessage ($msg, "Group '$self->{shmgroup}->{name}' already exists");
    }
    else{
        $self->AddSubMessage ($msg, "Group '$self->{shmgroup}->{name}' doesn't exist");
        return 0;
    }
    my ($groupname,$group_found);
    foreach my $user (@{$self->getUsers()}){
        $msg = $self->AddMessage ("Checking user '$user->{name}'");
        $group_found = 0;
        foreach $groupname (@{$user->getGroups}){
            if ($groupname eq $self->{shmgroup}->{name}){
                $group_found = 1;
                $self->AddSubMessage ($msg, "User '$user->{name}' is already a member of group '$self->{shmgroup}->{name}'");
                last;
            }
        }
        if (!$group_found){
            $self->AddSubMessage ($msg, "User '$user->{name}' is not a member of group '$self->{shmgroup}->{name}'");
            $rc = 0;
        }
    }
    return $rc;
}


sub configureHome{
	my ($self, $templatedir,$instanceDir,$hostname, $replacements, $msglst) = @_;
	
	if ($isWin){
		return 1;
	}

	if (!defined $msglst){
		$msglst = $self;
	}
	
	my $home = $self->{admuser}->{home};
	my $uid = $self->uidUx();
	my $gid = $self->{admuser}->{gid};
	
	my %copyitems = (
		'SAPSRC.SH' => '.sapsrc.sh',
		'SAPSRC.CSH' => '.sapsrc.csh',
		'CSHRC' => '.cshrc',
        'PROFILE' => '.profile',
        'BASHRC' => '.bashrc'
	);
	
	my %replaceitems = (
		'SAPENV.SH' => '.sapenv.sh',
		'SAPENV.CSH' => '.sapenv.csh',
	);
	
	my $string_map = {'TRX' => 'HDB', 'trex' => 'hdb', 'TREX' => 'HDB'};
	
	my $migrate_strings = {
		'.sapenv.sh' => $string_map,
		'.sapenv.csh' => $string_map
	};
	
	my $peppers = {
		'.sapenv.sh' => [ 
			'. /usr/sap/$SAPSYSTEMNAME/HDB[0-9][0-9]/HDBSettings.sh'
		],
		'.sapenv.csh' => [
			'source /usr/sap/$SAPSYSTEMNAME/HDB[0-9][0-9]/HDBSettings.csh'
		]
	};
	my $msg = $msglst->AddMessage ("Configuring user home '$home'");
	
	$msg->setEndTag ('Configure Home Dir');
	
	my $submsglst = $msg->getSubMsgLst ();
	
	if (!-d $home){
		my $config = {'uid' => $uid, 'gid' => $gid, 'mode' => 0750};
		if (!defined makedir ($home,$config)){
			$msglst->AddError ("Cannot create home directory $home" , $config);
            $msg->endMessage ();
			return undef;
		}
	
	}
	my $dst;
	foreach my $src (keys %copyitems){
		$dst = $home . '/' . $copyitems{$src};
		$src = $templatedir . '/' . $src;
		if (-f $dst){
			if ($self->{integratedSystem}){
				next;
			}
			$submsglst->AddMessage ("$dst already exists -> overwriting");
		}
	
		$submsglst->AddMessage ("Copy $src to $dst");		

		my $config = {};
		if (!copy_file ($src,$dst, $config)){
			$submsglst->AddWarning ("Could not copy $dst", $config);
		}
	}

	
	my $sid = uc ($self->{sid});
	my $usrsapsidexe = '/usr/sap/' . $sid . '/SYS/exe/run'; 
	my @lines;
	my $line;
	my $_submsglst;
	my $_msg;
	my $basename;
	
	
	my $repl = {'\@SID\@' => $sid,
			'\@DEF_EXE\@' => $usrsapsidexe};
			
	if (defined $instanceDir){
		$repl->{'TARGETDIR'} = $instanceDir;
	}

	if (defined $hostname){
		$repl->{'CONFIGDIR'} = $hostname;
	}

	foreach my $src (keys %replaceitems){
		$basename = $replaceitems{$src};
		$dst = $home . '/' . $basename;
		$src = $templatedir . '/' . $src;
		if (-f $dst && $basename !~ /hdb/i){
			if (defined $replacements){
				$_submsglst = new SDB::Install::MsgLst ();
				$self->replace_in_file ($dst,$dst, $replacements, $_submsglst);
				$self->AddSubMsgLst ($msg, $_submsglst);
				chmod (0750, $dst);
				next;
			}
			if ($self->{integratedSystem}){
				next;
			}
			$_msg = $submsglst->AddMessage ("$dst already exists -> enhancing");
			$self->enhance($dst, $peppers->{$basename}, $migrate_strings->{$basename}, $_msg->getSubMsgLst());
			next;
		}
		$_submsglst = new SDB::Install::MsgLst ();
		if (!$self->replace_in_file ($src,$dst,$repl, $_submsglst)){
			$self->AddError ($_submsglst);
            $msg->endMessage ();
			return undef;
		}
		chmod (0750, $dst);
	}
    my $errlst = new SDB::Install::MsgLst ();
    if (!changeOwn (undef, $uid, undef, $gid, $home, $errlst, $msg->getSubMsgLst ())){
        $self->setErrorMessage ("Change owner of home directory '$home' failed", $errlst);
    }
    $msg->endMessage ();
	return 1;
}

sub enhance{
	my ($self, $path, $peppers, $migrate_strings, $msglst) = @_;
	
	if (!defined $msglst){
		$msglst = $self;
	}

	chmod (0740, $path);
	
	if (!open (FD, $path)){
		$msglst->AddError ("Cannot open file $path: $!");
		return undef;
	}
	
	my @content;
	while(<FD>){
		chomp;
		push @content, $_;
	}
	close (FD);
	
	my @append;
	my $line;
	my $found;
	my $fileline;

	my $migrated = 0;
	if (defined $migrate_strings){
		my $pattern;
		my $repl;
		foreach my $i (0..$#content){
			foreach my $str (keys (%$migrate_strings)){
				$pattern = quotemeta ($str);
				$repl = $migrate_strings->{$str};
				if ($content[$i] =~ s/$pattern/$repl/g){
					$migrated = 1;
				}
			}
		}
	}
	
	if (defined $peppers){
		foreach $line (@{$peppers}){
			$found = 0;
			foreach $fileline (@content){
				if ($line eq $fileline){
					$found = 1;
					last;
				}
			}
			if (!$found){
				push @append, $line;
			}	
		}
	}
	
	if (!@append && !$migrated){
		$msglst->AddMessage ("File $path is up to date");
		return 1;
	}


	if ($migrated){
		$msglst->AddMessage ("Migration of file '$path'");
		if (!open (FD, '>' . $path)){
			$msglst->AddError ("Cannot create file $path: $!");
			return undef;
		}
		print FD join ("\n", @content) . "\n";
	}
	else{
		if (!open (FD, '>>' . $path)){
			$msglst->AddError ("Cannot open file $path for append: $!");
			return undef;
		}
	}
	
	foreach my $line (@append){
		print FD $line . "\n";
		$msglst->AddMessage ("Adding: $line");
	}
	close (FD);
	return 1;
}

sub switchto{
	
	if ($isWin){
		return 1;
	}

	my ($self) = @_;
		
	if ($> != $self->uidUx()){
		my $msg = $self->AddMessage ("Switching to user " . $self->getSidAdmName() . "/$self->{admuser}->{group}");
		require SAPDB::Install::SetUser;
		if (SAPDB::Install::SetUser::SetUser ($self->uidUx(), $self->{admuser}->{gid}) != 0){
			$self->AddError ("SetUser() failed");
			return undef;
		}
		my $msglst = new SDB::Install::MsgLst ();
		$msglst->AddMessage ("uid = $<, gid = $(");
		$msglst->AddMessage ("euid = $>, egid = $)");
		$self->AddSubMsgLst ($msg, $msglst);

	}
	return 1;
}

sub switchback{

	if ($isWin){
		return 1;
	}

	if ($) != $( || $> != $<){
		my ($self) = @_;
		my $msg = $self->AddMessage ("Switching back user root");
		require SAPDB::Install::SetUser;
		if (SAPDB::Install::SetUser::SetUser ()){
			$self->AddError ("SetUser() failed");
			return undef;
		}
		my $msglst = new SDB::Install::MsgLst ();
		$msglst->AddMessage ("uid = $<, gid = $(");
		$msglst->AddMessage ("euid = $>, egid = $)");
		$self->AddSubMsgLst ($msg, $msglst);
	}
	return 1;
}


sub possessUx{
	my ($self, $path, $recursive, $permissions, $visited,$msglst) = @_;

	if (!defined $msglst){
		$msglst = $self;
	}

	if (!defined $recursive){
		$recursive = 1;
	}
	if (!defined $permissions){
		$permissions = 0750;
	}
	

	if (!defined $visited){
		if ($recursive){
			$msglst->AddMessage ('Possess recursive: ' . $path)
		}
		else{
			$msglst->AddMessage('Possess: ' . $path);
		}
		$msglst->AddMessage (sprintf ("Setting permissions to 0%o", $permissions));

	}
	
	my $uid = $self->uidUx();
	my $gid = $self->gidUx();		
	if (-e $path){
		my @statbuf = stat _ ;

		if (defined $visited && exists $visited->{$statbuf[0]} && 
			exists $visited->{$statbuf[0]}->{$statbuf[1]}){
			# symlink cycle
			return 1;
		}

		if ((defined $uid && ($statbuf[4] != $uid)) ||
		    (defined $gid && ($statbuf[5] != $gid))) {
			chown ($uid, $gid, $path);
		}
		if (($statbuf[2] & 07777) != $permissions){
			chmod ($permissions, $path);
		}
		if (-d _  && $recursive){
			if (!defined $visited){
				$visited = {};
			}
			$visited->{$statbuf[0]}->{$statbuf[1]} = undef;
			if (!opendir (DH, $path)){
				$msglst->AddError ("Cannot open directory $path: $!");
				return undef;
			}
			my @items = grep {!/^\.{1,2}$/} readdir (DH);
			closedir (DH);
			foreach my $item (@items){
				$self->possess ("$path/$item", 1, $permissions, $visited,
				$msglst == $self ? undef : $msglst);
			} 
		}
	}
	return 1;
}

sub possessShare{
	my ($self, $share, $msglst) = @_;
	use SAPDB::Install::System::Win32::SecurityInfo;	
	my $msg = $msglst->AddMessage ('Possess share: ' . $share);
	
	my $si = new SAPDB::Install::System::Win32::SecurityInfo ($share, 1);
	my  @list = (
				{'sid' => $self->{admingroup}->{sid}, 'mask' => GENERIC_ALL},
				{'sid' => $self->{hostadmingroup}->{sid}, 'mask' => GENERIC_ALL},
				{'trustee' => 'CURRENT_USER', 'mask' => GENERIC_ALL}
				);
				
	my $_msglst = new SDB::Install::MsgLst();			
	my $rc = set_dacl ($share,$si, \@list,$_msglst);
	$msglst->AddSubMsgLst ($msg,$_msglst);
		
	if (!$rc){
		$msglst->AddError (undef, $_msglst);
	}
	return $rc;
}

sub possessWin{
	my ($self, $path, $recursive, $permissions, $msglst) = @_;

	use SAPDB::Install::System::Win32::SecurityInfo;
	
	if (!defined $msglst){
		$msglst = $self;
	}
	
	if (!defined $recursive){
		$recursive = 1;
	}
	if (!defined $permissions){
		$permissions = 0750;
	}

	my $msg;

	if ($recursive){
		$msg = $msglst->AddMessage ('Possess recursive: ' . $path)
	}
	else{
		$msg = $msglst->AddMessage('Possess: ' . $path);
	}

	my $inheritance = $recursive ? SUB_CONTAINERS_AND_OBJECTS_INHERIT : NO_INHERITANCE;
		
	my @list = (
		{'sid' => $self->{admuser}->{sid}, 'mask' => GENERIC_ALL, 'inheritance' => $inheritance},
		{'sid' => $self->{serviceuser}->{sid}, 'mask' => GENERIC_ALL, 'inheritance' => $inheritance},
		{'sid' => $self->{hostadmingroup}->{sid}, 'mask' => GENERIC_ALL, 'inheritance' => $inheritance},
		{'sid' => $self->{admingroup}->{sid}, 'mask' => GENERIC_ALL, 'inheritance' => $inheritance},
		{'sid' => $self->{localadmingroup}->{sid}, 'mask' => GENERIC_ALL, 'inheritance' => $inheritance},
		{'trustee' => 'CURRENT_USER', 'mask' => GENERIC_ALL, 'inheritance' => $inheritance},
		{'sid' => new SDB::Install::User ('BUILTIN\System')->{sid}, 'mask' => GENERIC_ALL, 'inheritance' => $inheritance}
	);
	
	my $_msglst = new SDB::Install::MsgLst();
	
	my $si = new SAPDB::Install::System::Win32::SecurityInfo ($path);
	
	my $rc = set_dacl ($path,$si, \@list,$_msglst);
	
	$msglst->AddSubMsgLst ($msg,$_msglst);
		
	if (!$rc){
		$msglst->AddError (undef, $_msglst);
	}
	
	return 1;
	
	
	#    rc = ntos.changeOwner(path, recursive,self.admuser.name,self.localadmingroup.name)
	#    self._logger.info('change owner with ret: %d'%(int(rc)))
	#    rc = ntos.shareACL("sapmnt",[("Administrators","SHARE_FULL"),("SAP_LocalAdmin","SHARE_FULL"),("Everyone","NONE")],[])
	#    self._logger.info('set share acl with ret: %d'%(int(rc)))
	#    rc = ntos.shareACL("saploc",[("Administrators","SHARE_FULL"),("SAP_LocalAdmin","SHARE_FULL"),("Everyone","NONE")],[])
	#    self._logger.info('set share acl with ret: %d'%(int(rc)))
}



sub getSidAdm{
    return $_[0]->{admuser}; # returns SDB::Install::User containing the properties of system admin
}

sub getSidAdmName{
    return $_[0]->{admuser}->getname (); # returns the name of the system admin user
}

sub getSidAdmHome{
    return $_[0]->getSidAdm()->{home};
}

if ($isWin){
	*exists = \&existsWin;
	*create = \&createWin;
	*possess = \&possessWin;
	*getUsers = \&getUsersWin;
}
else{
	*exists = \&existsUx;
	*create = \&createUx;
	*possess = \&possessUx;
	*getUsers = \&getUsersUx;
}

1;
