#!/usr/bin/perl

package SDB::Install::ControlFile;

use strict;
use SDB::Install::BaseLegacy;
use SDB::Install::System qw (exec_program);
use SDB::Install::Globals qw ($gProductNameEngine);
use SDB::Install::SysVars qw ($isPowerPC);
require File::stat;

our @ISA = qw (SDB::Install::BaseLegacy);


sub new{
	my $self = shift->SUPER::new ();
	my ($filename) = @_;
	
	if (!defined $filename){
		$filename = '/etc/sysctl.conf';
	}
	$self->{filename} = $filename;
	$self->{lines} = [];
	$self->{read} = 0;
	$self->{changed} = 0;
	return $self;
}


sub read{
    my ($self) = @_;
    if (!$self->{read}){
        if (!-f $self->{filename}){
            $self->AddError ("File '$self->{filename}' not found");
            return undef;
        }

        # if file > 50k, then cleanup mess caused by bug#31173
        my $removeEmptyLinesBlock = (stat (_))[7] > 51200;

        if (!open (FD, $self->{filename})){
            $self->AddError ("Cannot open file '$self->{filename}': $!");
            return undef;
        }

        if ($removeEmptyLinesBlock){
            #
            # remove more than one empty line in a block
            #
            my $lastEmpty = 0;
            my $currentEmpty;
            foreach my $line (<FD>){
                chomp ($line);
                $currentEmpty = $line eq '';
                if ($currentEmpty && $lastEmpty){
                    next;
                }
                $lastEmpty = $currentEmpty;
                push @{$self->{lines}}, $line;
            }
        }
        else{
            my @lines = <FD>;
            foreach my $line (@lines){
                chomp ($line);
            }
            $self->{lines} =  \@lines;
        }
        close (FD);
        $self->{read} = 1;
    }
    return 1;
}



sub get{
	my ($self, $entry) = @_;
	if (! defined $self->read ()){
		return undef;
	}
	if (!@{$self->{lines}}){
		return (-1,'');
	}
	#m = re.compile("^[ \t]*%s[ \t]*=[ \t]*(.*)" % entry)
	my $pattern = '^\s*' . quotemeta ($entry) . '\s*=\s*(.*)';
	my $r;
	my $i = -1;
	foreach my $line (@{$self->{lines}}){
		$i++;
		($r) = ($line =~ /$pattern/);
		if (defined $r){
			$r =~ s/\s*$//;
			return ($i,$r);
		}
	}
	return (-1,'');
}

sub set{
	my ($self, $index, $entry, $value) = @_;
	
	# replace with two lines
	my $line = "$entry=$value";
	$self->AddMessage ("Add to '$self->{filename}' text: $line");
	my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) =
			localtime(time);
	my $timestamp = sprintf ("%d.%02d.%02d_%02d.%02d.%02d", $year + 1900, $mon + 1,
			$mday, $hour, $min, $sec); #   +time.strftime("%Y.%m.%d_%H.%M.%S\n")
	if ($index == -1){
		my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) =
			localtime(time);
		my $comment = "# Next line added for $gProductNameEngine on $timestamp\n";
		push @{$self->{lines}}, "$comment$line";
	}
	else{
		my $comment = "# Next line modified for $gProductNameEngine on $timestamp\n";
		$self->{lines}->[$index] = "$comment$line";
	}
	$self->{changed}++;
	return $self->{changed};
}

sub submit{
	my ($self) = @_;
	if (!$self->{changed}){
		$self->AddMessage ("sysctl is up-to-date");
		return 1;
	}
	$self->AddMessage ("Adapting file '$self->{filename}'");
	
	my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) =
			localtime(time);
	my $now = sprintf ("%d.%02d.%02d_%02d.%02d.%02d", $year + 1900, $mon + 1,
		$mday, $hour, $min, $sec); #   +time.strftime("%Y.%m.%d_%H.%M.%S\n")
	# write sysctl.conf.new independent, if /etc/sysctl.conf exists or not
	if (!open (FD, '>'.$self->{filename}.'.new')){
		$self->AddError ("Cannot create file '$self->{filename}.new': $!");
		return undef;
	}
	foreach my $line (@{$self->{lines}}){
		print FD "$line\n";
	}
	close (FD);
	if (!rename ($self->{filename}, $self->{filename}.$now.'.bak')){
		$self->AddError ("Cannot rename file '$self->{filename}' to '$self->{filename}$now.bak': $!");
		return undef;
	}
	
	if (!rename ($self->{filename}.'.new', $self->{filename})){
		$self->AddError ("Cannot rename file '.$self->{filename}.new' to '$self->{filename}': $!");
		return undef;
	}
	my $prog = 'sysctl';
	my @args = ('-p');
	my $cfg = {};
	my $msg = $self->AddMessage ("Refreshing kernel parameters");
	$msg->getSubMsgLst ()->injectIntoConfigHash ($cfg);
	
	local %ENV = %ENV;
	$ENV{PATH} = join (':', qw(/sbin /usr/sbin), $ENV{PATH});
	my $rc = exec_program ($prog,\@args,$cfg);
	if ($rc != 0){
		$self->setErrorMessage ("Error refreshing kernel parameters ('sysctl -p' failed)", $msg->getSubMsgLst ());
		return undef;
	}
	return 1;
}

package SDB::Install::OSConfig;

use strict;
use SDB::Install::BaseLegacy;
use SDB::Install::SysVars;
use SDB::Install::Globals;
use SAPDB::Install::System::Unix;
use SDB::Install::Tools qw (trim);
use Exporter;

our @ISA = qw (SDB::Install::BaseLegacy Exporter);

our @EXPORT = qw ($SOFTLIMIT_NOFILE $SOFTLIMIT_STACK);

sub new{
	my $self = shift->SUPER::new ();
	$self->{sysctl} = new SDB::Install::ControlFile ();
	return $self;
}

#@statuschecked("adapt OS configuration")
sub adaptSystemConfigNewDB{
	my ($self,$SIDadm, $checkOnly, $processPortRange) = @_;
	if ($isWin){
		return 1;
	}

    $checkOnly //= 0;
    my $checkResult;
    my $action = $checkOnly ? 'Check' : 'Adapt';
    my $actionProgressive = "${action}ing";

	my $msg = $self->AddMessage ("$actionProgressive os configuration for $gProductName");

	my $msglst = $msg->getSubMsgLst ();
	$checkResult = $self->fileDescriptorsPerUser ($SIDadm, $msglst, $checkOnly);
    if( $isPowerPC ) {
        $checkResult &&= $self->stackPerUser ($SIDadm, $msglst, $checkOnly);
    }
    $checkResult &&= $self->fileDescriptorsPerHost ($msglst, $checkOnly);
	$checkResult &&= $self->processesPerSapSysGroup ($msglst, $checkOnly);
	$checkResult &&= $self->increaseAsyncIoRequests ($msglst, $checkOnly);
	$checkResult &&= $self->setMemoryFailureEarlyKill ($msglst, $checkOnly);
	$checkResult &&= $self->increaseSharedMemory ($msglst, $checkOnly);
    $checkResult &&= $self->increaseMaxMapCount ($msglst, $checkOnly);
    $checkResult &&= $self->adaptNetworkParameters ($msglst, $checkOnly);
    if($processPortRange) {
	    $checkResult &&= $self->increasePortRange ($msglst, $checkOnly);
    }
	$checkResult &&= $self->pamForSapstartsrv ($msglst, $checkOnly);
	$self->messagesReadForAll ($msglst);
    if (!$checkOnly){
        my $submitMsg = $msglst->addMessage ("Submitting sysctl changes");
        $self->{sysctl}->setMsgLstContext([$submitMsg->getSubMsgLst()]);
	    $self->{sysctl}->submit ();
    }
    $msg->endMessage (0, "$action os configuration");
	return $checkOnly ? $checkResult : 1;
}


sub contained{
	my (
	    $self,
	    $text,
	    $name,
	    $dontquote # optional, boolean, if set, $text is treated as a pattern
	) = @_;
	my $pattern = $text;
	if(not $dontquote) {
	    $pattern = quotemeta ($text);
	}
	my $fd = new IO::File($name);
	if (!defined $fd){
		$self->PushError ("Cannot open file '$name': $!");
		return undef;
	}
	my @lines = $fd->getlines();
	$fd->close();
	my @s;
	foreach my $l (@lines){
		if ($l =~ /$pattern/){
			@s = split($pattern,$l);
			if ($s[0] !~ /#/){
				return 1;
			}
		}
	}
	return 0;
}


our $sudoers = '/etc/sudoers';

sub adaptSudoers{
    my (
        $self,
        $user,
        $addcmds,
        $removecmds,
        $sudoersPrefix
    ) = @_;
    my $user_all_pattern = '^\s*' . $user . '\s+.*:(.*)$';
    if (!open (FD, $sudoers)){
        $self->AddError ("Cannot open file '$sudoers': $!");
        return undef;
    }
    my @buffer = <FD>;
    close (FD);
    my @existing_cmds;
    my $existing_cmds;
    my $cmd,
    my $line_index;
    foreach my $i (0..$#buffer){
        ($existing_cmds) = ($buffer[$i] =~ /$user_all_pattern/);

        next if(!$existing_cmds || ($buffer[$i] =~ /#HDB_XSA$/)); # Do not include commands for XSA OS users handling

        foreach $cmd (split (',' ,$existing_cmds)){
            trim (\$cmd);
            push @existing_cmds, $cmd;
            $line_index = $i;
        }
    }
    if (!@existing_cmds && (!defined $addcmds || !@$addcmds)){
        $self->AddMessage ("'$sudoers' is up-to-date");
        return 1;
    }

    my @newCmds;
    my $remove = 0;
    my $already_there = 0;
    my $changed = 0;
    my $i = 0;
    foreach $cmd (@existing_cmds){
        if (defined $removecmds){
            $remove = 0;
            foreach $i (0..(scalar (@$removecmds) -1)){
                if ($removecmds->[$i] eq $cmd){
                    $remove = 1;
                    last;
                }
            }
            if ($remove){
                $changed = 1;
                next;
            }
       }
       $already_there = 0;
       foreach $i (0..$#newCmds){
            if ($newCmds[$i] eq $cmd){
                   $already_there = 1;
                   last;
            }
        }
        if (!$already_there){
            push @newCmds, $cmd;
        }
    }

    if (defined $addcmds){
        foreach $cmd (@$addcmds){
            $already_there = 0;
            foreach $i (0..$#existing_cmds){
                if ($existing_cmds[$i] eq $cmd){
                    $already_there = 1;
                    last;
                }
            }
            if ($already_there){
                next;
            }
            $changed = 1;
            push @newCmds, $cmd;
        }
    }

    my $nl = $isWin ? "\r\n" : "\n";
    if ($changed){
        my $newLine = "$user $sudoersPrefix ". join (',', @newCmds) . $nl;
        my $prettyNewLine = $newLine;
        my $prettyOldLine = $buffer[$line_index];
        chomp($prettyNewLine);
        chomp($prettyOldLine);
        my $whatWeHaveDone = undef; 
        my $whatWeHaveDone2ndLine = undef; 
        if (defined $line_index){
            if (@newCmds){
                $whatWeHaveDone =        "replaced line: '$prettyOldLine'";
                $whatWeHaveDone2ndLine = "with     line: '$prettyNewLine'";
                $buffer[$line_index] = $newLine;
            }
            else{
                $whatWeHaveDone = "deleted line: '$prettyOldLine'";
                splice (@buffer, $line_index, 1);
            }
        }
        else{
            if (@newCmds){
                $whatWeHaveDone = "added   line: '$prettyNewLine'";
                push @buffer, $newLine;
            }
        }
        my $tmp_file = $sudoers.'_hdb' . $$;
        if (!open (FD, '>' . $tmp_file)){
            $self->AddError ("Cannot create temp. file '$tmp_file': $!");
            return undef;
        }
        foreach $i (0..$#buffer){
            print FD $buffer[$i];
        }
        close (FD);
        my ($sec, $min, $hour, $mday, $mon, $year) = localtime(time ());
        $mon++;
        $year += 1900;
        my $datetime = sprintf ("%d-%02d-%02d_%02d.%02d.%02d",
                                $year, $mon, $mday, $hour, $min, $sec);
        my $sudoersBackup = $sudoers.'_backup_hdb_' . $datetime;
        rename ($sudoers, $sudoersBackup);
        chmod (0440, $sudoersBackup);
        rename ($tmp_file, $sudoers);
        $self->AddMessage ("'$sudoers' updated:");
        $self->AddMessage ($whatWeHaveDone);
        if(defined $whatWeHaveDone2ndLine) {
            $self->AddMessage ($whatWeHaveDone2ndLine);
        }
        $self->AddMessage ("Wrote backup file '$sudoersBackup'.");
        chmod (0440, $sudoers);
    }
    else{
        $self->AddMessage ("'$sudoers' is up-to-date");
    }
    return 1;
}


sub addLineToSudoers {
    my (
        $self,
        $line,    # the line to append
        $pattern  # do nothing if pattern is found in sudoers
    ) = @_;
    if (!$self->contained($pattern, $sudoers, 'dontquote')){
        my $nl = $isWin ? "\r\n" : "\n";
        my $insertLine = $line.$nl;
        my $prettyNewLine = $line;
        chomp($prettyNewLine);
        $self->AddMessage ("Adapting  '$sudoers'");
        my $origin_mode;
        if (!-w $sudoers){
            $origin_mode = (stat (_))[2] & 07777;
            if (!chmod ($origin_mode | 0400, $sudoers)){
                $self->AddError ("Cannot enable write permissions of file '$sudoers': $!");
                return undef;
            }
        }
        if (!open (FD,">>$sudoers")){
            $self->AddError ("Cannot open to append file '$sudoers': $!");
            return undef;
        }
        print FD $insertLine;
        close (FD);
        if (defined $origin_mode){
            if (!chmod ($origin_mode, $sudoers)){
                $self->AddError ("Cannot restore permissions of file '$sudoers': $!");
                return undef;
            }
        }
        $self->AddMessage ("'$sudoers'  updated:");
        $self->AddMessage ("added line:  '$prettyNewLine'");
    }
    return 1;
}

sub removeLineFromSudoers {
    my (
        $self,
        $pattern  # all lines matching the pattern will be removed
    ) = @_;
    if(!open (FD, $sudoers)) {
        $self->AddError("Cannot open file '$sudoers': $!");
        return undef;
    }
    my @lines = <FD>;
    close (FD);
    my @out;
    $self->AddMessage ("Adapting  '$sudoers'");
    foreach my $line (@lines) {
        if ($line =~ /$pattern/) {
            my $prettyLine = $line;
            chomp($prettyLine);
            $self->AddMessage ("Removing line '$prettyLine'");
            next;
        }
        push @out, $line;
    }
    my $tmp_file = $sudoers.'_hdb' . $$;
    if(!open (FD, '>' . $tmp_file)) {
        $self->AddError("Cannot create temporary file '$tmp_file': $!");
        return undef;
    }
    foreach my $i (0..$#out) {
        print FD $out[$i];
    }
    close (FD);
    unlink($sudoers);
    rename ($tmp_file, $sudoers);
    chmod (0440, $sudoers);
    return 1;
}



our $SOFTLIMIT_NOFILE = 1048576;

sub fileDescriptorsPerUser{
	my ($self,$SIDadm, $msglst,$checkOnly) = @_;
	# allow the SIDadm user to have up to 8192 open file descriptors
	# by adding this to /etc/security/limits.conf
	# the dash in linea means to set both limits, soft and hard, in one line
	my $file   = '/etc/security/limits.conf';
	my $line1  = "$SIDadm          -       nofile          $SOFTLIMIT_NOFILE";
	#my $line2 = "$SIDadm          hard    nofile          8000\n";
    my $action = $checkOnly ? 'Checking' : 'Adjusting';
	my $msg = $msglst->AddMessage ("$action maximal number of open files per user");
    my $checkResult = 1;
	if (!$self->contained ($line1, $file)){
        if ($checkOnly){
            $self->appendErrorMessage ("'$file' needs line '$line1'");
            $checkResult = 0;
        }
        else{
		    if (!open (FD,'>>'.$file)){
			    $self->PushError ("Cannot open file '$file': $!");
			    return undef;
		    }
		    print FD "$line1\n";
		    $msglst->AddSubMessage ($msg, "Adding line to $file: $line1");
		    #print FD "$line2\n";
		    close (FD);
        }
	}
	my ($rc,$soft,$hard) = getrlimit (RLIMIT_NOFILE);
	
	if ($rc){
		$self->AddWarning ("getrlimit (RLIMIT_NOFILE) failed: errno = $rc");
		return $checkOnly ? $checkResult : 1;
	}
	
	$msglst->AddSubMessage ($msg, "nofile max = $hard");
	$msglst->AddSubMessage ($msg, "nofile cur = $soft");
	
	if ($soft < $SOFTLIMIT_NOFILE){
		if ($SOFTLIMIT_NOFILE > $hard){
            if (!$checkOnly){
			    $msglst->AddSubMessage ($msg, "Setting max number of open files (hard limit) from $hard to $SOFTLIMIT_NOFILE");
			    $hard = $SOFTLIMIT_NOFILE;
            }
		}
        if (!$checkOnly){
		    $msglst->AddSubMessage ($msg, "Setting current number of open files (soft limit) from $soft to $SOFTLIMIT_NOFILE");
		    $rc = setrlimit (RLIMIT_NOFILE, $SOFTLIMIT_NOFILE, $hard);
		    if ($rc){
			    my $msg = $self->AddWarning ("Cannot set current number of open files (soft limit) to $SOFTLIMIT_NOFILE for the current process");
			    $self->AddSubMessage ($msg,"setrlimit (RLIMIT_NOFILE) failed: errno = $rc", undef, 'ERR');
		    }
        }
	}
	return $checkOnly ? $checkResult : 1;
}

our $SOFTLIMIT_STACK = 16384;

sub stackPerUser{
	my ($self,$SIDadm, $msglst, $checkOnly) = @_;
    # set only soft limit so hard limit can be raised even more
	my $file   = '/etc/security/limits.conf';
	my $line1  = "$SIDadm          soft       stack          $SOFTLIMIT_STACK";
    my $action = $checkOnly ? 'Checking' : 'Adjusting';
	my $msg = $msglst->AddMessage ("$action maximal stack per user");
    my $checkResult = 1;
	if (!$self->contained ($line1, $file)){
        if ($checkOnly){
            $self->appendErrorMessage ("'$file' needs line '$line1'");
            $checkResult = 0;
        }
        else{
		    if (!open (FD,'>>'.$file)){
			    $self->PushError ("Cannot open file '$file': $!");
			    return undef;
		    }
		    print FD "$line1\n";
		    $msglst->AddSubMessage ($msg, "Adding line to $file: $line1");
		    close (FD);
        }
	}
    # we do not have to change this for the installation process itself
	return $checkOnly ? $checkResult : 1;
}

our $limitsDircectory = '/etc/security/limits.d';
our $sapsysConfFile = '99-sapsys.conf';

sub processesPerSapSysGroup{
    my ($self, $msglst, $checkOnly) = @_;
    my $stat = File::stat::stat($limitsDircectory);
    if (!defined $stat){
        $msglst->addMessage ("Skipping nproc due to '/etc/security/limits.d' is not accessible: $!");
        return 1;
    }

    if (!-d $stat){
        $msglst->addMessage ("Skipping nproc due to '/etc/security/limits.d' is not a directory");
        return 1;
    }

    my $action = $checkOnly ? 'Checking' : 'Setting';
    my $checkResult = 1;
    my $msg = $msglst->addMessage ("$action number of processes (nproc) for users in sapsys group to unlimited");
    my $line = '@sapsys    soft    nproc    unlimited';
    my $file = $limitsDircectory . '/' . $sapsysConfFile;
    $stat = File::stat::stat($file);
    my $fh;
    if (defined $stat && -f $stat){
        my $pattern = '^\s*' . join('\s+', split(/\s+/,$line)). '\s*$';
        if ($self->contained ($pattern, $file, 1)){
            $msg->getSubMsgLst ()->addMessage ("File '$file' is up-to-date.");
            return 1;
        }
        if ($checkOnly){
            $self->appendErrorMessage ("'$file' should contain '$line'");
            $checkResult = 0;
        }
        else{
            $fh = new IO::File(">>$file");
            if (!defined $fh){
                $self->appendErrorMessage ("Cannot open file '$file': $!");
                return undef;
            }
        }
    }
    else{
        if ($checkOnly){
            $self->appendErrorMessage ("'$file' should be there and contain '$line'");
            $checkResult = 0;
        }
        else{
            $fh = new IO::File(">$file");
            if (!defined $fh){
                $self->appendErrorMessage("Cannot create file '$file': $!");
                return undef;
            }
        }
    }
    if (!$checkOnly){
        $msg->getSubMsgLst ()->addMessage ("File '$file' updated.");
        $fh->print($line . "\n");
        $fh->close();
    }
    return $checkOnly ? $checkResult : 1;
}


our $file_max = 20000000;
our $sysctl_file_max = 'fs.file-max';
our $proc_file_max = '/proc/sys/fs/file-max';

sub fileDescriptorsPerHost{
    my ($self, $msglst, $checkOnly) = @_;

    if (!defined $msglst){
        $msglst = $self;
    }
    my $action = $checkOnly ? 'Checking' : 'Adjusting';
    my $msg = $msglst->AddMessage ("$action maximal number of open files per host");
    my $checkResult = 1;
    my $value;

    my $fh = new IO::File($proc_file_max, "r");
    if (!defined $fh) {
        $self->PushError("Cannot open file '$proc_file_max': $!");
        return undef;
    }
    chomp($value = $fh->getline());
    $fh->close();

    if ($value < $file_max) {
        if ($checkOnly) {
            $self->appendErrorMessage ("systcl '$sysctl_file_max' has to be increased from $value to $file_max");
            $checkResult = 0;
        } else {
            my ($index) = $self->{sysctl}->get ($sysctl_file_max);
            $self->{sysctl}->set ($index, $sysctl_file_max, $file_max);
            $msglst->AddSubMessage ($msg, "Increase value from $value to $file_max",$self->{sysctl});
        }
    } else {
        $msglst->AddSubMessage ($msg, "$sysctl_file_max is up-to-date ($value)");
    }
    return $checkOnly ? $checkResult : 1;
}

sub pamForSapstartsrv{
	my ($self, $msglst, $checkOnly) = @_;
	# autentification PAM entry for sapstartsrv
	my $file = '/etc/pam.d/sapstartsrv';
	if (-d '/etc/pam.d' && !-f $file){
        if ($checkOnly){
            $self->appendErrorMessage ("file '$file' doesn't exist");
            return 0;
        }

		if (defined $msglst){
			$msglst->AddMessage ("Creating PAM entry for sapstartsrv");
		}
		my $restore = umask (0133);
		if (!open (FD, '>' . $file)){
			$_[0]->AddError ("Cannot create file '$file': $!");
			umask ($restore);
			return undef;
		}
		print FD "#%PAM-1.0\n";
		print FD "auth requisite pam_unix_auth.so\n\n";
		close (FD);
		umask ($restore);
	}
	else{
		if (defined $msglst && stat _){
			$msglst->AddMessage ("File '$file' already exists");
		}
	}
	return 1;
}


sub messagesReadForAll{
	my ($self,$msglst) = @_;
	
	#
	# disable until security review
	#
	if (defined $msglst){
		$msglst->AddMessage ("Skipping access granting to messages file");
	}
	return 1;
	
	my $file = '/var/log/messages';
	my $mode = 0644;
	my @stat = stat ($file);
	if (!@stat){
		$self->AddError ("Cannot access file '$file': $!");
		return undef;
	}
	if (($stat[2] & $mode) != $mode){
		if (!chmod ($stat[2] | $mode, $file)){
			$self->AddError ("Cannot chmod file '$file': $!");
			return undef;
		}
	}
	return 1;
}


sub increaseKernelParameter{
    my ($self,$paramList, $msglst, $checkOnly) = @_;
    my $procval;
    foreach my $param (@$paramList){
        my $mode = $checkOnly ? "<" : "+<";
        if (!open (FD, $mode . $param->[1])){
            $self->PushError ("Cannot open file '$param->[1]': $!");
            return undef;
        }
        $procval = <FD>;
        chomp ($procval);
        $procval = int ($procval);
        if ($procval < $param->[3]){
            if ($checkOnly) {
                $self->appendErrorMessage ("$param->[2] should be increased from $procval to $param->[3]");
                return 0;
            } else {
                seek (FD,0,0);
                print FD "$param->[3]";
            }
        }
        close (FD);

        # now write the value to /etc/sysctl.conf
        # for loading into kernel after rebooting.
        my ($index, $value) = $self->{sysctl}->get($param->[2]);
        if ($index == -1){
            $value = $procval;
        }
        if (int($value) < $param->[3]){
            if ($checkOnly){
                $self->appendErrorMessage ("$param->[2] should be increased from $value to $param->[3]");
                return 0;
            }
            else{
                $self->{sysctl}->set($index, $param->[2], $param->[3]);
                $msglst->AddMessage ("Increase $param->[2] from $value to $param->[3]",$self->{sysctl});
            }
        }
        else{
            $msglst->AddMessage ("$param->[2] is up-to-date ($value), suggested $param->[3]");
        }
    }
    return 1;
}

#
# set to max value 2147483647 of signed int.
#

sub increaseMaxMapCount{
    my ($self,$msglst, $checkOnly) = @_;
    my $max_map_count = 2147483647;

    my @paramLst = (
         ['max_map_count', '/proc/sys/vm/max_map_count', 'vm.max_map_count', $max_map_count]
    );
    return $self->increaseKernelParameter (\@paramLst, $msglst, $checkOnly);
}

sub increaseSharedMemory{
	my ($self,$msglst, $checkOnly) = @_;
	# check for max shared memory settings, increase if needed
	# handle boot script sequence in /etc/init.d

    #max segment size in bytes
    my $CONST_SHMMAX = 18446744073709551615;

    my $CONST_SHMMNI = 32768;

    #system-wide total shared memory size
    my $CONST_SHMALL = 1152921504606846720;

    my @shmparams = (
        ['SHMMAX', '/proc/sys/kernel/shmmax','kernel.shmmax', $CONST_SHMMAX],
        ['SHMMNI','/proc/sys/kernel/shmmni','kernel.shmmni', $CONST_SHMMNI],
        ['SHMALL','/proc/sys/kernel/shmall','kernel.shmall', $CONST_SHMALL],
    );
    return $self->increaseKernelParameter (\@shmparams,$msglst, $checkOnly);
}

sub increasePortRange{
	my ($self, $msglst, $checkOnly) = @_;
	#
	# by default the ephemeral port range starts at 32k which
	# occasionally results in other processes picking a tcp port
	# that BIA/TREX considers a well-defined application port;
	# so change the range to start at 40k and set the upper
	# limit to 65500 if it is less than 40k
	#
	my $fn = '/proc/sys/net/ipv4/ip_local_port_range';
    my $key = 'net.ipv4.ip_local_port_range';
    my $line;

    my $fh = new IO::File($fn, "r");
    if (!defined $fh){
        $self->PushError ("Cannot open file '$fn': $!");
        return undef;
    }
    chomp($line = $fh->getline());
    $fh->close();

    my $checkResult = 1;
	my ($port_low,$port_high) = split (/\s+/,$line);
	my $limit_low = 40000;
    my $default_high = 65500;

    if (int ($port_low) < $limit_low) {
        if ($checkOnly){
            $self->appendErrorMessage ("'/proc/sys/net/ipv4/ip_local_port_range' is not set correctly");
            $self->appendErrorMessage ("expected lower limit is '$limit_low'");
            $checkResult = 0;
        } else {
            my ($index, $values) = $self->{sysctl}->get ($key);
            my $msg = "Increase lower limit of ephemeral port range from $port_low to $limit_low";
            if (int ($port_high) < $limit_low) {
                $msg .= " and higher limit from $port_high to $default_high";
                $port_high = $default_high;
            }
            $values = "$limit_low $port_high";
            $self->{sysctl}->set ($index, $key, $values);
            $msglst->AddMessage ($msg, $self->{sysctl});
        }
    } else {
        $msglst->AddMessage ("Ephemeral port range ($port_low to $port_high) is up to date already.");
    }
    return $checkOnly ? $checkResult : 1;
}

sub increaseAsyncIoRequests{
	my ($self, $msglst, $checkOnly) = @_;
	my $AIO_MAX_NR = 18446744073709551615;  # ULONG_MAX = 2^64-1, Bug 169992
	my ($index,$value) = $self->{sysctl}->get ('fs.aio-max-nr');

	if ($index == -1 || int ($value) < $AIO_MAX_NR){
        if ($checkOnly){
            $self->appendErrorMessage ("Max number of asynchronous I/O requests should be increased to $AIO_MAX_NR");
            $self->appendErrorMessage ("Current value: fs.aio-max-nr = $value");
            return 0;
        }
		$self->{sysctl}->set ($index, 'fs.aio-max-nr', $AIO_MAX_NR);
		$msglst->AddMessage ("Increase max number of asynchronous I/O requests to $AIO_MAX_NR",$self->{sysctl});
	}
    else{
        $msglst->AddMessage ("fs.aio-max-nr is up to date ($value)");
    }
	return 1;
}


sub setMemoryFailureEarlyKill{
	my ($self,$msglst, $checkOnly) = @_;
	
	if (!-e '/proc/sys/vm/memory_failure_early_kill'){
		$msglst->AddMessage ('Skipping vm.memory_failure_early_kill: not supported by linux kernel');
		return 1;
	}
	
	my ($index,$value) = $self->{sysctl}->get ('vm.memory_failure_early_kill');
	
	if ($index == -1){
		my $val = 1;
        if ($checkOnly){
            $self->appendErrorMessage ("vm.memory_failure_early_kill isn't set");
            return 0;
        }
		$self->{sysctl}->set ($index, 'vm.memory_failure_early_kill', $val);
		$msglst->AddMessage ("Setting vm.memory_failure_early_kill to $val",$self->{sysctl});
	}
	else{
		$msglst->AddMessage ("vm.memory_failure_early_kill is already set (value=$value)",$self->{sysctl});
	}
	return 1;
}

sub adaptNetworkParameters {
    my ($self, $msglst, $checkOnly) = @_;

    my $sysctl = $self->{sysctl};
    my $paramToRecommendedValue = {
        'net.core.somaxconn' => 4096,
        'net.ipv4.tcp_max_syn_backlog' => 8192,
        'net.ipv4.tcp_slow_start_after_idle' => 0,
        'net.ipv4.tcp_window_scaling' => 1
    };

    for my $param (keys %$paramToRecommendedValue) {
        my $recommendedValue = $paramToRecommendedValue->{$param};
        my ($index, $currentValue) = $sysctl->get($param);

        if ($index != -1 && $currentValue == $recommendedValue) {
            $msglst->addMessage("Value of $param is already set as per recommendation (value=$currentValue)");
            next;
        }

        if ($checkOnly) {
            $self->appendErrorMessage("Value of $param should be set to $recommendedValue");
            $self->appendErrorMessage("Current value: $param = $currentValue") if $index != -1;
            return 0;
        }
        $msglst->addMessage("Setting value of $param to $recommendedValue");
        $sysctl->set($index, $param, $recommendedValue);
    }
    return 1;
}

1;
