#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#
# Desc: NewDB installation package (installed package)


package SDB::Install::Package::Installed;

use SDB::Install::Package;
use SAPDB::Install::MD5Sum;
use SDB::Install::System qw (removeBusyFiles getRealPathName removeEmptyDirs enableWritePermissions);
use SDB::Install::DebugUtilities;
use SDB::Install::SysVars qw ($path_separator);

use File::Basename qw (dirname);

our  @ISA = qw (SDB::Install::Package);


our @import_member = qw (name version description
			buildstring makeid script files
			requires size checksum subdir interfaceversion);

sub new{
	my $class = shift;
	my $data = pop;
	my $self = $class->SUPER::new (@_);

	if (ref ($data) eq 'HASH'){
		
		#
		#  create package with data from registry
		#
		$self->{id} = shift;
		$self->{data} = $data;
	}
	elsif (ref ($data) =~ /::/ and $data->isa ('SDB::Install::Package::Installable')){

		#
		# create package with data from installable package
		#
		$self->{data} = $data->{data};
		$self->{id} = $data->{id};
	}
	$self->init ();
	return $self;
}

sub init{
	return $_[0]->evaluateScript ();
	
}


sub UninstallForUpdate {
	my ($self) = @_;
	
	my $submsglist = new SDB::Install::MsgLst();
	
	if (!$self->SetValid(0,$submsglist)){
			return undef;
	}
		
	$self->AddMessage (undef, $submsglist);

	if (defined $self->{progress_handler}){
		$self->{progress_handler}->InitProgress($self->GetNumberOfFiles,0);
	}
	# during a software update we do not un/reregister the odbc driver
	# any more, but we want to do that on uninstallation.
	# Note that we have to exclude this case here and not in the odbc
	# 'script.pm' since we have to take care of old odbc clients and their
	# 'Unregister' methods which are persisted in the 'INSTREG'.
	if(not $self->{'id'} eq 'odbc') {	
    	eval{
    		$self->Unregister ();
    	};
	}
	
	if ($@){
		$self->AddError ('Unregistration failed: '.$@);
	}

	$self->removePackageShortcuts();

	if (!$self->{installation}->isa ('SDB::Install::SAPSystem') ||
		$self->isGlobal){
		$self->RemoveFiles ();
	}

	$self->{installation}->RemovePackage ($self,1);

	return 1;
}



sub Uninstall {
	my (
	   $self,
	   $keepFiles,
	   $removeEmptyDirs
    ) = @_;

    if (!defined $self->Preuninstall ()){    
        $self->AddError ("Preuninstall failed");    
        return undef;    
    }    

	my $submsglist = new SDB::Install::MsgLst();

	if (!$self->SetValid(0,$submsglist)){
			return undef;
	}
	$self->AddMessage (undef, $submsglist);

	if (defined $self->{progress_handler}){
		$self->{progress_handler}->InitProgress($self->GetNumberOfFiles,0);
	}	

	eval{
		$self->Unregister ();
	};

	if ($@){
		$self->PushError ('Unregistration failed: '.$@);
	}

	$self->removePackageShortcuts();

	if(not $keepFiles) {
	   $self->RemoveFiles (); 
	}
    # RemoveFiles() only deletes dirs, which were created at package extraction time,
    # and only if they are empty after package content files were deleted.
    # If $removeEmptyDirs is set, _all_ empty dirs are deleted:
	if($removeEmptyDirs) {
        my $path = $self->GetPath();
        removeEmptyDirs($path);
	}

	eval{
		$self->Postuninstall ();	
	};
	if ($@){
		$self->PushError ('Postuninstall failed: '.$@);
	}
	
	$self->{installation}->RemovePackage ($self);
	return 1;
}

sub Unregister (){
	return 1;
}


sub PreLockedFilesCheck (){
	return 1;
}

# when this method fails for a package, that particular package is not deinstalled,
# but other packages are.
sub Preuninstall (){
	return 1;
}

sub Postuninstall (){
    return 1;
}

# when this method fails for a package, the uninstallation is aborted before anything is deinstalled,
# even other packages. IMPORTANT: this is only effective in installations involving the LCM sdk, i.e.
# involving the package manager 'SDB::Install::Installation::Generic', or its sub classes.
sub PreemptivePreuninstall (){
    return 1;
}

# -----------------------------------------------------------------

# we need these hard coded package ids for backward compatibility
# during updates/upgrades:

sub isGlobal {
    my $retval = 0;
    my $id = $_[0]->{id};
    if(defined $_[0]->GetInstallation() && $_[0]->GetInstallation()->isServer() &&
    ($id eq 'installer' || $id eq 'doc' || $id eq 'languages')) {
        $retval = 1;
    }
    if(defined $_[0]->{'data'}->{'global'}) {
        $retval = $_[0]->{'data'}->{'global'};
    }
    return $retval;
}

# we need these hard coded package ids for backward compatibility
# during updates/upgrades:

sub isClientPackage{
    my $retval = 0;
    my $id = $_[0]->{id};
    if($id eq 'sqldbc' || $id eq 'jdbc' || $id eq 'odbc' || $id eq 'odbo') {
        $retval = 1;
    }
    if(defined $_[0]->{'data'}->{'isClientPackage'}) {
        $retval = $_[0]->{'data'}->{'isClientPackage'};
    }
    return $retval;
}

# -----------------------------------------------------------------

#
# remove installed files
#

our $rmfile = 'pgm/rmfile';

sub _deleteFile{
    my ($self, $file) = @_;

    if (! -f $file){
        $self->AddProgressMessage ("File '$file' is already gone");
        return 1;
    }

    if (unlink ($file)){
        $self->AddMessage ("File '$file' deleted");
        return 1;
    }
    my $errstring = $!;
    if (!$isWin && $>  && $!{'EACCES'} && (stat(_))[4] == $>){
        my $dir = dirname ($file);
        my @statbuf = stat ($dir);
        if (!@statbuf){
            $self->PushError ("Cannot access parent dir '$dir' of file '$file': $!");
            return 0;
        }
        my $mode = enableWritePermissions ($dir, \@statbuf);
        if (defined $mode){
            my $rc = $self->_deleteFile ($file);
            chmod ($mode, $dir);
            return $rc;
        }
    }
    elsif($^O =~ /mswin|hpux/i){
        return undef;
    }
    else{
        $self->PushError ("Cannot delete file '$file': $errstring");
        return 0;
    }
    return 1;
}



sub RemoveFiles{
	my ($self) = @_;
	
	
	my $path = $self->GetPath();
	my $files = $self->{data}->{files};
	
	
	my @busyFiles;
	my $dirsToRemove = {};

	my $isWin = $^O =~ /mswin/i;

	my ($file, $name);

	foreach $file (keys (%$files)){
		if (defined $files->{$file}->{substituteMacros} && defined $files->{$file}->{origin_checksum}){
			
			my $origin = $self->getOriginFileBackupPath () . $path_separator . $self->{data}->{files}->{$file}->{origin_checksum};
			$self->_deleteFile ($origin);
		}

		if (defined $files->{$file}->{specialFile}){
			if($isWin){
				$file =~ s/\//\\/g;
			}
			$self->UninstallSpecialFile ($file);
			next;	
		}
		elsif($isWin){
			$file =~ s/\//\\/g;
		}
		
		$name = $path . $path_separator . $file;
		my $dir = dirname($name);
		while((not exists $dirsToRemove->{$dir}) && length($dir) > length($path)) {
			$dirsToRemove->{$dir} = undef;
			$dir = dirname($dir);
		}

		
		if (! -f $name){
			$self->AddProgressMessage ("File $name is already gone");
			next;
		}

		if (!defined $self->_deleteFile ($name)){
			push @busyFiles, $name;
		}
	}
	if (@busyFiles){
		my $msglist = new SDB::Install::MsgLst();
		my $msg = $self->AddMessage ('Removing busy files');
		my $rc = removeBusyFiles (\@busyFiles,$msglist);
		if (!$rc){
			$self->PushError ('Error removing busy files', $msglist);
		}
		$self->AddSubMsgLst ($msg,$msglist);
	}
	# now, we have a list of directories which contained installed package files.
	# we delete the empty ones of them.
	#
	# So we only delete dirs, if we created them at package extraction time,
	# and only if they are empty now.
	#
	my $dir;
	foreach $dir (reverse(sort(keys (%$dirsToRemove)))) {
	    unless (opendir(DH,$dir)){
	        $self->AddWarning ("Cannot  open directory \"$dir\": $!");
	    }
	    my @content=readdir(DH);
	    closedir(DH);
	    unless (@content){
	        $self->AddWarning ("Cannot read directory \"$dir\": $!");
	    }
	    my $isEmpty = 1;
	    if ($#content > 1){
	        foreach my $element (@content){
	            if(($element ne '.') and ($element ne '..')){
	                    $isEmpty = 0;
	                    last;
	            }       
	        }
	    }
	    if ($isEmpty){
	        if (!rmdir($dir)){
	            if (!$isWin && $> && $!{'EACCES'}){
	                my $parent = dirname ($dir);
	                my $mode = enableWritePermissions ($parent);
	                if (defined $mode){
	                    my $rc = rmdir ($dir);
	                    chmod ($mode, $parent);
	                    if ($rc){
	                        next;
	                    }
	                }
	            }
	            if (-l $dir){
	                if (!unlink ($dir)){
	                    $self->AddWarning ("Cannot remove symbolic link \"$dir\": $!");
	                }
	            }
	            else{
	                $self->AddWarning ("Cannot remove directory \"$dir\": $!");
	            }
	        }
	    }
	}
}

sub Verify (){
	return 1;
}


sub UninstallSpecialFile ($){
	my ($self,$file) = @_;
	$self->AddWarning ('Method UninstallSpecialFile() not implemented in package "' .
						 $self->GetName . '" of installation "' .
						 $self->{installation}->GetName . '"');
	return 1;
}


#
# check files, file permissions, dependencies and run package verify script 
# (e.g. to verify custom registrations) 
#

sub CheckPackage{
	my ($self,$result,$repair) = @_;
	
	$self->ResetError ();
	
	if (defined $self->{progress_handler}){
		$self->{progress_handler}->InitProgress ($self->GetNumberOfFiles(),0);
	}	
		
	if (!$self->IsValid){
		$self->PushError ('Package is flagged as invalid');
		if (defined $result){
			$result->{invalid} = 1;
		}
	}	
		
	#
	# checking files
	#
	
	my $path = $self->GetPath;
	my $files = $self->{data}->{files};
	
	local $uid = $self->{installation}->{uid};
	local $gid = $self->{installation}->{gid};
	
	my $missing = {};
	my $modified = {};
	my $permissions = {};
	my $perm_repaired = {};	

	my ($file, @statbuf, $sum, $mode, $perm_changed, @perm_msgs,@repair_msgs);

	my $error = 0;

	foreach my $key (keys (%$files)){
	    if( (exists $files->{$key}->{specialFile}) && $files->{$key}->{specialFile}) {
	        $self->AddProgressMessage ('Not checking special file ' . $key);
	        next;
	    }
		$file = $path . '/' . $key;
		@statbuf = stat ($file);
		$self->AddProgressMessage ('Checking file ' . $file);
		if (!@statbuf || !-r $file){
			$missing->{$key} = {'path' => getRealPathName($file)};
			$self->PushError ('Cannot check file "'.$file.'": '.$!);
			$error = 1;
			next;
		}
        if ($files->{$key}->{checksum} eq '0000000000000000000000'){
            $self->AddMessage ("Skipping checksum check of file '$file'");
        }
        else{
            $sum = MD5Sum($file);
            unless ($sum eq $files->{$key}->{checksum}){
                $modified->{$key} = {'path' => getRealPathName($file),
                                     'msg' => 'file was modified: md5 checksum changed [' . $files->{$key}->{checksum} . ' => ' . $sum . ']'};

                $self->PushError ('File "'.$file.'" was modified: md5 checksum changed [' . $files->{$key}->{checksum} . ' => ' . $sum . ']');
                $error = 1;
            }
        }

		$mode = (07777 & $statbuf[2]);

		
		$perm_changed = 0;
		
		@perm_msgs = ();
		@repair_msgs = ();

		
		if ($^O !~ /mswin/i){
				
			local $uid = $files->{$key}->{uid}
				if exists $files->{$key}->{uid};
			
			
			local $gid = $files->{$key}->{gid}
				if exists $files->{$key}->{gid};
	
		
			if (defined $uid and $uid != $statbuf[4]) {
				$perm_changed = 1;
				push @perm_msgs, 'uid was changed ['.$uid .' => ' . $statbuf[4].']';
			
				$permissions->{$key} = {
						'path' => getRealPathName($file),
						'msg' => 'uid was changed ['.$uid .' => ' . $statbuf[4].']'
					};
				$self->PushError ('uid of file "'.$file.'" was changed ['.$uid .' => ' . $statbuf[4].']');
			}
		
			if (defined $gid and $gid != $statbuf[5]) {
				$perm_changed = 1;
				push @perm_msgs, 'gid was changed ['.$gid .' => ' . $statbuf[5].']';
				$self->PushError ('gid of file "'.$file.'" was changed ['.$gid .' => ' . $statbuf[5].']');
			}
			

			if ($perm_changed){
		 		if (defined $repair){
					if (chown ($uid,$gid,$file)){
						if (defined $result){
							push @repair_msgs, "Ownership of file \"$file\" repaired";
						}
						$self->AddMessage ("Ownership of file \"$file\" repaired");
					}
					else{
						$self->PushError ("Cannot repair ownership of file \"$file\": $!");
						$error = 1;
					}	
				}
				else{
					$error = 1;
				}
			}


			if ($files->{$key}->{mode} != $mode) {
				$perm_changed = 1;
				push @perm_msgs, 'access mask was changed ' . sprintf ('[0%o => 0%o]',$files->{$key}->{mode}, $mode);
				$self->PushError ('Access mask of file "'.$file.'" was changed '.sprintf ('[0%o => 0%o]',$files->{$key}->{mode}, $mode));
				if ($repair){
					if (chmod ($files->{$key}->{mode},$file)){
						if (defined $result){
							push @repair_msgs, "Access mask of file \"$file\" repaired";
						}
						$self->AddMessage ("Access mask of file \"$file\" repaired");
					}
					else{
						$self->AddError ("Cannot repair access mask of file \"$file\": $!");
						$error = 1;
					}
				}
				else{
					$error = 1;
				}
			}


			if ($perm_changed){
				$permissions->{$key} = {
						'path' => getRealPathName($file),
						'msgs' => \@perm_msgs
					};
			}
	
			if (@repair_msgs){
				$perm_repaired->{$key} = {
						'path' => getRealPathName($file),
						'msgs' => \@repair_msgs
					};
			}


		}	
	}

	my $dependencies = {};
		
	unless ($self->CheckDependenciesByInstallation ($self->{installation})){
		foreach my $dependency (keys (%{$self->{dependencies}})){
			unless ($self->{dependencies}->{$dependency}->{resolved_by_installation}){
				$self->AddWarning ('Unresolved dependency: ' . $self->{dependencies}->{$dependency}->{str});
				$dependencies->{$dependency} = {'str' => $self->{dependencies}->{$dependency}->{str}};
			}
		}
	}
	
	if (defined $result){
		$result->{missed} = $missing;
		$result->{modified} = $modified;
		$result->{perm_changed} = $permissions;
		$result->{dependencies} = $dependencies;
		$result->{perm_repaired} = $perm_repaired;
	}
	
	
	#
	#	checking Verify() function
	#
	
	$result->{verify_ok} = 1;
	
	unless (defined $self->Verify()){
		$self->PushError ('Function Verify() of package failed');
		$error = 1;
		$result->{verify_ok} = 0;
	}
	
	if ($error){
		return undef;
	}
	return 1;	
	
}

sub IsDummy (){
	0;
}

sub IsValid ($){
	$_[0]->{data}->{valid};
}


sub SetValid ($$$) {
	my ($self,$value,$msgl) = @_;
	
	if ($value){
	      $self->{data}->{valid} = 1;
	}
	else{
	      $self->{data}->{valid} = 0;
	}
	$self->{data}->{modtime} = time ();
	
	my $nmsg;
	if (defined $msgl){
		$nmsg = $msgl->AddMessage ('Set package ' . ($value ? 'valid' : 'invalid'));
	}	
	if (!$self->{installation}->FlushRegistry ($nmsg)){
		$self->AddError ('Cannot set valid flag of package', $self->{installation});
		return 0;
	}
	return 1;
}

#-------------------------------------------------------------------------------
# Changes the owner/group ID of the registry file if an old ID matches

sub changeFileOwnerInRegistry{
	my ($self,$oldUid, $uid, $oldGid, $gid) = @_;
	
	if ($isWin){
		return 1;
	}
	
	my $files = $self->{data}->{files};
	my $fileProperties;
	my $changed = 0;
	
	foreach my $file (keys (%$files)){
		$fileProperties = $files->{$file};
		if (defined $fileProperties->{gid} && $fileProperties->{gid} == $oldGid){
			$fileProperties->{gid} = $gid;
			$changed = 1;
		}
		
		if (defined $fileProperties->{uid} && $fileProperties->{uid} == $oldUid){
			$fileProperties->{uid} = $uid;
			$changed = 1;
		}
	}
	
	if ($changed){
		$self->{data}->{modtime} = time ();
	}
	return 1;
}


# allows to find and remove legacy shortcuts for studio installations
sub getShortcutDefinitions{
	my ($self, $packagemanager) = @_;

	if ($self->{id} eq 'newdbstudioDirector'){
		return [{'folder' => 'SAP HANA',
			'name' => 'SAP HANA Studio',
			'target' => $self->getPath() . $path_separator .'hdbstudio' . ($isWin ? '.exe' : '')
		},
		{'folder' => 'SAP HANA',
			'name' => 'Studio',
			'target' => $self->getPath() . $path_separator .'hdbstudio' . ($isWin ? '.exe' : '')
		} ];
	}
	return undef;
}

1;

