#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#
# Desc: A collection of all PackageManager instances which are involved in
#       an installer session (i.e. a run of hdbinst, hdbupd, hdbsetup, or hdbuninst) 


package SDB::Install::PackageManagerCollection;

use SDB::Install::PackageManager;
use SDB::Install::BaseLegacy;
use SDB::Install::Tools qw (printTableToArrayOfLines arrayToHash);
use SDB::Install::System qw (getFileSystemInfo);
use SAPDB::Install::ProcState qw(SDB_PROCSTATE_FLAG_MODULES);
use SDB::Install::System::FuserHangChecker;
use SDB::Install::SysVars qw($isLinux);
our @ISA = qw (SDB::Install::BaseLegacy);
use strict;



#----------------------------------------------------------------------------

sub END_OF_ITERATION () {-2;}

sub new {
    my $self = shift->SUPER::new ();
    # the options of the application, choosen by the user either on the
    # commandline or in the gui:
    $self->{'applicationOptions'}         = undef;
    $self->{'packageManagers'}            = {};
    # we want to add and retrieve PackageManager objects by key, but we
    # also want to keep the order in which they were added: 
    $self->{'packageManagersOrderedKeys'} = [];
    # for the 'preinstall' phase, we need an 'Installation' object
    # per 'PackageManager': 
    $self->{'installations'}              = {};
    # there is a special PackageManager, which, if added, is processed first (last)
    # in all iterations over this collection during installation (uninstallation).
    $self->{'globalPackageManager'}       = undef;
    $self->{'globalInstallation'}         = undef;
    # for iteration: current position in the ordered keys list
    # ranges from 0 to #{ordered keys}; special value -1 indicates the
    # global PackageManager being current; -2 indicates that the iterator
    # has no more elements.
    $self->{'iteratorState'}              = END_OF_ITERATION;
    return $self;
}

#----------------------------------------------------------------------------

=functionComment

Sets a 'ProgressHandler' on every PacketManager in this collection,
and to its associated 'Installation', if existent.
A value of 'undef' is not treated specially.

=cut
sub SetProgressHandler{
    my (
        $self,
        $handler
    ) = @_;
    $self->SUPER::SetProgressHandler($handler);
    for($self->setIteratorToBegin(); $self->iteratorHasMoreItems(); ) {
        my ($packageManager, $installation) = @{$self->nextIteratorItem()};
        $packageManager->SetProgressHandler($handler);
        if(defined $installation) {
            $installation->SetProgressHandler($handler);
        }
    }
}

#----------------------------------------------------------------------------

sub SetPackageProgressHandler{
    my (
        $self,
        $handler
    ) = @_;
    for($self->setIteratorToBegin(); $self->iteratorHasMoreItems(); ) {
        my ($packageManager, $installation) = @{$self->nextIteratorItem()};
        $packageManager->SetPackageProgressHandler($handler);
    }
}

#----------------------------------------------------------------------------

=functionComment

The 'applicationOptions' field contains the commandline options and their values,
as set application wide.

=cut
sub setApplicationOptions{
    my (
        $self,
        $options
    ) = @_;
    $self->{'applicationOptions'} = $options;
    for($self->setIteratorToBegin(); $self->iteratorHasMoreItems(); ) {
        foreach my $packageManager (@{$self->nextIteratorItem()}) {
            if(defined $packageManager) {
                $packageManager->setApplicationOptions($options);
            }
        }
    }
}

#----------------------------------------------------------------------------

sub addPackageManagerObject {
    my (
        $self,
        $key,
        $packageManager,
        $installation # optional, only needed in 'preinstall' phase.    
    ) = @_;
    my $packageManagers = $self->{'packageManagers'};
    my $orderedKeys     = $self->{'packageManagersOrderedKeys'};
    my $installations   = $self->{'installations'};
    $packageManagers->{$key}     = $packageManager;
    $installations->{$key}       = $installation;
    push @$orderedKeys, $key;
    my $handler = $self->GetProgressHandler();
    if(defined $packageManager) {
        $packageManager->setApplicationOptions($self->{'applicationOptions'});
    }
    if(defined $installation) {
        $installation->setApplicationOptions($self->{'applicationOptions'});
    }    
}

#----------------------------------------------------------------------------

sub setIteratorToBegin {
	my ($self) = @_;
    my $orderedKeys = $self->{'packageManagersOrderedKeys'};
    if(defined $self->{'globalPackageManager'}) {
        $self->{'iteratorState'} = -1;
    }
    else {
		if (%{$self->{'packageManagers'}}) {
			my $nonGlobalCount = scalar (@{$self->{packageManagersOrderedKeys}}); 
            $self->{'iteratorState'} = 0;
             for(; !defined $self->{'packageManagersOrderedKeys'}->[$self->{'iteratorState'}] && $self->{'iteratorState'} < $nonGlobalCount;
                 $self->{'iteratorState'}++) {};
        }
        else {
            $self->{'iteratorState'} = END_OF_ITERATION;
        }
    }
}

#----------------------------------------------------------------------------

sub setIteratorToEnd {
    my ($self) = @_;
    if (!%{$self->{'packageManagers'}} && !defined $self->{'globalPackageManager'}){
		   $self->{'iteratorState'} = END_OF_ITERATION;
    }else{
		if (%{$self->{'packageManagers'}}) {
			$self->{'iteratorState'} = scalar (@{$self->{packageManagersOrderedKeys}}) - 1;
			for(; !defined $self->{'packageManagersOrderedKeys'}->[$self->{'iteratorState'}] && $self->{'iteratorState'} > -1;
					$self->{'iteratorState'}--) {}
		}
		else{
			$self->{'iteratorState'} = -1;
		}
    
	
		
	}
}

#----------------------------------------------------------------------------

sub nextIteratorItem {
    my ($self) = @_;  
    my $retval = undef;
    if($self->{'iteratorState'} != END_OF_ITERATION) {
        my $nonGlobalCount = (scalar @{$self->{'packageManagersOrderedKeys'}});
        if($self->{'iteratorState'} == -1) {
            $retval = [$self->{'globalPackageManager'}, $self->{'globalInstallation'}];
       }
        else {
            my $key = $self->{'packageManagersOrderedKeys'}->[$self->{'iteratorState'}];
            $retval = [$self->{'packageManagers'}->{$key}, $self->{'installations'}->{$key}];
        }
        
        $self->{'iteratorState'}++;
        
        for(; !defined $self->{'packageManagersOrderedKeys'}->[$self->{'iteratorState'}] && $self->{'iteratorState'} < $nonGlobalCount;
                $self->{'iteratorState'}++) {}
        
        if($self->{'iteratorState'} >= $nonGlobalCount) {
            $self->{'iteratorState'} = END_OF_ITERATION;
        }
	 }
    return $retval;
}

#----------------------------------------------------------------------------

sub previousIteratorItem {
    my ($self) = @_;    
    my $retval = undef;
    if($self->{'iteratorState'} != END_OF_ITERATION) {
        if($self->{'iteratorState'} == -1) {
            $retval = [$self->{'globalPackageManager'}, $self->{'globalInstallation'}];
			$self->{'iteratorState'}--;
        }
        else {
			my $key = $self->{'packageManagersOrderedKeys'}->[$self->{'iteratorState'}];
            $retval = [$self->{'packageManagers'}->{$key}, $self->{'installations'}->{$key}];
			$self->{'iteratorState'}--;
			for(; !defined $self->{'packageManagersOrderedKeys'}->[$self->{'iteratorState'}] && $self->{'iteratorState'} > -1;
					$self->{'iteratorState'}--) {}
			if ($self->{'iteratorState'} == -1 && !defined $self->{globalPackageManager}){
				$self->{'iteratorState'} = 	END_OF_ITERATION;		
			}
			
        }
    }
    return $retval;
}

#----------------------------------------------------------------------------

sub iteratorHasMoreItems {
    my ($self) = @_;    
    my $retval = 1;
    if($self->{'iteratorState'} == END_OF_ITERATION) {
        $retval = 0;
    }
    return $retval;
}

#----------------------------------------------------------------------------

sub getPackageManagersByKey {
    my (
        $self,
        $key
    ) = @_;    
    my $retval = undef;
    if(defined $self->{'packageManagers'}->{$key}) {
        $retval = [$self->{'packageManagers'}->{$key}, $self->{'installations'}->{$key}];
    }
    return $retval;
}

#----------------------------------------------------------------------------

sub removePackageManagersByKey {
    my (
        $self,
        $key
    ) = @_;    
    my $retval = undef;
    if(defined $self->{'packageManagers'}->{$key}) {
        $retval = [$self->{'packageManagers'}->{$key}, $self->{'installations'}->{$key}];
        delete $self->{'packageManagers'}->{$key};
        delete $self->{'installations'}->{$key};
        
        my $i = 0;
        foreach my $pmKey (@{$self->{packageManagersOrderedKeys}}){
			if ($key eq $pmKey){
				undef $self->{packageManagersOrderedKeys}->[$i];
				last;
			}
			$i++;
        }
    }
	
	if (exists $self->{resume}){
		delete $self->{runningProcesses};
		delete $self->{procState};
		delete $self->{resume};
	}

    return $retval;
}

#----------------------------------------------------------------------------

sub removeGlobalPackageManagerObject {
    my (
        $self
    ) = @_;
    my $retval = [$self->{'globalPackageManager'}, $self->{'globalInstallation'}];
    if(defined $self->{'globalPackageManager'}) {
        delete $self->{'globalPackageManager'};
    }
    if(defined $self->{'globalInstallation'}) {
        delete $self->{'globalInstallation'};
    }
    return $retval;
}

#----------------------------------------------------------------------------

=functionComment

Creates a data structure containing all file systems (identified by mount point)
affected by the 'PackageManager's in this collection;
and per file system, the number of bytes used by 
this installation in this file system.
The return value is a reference to a hash with the mount points as keys and
refs to hashes containing filesystem utilization information as values.
NOTE: if > 1, 'scalingFactor' modifies the unit of fields
'*Blocks'.
e.g. if scalingFactor is 1024 (or 1024*1024), the unit is kiloblocks
(resp. megablocks). 

Example:
        {
            '/'=>
                {
                    'fileSystemTotalSizeBlocks'=>'7224600',
                    'filesystemBlockSizeBytes'=>'1024',
                    'actualInstallationFootprintBlocks'=>'288',
                    'estimatedUpdatedInstallationFootprintBlocks'=>'288',
                    'deviceId'=>'2050',
                    'filesystemAvailableSizeBlocks'=>'1890908',
                    'estimatedNewInstallationFootprintBlocks'=>'137964728',
                    'scalingFactor'=>'1',
                },
            '/home'=>
                {
                    'fileSystemTotalSizeBlocks'=>'27095116',
                    'filesystemBlockSizeBytes'=>'1024',
                    'actualInstallationFootprintBlocks'=>'130',
                    'estimatedUpdatedInstallationFootprintBlocks'=>'130',
                    'deviceId'=>'2052',
                    'filesystemAvailableSizeBlocks'=>'13063428',
                    'estimatedNewInstallationFootprintBlocks'=>'56547314',
                    'scalingFactor'=>'1024',
                },
            '/tmp/mptest/fs02'=>
                {
                    'fileSystemTotalSizeBlocks'=>'19362',
                    'filesystemBlockSizeBytes'=>'1024',
                    'actualInstallationFootprintBlocks'=>'170',
                    'estimatedUpdatedInstallationFootprintBlocks'=>'170',
                    'deviceId'=>'1794',
                    'filesystemAvailableSizeBlocks'=>'17157',
                    'estimatedNewInstallationFootprintBlocks'=>'81203400',
                    'scalingFactor'=>'1024',
                },
        }
 
=cut
sub getFootprintsByFilesystem {
    my ($self) = @_;
    # keys: mount points
    # values: refs to hashes containing filesystem footprint information.
    my $packageManagerCollectionFootprints     = undef;
    my $packageManagerFootprints               = undef;
    my $fileSystemRoot                         = undef;
    my $size                                   = undef;
    my $packageManagers                        = $self->{'packageManagers'};
    my $packageManagerCollectionFileSystemInfo = undef; 
    my $packageManagerFileSystemInfo           = undef;
    my $success                                = 1;
    my $errmsg                                 = undef;
    if(!defined $packageManagers) {
        $errmsg = "SDB::Install::PackageManagerCollection::getFootprintsByFilesystem: ".
                  "Fatal error: No PackageManager objects to process in this collection.\n";
        $success = 0;
        return (undef, $success, $errmsg);
    }
    $packageManagerCollectionFootprints = {};
    for($self->setIteratorToBegin(); $self->iteratorHasMoreItems(); ) {
        my ($packageManager) = @{$self->nextIteratorItem()};
        ($packageManagerFootprints, $success, $errmsg) =
            $packageManager->getFootprintsByFilesystem();
        if(!$success) {
            $self->PushError(undef, $packageManager);
            return (undef, $success, $errmsg);
        }
        foreach $fileSystemRoot (keys %$packageManagerFootprints) {
            $packageManagerCollectionFileSystemInfo = $packageManagerCollectionFootprints->{$fileSystemRoot};
            $packageManagerFileSystemInfo           = $packageManagerFootprints->{$fileSystemRoot};
            if(!defined $packageManagerCollectionFileSystemInfo) {
                # we have not seen this file system before:
                $packageManagerCollectionFootprints->{$fileSystemRoot} = $packageManagerFileSystemInfo;
            }
            else {
                # file system info for this file system already exists, merge:
                $packageManagerCollectionFileSystemInfo->{'estimatedNewInstallationFootprintBlocks'} +=
                    $packageManagerFileSystemInfo->{'estimatedNewInstallationFootprintBlocks'};
                $packageManagerCollectionFileSystemInfo->{'estimatedUpdatedInstallationFootprintBlocks'} +=
                    $packageManagerFileSystemInfo->{'estimatedUpdatedInstallationFootprintBlocks'};
                $packageManagerCollectionFileSystemInfo->{'actualInstallationFootprintBlocks'} +=
                    $packageManagerFileSystemInfo->{'actualInstallationFootprintBlocks'};
            }
        }
    }
    # we want to have the dynamic part of the file system info from the OS
    # (i.e. available space) as up to date as possible:
    foreach $fileSystemRoot (keys %$packageManagerCollectionFootprints) {
        my (undef, undef, undef, $filesystemAvailableSizeBlocks, undef, undef, $success, $errmsg) = 
            getFileSystemInfo($fileSystemRoot);
        if(!$success) {
            return (undef, $success, $errmsg);
        }
        $packageManagerCollectionFileSystemInfo = $packageManagerCollectionFootprints->{$fileSystemRoot};
        $packageManagerCollectionFileSystemInfo->{'filesystemAvailableSizeBlocks'} = $filesystemAvailableSizeBlocks;
    }
    return (
        $packageManagerCollectionFootprints,
        $success,
        $errmsg
    );
}

#----------------------------------------------------------------------------

=functionComment

Creates a data structure containing all files of the 'PackageManager's
in this collection which are currently in use and belong
to a package which is to be installed (i.e. does not have the
'skip' attribute set; this can only happen in the update case.),
together with the commandline of the locking process.
The return value is a reference to a hash where the keys are the PIDs of
the locking processes and the values are arrays of hashes,
one and only one for each locked file.
The keys in these subhashes point to the paths of the locked files and to
the respective commandline of the locking process.
Example:
        {
            '30000'=>
                {
                    'lockedFiles'=>
                        [
                            'bbb.jpg',
                            '/aaa.dll',
                        ],
                    'commandLine'=>'/usr/bin/program01 -someopt',
                },
            '100000'=>
                {
                    'lockedFiles'=>
                        [
                            'bbb.pbx',
                            '/aaa/zzz.htm',
                        ],
                    'commandLine'=>'/usr/bin/program02 -someotheropt',
                }
        }
=cut
sub getFilesInUseByPID {
    my (
        $self,
        $ps
    ) = @_;
    my $success           = 1;
    my $errmsg            = undef;
    my $packageManagers   = $self->{'packageManagers'};
    if(!defined $packageManagers) {
        $errmsg = "SDB::Install::PackageManagerCollection::getFilesInUseByPID: ".
                  "Fatal error: No PackageManager objects to process in this collection.\n";
        $success = 0;
        return (undef, $success, $errmsg);
    }
    my $packageManagerFilesInUse = undef;
    if(!defined $ps) {
        if ($isLinux){
            my $hangChecker = new SDB::Install::System::FuserHangChecker();
            $hangChecker->setMsgLstContext([$self->getMsgLst()]);
            my $rc = $hangChecker->check();
            if (defined $rc && $rc == 0){
                return (undef,0,$hangChecker->getErrorString());
            }
        }
        $ps = new SAPDB::Install::ProcState (SDB_PROCSTATE_FLAG_MODULES);
    }
    my %retval;
    my $progress_handler = $self->GetProgressHandler();
    if(defined $progress_handler) {
		$progress_handler->IncRange(int values %$packageManagers);
    }
    my $ignoreProgramsPattern;
    for($self->setIteratorToBegin(); $self->iteratorHasMoreItems(); ) {
        my ($packageManager,$inst) = @{$self->nextIteratorItem()};
        ($packageManagerFilesInUse, $success, $errmsg) =
            $packageManager->getFilesInUseByPID($ps);
        if(!$success) {
            $self->PushError(undef, $packageManager);
            return (undef, $success, $errmsg);
        }
        if (defined $inst){
            $ignoreProgramsPattern = $inst->getIgnoreBusyProgramsPattern ();
        }
        foreach my $processId (keys %$packageManagerFilesInUse) {
            my $lockedFiles = $packageManagerFilesInUse->{$processId}->{'lockedFiles'};
            my $commandLine = $packageManagerFilesInUse->{$processId}->{'commandLine'};
            if(exists $retval{$processId}) {
                foreach my $entry (@$lockedFiles) {
                    if(!exists $retval{$processId}->{'lockedFiles'}->{$entry}) {
                        $retval{$processId}->{'lockedFiles'}->{$entry} = $entry;
                    }
                }
            }
            else {
                $retval{$processId}->{'lockedFiles'} = arrayToHash($lockedFiles);
                $retval{$processId}->{'commandLine'} = $commandLine;
            }
            
            if ($packageManagerFilesInUse->{$processId}->{lock_check_failed}){
                if (!defined $ignoreProgramsPattern ||
                    ($commandLine !~ /$ignoreProgramsPattern/)){

                    $retval{$processId}->{lock_check_failed} = 1;
                }
            }
        }
    }
    # converting from hashes to arrays again:
    foreach my $processId (keys %retval) {
        my $lockedFilesAsHash = $retval{$processId}->{'lockedFiles'};
        my @fkeys = keys %$lockedFilesAsHash;
        $retval{$processId}->{'lockedFiles'} = \@fkeys;
        $retval{$processId}->{'commandLine'} = $retval{$processId}->{'commandLine'};
    }
    
    $self->{runningProcesses} = \%retval;
        
    return (
        \%retval,
        $success,
        $errmsg
    );
}

#----------------------------------------------------------------------------

=functionComment

Iterates over all PackageManagers in this collection and determines the required
disk space in every affected file system. Prints info/warnings/errors on the console
and into the log file.
Returns 0 if not enough disk space is available or if an error occurs, 1 otherwise. 

=cut
sub checkFileSystems {
    my ($self) = @_;
    my $headerMsg = "Checking required and available disk space per file system...";
    $self->AddMessage($headerMsg);
    my ($installationFootprints, $success, $errmsg) = 
        $self->getFootprintsByFilesystem();
    if(!$success) {
        $self->AddError($errmsg);
        return 0;
    }
    my $logTableAsArray = [];
    my @stakeholderArray;
    $self->AddMessage ('File System Utilization Summary:');
    my $consoleTableAsArray = [];
    my $tableHeader = ['| Mount Point', 'Required disk space (KB)', 'Available disk space (KB)', ''];
    push @$logTableAsArray, ($tableHeader);
    push @$consoleTableAsArray, ($tableHeader);
    my $mountPoint           = undef;
    my $mountPointInfo       = undef;
    my $mountPointMsg        = undef;
    my $requiredBlocks       = undef;
    my $availableBlocks      = undef;
    my $blockSize            = undef;
    my $scalingFactor        = undef;
    my $scaledBlockSize      = undef;
    my $remainingDenominator = undef;
    my $requiredBytes        = undef;
    my $availableBytes       = undef;
    my $failureNoSpace       = 0;
    my $failureNoSpaceMsg    = undef;
    my $currentRow           = undef;
    foreach $mountPoint (keys %$installationFootprints) {
        $mountPointInfo = $installationFootprints->{$mountPoint};
        $requiredBlocks = $mountPointInfo->{'estimatedNewInstallationFootprintBlocks'};
        $availableBlocks = $mountPointInfo->{'filesystemAvailableSizeBlocks'};
        $blockSize = $mountPointInfo->{'filesystemBlockSizeBytes'};
        $scalingFactor = $mountPointInfo->{'scalingFactor'};
        $scaledBlockSize = $blockSize * $scalingFactor;
        $remainingDenominator = 1024;
        while($scaledBlockSize % 2 == 0 && $remainingDenominator % 2 == 0) {
            $scaledBlockSize /= 2;
            $remainingDenominator /= 2;
        }
        $requiredBytes = ($requiredBlocks * $scaledBlockSize) / $remainingDenominator;
        $availableBytes = ($availableBlocks * $scaledBlockSize) / $remainingDenominator;
        if($requiredBlocks > $availableBlocks) {
            $failureNoSpace = 1;
            $currentRow = ['| '.$mountPoint, $requiredBytes, $availableBytes, '<== NOT ENOUGH SPACE LEFT'];
            push @$consoleTableAsArray, ($currentRow);
        }
        else {
            $currentRow = ['| '.$mountPoint, $requiredBytes, $availableBytes, ''];
        }
        push @$logTableAsArray, ($currentRow);
        if (defined $mountPointInfo->{stakeholder}){
            push @stakeholderArray, ['| ' . $mountPoint, join (', ', @{$mountPointInfo->{stakeholder}})];
        }
    }
    my $logTableLines;
    if (@stakeholderArray){
        unshift @stakeholderArray, ['| Mount Point', 'Used by'];
        $logTableLines = printTableToArrayOfLines(\@stakeholderArray, ' | ');
        $self->AddMessages(['', @$logTableLines, '']);
    }

    $logTableLines = printTableToArrayOfLines($logTableAsArray, ' | ');
    $self->AddMessages($logTableLines);

    if($failureNoSpace) {
        my @summaryMessages = ();
        push(@summaryMessages, 'Not enough disk space:');

        my $consoleTableLines = printTableToArrayOfLines($consoleTableAsArray, ' | ');
        push(@summaryMessages, @$consoleTableLines);

        if ($self->{_ignore_fs_check}){
            $self->getMsgLst()->addWarning($_) for @summaryMessages;
            $self->AddMessage ("Ignoring error due to command line switch '--ignore'");
            return 1;
        }
        $self->appendErrorMessage($_, undef, 4) for @summaryMessages;
        return 0;
    }
    $self->AddMessage('...DONE. Enough disk space available.');
    return 1;
}

#----------------------------------------------------------------------------

=functionComment

Iterates over all PackageManagers in this collection and determines whether there are
files locked by processes. Prints info/warnings/errors on the console
and into the log file.
Returns 0 if locked files exist or if an error occurs, 1 otherwise. 

=cut
sub checkForLockedFiles {
    my (
        $self,
        $procState
    ) = @_;
    $self->AddMessage('Checking for installation files locked by running processes...');
    my ($filesInUseByPID, $success, $errmsg) = 
        $self->getFilesInUseByPID($procState);
    if(!$success) {
        $self->AddError($errmsg);
        return 0;
    }
    my $filesInUseByPIDSize = int values %$filesInUseByPID;
    if($filesInUseByPIDSize > 0) {
        my $line = undef;
        my $pid = undef;
        my $pidInfo = undef;
        my $commandLine = undef;
        my $lockedFiles = undef;
        my $file = undef;

        my $logTableAsArray = [];
        push @$logTableAsArray, (['| File', 'Process Id', 'Command Line']);

        my $consoleTableAsArray = [];
        push @$consoleTableAsArray, (['| Process Id', 'Command Line']);

        my $currentLogRow = undef;
        my $currentConsoleRow = undef;
        foreach $pid (keys %$filesInUseByPID) {
            $pidInfo = $filesInUseByPID->{$pid};
            $commandLine = $pidInfo->{'commandLine'};
            $lockedFiles = $pidInfo->{'lockedFiles'};
            
            if ($filesInUseByPID->{$pid}->{lock_check_failed}){
				$currentConsoleRow = ['| '.$pid, $commandLine];
				push @$consoleTableAsArray, ($currentConsoleRow);
            }
            
            foreach $file (@$lockedFiles) {
                $currentLogRow = ['| '.$file, $pid, $commandLine];
                push @$logTableAsArray, ($currentLogRow);
            }
        }
        my $logTableLines = printTableToArrayOfLines($logTableAsArray, ' | ');
        $self->AddMessages($logTableLines);
        
        if (scalar @$consoleTableAsArray > 1){
            my @summaryMessages = ();
            push(@summaryMessages, 'There are files locked by the following processes:');

            my $consoleTableLines = printTableToArrayOfLines($consoleTableAsArray, ' | ');
            push(@summaryMessages, @$consoleTableLines);

			if ($self->{_ignore_busy_files}){
                $self->getMsgLst()->addWarning($_) for @summaryMessages;
				$self->AddMessage ("Ignoring error due to command line switch '--ignore'");
				return 1;
			}
            $self->appendErrorMessage($_, undef, 4) for @summaryMessages;
            $self->PushError('Terminate all locking processes and rerun the installer.');
            return 0;
		}
    }
    $self->AddMessage('...DONE. No locked files.');
    return 1;
}

#----------------------------------------------------------------------------

=functionComment

Iterates over all PackageManagers in this collection in the order in which they were added
and runs their resp. 'Preinstall' methods.
If a global PackageManager was added, it comes first in this iteration.
After these runs, global checks for disk space and locked files are performed. 

=cut
sub Preinstall{
    my ($self, $repair_mode,$force_downgrade)       = @_;
    my $success                    = 0;
    my $errmsg                     = undef;
    my $packageManagers            = $self->{'packageManagers'};
    my $packageManagersOrderedKeys = $self->{'packageManagersOrderedKeys'};
    my $packageManagerKey          = undef;
    my $packageManager             = undef;
    
    if(!defined $packageManagers) {
        $errmsg = "SDB::Install::PackageManagerCollection::Preinstall: ".
                  "Fatal error: No PackageManager objects to process in this collection.\n";
        $self->AddError($errmsg);
        $success = 0;
        return $success;
    }
    my %retval;
    my $progress_handler = $self->GetProgressHandler();
    if(defined $progress_handler) {
        $progress_handler->IncRange(int values %$packageManagers);
    }
    for($self->setIteratorToBegin(); $self->iteratorHasMoreItems(); ) {
		my ($packageManager, $installation) = @{$self->nextIteratorItem()};
        $packageManager->setMsgLstContext ($self->getMsgLstContext());
        $success = $packageManager->Preinstall($installation, $repair_mode, $force_downgrade);
        if(!$success) {
            return $success;
        }
    }
    if(!$self->checkFileSystems()) {
        $success = 0;
        $self->{resume} = 1;
        return $success;
    }
    if ($isLinux){
        my $hangChecker = new SDB::Install::System::FuserHangChecker();
        $hangChecker->setMsgLstContext($self->getMsgLstContext());
        my $rc = $hangChecker->check();
        if (defined $rc && $rc == 0){
            return 0;
        }
    }
    my $ps = new SAPDB::Install::ProcState (SDB_PROCSTATE_FLAG_MODULES);
    if(!$self->checkForLockedFiles($ps)) {
        $success = 0;
        $self->{resume} = 2;
        $self->{procState} = $ps; 
        return $success;
    }
    delete $self->{resume};
    delete $self->{procState};
    $success = 1;
    return $success;
}

#----------------------------------------------------------------------------

sub canResumePreinstallChecks ($){
	defined $_[0]->{resume} ? 1 : 0;
}

#----------------------------------------------------------------------------

=functionComment

Iterates over all PackageManagers in this collection in the reverse order
in which they were added and runs their resp. 'Uninstall' methods.
If a global PackageManager was added, it comes last in this iteration.

=cut
sub Uninstall{
    my ($self, @args) = @_;
    my $success                    = 1;
    my $errmsg                     = undef;
    my $packageManagers            = $self->{'packageManagers'};
    my $packageManagersOrderedKeys = $self->{'packageManagersOrderedKeys'};
    my $packageManagerKey          = undef;
    my $packageManager             = undef;
    if(!defined $packageManagers) {
        $errmsg = "SDB::Install::PackageManagerCollection::Uninstall: ".
                  "Fatal error: No PackageManager objects to process in this collection.\n";
        $self->AddError($errmsg);
        $success = 0;
        return $success;
    }
    my %retval;
    my $progress_handler = $self->GetProgressHandler();
    if(defined $progress_handler) {
        $progress_handler->IncRange(int values %$packageManagers);
    }
    for($self->setIteratorToEnd(); $self->iteratorHasMoreItems(); ) {
        my ($packageManager) = @{$self->previousIteratorItem()};
        if(!defined $packageManager->Uninstall (@args)) {
            $self->PushError(undef, $packageManager);
			return undef;
        }
    }
    return $success;
}

#----------------------------------------------------------------------------

sub FreePackages {
    my ($self) = @_;
    my $success                    = 1;
    my $errmsg                     = undef;
    my $packageManagers            = $self->{'packageManagers'};
    my $packageManagersOrderedKeys = $self->{'packageManagersOrderedKeys'};
    my $packageManagerKey          = undef;
    my $packageManager             = undef;
    if(!defined $packageManagers) {
        $errmsg = "SDB::Install::PackageManagerCollection::FreePackages: ".
                  "Fatal error: No PackageManager objects to process in this collection.\n";
        $self->AddError($errmsg);
        $success = 0;
        return $success;
    }
    for($self->setIteratorToEnd(); $self->iteratorHasMoreItems(); ) {
        my ($packageManager, $installation) = @{$self->previousIteratorItem()};
        $packageManager->setMsgLstContext ($self->getMsgLstContext());
        $success = $packageManager->FreePackages();
        if(!$success) {
            return $success;
        }
        if(defined $installation) {
            $success = $installation->FreePackages();
            if(!$success) {
                return $success;
            }
        }
    }
    $success = 1;
    return $success;
}

#----------------------------------------------------------------------------

sub CleanUp{
    my ($self) = @_;
    my $success                    = 1;
    my $errmsg                     = undef;
    my $packageManagers            = $self->{'packageManagers'};
    my $packageManagersOrderedKeys = $self->{'packageManagersOrderedKeys'};
    my $packageManagerKey          = undef;
    my $packageManager             = undef;
    if(!defined $packageManagers) {
        $errmsg = "SDB::Install::PackageManagerCollection::CleanUp: ".
                  "Fatal error: No PackageManager objects to process in this collection.\n";
        $self->AddError($errmsg);
        $success = 0;
        return $success;
    }
    $self->SetProgressHandler(undef);
    $self->FreePackages();
    $self->{'packageManagers'}            = undef;
    $self->{'packageManagersOrderedKeys'} = undef;
    $self->{'installations'}              = undef;
    $self->{'globalPackageManager'}       = undef;
    $self->{'globalInstallation'}         = undef;
    $self->{'iteratorState'}              = END_OF_ITERATION;
    $success = 1;
    return $success;
}

#----------------------------------------------------------------------------

1;
