package SDB::Install::RemoteHosts;

use strict;
use base "SDB::Install::BaseLegacy";
use SDB::Install::Globals qw($gProductNameSystem);
use SDB::Install::System qw (exec_program loadSSLRequiringPackage);
use SDB::Install::SysVars qw ($path_separator $isWin);
use Fcntl qw(:DEFAULT :mode);
use File::Spec;
use Time::HiRes qw (usleep);
use SDB::Install::HdbInstallerOutputParser qw (parseHdbInstallerErrorMessages);
use SDB::Install::SSH::Factory;

our $sshFactory;

sub new{
    my $self = shift->SUPER::new();
    $self->{_hosts} = \@_;
    $self->{_privkey} = undef;
    $self->{_pubkey}  = undef;

    $sshFactory = SDB::Install::SSH::Factory->new();

    return $self;
}

sub setUserName{
    my ($self, $name, $generateSSHKeys) = @_;
    if (defined $self->{_user} && ($self->{_user} eq $name)) {
        return 1;
    }
    $self->{_user} = $name;
    my @ent = getpwnam ($name);
    if (@ent){
        my $home = $ent[7];
        my $privateKeyPath = File::Spec->catfile($home, '.ssh', 'id_rsa');
        my $publicKeyPath = File::Spec->catfile($home, '.ssh', 'id_rsa.pub');

        if (-f $privateKeyPath){
            $self->{_privkey} = $privateKeyPath;
        }
        if (-f $publicKeyPath){
            $self->{_pubkey} = $publicKeyPath;
        }

        if (defined $self->{_pubkey} && defined $self->{_privkey}){
            $self->AddMessage ("rsa keys found");
        } else {
            $self->AddMessage ("no rsa keys found");
            return 1 if (!$generateSSHKeys);

            my $msglst = SDB::Install::MsgLst->new();
            $self->AddMessage ("Generating rsa keys", $msglst);
            if (defined $privateKeyPath) {
                unlink $privateKeyPath;
            }
            my $session = $sshFactory->getSshSession();
            if (defined $session && $session->generateSSHKeys($privateKeyPath, $msglst)) {
                $self->AddMessage("Keys generated successfully");
                $self->{_privkey} = $privateKeyPath;
                $self->{_pubkey}  = $publicKeyPath;
            } else {
                $self->AddWarning("Error occured while geneating SSH keys. Remote hosts authentication will be performed with the root credentials");
            }
        }
    }
    return 1;
}

sub connect{
    my ($self) = @_;
    $self->{_sessions} = [];
    my $socket;
    my $rc = 0;
    my $errlst = new SDB::Install::MsgLst ();

    if (!$sshFactory->loadLibsshPackage($errlst)) {
        $self->setErrorMessage ("Loading libssh packages failed", $errlst);
        return 2;
    }

    for my $host (@{$self->getHostNames()}) {
        $rc = $self->_connectToHost($host) ? $rc : 1;
    }

    return $rc;
}

sub _connectToHost {
    my ($self, $hostname) = @_;
    my $session = $sshFactory->getSshSession();
    $session->setMsgLstContext($self->getMsgLstContext(), 1);

    if (!$session->connect($hostname)) {
        my $rc = $session->sessionError();
        my $errorMessage = "Connect to host '$hostname' failed" . ((defined $rc && $rc ne '') ? ": rc=$rc" : '');
        $self->appendErrorMessage($errorMessage);
        return 0;
    }
    $self->{_sessions}->[$self->getIndexByHostName($hostname)] = $session;
    return 1;
}

sub getSession {
    my ($self, $hostname) = @_;
    my $idx = $self->getIndexByHostName($hostname);
    return $self->{_sessions}->[$idx];
}

sub getSessionByIndex {
    my ($self, $idx) = @_;
    return $self->{_sessions}->[$idx];
}

#
# key passphrase
#

sub setPassPhrase{
    $_[0]->{_passphrase} =  $_[1];
}

#
# user password
#

sub setUserPass{
    $_[0]->{_password} = $_[1];
}

sub addHost{
    push @{$_[0]->{_hosts}}, $_[1];
}

sub getNumberOfHosts{
    return scalar @{$_[0]->{_hosts}};
}

sub authKey {
    my ($self, $ignorePubKeyMissing) = @_;
    my $rc = 0;

    my $session;
    foreach my $host (@{$self->getHostNames()}) {
        $session = $self->getSession($host);
        if (!defined $session){
            my $msg = "authKey failed: host '$host' is not connected";
            if ($ignorePubKeyMissing) {
                $self->AddMessage($msg);
            }
            else {
                $self->AddError($msg);
            }
            $rc |= 1;
            next;
        }
        if (!$session->authKey($ignorePubKeyMissing, $self->{_user}, $self->{_pubkey},
                                 $self->{_privkey}, $self->{passphrase})) {
            $rc |= 2;
        }
    }
    return $rc;
}

sub getPubKey{
    my ($self) = @_;
    if (!$self->{_pubkey}){
        $self->AddError ("No public rsa key");
        return undef;
    }
    if (!open (FH, $self->{_pubkey})){
        $self->AddError ("Cannot open public rsa key '$self->{_pubkey}': $!");
        return undef;
    }
    my $keystr = <FH>;
    close (FH);
    return $keystr;
}

sub areHostsAuthenticated{
    my ($self) = @_;
    my $success = 1;
    foreach my $host (@{$self->getHostNames()}) {
        my $session = $self->getSession($host);
        if (!defined $session) {
            $self->appendErrorMessage("Host '$host' is not accessible");
            $success = 0;
            next;
        }
        if (!$session->authOk()) {
            $self->appendErrorMessage("Connection to host '$host' is not authorized");
            $success = 0;
        }
    }
    return $success;
}

sub authPassword {
    my ($self, $generateSSHKeys) = @_;
    my $rc = 0;
    my $session;
    foreach my $host (@{$self->getHostNames()}) {
        $session = $self->getSession($host);
        if (!$session) {
            $self->PushError ("Host '$host' not accessible");
            $rc |= 1;
            next;
        }

        if (!$session->authPassword($self->{_user}, $self->{_password}, $generateSSHKeys, $self->getPubKey())){
            $rc |= 2;
            if ($self->{_userCouldBeChanged}){
                # support several attempts with different users
                # => reconnect
                $self->_connectToHost($host);
            }
        }
    }

    return $rc;
}


#-------------------------------------------------------------------------------
# Executes the specified command parallel on each remote host or on selected
# hosts (parameter $executeOnlyOnHosts).
#
# Calling a specific main program
# -------------------------------
# If a '-main' parameter is specified, the main method of that Perl class
# is executed instead of the specified program.
# Example:
#    '-main SDB::Install::App::Console::DBUpgradeHost::main'
# Note:
#    The C-program (helpers.c) does not detect a main parameter followed by
#    an equal sign:
#         '-main=SDB...' is wrong, but '-main SDB...' is correct
#
# Passwords are passed via XML using the input stream to the host.
# --------------------------------------------------
# '--read_password_from_stdin[=xml]' command:
# The function '$instconfig->getXmlPasswordStream' returns the required array
# for the parameter $stdin_input.
#
#
# Template values
# ---------------
# If $cmd contains template symbols '%s', these placeholders are replaced by
# the values contained in an array for each host.
# Example: $cmd        = '... --hostname=%s --roles=%s%s'
#          $cmdTmplRpl = [['lu000111', 'worker', ' --group=default'],
#                         ['lu000222', 'standby', '']]
#
#          '... --hostname=lu000111 --roles=worker --group=default' is sent to $self->{_hosts}->[0]
#          '... --hostname=lu000222 --roles=standby'                is sent to $self->{_hosts}->[1]

sub executeParallel {
    my $rc = 0;
    my ($self,
        $cmd,                # e.g. 'hana/shared/<SID>/global/hdb/install/bin/hdbreg <parameters>'
        $progress_message,   # e.g. "Upgrading host '%s'..."
        $done_message,       # e.g. "Upgrade host '%s' done"
        $error_message,      # e.g. "Upgrade host '%s' failed"
        $stdin_input,        # Password input stream (see above)
        $cmdTmplRpl,         # Template values (see above)
        $executeOnlyOnHosts, # Array of hostnames, on which to execute the command
                             # (use when you don't want to execute it on all hosts)
        $doNotFailOnError,
        $isNotVerbose,       # suppress console output
        $isCollectHostInfo   # suppress logging of the remote output
       ) = @_;

    my (@rcs, @rest, @output_buffer, $hostname, $session);
    my ($msglst, $errlst);


    my $indexes = $self->filterIndexes($executeOnlyOnHosts);
    my $commands = $self->createCommandsForHosts($indexes, $cmd, $cmdTmplRpl);

    my @mainmsg;
    foreach my $idx (@$indexes) {
        $session = $self->getSessionByIndex($idx);
        $hostname = $self->getHostNameByIndex($idx);
        if (!defined $session) {
            $self->PushError ("executeParallel failed: host '$hostname' is not connected");
            $rc |= 1;
            next;
        }
        $cmd = $commands->[$idx];
        $output_buffer[$idx] = '';
        if ($progress_message) {
            if ($isNotVerbose) {
                $mainmsg[$idx] = $self->AddMessage (sprintf ($progress_message ,$hostname));
            } else {
                $mainmsg[$idx] = $self->AddProgressMessage (sprintf ($progress_message ,$hostname));
            }
        }
        else{
            $mainmsg[$idx] = $self->AddMessage ("Performing '$cmd' on host '$hostname'");
        }
        my $stdin_buffer;
        if (defined $stdin_input) {
            $stdin_input->[0] .= "\n";
            $stdin_buffer = [@$stdin_input];
        }
        $rc |= $session->executeCommand($cmd, $stdin_buffer, 0);
    }

    my ($outputMsgLst, $outputHandler, $outputString);
    while (1) {
        foreach my $idx (@$indexes) {
            next if defined $rcs[$idx];
            $session = $self->getSessionByIndex($idx);
            next if (!defined $session);
            $hostname = $session->getHostname();
            if (!$session->isEmptyInputBuffer()) {
                $session->write();
            }
            if ($session->finishedExecution()) {
                $output_buffer[$idx] = $session->getOutputBuffer();
                if ($rest[$idx]){
                    $outputHandler = $self->getOutputHandler ($hostname);
                    if (defined $outputHandler) {
                        $outputHandler->addLine($rest[$idx]);
                    }
                }
                if($progress_message && !$isNotVerbose) {
                    $mainmsg[$idx]->endMessage();
                } else {
                    if ($rest[$idx]){
                        $self->AddMessage ("$hostname: $rest[$idx]");
                    }
                }
                $cmd = $commands->[$idx];

                my $signal;
                ($rcs[$idx], $signal) = $session->exitChannel();

                $msglst = new SDB::Install::MsgLst ();
                $outputMsgLst = $self->createOutputMsgLstFromCmd($cmd, $hostname, $isCollectHostInfo, $output_buffer[$idx], $rcs[$idx]);
                $self->AddSubMsgLst ($mainmsg[$idx], $outputMsgLst);

                if ($rcs[$idx] != 0 && !$doNotFailOnError) {
                    $msglst = parseHdbInstallerErrorMessages ([split("\n", $output_buffer[$idx])]);
                    if ($error_message) {
                        $self->PushError (sprintf ($error_message, $hostname), $msglst);
                    }
                    else {
                        $self->PushError ("Command '$cmd' finished on host '$hostname' with error (rc = $rcs[$idx]". ($signal ? " signal = $signal" : '') .")", $msglst);
                    }
                    $rc |= 8;
                }
                else {
                    if ($done_message) {
                        if ($isNotVerbose) {
                            $self->AddMessage (sprintf ($done_message ,$hostname), $msglst);
                        }
                        else {
                            $self->AddProgressMessage (sprintf ($done_message ,$hostname), $msglst);
                        }
                    }
                    else {
                        $self->AddMessage ("Cmd '$cmd' done on host '$hostname'", $msglst);
                    }
                }
                next;
            }
            else {
                $outputString = $session->getOutput();
                usleep (300000);
            }
            $self->addBufferToOutputHandler($outputString, \@rest, $idx, $hostname);
        }

        last if ($self->hasFinishedExecution($indexes, \@rcs));
    }
    $self->{exit_codes} = \@rcs;
    $self->{output_buffer} = \@output_buffer;
    return $rc;
}

sub filterIndexes {
    my ($self, $executeOnlyOnHosts) = @_;
    my (@result, $hostname);
    foreach my $idx (0 .. scalar(@{$self->getHostNames()} - 1)) {
        $hostname = $self->getHostNameByIndex($idx);
        if (defined $executeOnlyOnHosts) {
            push (@result, $idx) if grep {$hostname eq $_ } @$executeOnlyOnHosts;
            next;
        }
        push (@result, $idx);
    }
    return \@result;
}

sub createCommandsForHosts {
    my ($self, $indexes, $cmd, $cmdTmplRpl) = @_;
    my @commands;
    foreach my $idx (@$indexes) {
        if (defined $cmdTmplRpl) {
            $commands[$idx] = sprintf($cmd, @{$cmdTmplRpl->[$idx]});
        }
        else {
            $commands[$idx] = $cmd;
        }
    }
    return \@commands;
}

sub createOutputMsgLstFromCmd {
    my ($self, $cmd, $hostname, $isCollectHostInfo, $output_buffer, $rc) = @_;
    my $outputmsglst = new SDB::Install::MsgLst();
    $outputmsglst->addMessage("Executing command on host '$hostname':");
    my $isFirst = 1;
    foreach my $cmdItem (split(/[\s*]--/, $cmd)) {
        if ($isFirst) {
            $outputmsglst->addMessage("    $cmdItem");
            $isFirst = 0;
        }
        else {
            $outputmsglst->addMessage("        --$cmdItem");
        }
    }
    if (!$isCollectHostInfo) {
        $self->addRemoteOutputToMsglst($outputmsglst,
                                   $hostname,
                                   [split("\n", $output_buffer)]);
    }
    $outputmsglst->addMessage('');
    $outputmsglst->addMessage("Command finished on host '$hostname' with exit code $rc");
    return $outputmsglst;
}

sub addBufferToOutputHandler {
    my ($self, $buffer, $rest, $idx, $hostname) = @_;
    my $currLines = (defined $rest->[$idx])
                    ? $rest->[$idx] . $buffer : $buffer;
    my @lines = split ("\n", $currLines);
    my ($endNewLines) = ($currLines =~ /(\n+)$/);
    if ($endNewLines){
        foreach my $n (1 .. length ($endNewLines)){
            push @lines, '';
        }
    }
    if (@lines == 1){
            $rest->[$idx] = $currLines;
            return;
    }
    $rest->[$idx] = pop @lines;
    my $outputHandler = $self->getOutputHandler ($hostname);
    if (defined $outputHandler) {
        foreach my $newLine (@lines) {
            $outputHandler->addLine($newLine);
        }
    }
}

sub hasFinishedExecution {
    my ($self, $indexes, $rcs) = @_;
    foreach my $idx (@$indexes) {
        my $session = $self->getSessionByIndex($idx);
        next if !$session->hasChannel();
        if (!defined $rcs->[$idx]) {
            return 0;
        }
    }
    return 1;
}

sub getOutputBuffer{
    return $_[0]->{output_buffer};
}

sub freeOutputBuffer{
    delete $_[0]->{output_buffer};
}


#-------------------------------------------------------------------------------
# Executes the specified command serial on each remote host (see executeParallel)

sub executeSerial {
    my $rc = 0;
    my ($self,
        $cmd,                # e.g. 'hana/shared/<SID>/global/hdb/install/bin/hdbreg <parameters>'
        $force,              # do not fail on error
        $progress_message,   # e.g. "Upgrading host '%s'..."
        $done_message,       # e.g. "Upgrade host '%s' done"
        $error_message,      # e.g. "Upgrade host '%s' failed"
        $stdin_input,        # Password input stream (see executeParallel)
        $cmdTmplRpl,         # Template values (see executeParallel)
       ) = @_;

    my ($session, $hostname, $output_buffer, $msglst,
        $outputmsglst, $stdin_buffer);

    my $indexes = $self->filterIndexes();
    my $commands = $self->createCommandsForHosts($indexes, $cmd, $cmdTmplRpl);

    my $mainmsg;
    foreach my $idx (@$indexes) {
        $session = $self->getSessionByIndex($idx);
        $hostname = $self->getHostNameByIndex($idx);
        if (!defined $session) {
            $self->PushError ("executeSerial failed: host '$hostname' is not connected");
            $rc |= 1;
            next;
        }
        $cmd = $commands->[$idx];
        if ($progress_message) {
            $mainmsg = $self->AddProgressMessage (sprintf ($progress_message, $hostname));
        }
        else {
            $mainmsg = $self->AddMessage ("Performing '$cmd' on host '$hostname'");
        }
        if (defined $stdin_input) {
            $stdin_input->[0] .= "\n";
            $stdin_buffer = [@$stdin_input];
        }
        $rc |= $session->executeCommand($cmd, $stdin_buffer, 0);
        while (1) {
            if (!$session->isEmptyInputBuffer()) {
                $session->write();
            }
            if ($session->finishedExecution()) {
                $mainmsg->endMessage();
                my ($rc, $signal) = $session->exitChannel();
                $msglst = new SDB::Install::MsgLst ();
                $outputmsglst = new SDB::Install::MsgLst ();
                $outputmsglst->addMessage("Running command '$cmd'");
                $self->addRemoteOutputToMsglst($outputmsglst,
                                               $hostname,
                                               [split("\n", $output_buffer)]);
                $outputmsglst->addMessage('');
                $outputmsglst->addMessage("Command finished on host '$hostname' with exit code $rc");
                $self->AddSubMsgLst ($mainmsg, $outputmsglst);
                if ($rc != 0) {
                    if ($error_message) {
                        $self->PushError (sprintf ($error_message, $hostname), $msglst);
                    }
                    else {
                        $self->PushError ("Command '$cmd' finished on host '$hostname' with error (rc = $rc". ($signal ? " signal = $signal" : '') .")",$msglst);
                    }
                    $rc |= 8;
                    if (!$force){
                        return $rc;
                    }
                }
                else {
                    if ($done_message) {
                        $self->AddProgressMessage (sprintf ($done_message, $hostname), $msglst);
                    }
                    else {
                        $self->AddMessage ("Cmd '$cmd' done on host '$hostname'", $msglst);
                    }
                }
                last;
            }
            else {
                $output_buffer .= $session->getOutput();
                usleep(300000);
            }
        }
    }
}


#-------------------------------------------------------------------------------
# Writes a header and each output line to the specified message list.

sub addRemoteOutputToMsglst {
    my ($self,
        $outputMsgLst,
        $host,
        $outputLines, # reference to an array of output line
       ) = @_;

    $outputMsgLst->addMessage('');
    $outputMsgLst->addMessage(" ----- Output from host '$host' -----");
    $outputMsgLst->addMessage('|');
    my $isFirst = 1;

    foreach my $currLine (@$outputLines) {
        if (!$isFirst || ((length($currLine) > 0) && ($currLine !~ /^\s*$/))) {
            $outputMsgLst->addMessage('|  ' . $currLine);
            $isFirst = 0;
        }
    }
}

sub copyTree{
    my ($self,$localSrcDir, $remoteDestDir, $items) = @_;
    my ($lPath, $rPath);
    my $rc = 0;
    foreach my $item (@$items){
        $lPath = File::Spec->catfile($localSrcDir, $item);
        $rPath = File::Spec->catfile($remoteDestDir, $item);
        my @statbuf = stat ($lPath);
        if (! -e _){
            $self->PushError ("local path '$lPath' doesn't exist");
            $rc ||= 1;
            next;
        }
        if (-f _){
            foreach my $host (@{$self->getHostNames()}){
                my $session = $self->getSession($host);
                $rc ||= $session->copyFile($lPath, $rPath, \@statbuf);
            }
        }
        elsif (-d _){
            if (!opendir (DH, $lPath)){
                $self->PushError ("Cannot open local directory '$lPath': $!");
                $rc ||= 1;
                next;
            }
            my @subitems = grep {!/^\.{1,2}$/} readdir (DH);
            closedir (DH);
            foreach my $host (@{$self->getHostNames()}) {
                my $session = $self->getSession($host);
                if (!defined $session->stat($rPath)) {
                    if (!$session->mkdir($rPath, $statbuf[2])) {
                            my $error = $session->sftpError();
                            $self->PushError("Cannot create remote directotry '$rPath': $error");
                    }
                }
            }
            $rc ||= $self->copyTree ($lPath, $remoteDestDir . $path_separator . $item, \@subitems);
        }
    }
    return $rc;
}

sub deltree{
    my ($self,$remotePath, $errlst) = @_;
    my $rc = 0;
    if (!defined $errlst) {
        $errlst = $self;
    }
    my ($hosterrlst, $lrc);
    foreach my $host (@{$self->getHostNames()}) {
        my $session = $self->getSession($host);
        if (!defined $session) {
            $errlst->PushError ("Host '$host' is not connected");
            next;
        }
        $hosterrlst = new SDB::Install::MsgLst ();
        $lrc = $session->deltree ($remotePath, $hosterrlst);
        if ($lrc != 0){
            $errlst->PushError ("Cannot remove $remotePath on host '$host'", $hosterrlst);
            $rc ||= $lrc;
        }
    }
    return $rc;
}

sub mkdir{
    my ($self,$rPath, $mode) = @_;
    my $rc = 0;
    my ($session, $code, $str, $dsc);
    foreach my $host (@{$self->getHostNames()}) {
        my $session = $self->getSession($host);
        if (!defined $session) {
            $self->PushError ("Host '$host' is not connected");
            next;
        }
        if (!$session->mkdir($rPath, $mode)) {
            my $error = $session->sftpError();
            $self->PushError("Cannot create remote directotry '$rPath' on host '$host': $error");
            $rc ||= 1;
            next;
        }
    }
    return $rc;
}

sub existsFile{
    my ($self,$rPath, $isDirectory, $outStatError, $hosts) = @_;
    my $rc = 1;

    $self->ResetError();
    if (!defined $hosts){
        $hosts = $self->{_hosts};
    }

    my $session;
    foreach my $host (@$hosts){
        $session = $self->getSession($host);
        if (!defined $session){
            $self->appendErrorMessage("host '$host' is not handled in this RemoteHosts instance");
            $rc &&= 0;
            next;
        }
        $rc &&= $session->existsFile($rPath, $isDirectory, $self, $outStatError);
    }
    return $rc;
}

sub resetMsgLstContext{
    my ($self) = @_;
    $self->SUPER::resetMsgLstContext();
    my $session;
    foreach my $i (0 .. (scalar @{$self->{_hosts}} -1)) {
        $session = $self->getSessionByIndex($i);
        if (defined $session) {
            $session->setMsgLstContext($self->getMsgLstContext());
        }
    }
}

sub getHostNameByIndex{
    return $_[0]->{_hosts}->[$_[1]];
}

sub getIndexByHostName{
    my $index;
    my $hosts = $_[0]->{_hosts};
    foreach my $i (0 .. (scalar @{$hosts} -1)){
        if ($hosts->[$i] eq $_[1]){
            $index = $i;
            last;
        }
    }
    return $index;
}

sub getExitCode{
    my ($self, $index) = @_;
    if (!defined $self->{exit_codes}){
        return undef;
    }
    return $self->{exit_codes}->[$index];
}


sub getHostNames{
    return $_[0]->{_hosts};
}

sub getUserName{
    return $_[0]->{_user};
}


#-------------------------------------------------------------------------------
# Copys connections from the class $allHosts to this class.
#
# Returns 0 in case of host not found in $allHosts.

sub copyConnections {

    my ($self,
        $allHosts,  # SDB::Install::RemoteHosts
        $msglst     # for writing error message
        ) = @_;

    if (!defined $self->{_hosts} || ($self->getNumberOfHosts() == 0)) {
        return 1;
    }

    my %idxMapAllHosts = ();

    my $i = 0;
    foreach my $curr (@{$allHosts->{_hosts}}) {
        $idxMapAllHosts{$curr} = $i;
        $i++;
    }

    my $rc          = 1;
    my @sessions = ();

    foreach my $currHost (@{$self->{_hosts}}) {

        my $allHostIdx = $idxMapAllHosts{$currHost};

        if (defined $allHostIdx) {
            push @sessions, $allHosts->{_sessions}->[$allHostIdx];
        }
        else {
            push @sessions, undef;
            $msglst->PushError
                    ("Host '$currHost' is not part of the $gProductNameSystem.");
            $rc = 0;
        }
    }

    $self->{_sessions} = \@sessions;

    return $rc;
}


#-------------------------------------------------------------------------------
# removes hosts to the internal host array.

sub removeRemoteHosts {

    my ($self,
        $removeHosts # reference to an array containing hosts to be removed
        ) = @_;

    my @newHosts       = ();
    my @newSessions = ();
    my %removeMap      = ();

    foreach my $curr (@$removeHosts) {
        $removeMap{$curr} = 1;
    }

    my $i = 0;
    foreach my $currHost (@{$self->{_hosts}}) {

        if (!exists $removeMap{$currHost}) {
            push @newHosts,       $currHost;
            push @newSessions, $self->{_sessions}->[$i];
        }
        $i++;
    }
    $self->{_hosts}       = \@newHosts;
    $self->{_sessions} = \@newSessions;
}


#-------------------------------------------------------------------------------

sub getOutputHandler {
    my ($self, $host) = @_;
    if (!defined $self->{_outputHandler}){
        return undef;
    }
    return $self->{_outputHandler}->{$host};
}

#-------------------------------------------------------------------------------
sub setOutputHandler {
    my ($self, $host, $outputHandler) = @_;
    if (!defined $self->{_outputHandler}){
        $self->{_outputHandler} = {};
    }
    $self->{_outputHandler}->{$host} = $outputHandler;
    return 1;
}


#-------------------------------------------------------------------------------
sub destroy{
    my $self = shift;
    if (defined $self->{_sessions}) {
        foreach my $session (@{$self->{_sessions}}) {
            if (defined $session) {
                $session->disconnect();
            }
        }
        $self->{_sessions} = undef;
    }
}


#-------------------------------------------------------------------------------

sub isHostctrl {
    return 0;
}

1;
