package SDB::Install::SAPControl::ProcessCollection;

use strict;
use parent qw(SDB::Install::Base Exporter);
use SDB::Install::SAPControl::Process;

use constant {
    PROC_NAME_DAEMON    => 'hdbdaemon',
};

use constant {
    PROC_ACTION_START   => 'Start',
    PROC_ACTION_STOP    => 'Stop',
};

our @EXPORT = qw (PROC_NAME_DAEMON);

sub new {
    my ($class) = @_;
    return $class->SUPER::new();
}

sub getSubset {
    my ($self, $services) = @_;
    my $subset = __PACKAGE__->new();
    for my $service (@{$services // []}) {
        my $processObject = $self->getProcessByName($service);
        $subset->addProcess($processObject) if defined($processObject);
    }
    return $subset;
}

sub addProcess {
    my ($self, $process) = @_;
    $self->{processes} //= [];
    push @{$self->{processes}}, $process;
}

sub getAll {
    my ($self) = @_;
    my @sortedProcesses = sort { $a->getName() <=> $b->getName() } @{$self->{processes} // []};
    return \@sortedProcesses
}

sub getProcessByName {
    my ($self, $processName) = @_;
    return undef if !$processName;

    for my $process (@{$self->getAll()}) {
        return $process if $process->getName() =~ /$processName/;
    }
    return undef;
}

sub isProcessRunning {
    my ($self, $processName) = @_;
    my $process = $self->getProcessByName($processName);
    return $process ? $process->isRunning() : undef;
}

sub isProcessStopped {
    my ($self, $processName) = @_;
    my $process = $self->getProcessByName($processName);
    return $process ? $process->isStopped() : undef;
}

sub _getPendingProcessesByAction {
    my ($self, $action) = @_;
    my @pendingProcesses;
    if ($action eq PROC_ACTION_START) {
        @pendingProcesses = grep { !$_->isRunning() } @{$self->getAll()};
    } else {
        @pendingProcesses = grep { !$_->isStopped() } @{$self->getAll()};
    }
    return \@pendingProcesses;
}

sub getStoppedProcesses {
    my ($self) = @_;
    my @stoppedProcesses = grep { $_->isStopped() } @{$self->getAll()};
    return \@stoppedProcesses;
}

sub getStartPendingProcesses {
    my ($self) = @_;
    return $self->_getPendingProcessesByAction(PROC_ACTION_START);
}

sub getStopPendingProcesses {
    my ($self) = @_;
    return $self->_getPendingProcessesByAction(PROC_ACTION_STOP);
}

sub getRunningProcesses {
    my ($self) = @_;
    my @runningProcesses = grep { $_->isRunning() } @{$self->getAll()};
    return \@runningProcesses;
}

sub hasStoppedProcesses {
    my ($self) = @_;
    my $stoppedProcesses = $self->getStoppedProcesses();
    return @{$stoppedProcesses} > 0;
}

sub isHDBDaemonStopped {
    my ($self) = @_;
    return $self->isProcessStopped(PROC_NAME_DAEMON);
}

sub getProcessHDBDaemon {
    my ($self) = @_;
    return $self->getProcessByName(PROC_NAME_DAEMON);
}

sub asString {
    my ($self) = @_;
    my @processStrings = map { $_->asString() } @{$self->getAll()};
    return join "\n", @processStrings;
}

sub isEmpty {
    my ($self) = @_;
    return @{$self->getAll()} == 0;
}

1;