package SDB::Install::SAPSystemContainerHelper::CheckUpdateContainerHelper;

use strict;
use SDB::Install::Globals qw ($gProductNameEngine);
use SDB::Install::SAPSystemUtilities;

our @ISA = qw (SDB::Install::Base SDB::Install::SAPSystemUtilities);


#-------------------------------------------------------------------------------
# checks whether a hana component could be updated from version A to version B,
# when both versions are given as component manifest files.
# when plugin manifests are given, checks each of them against the target manifest for compatibility.
#
# Parameters: $instconfig  SDB::Install::Configuration::ContainerHelper
#
# Returns 1 if update possible and plugins are compatible, 0 otherwise, or undef if an internal error occurred.
#-------------------------------------------------------------------------------
# Synopsis of the app, in compatible extension to TIPHANA15IT10-2587 :
# 
# path/to/hdbinst --main SDB::Install::App::Console::HdbContainer::main
#     --action=check_update
#   [ --src_manifest=<src manifest> ] 
#     --tgt_manifest=<target manifest>
#   [ --plugin_manifests=<manifest1,manifest2,...> ]
#  
# Manifest paths are either absolute, or relative to cwd.
#
# With --tgt_manifest, at least one of the other optional manifest parameters must be present.
#
# All checks that can be performed with the given set of manifests, will be performed:
#  - server upgrade check: Src and tgt manifest are checked for upgradeability.
#  - plugin compatibility: Each plugin manifest is checked against the tgt manifest.
#
# Returncode of hdbinst is 0 iff all checks given by the manifest set passed and no error occurred.
#-------------------------------------------------------------------------------

sub performContainerAction {
    my ($self, $instconfig, undef, undef, $app) = @_;
    # basic checks whether parameters are provided:
    my $srcManifestPath = $instconfig->getValue('SrcManifest');
    my $tgtManifestPath = $instconfig->getValue('TgtManifest');
    my $pluginManifestPathsCSVString = $instconfig->getValue('PluginManifests');
    my $doUpdateCheck = 0;
    my $doPluginChecks = 0;
    if(!defined $tgtManifestPath) {
        my $errstr = "Target manifest path not provided.";
        $self->appendErrorMessage($errstr);
        return undef;
    }
    $tgtManifestPath = $app->getAbsPath($tgtManifestPath);
    $self->getMsgLst()->addProgressMessage("Upgrade target release given by manifest '$tgtManifestPath'.");
    if(defined $srcManifestPath) {
        $doUpdateCheck = 1;
    }
    if(defined $pluginManifestPathsCSVString) {
        $doPluginChecks = 1;
    }
    if(not ($doUpdateCheck || $doPluginChecks)) {
        my $errstr = "At least one of source manifest or plugin manifests must be provided.";
        $self->appendErrorMessage($errstr);
        return undef;
    }
    # checks whether manifest paths represent readable manifests of appropriate type.
    # we check all given paths to create a comprehensive summary at once instead of
    # bailing out at first error, since the app will be used in some automation:
    my $errcon = 0;
    my $tgtManifest = $self->_readManifest($tgtManifestPath, 'target');
    if(!defined $tgtManifest) {
        $errcon = 1;
    }
    elsif($doPluginChecks && !$tgtManifest->isServer()) {
        my $errstr = "'$tgtManifestPath' is not a server manifest.";
        $self->appendErrorMessage($errstr);
        $errcon = 1;
    }
    my $srcManifest = undef;
    if($doUpdateCheck) {
        $srcManifestPath = $app->getAbsPath($srcManifestPath);
        $self->getMsgLst()->addProgressMessage("Upgrade source release given by manifest '$srcManifestPath'.");
        $srcManifest = $self->_readManifest($srcManifestPath, 'source');
        if(!defined $srcManifest) {
            $errcon = 1;
        }
        else {
	        my $srcComponent = $srcManifest->getKeyname();
	        my $tgtComponent = $tgtManifest->getKeyname();
	        if(!($srcComponent eq $tgtComponent)) {
	            my $errstr = "Manifest components are not compatible: '$srcComponent', '$tgtComponent'";
	            $self->appendErrorMessage($errstr);
	            $errcon = 1;
	        }
        }
    }
    my %pluginManifestsByPath = ();
    if($doPluginChecks) {
	    my @pluginManifestPaths = split(/\s*,\s*/, $pluginManifestPathsCSVString);
	    for my $pluginManifestPath (@pluginManifestPaths) {
	        $pluginManifestPath = $app->getAbsPath($pluginManifestPath);
	        my $manifest = $self->_readManifest($pluginManifestPath, 'plugin');
	        if(!defined $manifest) {
	            $errcon = 1;
	            next;
	        }
		    if(!$manifest->isServerPlugin()) {
		        my $errstr = "'$pluginManifestPath' is not a server plugin manifest.";
		        $self->appendErrorMessage($errstr);
		        $errcon = 1;
                next;
		    }
	        $pluginManifestsByPath{$pluginManifestPath} = $manifest;
	    }
    }
    if($errcon) {
        return undef;
    }
    # the actual checks:
    if($doUpdateCheck) {
        my $msgList = new SDB::Install::MsgLst();
        $instconfig->setMsgLstContext([$msgList]);
	    my $retcode = $instconfig->canUpgradeComponent($srcManifest, $tgtManifest);
	    $self->getMsgLst()->addProgressMessage(${$msgList->getMsgLstString()});
	    if (!$retcode) {
	        $self->appendErrorMessage("Checking the update path failed.", $instconfig->getErrMsgLst());
	        return $retcode;
	    }
    }
    my $nocompat = 0;
    if($doPluginChecks) {
        for my $pluginManifestPath (keys %pluginManifestsByPath) {
            my $pluginManifest = $pluginManifestsByPath{$pluginManifestPath};
            my $retcode = $pluginManifest->checkComponentDependency($tgtManifest, $gProductNameEngine);
            if(!$retcode) {
                $self->appendErrorMessage("Checking plugin compatibility failed for manifest '$pluginManifestPath'.",
                                          $pluginManifest->getErrMsgLst());
                $nocompat = 1;
                next;
            }
            else {
                $self->getMsgLst()->addProgressMessage("plugin by manifest '$pluginManifestPath' is compatible with target release.");
            }
        }
    }
    if($nocompat) {
        return 0;
    }
    return 1;
}

sub _readManifest {
    my ($self, $absoluteManifestPath, $manifestCaption) = @_;
    my $manifest = new SDB::Install::Manifest();
    $manifest->read($absoluteManifestPath);
    if(!defined $manifest || $manifest->errorState()) {
        my $mferrstr = (defined $manifest) ? ": ".$manifest->getErrorString() : ".";
        my $errstr = "Could not open $manifestCaption manifest".$mferrstr;
        $self->appendErrorMessage($errstr);
        return undef;
    }
    my $component = $manifest->getKeyname();
    if(!$manifest->isKnownComponent()) {
        my $errstr = "$manifestCaption manifest '$absoluteManifestPath': unknown component '$component'";
        $self->appendErrorMessage($errstr);
        return undef;
    }
    return $manifest;
}

1;
