package SDB::Install::SAPSystemUtilities;
use parent SDB::Install::Base;

use strict;
use SDB::Install::MsgLst;
use SDB::Install::NewDBUser;
use SDB::Install::Saphostagent qw(installSaphostagent);
use SDB::Install::SAPInstance::TrexInstance;
use SDB::Install::SAPSystem;
use SDB::Install::System  qw (deltree);
use SDB::Install::Globals qw ($gHostRoleWorker
                              $gLogDir
                              $gOperationRegisterHost
                              $gProductName
                              $gProductNameEngine
                              $gProductNameSystem);
use SDB::Install::DirectoryWalker;
use SDB::Install::SysVars qw($isWin $path_separator);
use SDB::Install::Recovery;


#-------------------------------------------------------------------------------
# Constructor

sub new {
    my $class = shift;
    my $self  = {};
    bless ($self, $class);
    return $self;
}


#-------------------------------------------------------------------------------
# Adapts the operating system configuration and tries to install SAPHostagent
# if the parameter 'HostagentPassword' is defined.
#
# Parameter: $sapSys      SDB::Install::SAPSystem
#            $instconfig  SDB::Install::Configuration
#
# Returns int retCode

sub adaptOperatingSystemConfig {

    my ($self, $sapSys, $instconfig) = @_;

    my $msg = $sapSys->AddProgressMessage ("Adapting system configuration...");
    my $saveSapSysMsgLstCtxt = $sapSys->setMsgLstContext([$msg->getSubMsgLst()]);

    if (!$sapSys->adaptSystemConfiguration(undef, $instconfig)) {
        $sapSys->setMsgLstContext ($saveSapSysMsgLstCtxt);
        $sapSys->AddError ('Cannot adapt system configuration', $sapSys);
        return undef;
    }
    $sapSys->setMsgLstContext ($saveSapSysMsgLstCtxt);

    my $installParam = $instconfig->getValue('InstallHostagent');

    if (!defined $installParam || $installParam) {
        # skip if option '--install_hostagent=off' is set

        if (!installSaphostagent($sapSys->get_globalSidDir(),
                                 $instconfig->getValue('HostagentPassword'),
                                 $sapSys->getMsgLst(),
                                 $instconfig->getValue('HostagentUserName'),
                                 1, # isProgressMsg
                                 $gLogDir)) { # the cwd to restore to (after a temp. chdir)

            $sapSys->AddError ('Cannot install SAP Host Agent', $sapSys);
            return undef;
         }
    }
    else {
        $sapSys->AddMessage ('Installation/update of SAP Host Agent disabled');
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Changes the password of the system user of the database

sub changeSystemUserPassword {

    my ($self,
        $sapSys,     # SDB::Install::SAPSystem
        $instance,   # SDB::Install::SAPInstance::TrexInstance
        $instconfig, # SDB::Install::Configuration
       ) = @_;

    my $msg = $sapSys->AddProgressMessage ("Setting database user SYSTEM password...");
    $instance->setMsgLstContext ([$msg->getSubMsgLst ()]);
    if (!defined $instance->setEnginePassword ($instconfig->getValue ('TrgSQLSysPasswd'),
                                               $instconfig->getValue ('SrcSQLSysPasswd'),
                                               $instconfig->getValue ('SystemUser'))){
        $sapSys->AddError (undef, $instance);
        return undef;
    }
    return 1;
 }


#-------------------------------------------------------------------------------
# Deletes or unlinks the specified directory.
#
# Parameter: SDB::Install::MsgLst $msgList
#            string $dir
#
# Returns int retCode

sub deleteOrUnlinkDirectory {

    my ($self, $msgList, $errLst, $dir) = @_;

    my $rc  = 1;

    if (-l $dir) {

        $rc = unlink($dir);

        if ($rc) {
            $msgList->addMessage ("Link '$dir' removed");
        }
        else {
            my $msg = $msgList->addError ("Cannot unlink '$dir': $!");
            if (defined $errLst){
                $errLst->appendMsg ($msg);
            }
        }
    }
    elsif (-e $dir) {

        my $msg     = $msgList->addProgressMessage("Deleting directory '$dir'...");
        my $subErrList = new SDB::Install::MsgLst ();

        $rc = deltree ($dir, $subErrList);

        if ($rc) {
            $msg->getSubMsgLst ()->addMessage ("Directory '$dir' deleted");
        }
        else {  
            my $errmsg = $msg->getSubMsgLst ()->addError ("Cannot remove directory '$dir'", $subErrList);
            if (defined $errLst){
                $errLst->appendMsg ($errmsg);
            }
        }
    }
    else {
        $msgList->addMessage ("Directory '$dir' is already removed");
    }

    return $rc;
}


#-------------------------------------------------------------------------------
# Removes the "/usr/sap/<SID>" entry, if it is a symlink,
# or cleans out every known installation related entry, if not:
# in this case all symlinks are removed, and all empty directories 
# are deleted. user provided (actual) files are kept.
#
# Parameter: SDB::Install::MsgLst $msgList : for log messages
#            SDB::Install::MsgLst $errList  : for errors
#            string $sid : the SID
#
# Returns int retCode, 1 for success, 0 or undef for failure.

sub cleanupUsrSapSidDirectory {
    my (
        $self,
        $msgList,
        $errLst,
        $sid,
        $keepUserHomeDir,
        $homeDir
    ) = @_;
    my $rc  = 1;
    my $dir = '/usr/sap/'.$sid;
    if (-l $dir) {
        $rc = unlink($dir);
        if ($rc) {
            $msgList->addMessage ("Link '$dir' removed");
        }
        else {
            my $msg = $msgList->addError ("Cannot unlink '$dir': $!");
            if (defined $errLst){
                $errLst->appendMsg ($msg);
            }
        }
    }
    elsif (-e $dir) {
        my $msg     = $msgList->addProgressMessage("Cleaning up  directory '$dir'...");
        my $subErrList = new SDB::Install::MsgLst ();
        # we delete all symlinks and empty directories (depth-first):
        my $dirWalker = SDB::Install::DirectoryWalker->new(undef, undef, undef,
                                                           sub { # pruneMatcher
                                                               if ($keepUserHomeDir) {
                                                                  my $fullpath = $_[3].$path_separator.$_[4];
                                                                  return 1 if ($fullpath=~/^$homeDir/);
                                                               }
                                                               return 0;
                                                           },
                                                           undef, undef,
                                                           sub { # action 
                                                               my $fullpath = $_[3].$path_separator.$_[4];
                                                               my $isDir = $_[6];
                                                               my $isSymlink = $_[8];
                                                               if($isDir && !$isSymlink) {
                                                                    if(!rmdir($fullpath)) {
                                                                       $subErrList->addError ("Cannot delete directory $fullpath: $!");
                                                                    }
                                                               }
                                                               elsif($isSymlink) {
                                                                   if(!unlink($fullpath)) {
                                                                       $subErrList->addError ("Cannot delete symlink $fullpath: $!");
                                                                       return undef;
                                                                   }
                                                               }
                                                               return 1;
                                                            }, undef, undef,
                                                            0,    # depth-first
                                                            1,    # dont collect list
                                                            0     # dont follow symlinks
                                                           );
        my $rc = $dirWalker->findAndProcess($dir);
        if(not defined $rc) {
            $subErrList->appendMsgLst ($dirWalker->getErrMsgLst ());
        }
        elsif (!$keepUserHomeDir) {
            my $rc = rmdir($dir);
            if(!$rc) {
                $subErrList->addError ("Cannot delete directory '$dir': $!");
            }
        }
        if ($rc) {
            my $warningsList = new SDB::Install::MsgLst();
            for my $messageObject (@{$subErrList->getMessages()}){
                $warningsList->addWarning($messageObject->getRawText());
            }
            $msg->getSubMsgLst ()->addMessage ("Directory '$dir' cleaned.", $warningsList);
        }
        else {  
            my $errmsg = $msg->getSubMsgLst ()->addError ("Cannot clean up directory '$dir'.", $subErrList);
            if (defined $errLst){
                $errLst->appendMsg ($errmsg);
            }
        }
    }
    else {
        $msgList->addMessage ("Directory '$dir' is already removed");
    }
    return $rc;
}

#-------------------------------------------------------------------------------
# Creates the system administrator user.
#
# Returns (int retCode, int configureUser [1 user created, 2 configure])

sub tryCreateSysAdminUser {

    my ($self,
        $instconfig,    # SBD::Install::Configuration::AnyRegisterRename
        $sapSys,        # SDB::Install::SAPSystem
        $password
        ) = @_;

    my $rc = 1;

    my $mdcUsers = $instconfig->getTenantDatabaseOSUsers ();

    if ($sapSys->getUser()->exists()) {

        my $configureUser = 0;
        my $homeDir = ($isWin) ? undef : $sapSys->getUser()->getSidAdmHome();

        if (defined $homeDir && !-d $homeDir) {
            # create home directory with SDB::Install::NewDBUser::configureHome
            $configureUser = 2;
        }
        return ($rc, $configureUser);
    }

    my $userName = $sapSys->getUser()->getSidAdmName();
    my $msgUser  = $sapSys->AddProgressMessage
                          ("Creating system administrator user '$userName'...");

    if (!$instconfig->{isSlave} && !defined $instconfig->{existingSidAdm}) {
        $sapSys->getUser()->setMsgLstContext ([$msgUser->getSubMsgLst ()]);
        $rc = $sapSys->getUser()->create (undef, # no template directory
                                                 # --> call configureHome later
                                    "$gProductNameEngine System Administrator",
                                    $instconfig->getValue('UID'),
                                    $password,
                                    $instconfig->getValue('GID'),
                                    $instconfig->getValue('HomeDir'),
                                    $instconfig->getValue('Shell'),
                                    $mdcUsers);

        if (!defined $rc) {
            $sapSys->AddError ("Cannot create user '$userName'",
                               $sapSys->getUser());
        }
    }
    else {
        $rc = $sapSys->createuserfromconfig ($password, $mdcUsers);
        if (!$rc) {
            $sapSys->AddError ("Cannot create user '$userName' from configuration", $sapSys);
        }
        else {
            $rc = $sapSys->tryChangeNfsAccess($instconfig->getOwnInstance());
        }
    }
    $msgUser->endMessage(0);
    return ($rc, 1); # retcode, created
}


#-------------------------------------------------------------------------------
# Deletes the system administrator user
#
# Returns int retCode

sub deleteSysAdminUser {

    my ($self,
        $msgList,         # SDB::Install::MsgLst
        $sid,             # string
        $keepUserHomeDir, # bool
        $keepGroup,       # bool
        $strUserKind      # string (e.g. 'origin', 'source' or undef)
        ) = @_;

    my $user     = new SDB::Install::NewDBUser($sid);
    my $userName = $user->getSidAdmName();
    my $userInfo = "system administrator user '$userName'";

    if (defined $strUserKind) {
        $userInfo = $strUserKind . ' ' . $userInfo;
    }

    my $rc  = 1;
    my $msg = $msgList->addProgressMessage ("Deleting $userInfo...");
    $user->setMsgLstContext ([$msg->getSubMsgLst ()]);

    if ($user->exists()) {

        if (!defined $user->delete(1, $keepUserHomeDir, $keepGroup)) {
            $msgList->addError ("Cannot delete $userInfo", $user->getErrMsgLst());
            $rc = undef;
        }
    }
    else{
        $msg->getSubMsgLst()->addMessage ("$userInfo is already deleted");
    }

    return $rc;
}


#-------------------------------------------------------------------------------
# Removes SAPServices entry.
# In case of rename, old and new SIDs can be specified.
#
# Parameter: SDB::Install::SAPSystem                 $sapSys
#            SDB::Install::SAPInstance::TrexInstance $instance
#            string $oldSid   # may be omitted
#            string $newSid   # may be omitted
#
# Returns int retCode

sub cleanupSAPServicesEntry {

    my ($self, $sapSys, $instance, $oldSid, $newSid) = @_;

    my $msg = $sapSys->AddProgressMessage ('Removing sapservices entry...');

    my $profilePattern = $instance->get_profilePattern($oldSid);

    if (defined $newSid && ($newSid ne $oldSid)) {
        $profilePattern .= '|'. $instance->get_profilePattern($newSid);
    }
    $instance->resetMsgLstContext();
    my $rc = $instance->removeSapservicesEntry(undef, $profilePattern);
    $sapSys->AddSubMsgLst ($msg, $instance);

    return $rc;
}


sub _revertToSnapshot{
    my ($self,$instance,$instconfig,$msglst,$errlst) = @_;
    my $recovery = new SDB::Install::Recovery();
    $recovery->setMsgLstContext([$msglst,$errlst]);
    return $recovery->recoverSnapshot($instance,$instconfig);
}

sub _revertTenantsToSnapshot{
    my ($self,$instance,$instconfig,$msglst,$errlst) = @_;
    my $recovery = new SDB::Install::Recovery();
    $recovery->setMsgLstContext([$msglst,$errlst]);
    return $recovery->recoverSnapshotTenants($instance,$instconfig,$msglst);
}


sub _isWorkerHost{
    my ($self,$hdbInstance) = @_;
    my $hostRoles = $hdbInstance->getHostRolesByIniFile();
    if (!defined $hostRoles){
        $self->setErrorMessage("Cannot get host roles", $hdbInstance->getErrMsgLst);
        return undef;
    }
    if ((grep {$_ eq $gHostRoleWorker} @$hostRoles)){
        return 1;
    }
    return 0;
}

#-------------------------------------------------------------------------------
# Starts the local instance
#
# Parameter: $sapSys   SDB::Install::SAPSystem
#            $instance SDB::Install::SAPInstance::TrexInstance
#            $sid      string
#            $password string
#            $timeout  number from $instconfig->getTimeout('start_instance')

#
# Returns int retCode

sub startLocalInstance {

    my ($self, $sapSys, $instance, $sid, $passwordForWindows, $instconfig) = @_;

    my $localHost    = $instance->get_host();
    my $nr           = $instance->get_nr();
    my $instanceInfo = "instance $sid (HDB$nr) on host '$localHost'";
    my $msg          = $sapSys->AddProgressMessage("Starting $instanceInfo...");

    if($instconfig->getValue('RevertToSnapshot') && !$instconfig->{isSlave}){
        if(!defined $self->_revertToSnapshot($instance,$instconfig,$msg->getSubMsgLst(),$sapSys->getErrMsgLst())){
            return undef;
        }
        return 1;
    }

    $instance->setMsgLstContext([$msg->getSubMsgLst()]);

    if (!defined $instance->startHost($sapSys->getUser()->getSidAdmName(),
                                      $passwordForWindows,
                                      undef, # localhost
                                      $instconfig->getTimeout('start_instance'),
                                      $instconfig->isUseHttps())) {
        $sapSys->AddError ("Cannot start $instanceInfo", $instance);
        return undef;
    }

    if($instconfig->getValue('RevertToSnapshot') && $instconfig->{isSlave} && $self->_isWorkerHost($instance)){
        if(!defined $self->_revertTenantsToSnapshot($instance,$instconfig,$msg->getSubMsgLst(),$sapSys->getErrMsgLst())){
            return undef;
        }
    }

    $msg->endMessage(0);

    return 1;
}

#-------------------------------------------------------------------------------
# Starts sapstartsrv
#
# Returns int retCode

sub startSapstartsrv {

    my ($self, $sapSys, $instance, $instconfig) = @_;

    my $msg = $sapSys->AddProgressMessage("Starting service (sapstartsrv)...");

    $instance->setMsgLstContext([$msg->getSubMsgLst ()]);

    my $rc = 1;

    if (!defined $instance->startLocalSapStartSrv(undef,
                                   $instconfig->getTimeout ('start_service'),
                                   $instconfig->isUseHttps())) {
        $sapSys->AddError ("Cannot start service (sapstartsrv)", $instance);
        $rc = undef;
    }
    $msg->endMessage(0);
    return $rc;
}


#-------------------------------------------------------------------------------
# Sends 'hdbreg --start/stop_only=instance' to remote hosts.
#
# Returns int retCode

sub startStopRemoteHosts {

    my ($self,
        $instconfig,        # SDB::Install::Configuration::AnyConfig
        $sapSys,            # SDB::Install::SAPSystem
        $startStopParamID,  # 'StartOnly' or 'StopOnly'
        $actionProgressive, # string (e.g. 'Starting')
        $actionDone,        # string (e.g. 'Started')
        $theseHostsOnly,    # array containing subset host names to start/stop
       ) = @_;

    my $savedScope = $instconfig->{params}->{$startStopParamID}->{value};
    $instconfig->{params}->{$startStopParamID}->{value} = 'instance';
    $instconfig->setMsgLstContext ([$sapSys->getMsgLst ()]);

    my $rc = $self->tryRegisterRemoteHosts($instconfig,
                                           $actionProgressive,
                                           $actionDone,
                                           $gOperationRegisterHost,
                                           [$startStopParamID, 'NoStart'],
                                           undef,
                                           $theseHostsOnly);
    if (!$rc) {
        $sapSys->AddError(undef, $instconfig);
    }
    $instconfig->{params}->{$startStopParamID}->{value} = $savedScope;

    return $rc;
}


#-------------------------------------------------------------------------------
# Starts SAP HANA system
#
# Returns int retCode

sub startSystem {

    my ($self,
        $sapSys,     # SDB::Install::SAPSystem
        $instconfig, # e.g. SDB::Install::Configuration::HdbReg
        $password,   # system administrator password
        $action      # e.g. 'Register'
       ) = @_;

    my $instance         = $instconfig->getOwnInstance();
    my $outstandingHosts = $instconfig->getOutstandingHosts();
    my $systemInfo       = 'system ' . $instance->get_sid()
                           .  sprintf(' (HDB%02d',$instance->get_nr()) . ')';

    if (defined $outstandingHosts) {
        $sapSys->AddProgressMessage("$action $outstandingHosts");
        $sapSys->AddProgressMessage("Start $gProductName $systemInfo manually.");
        return 1;
    }

    my $msg = $sapSys->AddProgressMessage("Starting $systemInfo...");
    $instance->setMsgLstContext([$msg->getSubMsgLst()]);


    if($instconfig->getValue('RevertToSnapshot')){
        if(!defined $self->_revertToSnapshot($instance,$instconfig,$msg->getSubMsgLst(),$sapSys->getErrMsgLst())){
            return undef;
        }
        return 1;
    }

    my $pHandler = $sapSys->getMsgLst()->getProgressHandler();
    $instance->getMsgLst()->setProgressHandler($pHandler);
    my $rc = defined $instance->startAllHosts
                                   ($sapSys->getUser()->getSidAdmName(),
                                    $password,
                                    $instconfig->getTimeout ('start_instance'),
                                    $instconfig->isUseHttps(),
                                    $instconfig->getValue ('SSOCertificate'));
    if (!$rc) {
        $sapSys->setErrorMessage("Cannot start $systemInfo",
                                 $instance->getErrMsgLst());
    }
    $msg->endMessage(0, "Start $systemInfo");
    return $rc;
}


#-------------------------------------------------------------------------------
# Stops the system
#
# Returns int retCode

sub stopSystem {

    my ($self,
        $sapSys,           # SDB::Install::SAPSystem
        $instance,         # SDB::Install::SAPInstance::TrexInstance
        $sidadmPassword,
        $timeout_service,  # number from $instconfig->getTimeout('stop_service')
        $timeout_instance, # number from $instconfig->getTimeout('stop_instance')
        $useHttps,
        $sso_cert,
        $localHostOnly,    # optional boolean
        $setSapstartsrvErr,# does not ignore error when stopping sapstartsrv
        $skipSapstartsrv
       ) = @_;

    if ($instance->hasLocalHost() && $instance->sapstartsrvIsRunning()) {

        my $scope  = ($localHostOnly) ? 'instance' : 'system';
        my $msg    = $sapSys->AddProgressMessage ("Stopping $scope...");
        my $sidadm = $sapSys->getUser()->getSidAdmName();
        my $rc;

        $instance->resetMsgLstContext();

        my $pHandler = $sapSys->getMsgLst()->getProgressHandler();
        $instance->getMsgLst()->setProgressHandler($pHandler);
        if ($localHostOnly) {
            $rc = $instance->stopHost($sidadm,
                                      $sidadmPassword,
                                      undef, # default is local host
                                      $timeout_instance,
                                      $useHttps);
        }
        else {
            $rc = $instance->stopAllHosts($sidadm,
                                          $sidadmPassword,
                                          1, # ignore not running nodes
                                          $timeout_instance,
                                          $useHttps,
                                          $sso_cert);
        }

        if (!defined $rc) {
            $sapSys->AddError("Cannot stop $scope", $instance);
            $sapSys->AddSubMsgLst($msg, $instance);
            return undef;
        }

        if (!$skipSapstartsrv) {

            if (!defined $instance->stopSapStartSrv
                                ($sapSys->getUser()->getSidAdmName(),
                                 $sidadmPassword,
                                 $timeout_service,
                                 $localHostOnly,
                                 $setSapstartsrvErr,
                                 $useHttps,
                                 $sso_cert)) {

                $sapSys->AddError('Cannot stop sapstartsrv', $instance);
                $sapSys->AddSubMsgLst($msg, $instance);
                return undef;
            }

            # workaround to get rid of wdisp semaphore access problem during
            # restart of the renamed system
            $instance->cleanipc();
        }

        $sapSys->AddSubMsgLst ($msg, $instance);
    }

    return 1;
}

#-------------------------------------------------------------------------------
# Renames the HANA data and log paths on a slave node
#
# Returns 1 on success or undef on failure
#
# This is necessary in the case when the HANA data and log volume paths are not shared
# across all the hosts in a distributed landscape. The 'adaptSlaveVolumePaths' will rename
# the path and change its ownership accordingly.
# In the case, when the paths are shared, it will just do nothing.
sub adaptSlaveVolumePaths {
    my ($self, $instconfig, $chownList, $sapSys) = @_;
    my $instance = $instconfig->getOwnInstance();
    my $newSid = $sapSys->get_sid();
    my $newInstanceNr = $instance->get_nr();
    my $oldSid = $instconfig->{oldSID};
    my $oldInstanceNr = $instconfig->{oldInstanceNumber};
    my $target = $instconfig->getValue('Target');
    my $errLst = SDB::Install::MsgLst->new();
    my $msg = $sapSys->AddMessage('Adapting data and log volume paths...');

    for my $paramID ('OldBasePathDataVolumes', 'OldBasePathLogVolumes') {
        my $oldPath = $instconfig->getValue($paramID);
        next if !$oldPath;

        my $newPath = $instance->getNewVolumePath($oldPath, $newSid, $newInstanceNr, $oldSid, $oldInstanceNr, $target);
        my $paramString = $instconfig->getString($paramID);
        if (!$instance->changeOwnAndRenameVolumePath($paramString, $chownList, $oldPath, $newPath, $msg, $errLst)) {
            return undef;
        }
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Restarts SAP HANA system
#
# Returns int retCode

sub restartSystem {
    my ($self,
        $sapSys,           # SDB::Install::SAPSystem
        $instance,         # SDB::Install::SAPInstance::TrexInstance
        $userName,
        $password,
        $start_timeout,    # number from $instconfig->getTimeout('start_instance')
        $stop_timeout,     # number from $instconfig->getTimeout('stop_instance')
        $nostart,
        $useHttps) = @_;

    my $message = $sapSys->AddProgressMessage('Restarting System ...');
    $instance->setMsgLstContext([$message->getSubMsgLst()]);

    my $progressHandler = $sapSys->getMsgLst()->getProgressHandler();
    $instance->getMsgLst()->setProgressHandler($progressHandler);

    if ( !$instance->stopAllHosts ($userName, $password, undef, $start_timeout, $useHttps)) {
        $sapSys->setErrorMessage('Cannot stop system', $instance->getErrMsgLst());
        $message->endMessage(0);
        return 0;
    }

    if (!$nostart && !$instance->startAllHosts ($userName, $password, $stop_timeout, $useHttps)) {
        $sapSys->setErrorMessage('Cannot start system', $instance->getErrMsgLst());
        $message->endMessage(0);
        return 0;
    }

    $message->endMessage(0);
    return 1;
}

#-------------------------------------------------------------------------------
# Starts and stops the SAP HANA system

sub startStopSystem {
    my ($self,
        $instance,         # SDB::Install::SAPInstance::TrexInstance
        $userName,           # SDB::Install::SAPSystem
        $password,
        $start_timeout,    # number from $instconfig->getTimeout('start_instance')
        $stop_timeout,     # number from $instconfig->getTimeout('stop_instance')
        $useHttps) = @_;

    my $message = $self->getMsgLst()->addProgressMessage('Starting and stopping the system...');
    $instance->setMsgLstContext([$message->getSubMsgLst()]);

    my $progressHandler = $self->getMsgLst()->getProgressHandler();
    $instance->getMsgLst()->setProgressHandler($progressHandler);

    if (!$instance->startAllHosts ($userName, $password, $start_timeout, $useHttps)) {
        $self->setErrorMessage('Cannot start system', $instance->getErrMsgLst());
        $message->endMessage(0);
        return 0;
    }

    if ( !$instance->stopAllHosts ($userName, $password, undef, $stop_timeout, $useHttps)) {
        $self->setErrorMessage('Cannot stop system', $instance->getErrMsgLst());
        $message->endMessage(0);
        return 0;
    }

    $message->endMessage(0);
    return 1;
}

#-------------------------------------------------------------------------------
# kills the process that is provided to speed up restart.

sub tryKillProcessesRestartHandover {

     my ($self,
         $sapSys,   # SDB::Install::SAPSystem
         $instance, # SDB::Install::SAPInstance::TrexInstance
        ) = @_;

    my $msg = $sapSys->AddMessage
            ("Checking outstanding processes of $gProductNameSystem...");

    my $saveContext = $instance->setMsgLstContext([$msg->getSubMsgLst()]);
    $instance->scanProcessesKillRestartHandover($msg);
    $instance->setMsgLstContext($saveContext);
    return 1;
}


#-------------------------------------------------------------------------------
# In case of distributed system and if parameter 'root_user' is specified,
# HdbRegHost is called for each remote host in parallel.
#
# These parameters are set implicitly:
#    --batch
#    --check_only
#    --install_hostagent (SSH connection only)
#    --keep_user
#    --keep_user_home_dir
#    --nostart
#    --read_password_from_stdin=xml
#
# Furthermore ignore and timeout options are set implicitly.
#
# Returns undef in case of an error

sub tryRegisterRemoteHosts {

    my ($self,
        $instconfig,        # SDB::Install::Configuration::AnyConfig
        $actionProgressive, # string  (e.g. 'Registering')
        $actionDone,        # string  (e.g. 'registered')
        $hostctrlOperation,
        $paramIDs,          # string array containing parameter IDs of additional options
        $cmdLineParams,     # string containing additional options for SSH only
        $theseHostsOnly,    # array containing subset of host names for register
        $isSerialExecution,
       ) = @_;

    if (!defined $instconfig->getRemoteHosts()) {
        return 1;
    }

    return $instconfig->excuteRemoteSerialOrParallel($actionProgressive,
                                             $actionDone,
                                             'hdbreg',     # executable
                                             'HdbRegHost', # main program
                                             $hostctrlOperation,
                                             ['HostagentPassword', 'Password'],
                                             $paramIDs,
                                             $cmdLineParams,
                                             undef, # $remoteHosts
                                             undef, # $remoteInstconfig
                                             undef, # $hostctrlParamMap
                                             undef, # $templateParamRepl
                                             $theseHostsOnly,
                                             $isSerialExecution);
}


sub setRenameMasterStartedHandler{
    my ($self, $instconfig) = @_;

    my $instance = $instconfig->getOwnInstance ();
    my $instanceRenameArgs = $self->getInstanceRenameArgs ($instconfig);
    my $renameMasterStartedHandlerStreaming =
        sub {

            my $script = $instance->getStreamingRenameScript ();
            if (! -f $script){
                return 1;
            }

            my ($msglst, $errlst) = @_;
            my $strSidRenameNotifyArgs = $instance->getArgsStrSidRenameNotify(
                        @$instanceRenameArgs
                );
            if (!@$strSidRenameNotifyArgs){
                $msglst->addMessage ("Skipping '$script' (online mode)");
                return 1;
            }
            push @$strSidRenameNotifyArgs, ('-phase', 'online');
            my $msg = $msglst->addMessage ("Performing '$script' (online mode)");
            my $rc = $instance->runUtilityInEnv (
                $script,
                $strSidRenameNotifyArgs,
                $msg->getSubMsgLst ());
            if (!$rc){
                $errlst->addError ("$script (online mode) failed", $msg->getSubMsgLst ());
                return undef;
            }
        };

    $instance->addMasterStartedHandler ($renameMasterStartedHandlerStreaming);
    return 1;
}


1;
