package SDB::Install::SHAOperationsManager;

use strict;
use SDB::Install::SysVars;
use SDB::Install::System qw (copy_file copy_tree deltree );
use SDB::Install::Globals qw($gSignatureManifestName determineSignedManifestPath);
use File::Basename;

sub new{
    my $class = shift();
    my ($sapSystem, $serverManifestDir, $globalSidDir) = @_;
    my $installationKitDir = File::Basename::dirname($serverManifestDir);
    my $sharedTargetDir = join($path_separator, ($globalSidDir, 'global', 'hdb'));
    my $sharedOperationsDir = join($path_separator, ($sharedTargetDir, 'operations.d'));

    my $self = {
        _sid => $sapSystem->get_sid(),
        _sourceOperationsDir => join($path_separator, ($installationKitDir, 'operations.d')),
        _sourceSignatureFile => determineSignedManifestPath($installationKitDir),
        _sharedTargetDir => $sharedTargetDir,
        _sharedOperationsDir => $sharedOperationsDir,
        _sapSystem => $sapSystem,
    };

    return bless($self, $class);
}

sub copySHAOperationsAndSignature {
    my ($self) = @_;
    my $sapSystem = $self->getSapSystem();
    my $msg = $sapSystem->AddMessage ("Copying SAP Host Agent operations");
    my $submsglst = new SDB::Install::MsgLst ();
    my ($uid, $gid) = (undef, undef);

    $sapSystem->SetFormatInfo ($msg, 'h1', 'Copy SAP Host Agent Operations');

    if(!$isWin){
        require SDB::Install::NewDBUser;
        my $user = new SDB::Install::NewDBUser ($self->getSid());
        ($uid, $gid) = ($user->uid(), $user->gid());
    }
    if(!$self->_clearOldConfigurations($submsglst)){
        $sapSystem->AddSubMsgLst ($msg, $submsglst);
        return undef;
    }
    if(!$self->_clearOldSignatureFile($submsglst)){
        $sapSystem->AddSubMsgLst ($msg, $submsglst);
        return undef;
    }
    if(!$self->_copyOperations($submsglst, $uid, $gid)){
        $sapSystem->AddSubMsgLst ($msg, $submsglst);
        return undef;
    }
    if(!$self->_copySignature($submsglst, $uid, $gid)){
        $sapSystem->AddSubMsgLst ($msg, $submsglst);
        return undef;
    }

    $sapSystem->AddSubMsgLst ($msg, $submsglst);
    return 1;
}

sub getSid { return $_[0]->{_sid}; }

sub getSourceOperationsDir { return $_[0]->{_sourceOperationsDir}; }

sub getSourceSignatureFile { return $_[0]->{_sourceSignatureFile}; }

sub getSharedTargetDir { return $_[0]->{_sharedTargetDir}; }

sub getSharedOperationsDir { return $_[0]->{_sharedOperationsDir}; }

sub getSapSystem { return $_[0]->{_sapSystem}; }

sub _clearOldConfigurations {
    my ($self, $submsglst) = @_;
    my $targetOperationsDir = $self->getSharedOperationsDir();

    if(-d $targetOperationsDir && !deltree ($targetOperationsDir, $submsglst)) {
        $self->getSapSystem()->AddError("Could not delete already existing directory '$targetOperationsDir'", $submsglst);
        return undef;
    }
    return 1;
}

sub _clearOldSignatureFile {
    my ($self, $submsglst) = @_;
    my $targetDir = $self->getSharedTargetDir();
    my $targetFilename = join($path_separator, ($targetDir, $gSignatureManifestName));

    if(-e $targetFilename && !unlink($targetFilename)) {
        $self->getSapSystem()->AddError("Could not delete existing file '$targetFilename': $!");
        return undef;
    }
    return 1;
}

sub _copyOperations {
    my ($self, $submsglst, $uid, $gid) = @_;
    my $sapSystem = $self->getSapSystem();
    my $sourceOperationsDir = $self->getSourceOperationsDir();
    my $targetOperationsDir = $self->getSharedOperationsDir();

    if (!copy_tree($sourceOperationsDir, $targetOperationsDir, $submsglst)) {
        $self->getSapSystem()->AddError("Could not copy '$sourceOperationsDir' => '$targetOperationsDir'", $submsglst);
        return undef;
    }

    return 1 if($isWin);

    if(!chmod(0755, $targetOperationsDir)) {
        $sapSystem->AddError("Could not change mode of directory '$targetOperationsDir': $!");
        return undef;
    }

    if (!defined $self->_chownRecursively($uid, $gid, $targetOperationsDir)) {
        $sapSystem->AddError("Could not change ownership of the files in directory '$targetOperationsDir'.");
        return undef;
    }
    if (!defined $self->_chmodRecursively(0644, $targetOperationsDir)) {
        $sapSystem->AddError("Could not change mode of the files in directory '$targetOperationsDir'.");
        return undef;
    }
    return 1;
}

sub _copySignature {
    my ($self, $submsglst, $uid, $gid) = @_;
    my $sapSystem = $self->getSapSystem();
    my $sourceFile = $self->getSourceSignatureFile();
    my $targetDir = $self->getSharedTargetDir();
    my $targetFilename = join($path_separator, ($targetDir, $gSignatureManifestName));

    if(-e $sourceFile) {
        $submsglst->AddMessage("Copying signature file '$sourceFile' to '$targetDir'");
        my $copyMsglst = new SDB::Install::MsgLst ();
        if (!copy_file($sourceFile, $targetDir, $copyMsglst)) {
            $self->getSapSystem()->AddError("Could not copy '$sourceFile' => '$targetDir'", $copyMsglst);
            $submsglst->appendMsgLst ($copyMsglst);
            return undef;
        }
        $submsglst->appendMsgLst ($copyMsglst);
    }
    else {
        $sapSystem->AddWarning("Could not copy file '$sourceFile': '$sourceFile' is missing.");
    }

    return 1 if($isWin);

    if(-e $targetFilename) {
        if(!chmod(0644, $targetFilename)) {
            $sapSystem->AddError("Could not change mode of file '$targetFilename': $!");
            return undef;
        }
        if(!chown($uid, $gid, $targetFilename)) {
            $sapSystem->AddError("Could not change ownership of file '$targetFilename': $!");
            return undef;
        }
    }
    else {
        $sapSystem->AddWarning("Could not change mode and ownership of file '$targetFilename': '$targetFilename' is missing.");
    }
    return 1;
}

sub _chownRecursively {
    my ($self, $uid, $gid, $path) = @_;
    
    require SDB::Install::DirectoryWalker;
    
    chown $uid, $gid, $path or return undef;

    my $dirWalker = new SDB::Install::DirectoryWalker(undef, undef, undef, undef, undef, undef,
                                                      sub {
                                                            my $entry = $_[3].$path_separator.$_[4];
                                                            chown $uid, $gid, $entry or return undef;
                                                      }, 
                                                      undef, undef, undef, 1, 1, 0);
    my $rc = $dirWalker->findAndProcess($path);
    return $rc;
}

sub _chmodRecursively {
    require SDB::Install::DirectoryWalker;

    my ($self, $mode, $path) = @_;
    my $dirWalker = new SDB::Install::DirectoryWalker(undef, undef, undef, undef, undef, undef,
                                                      sub {
                                                            my $entry = $_[3].$path_separator.$_[4];
                                                            chmod($mode, $entry) or return undef;
                                                      },
                                                      undef, undef, undef, 1, 1, 0);
    my $rc = $dirWalker->findAndProcess($path);
    return $rc;
}

1;