package SDB::Install::ServiceManager;
use parent qw(SDB::Install::Base);
use strict;

use SDB::Install::SysVars qw($isWin);
use SDB::Install::SAPControl;
use SDB::Install::LayeredConfig;
use SDB::Install::Configuration qw($bool_true_pattern);
use File::Spec;

sub new {
    my $class = shift;
    my $self  = {};
    bless ($self, $class);
    return $self;
}

sub startServices {
	my ($self, $config, $serviceNames, $specificHosts) = @_;
	my $csvServices = join(', ', @$serviceNames);
	$self->getMsgLst->addMessage("Starting the following services: $csvServices");
	return $self->_startOrStopServices($config, $serviceNames, 1,$specificHosts);
}

sub stopServices {
	my ($self, $config, $serviceNames, $specificHosts) = @_;
	my $csvServices = join(', ', @$serviceNames);
	$self->getMsgLst->addMessage("Stopping the following services: $csvServices");
	return $self->_startOrStopServices($config, $serviceNames, 0,$specificHosts);
}

sub disableServicesInDaemonIni {
    my ($self, $config, $serviceNames) = @_;
    my $csvServices = join(', ', @$serviceNames);
    $self->getMsgLst->addMessage("Disabling the following services in 'daemon.ini': $csvServices");
    return $self->_disableEnableServices($config, $serviceNames, 0);
}

sub enableServicesInDaemonIni {
    my ($self, $config, $serviceNames) = @_;
    my $csvServices = join(', ', @$serviceNames);
    $self->getMsgLst->addMessage("Enabling the following services in 'daemon.ini': $csvServices");
    return $self->_disableEnableServices($config, $serviceNames, 1);
}

sub cleanServicesEntriesInDaemonIni {
    my ($self, $config, $serviceNames) = @_;
    my $csvServices = join(', ', @$serviceNames);
    $self->getMsgLst->addMessage("Cleaning up releveant 'daemon.ini' sections:");
    return $self->_disableEnableServices($config, $serviceNames, 1, 'clean up');
}

sub _startOrStopServices {
	my ($self, $config, $serviceNames, $startFlag,$specificHosts) = @_;

	my $action = ($startFlag) ? 'start' : 'stop';
	my $trexInstance = $config->getOwnInstance();
	if (!defined $trexInstance) {
		$self->getErrMsgLst()->addMessage("Could not initiate instance object. Failed to reconfigure system and $action services.");
		return undef;
	}
    return undef if (!$self->_reconfigureSystem($trexInstance));
	my $hosts = $specificHosts // $trexInstance->get_allhosts();
	return $self->startStopViaSapcontrol($config, $hosts, $startFlag, $serviceNames);
}

sub _getHostDaemonIni {
	my ($self, $config, $host) = @_;

	my $trexInstance = $config->getOwnInstance();
	my $layered_cfg = new SDB::Install::LayeredConfig(
        $trexInstance->get_instanceDir(),
        $trexInstance->get_globalSidDir(),
        undef,
        $host,
        File::Spec->catfile($trexInstance->get_globalTrexExeDirAbs(), 'config'),
        $isWin ? undef : $trexInstance->get_user()->uidUx(),
        $isWin ? undef : $trexInstance->get_user()->gidUx()
    );
    if($layered_cfg->ErrorState()) {
        $self->getErrMsgLst()->addError("Error initializing layered configuration for host '$host'", $layered_cfg);
        return undef;
    }
    my $daemon_ini = $layered_cfg->getIniFile('daemon.ini');
    if(!defined $daemon_ini){
        $self->getErrMsgLst()->addError(undef, $layered_cfg);
        return undef;
    }

    $self->getMsgLst()->addMessage("Reading daemon.ini for host '$host'");
    $daemon_ini->resetMsgLstContext();
    if(!defined $daemon_ini->readValues()){
        $self->getErrMsgLst()->addError(undef, $daemon_ini);
        return undef;
    }
    $self->getMsgLst()->appendMsgLst($daemon_ini->getMsgLst());
    $daemon_ini->resetMsgLstContext();

    return $daemon_ini;
}

sub _disableEnableServices {
    my ($self, $config, $serviceNames, $startFlag, $action) = @_;
    $action = ($startFlag) ? 'enable' : 'disable' if (!defined $action);
    my $trexInstance = $config->getOwnInstance();
    if (!defined $trexInstance) {
        $self->getErrMsgLst()->addMessage("Could not initiate instance object. Failed to $action services.");
        return undef;
    }
    for my $host (@{$trexInstance->get_allhosts()}) {
        my $daemon_ini = $self->_getHostDaemonIni($config, $host);
        return undef if (!defined $daemon_ini);

        $self->getMsgLst()->addMessage("Adjusting daemon.ini for host '$host' to $action services.");
        return undef if (!$self->_manageDaemonIniEntries($daemon_ini, $serviceNames, $startFlag));
    }
    return 1;
}

sub _manageDaemonIniEntries {
    my ($self, $daemon_ini, $serviceNames, $startFlag) = @_;
    my $hasAlteredDaemonIni = 0;
    foreach my $service (@{$serviceNames}) {
        my $isDisabledValue = $self->_getDaemonIniValue($daemon_ini, $service, 'disabled', CFG_LAYER_HOST);
        # Check if service is stopped by us
        my $wasStoppedByUs = (1 == $self->_getDaemonIniValue($daemon_ini, $service, 'hplm_stopped', CFG_LAYER_HOST));

        if($startFlag) {
            if(defined $isDisabledValue && $isDisabledValue =~ /true/ && $wasStoppedByUs) {
                $hasAlteredDaemonIni = 1;
                $daemon_ini->removeKey(CFG_LAYER_HOST, $service, 'disabled');
                $daemon_ini->removeKey(CFG_LAYER_HOST, $service, 'hplm_stopped'); # Remove mark that service is stopped by us
            }
        }
        else {
            if(!defined $isDisabledValue || $isDisabledValue !~ /true/) {
                $hasAlteredDaemonIni = 1;
                $daemon_ini->setValue(CFG_LAYER_HOST, $service, 'disabled', 'true');
                $daemon_ini->setValue(CFG_LAYER_HOST, $service, 'hplm_stopped', 1); # Mark that service is stopped by us
            }
        }
    }
    if($hasAlteredDaemonIni && !defined $daemon_ini->write()) {
        $self->getErrMsgLst()->addError (undef, $daemon_ini);
        return undef;
    }
    $self->getMsgLst()->appendMsgLst($daemon_ini->getMsgLst());
    return 1;
}

# Doesn't generate errors in the log file
sub _getDaemonIniValue {
    my ($self, $daemon_ini, $section, $key, $layer) = @_;
    return undef if (!$daemon_ini->existsValue($section, $key));
    return $daemon_ini->getValue($section, $key, $layer);
}

sub _getRelevantDaemonIniSections {
    my ($self, $daemon_ini, $serviceNames) = @_;
    my $allSections = $daemon_ini->getSections();
    my @relevantSections = ();
    for my $service (@$serviceNames) {
        # sections may have format '<service>.TenantId'
        my @filteredServices = grep {$_ =~ /$service\.*/} @$allSections;
        push (@relevantSections, @filteredServices);
    }
    return [@relevantSections];
}

sub _reconfigureSystem {
	my ($self, $trexInstance) = @_;

	my $msg = $self->getMsgLst()->addMessage("Reconfiguring system");
    my $msglst = $msg->getSubMsgLst();

	my $args = ['-reconfig'];
	my $rc = $trexInstance->runNameServerUtility($args, $msglst);
    if (!defined $rc){
        $self->getErrMsgLst()->addError("Cannot reconfigure system", $msglst);
        return undef;
    }
    return 1;
}

sub _getFilteredServices {
    my ($self, $originalServices, $host, $config) = @_;
    my $daemonIni = $self->_getHostDaemonIni($config, $host);
    my $filteredServices = [];

    for my $service (@{$originalServices}) {
        my $disabledEntry = $self->_getDaemonIniValue($daemonIni, $service, 'disabled', CFG_LAYER_HOST);
        my $isDisabled = defined($disabledEntry) && $disabledEntry =~ $bool_true_pattern;
        my $instances = $self->_getDaemonIniValue($daemonIni, $service, 'instances', CFG_LAYER_HOST);
        my $shouldExistAsNormalService = defined($instances) && $instances > 0;
        my $shouldExistAsTenantService = $self->_existsTenantService($service, $daemonIni);
        push (@{$filteredServices}, $service) if (($shouldExistAsNormalService || $shouldExistAsTenantService) && !$isDisabled);
    }

    return $filteredServices;
}

sub _existsTenantService {
    my ($self, $service, $daemonIni) = @_;
    my $allSections = $daemonIni->getSections();
    return grep { $_ =~ /^$service\.\w+/} @{$allSections};
}

sub startStopViaSapcontrol {
	my ($self, $config, $hosts, $startFlag, $serviceNames) = @_;
	my $defaultTimeout = 7200; #legacy magic number
	my $trexInstance = $config->getOwnInstance();
    my $start_timeout = $config->getTimeout ('start_service') // $defaultTimeout;
    my $start_wait_delay = 2;
    my $start_wait_retries = int ($start_timeout / $start_wait_delay);
    my $stop_timeout = $config->getTimeout ('stop_service') // $defaultTimeout;
    my $stop_wait_delay = 2;
    my $stop_wait_retries_plus_60 = int (($stop_timeout + 60)/ $stop_wait_delay);

    my $nr = $trexInstance->get_nr();
    my $userName = $trexInstance->get_user()->getSidAdmName();
    my $password = $config->getValue('Password');
    my $useHttps = $config->isUseHttps();
    my $sso_cert = $config->getValue('SSOCertificate');

    my $csvServices = join(', ', @$serviceNames);
    $trexInstance->setMsgLstContext($self->getMsgLstContext());
    my $msg;
    foreach my $host (@$hosts) {
        $trexInstance->takeTraceDirSnapshot($host);
        my $sapcontrol = new SDB::Install::SAPControl(
            $trexInstance->_getSAPControlHost($host),
            $nr,
            $userName,
            $password,
            $useHttps,
            $sso_cert
        );
        $sapcontrol->set_callback($trexInstance->{_f_callback});
        if($startFlag) {
            $msg = $self->getMsgLst()->addMessage ("Starting $csvServices at host '$host', parameters: instance number = $nr, user = $userName");
            $sapcontrol->setMsgLstContext([$msg->getSubMsgLst()]);
            my $filteredServices = $self->_getFilteredServices($serviceNames, $host, $config);
            # Don't even call SAPControl if $filteredServices is empty
            next if !@{$filteredServices};
            my $rc = $sapcontrol->WaitforStarted ($start_wait_retries, $start_wait_delay, undef, $filteredServices);
            if(!$rc) {
                $self->getErrMsgLst()->addError(undef, $sapcontrol);
                $trexInstance->analyzeTraceDirDeltas($host);
                return undef;
            }
        }
        else {
            $msg = $self->getMsgLst()->addMessage ("Stopping $csvServices at $host, parameters: instance number = $nr, user = $userName");
            $sapcontrol->setMsgLstContext([$msg->getSubMsgLst()]);
            my $rc = $sapcontrol->WaitforStopped ($stop_wait_retries_plus_60, $stop_wait_delay, $serviceNames);
            if(!$rc) {
                $self->getErrMsgLst()->addError(undef, $sapcontrol);
                $trexInstance->analyzeTraceDirDeltas($host);
                return undef;
            }
        }
    }
    return 1;
}

1;
