#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#

package SDB::Install::Sql;

use strict;
use SDB::Install::BaseLegacy;
use SDB::Install::SysVars qw($isWin);

our @ISA = qw (SDB::Install::BaseLegacy);

our $SQLDBC_OK;
our $SQLDBC_SUCCESS_WITH_INFO;
our $SQLDBC_HOSTTYPE_UTF8;
our $SQLDBC_StringEncodingUTF8;
our $runtime;
our $sqldbc_env;

our $default_port = '3%02d15'; # IndexServer
our $systemserver_port = '3%02d13'; # SystemServer

our $initialized = 0;

sub SQLDBC_OK;
sub SQLDBC_SUCCESS_WITH_INFO;
sub SQLDBC_HOSTTYPE_UTF8;
sub SQLDBC_StringEncodingUTF8;

sub init{
    return if ($initialized);
    require SQLDBC;
    import SQLDBC qw (SQLDBC_OK SQLDBC_SUCCESS_WITH_INFO SQLDBC_HOSTTYPE_UTF8 SQLDBC_StringEncodingUTF8);
    $SQLDBC_OK = SQLDBC_OK;
    $SQLDBC_SUCCESS_WITH_INFO = SQLDBC_SUCCESS_WITH_INFO;
    $SQLDBC_HOSTTYPE_UTF8 = SQLDBC_HOSTTYPE_UTF8;
    $SQLDBC_StringEncodingUTF8 = SQLDBC_StringEncodingUTF8;
    $runtime = SQLDBC::GetClientRuntime();
    $sqldbc_env = new SQLDBC::Environment ($runtime);
    $initialized = 1;
}


sub connect{
    my ($self,$host,$instance_nr,$user,$passwd,$dbname, $propertiesHash) = @_;
    init();
    if ($self->{conn}){
        $self->release ();
    }

    $self->{conn} = $sqldbc_env->createConnection ();

    my $port = $default_port;

    my $connectProperties = new SQLDBC::ConnectProperties ();

    if (defined $dbname){
        $port = '3%02d13';
        $connectProperties->setProperty ( 'DATABASENAME', $dbname );
    }

    if (defined $propertiesHash){
        foreach my $connectProperty (keys (%$propertiesHash)){
            $connectProperties->setProperty (
                $connectProperty,
                $propertiesHash->{$connectProperty});
        }
    }

    if ($SQLDBC_OK != $self->{conn}->connect (
        sprintf ("%s:$port",$host,int ($instance_nr)),
        '',
        $user,
        $passwd,
        $SQLDBC_StringEncodingUTF8,
        $connectProperties
        )){
        my $err = $self->{conn}->error();
        $self->AddError ("Connect failed (code = " . $err->getErrorCode () .
                "): " . $err->getErrorText());
         return undef;
    }
    return 1;
}


sub connectWithUserstoreKey{
    my ($self,$key, $propertiesHash) = @_;
    init();
    if ($self->{conn}){
        $self->release ();
    }

    $self->{conn} = $sqldbc_env->createConnection ();

    my $connectProperties = new SQLDBC::ConnectProperties ();

    $connectProperties->setProperty ('KEY', $key);

    if (defined $propertiesHash){
        foreach my $connectProperty (keys (%$propertiesHash)){
            $connectProperties->setProperty (
                $connectProperty,
                $propertiesHash->{$connectProperty});
        }
    }

    if ($SQLDBC_OK != $self->{conn}->connectWithProperties ($connectProperties)){
        my $err = $self->{conn}->error();
        $self->AddError ("Connect failed (code = " . $err->getErrorCode () .
                "): " . $err->getErrorText());
         return undef;
    }
    return 1;
}

sub getConnErrorCode{
	my ($self) = @_;

	if (!defined $self->{conn}) {
		return undef;
	}
	my $err = $self->{conn}->error();
	if (!defined $err) {
		return undef;
	}
	return $err->getErrorCode ();
}

sub isErrorSSLEnforced{
    my $errorCode = $_[0]->getConnErrorCode ();
    if (!defined $errorCode){
        return undef;
    }
    return ($errorCode == 4321);
}

#
# SQLDBC does not distinguish between several socket connect errors.
# Try to parse the error text for the os related error code
# of 'connection refused' here.
#


our $ECONNREFUSED_pattern;
if ($isWin){
    require SAPDB::Install::System::Win32::API;
    $ECONNREFUSED_pattern = sprintf ('\ rc=%d:',
        SAPDB::Install::System::Win32::API::WSAECONNREFUSED());
}
else{
    require Errno;
    $ECONNREFUSED_pattern = sprintf ('\ rc=%d:', Errno::ECONNREFUSED());
}

sub isECONNREFUSED{
    my ($self) = @_;
    if (!defined $self->{conn}){
        $self->AddError ("No connection\n");
        return undef;
    }
    my $err = $self->{conn}->error();
    if (!defined $err){
        $self->AddError ("No error\n");
        return undef;
    }
    if ($err->getErrorCode() != -10709){
        return 0;
    }
    if ($err->getErrorText() =~ /$ECONNREFUSED_pattern/){
        return 1;
    }
    return 0;
}

sub isInvalidCredentials{
    my ($self) = @_;
    if (!defined $self->{conn}){
        $self->AddError ("No connection\n");
        return undef;
    }
    my $err = $self->{conn}->error();
    if (!defined $err){
        $self->AddError ("No error\n");
        return undef;
    }
    if ($err->getErrorCode() != 10){
        return 0;
    }
    if ($err->getErrorText() =~ /invalid|authentication/i){
        return 1;
    }
    return 0;
}

sub prepare{
    my ($self,$stmnt) = @_;
    if (!$self->{conn}){
        $self->AddError ("No connection\n");
        return undef;
    }
    $self->_releasePreparedStatement();
    $self->{prepared_stmnt} = $self->{conn}->createPreparedStatement ();
    my $prepRetcode = $self->{prepared_stmnt}->prepare ($stmnt);
    if ($prepRetcode != $SQLDBC_OK && $prepRetcode != $SQLDBC_SUCCESS_WITH_INFO){
        my $err = $self->{prepared_stmnt}->error ();
        $self->AddError ("Prepare failed with SQLDBC code $prepRetcode: (". $err->getErrorCode () .
                '): ' . $err->getErrorText());
        $self->_releasePreparedStatement();
        return undef;
    }
    return 1;
}

sub bindParam{
    my ($self, $index, $buff, $size, $refLengthIndicator, $type) = @_;
    if (!defined $self->{prepared_stmnt}){
        $self->setErrorMessage ("Cannot bind parameters: no prepared statement");
        return undef;
    }
    if (!defined $type){
        $type =  $SQLDBC_HOSTTYPE_UTF8;
    }
    if (!defined $size){
        if (ref($buff)){
            $size = length ($$buff);
        }
        else{
            $size = length ($buff);
        }
    }

    if (!defined $refLengthIndicator){
        my $tmp = $size;
        $refLengthIndicator = \$tmp;
    }

    my $bindRetcode = $self->{prepared_stmnt}->bindParameter ($index, $type, $buff, $$refLengthIndicator, $size, 1);
    if ($bindRetcode != $SQLDBC_OK && $bindRetcode != $SQLDBC_SUCCESS_WITH_INFO){
        my $err = $self->{prepared_stmnt}->error ();
        $self->AddError ("Bind failed with SQLDBC code $bindRetcode: (". $err->getErrorCode () .
                '): ' . $err->getErrorText());
        return undef;
    }
    if(!ref($buff)){
        # make sure, that the scalar survives until execute() is performed
        $self->{_boundParams}->[$index] = \$buff;
    }
    return 1;
}

sub execute{
    my ($self,$stmnt) = @_;
    if (!$self->{conn}){
        $self->AddError ("No connection\n");
        return undef;
    }

    if (defined $self->{stmnt}){
        $self->{conn}->releaseStatement ($self->{stmnt});
        $self->{stmnt} = undef;
    }

    if (!defined $stmnt && defined $self->{prepared_stmnt}){
        my $execRetcode = $self->{prepared_stmnt}->execute ();
        if ($execRetcode != $SQLDBC_OK && $execRetcode != $SQLDBC_SUCCESS_WITH_INFO){
            my $err = $self->{prepared_stmnt}->error ();
            $self->AddError ("Execute prepared statement failed with SQLDBC code $execRetcode: (". $err->getErrorCode () .##########################
                    '): ' . $err->getErrorText());
            return undef;
        }
        return 1;
    }

    $self->_releasePreparedStatement();

    $self->{stmnt} = $self->{conn}->createStatement ();
    my $execRetcode = $self->{stmnt}->execute ($stmnt);
    if ($execRetcode != $SQLDBC_OK && $execRetcode != $SQLDBC_SUCCESS_WITH_INFO){
        my $err = $self->{stmnt}->error ();
        $self->AddError ("Execute failed with SQLDBC code $execRetcode: (". $err->getErrorCode () .
                '): ' . $err->getErrorText());
        return undef;
    }
    return 1;
}

sub getStmntErrorCode{
	my ($self) = @_;

	if (!defined $self->{stmnt}) {
		return undef;
	}
	my $err = $self->{stmnt}->error();
	if (!defined $err) {
		return undef;
	}
	return $err->getErrorCode ();
}

sub getResultSet{
    my ($self, $withCaption) = @_;
    return $self->_getResultSet($self->{stmnt}, $withCaption);
}

sub getResultSetOfPreparedStatement{
    my ($self, $withCaption) = @_;
    return $self->_getResultSet($self->{prepared_stmnt}, $withCaption);
}

sub _getResultSet{
    my ($self, $stmnt, $withCaption) = @_;
    $withCaption //= 0;
    if (!defined $stmnt){
        $self->AddError ("No statement\n");
        return undef;
    }
    my $resultSet = $stmnt->getResultSet();
    if (!defined $resultSet){
        $self->AddError ("No resultset\n");
        return undef;
    }
    my $rc = $SQLDBC_OK;
    my $error;
    my ($cIndex);
    my $buff = '';
    my $resultMeta = $resultSet->getResultSetMetaData ();
    my $cols = $resultMeta->getColumnCount();
    my @result;
    if ($withCaption){
        my @caption;
        my $columnName;
        for my $i (1..$cols){
            $resultMeta->getColumnName($i,$columnName);
            push @caption, $columnName;
        }
        push @result, \@caption;
    }
    $rc =  $resultSet->next ();
    my $i = 0;
    while (1){
        $i++;
        if ($rc != $SQLDBC_OK){
            $error = $resultSet->error();
            if ($error->getErrorCode() == $SQLDBC_OK){
                # eof
                last;
            }
            else{
                $self->AddError ('Fetch failed ('. $error->getErrorCode () .
                '): ' . $error->getErrorText());
                return undef;
            }
        }
        my @row = ();
        foreach $cIndex (1..$cols){
            $rc = $resultSet->getObject ($cIndex, $SQLDBC_HOSTTYPE_UTF8, $buff, 1);
            if ($rc != $SQLDBC_OK){
                my $error = $resultSet->error;
                $self->AddError ('Fetch failed ('. $error->getErrorCode () .
                '): ' . $error->getErrorText());
                return undef;
            }
            push @row, $buff;
        }
        push @result, \@row;
        $rc = $resultSet->next ();
    }
    return \@result;
}


sub release{
    my ($self) = @_;
    if (!$self->{conn}){
        $self->AddError ("No connection\n");
        return undef;
    }

    if (defined $self->{stmnt}){
        $self->{conn}->releaseStatement ($self->{stmnt});
        undef ($self->{stmnt})
    }

    $self->_releasePreparedStatement();

    $sqldbc_env->releaseConnection ($self->{conn});
    undef $self->{conn};
    return 1;
}

sub _releasePreparedStatement{
    my ($self) = @_;
    if (defined $self->{prepared_stmnt}){
        $self->{conn}->releasePreparedStatement($self->{prepared_stmnt});
        $self->{prepared_stmnt} = undef;
        if (exists $self->{_boundParams}){
            delete $self->{_boundParams};
        }
    }
    return 1;
}

sub DESTROY;
*DESTROY = \&release;

1;
