package SDB::Install::Sql::SqlConnectionProvider;

use strict;
use base 'SDB::Install::Base';
use SDB::Install::Sql::SSLConnectProperties;
use SDB::Install::Sql;

our $_sql_module_loaded = 0;
our $_sql_module_error  = '';

sub _getConnectDetails{
    my ($self, $dbName, $hdbInstance, $user, $password, $userStoreKey, $properties, $timeout) = @_;
    my $timeout_string = defined $timeout ? " start_timeout=$timeout" : '';
    my $ssl_string = (defined $properties && $properties->{'ENCRYPT'}) ? " encryption=on" : " encryption=off";
    if (defined $userStoreKey){
        return "user store key '$userStoreKey'$timeout_string$ssl_string";
    }
    my $port = defined $dbName ? sprintf("3%02d13", $hdbInstance->get_nr()) : sprintf("3%02d15", $hdbInstance->get_nr());
    my $host = $hdbInstance->getSqlHost();
    my $connectString = "$host:$port";
    if (defined $dbName){
        $connectString = "$dbName\@$connectString";
    }
    $connectString .= "$timeout_string$ssl_string";
    return $connectString;
}


sub waitForSqlConnection{
    my ($self, $dbName, $hdbInstance, $user, $password, $userStoreKey, $properties, $timeout) = @_;
    $timeout //= 900;
    my $counter = 0;
    my $connected = 0;
    my $sql;
    my $msglst = $self->getMsgLst();
    my $_f_check_alive;
    my $serviceName;
    $self->{_was_a_timeout} = 0;
    if (!defined $dbName || $dbName eq $hdbInstance->get_sid()){
        $_f_check_alive = $hdbInstance->can('getIndexServerPid');
        $serviceName = 'hdbindexserver';
    }
    elsif($dbName eq 'SYSTEMDB'){
        $_f_check_alive = $hdbInstance->can('getSystemServerPid');
        $serviceName = 'hdbnameserver';
    }
    shift @_;
    my $connectDetails = $self->_getConnectDetails(@_);
    my $msg = $self->getMsgLst()->addMessage ("Connecting to database $connectDetails");
    while ($counter < $timeout || $timeout == 0){
        $sql = $self->getSqlConnection($dbName, $hdbInstance, $user, $password, $userStoreKey, $properties, 1, $msg->getSubMsgLst());
        if (defined $sql){
            return $sql;
        }
        if (!$self->{_is_ECONNREFUSED}){
            return undef;
        }
        $msglst->addMessage ("Connection refused: retrying... ($counter)");
        if (defined $self->{_f_callback}){
            &{$self->{_f_callback}} ();
        }
        $self->sleep(1);
        if (defined $self->{_f_callback}){
            &{$self->{_f_callback}} ();
        }
        $counter++;
        if (defined $_f_check_alive){
            if (!$_f_check_alive->($hdbInstance, undef, $msglst)){
                $self->setErrorMessage("Connect failed: $serviceName is not running");
                return undef;
            }
        }
    }
    $self->setErrorMessage ("$serviceName not properly running, connect timeout reached");
    $self->{_was_a_timeout} = 1;
    return undef;
}

sub sleep{
    CORE::sleep($_[1]);
}

sub wasATimeout{
    return $_[0]->{_was_a_timeout};
}

sub isECONNREFUSED{
    return $_[0]->{_is_ECONNREFUSED};
}

sub getSqlObj{
    my ($self) = @_;
    if (!defined $self->{_sql}){
       $self->{_sql} = new SDB::Install::Sql();
    }
    return $self->{_sql};
}

sub getSqlConnection{
    my ($self, $dbName, $hdbInstance, $user, $password, $userStoreKey, $properties, $expectECONNREFUSED, $msglst) = @_;
    my $sql = $self->getSqlObj();
    my $rc;
    my $msg;
    if (!defined $msglst){
        my $connectDetails = $self->_getConnectDetails($dbName, $hdbInstance, $user, $password, $userStoreKey, $properties);
        $msg = $self->getMsgLst()->addMessage("Connecting to database $connectDetails");
        $msglst = $msg->getSubMsgLst();
    }

    $properties //= {};
    $properties->{'APPLICATION'} //= 'HDBLCM';

    if (defined $userStoreKey){
        $rc = $sql->connectWithUserstoreKey($userStoreKey, $properties);
    }
    else{
        $rc = $sql->connect(
                $hdbInstance->getSqlHost(),
                $hdbInstance->get_nr(),
                $user,
                $password,
                $dbName,
                $properties);
    }
    if (defined $rc){
        return $sql;
    }

    $self->{_is_ECONNREFUSED} = $sql->isECONNREFUSED;

    if (!($properties->{'ENCRYPT'}) && $sql->isErrorSSLEnforced()){
        $msglst->addMessage("Sql encryption is enforced");
        return $self->getSecureSqlConnection($dbName, $hdbInstance, $user, $password, $userStoreKey, $properties, $expectECONNREFUSED);
    }
    if (!$self->{_is_ECONNREFUSED} || !$expectECONNREFUSED){
        $self->setErrorMessage ("Connect failed: " . $sql->getErrorString());
    }
    return undef;

}

sub getSecureSqlConnection{
    my ($self, $dbName, $hdbInstance, $user, $password, $userStoreKey, $properties, $expectECONNREFUSED, $msglst) = @_;
    my $sslMsg = $self->getMsgLst()->addMessage("Trying to establish a secure connection");
    my $secureConnectProperties = $hdbInstance->getSqlSSLProperties($sslMsg->getSubMsgLst());
    if (!defined $secureConnectProperties){
        $self->setErrorMessage('Cannot get SSL connect parameters',$hdbInstance->getErrMsgLst());
        return undef;
    }
    my ($sslConnectPropertiesHash, $sslEnv) = @$secureConnectProperties;
    local %ENV = %ENV if defined $sslEnv;
    if (defined $sslEnv){
        foreach my $envVarName (keys (%$sslEnv)){
	        $ENV{$envVarName} = $sslEnv->{$envVarName};
        }
    }
    if (defined $properties){
        foreach my $key (keys (%$sslConnectPropertiesHash)){
            $properties->{$key} = $sslConnectPropertiesHash->{$key};
        }
    }
    else{
        $properties = $sslConnectPropertiesHash;
    }
    return $self->getSqlConnection($dbName, $hdbInstance, $user, $password, $userStoreKey, $properties, $expectECONNREFUSED, $msglst)
}

1;


