package BackSlasher;
use SAPDB::Install::Hostname;

sub new{
    my $self = bless ({},shift);
    my ($value, $defaultFirst) = @_;

    my $hostname = hostname();

    if ($value =~ /\\/){
        ($self->{first},$self->{second}) = split(/\\/, $value);
    }
    else{
        $self->{first} = defined $defaultFirst ? $defaultFirst : $hostname;
        $self->{second} = $value;
    }
    if (lc ($self->{first}) eq  'builtin'){
        $self->{wellknown} = 1;
        $self->{local} = 1;
        $self->{first} = $hostname;
    }
    else{
        $self->{local} =  lc ($self->{first}) eq lc ($hostname);
    }
    $self->{concat} = lc($self->{first}) . '\\' . lc($self->{second});
    return $self;
}

sub equals{
    return $_[0]->{concat} eq $_[1]->{concat};
}

sub repr{
    return $_[0]->{concat};
}

##############################################################################


package SDB::Install::User;

use strict;

use SDB::Install::BaseLegacy;
use SDB::Install::System qw (exec_program makedir);
use SAPDB::Install::System::Win32::API;
use SDB::Install::SysVars;
use SDB::Install::Tools qw (readini);
use File::Basename qw (basename dirname);
use SDB::Common::BuiltIn;

our  @ISA = qw (SDB::Install::BaseLegacy);

our $useradd_defaults;

sub get_useradd_defaults{
    if (defined $useradd_defaults || $isWin){
        return $useradd_defaults;
    }
    my $data = readini ('/etc/default/useradd');
    if (defined $data && %$data){
        $useradd_defaults = $data->{''};
    }
    return $useradd_defaults;
}

sub newUx{
    my $self = shift->SUPER::new ();
    my ($name) = @_;

    my ($passwd,$uid,$gid,$quota,$comment,$gcos,
        $dir,$shell,$expire);


    if (!defined $name){
        ($name,$passwd,$uid,$gid,$quota,$comment,$gcos,
            $dir,$shell,$expire) = getpwuid ( $> );
        $self->{name} = $name;
        $self->{uid} = $>;
    }
    else{
        $self->{name} = $name;
        ($name,$passwd,$uid,$gid,$quota,$comment,$gcos,
            $dir,$shell,$expire) = getpwnam ($name);
        $self->{uid} = $uid;
    }
    if (defined $uid){
        $self->{gid} = $gid;
        $self->{group} = getgrgid ($gid);
        $self->{comment} = $gcos;
        $self->{home} = $dir;
        $self->{shell} = $shell;
        $self->{crypted_password} = $passwd;
    }
    return $self;
}

sub getSid{
    my ($self) = @_;
    if (defined $self->{sid}){
        return $self->{sid};
    }

    my ($name,$rc,$sid,$type);

    if ($self->{bsname}->{wellknown}){
        if (lc ($self->{bsname}->{second}) eq 'system'){
            ($name,$sid) = 	LookupBuiltinAccount (SECURITY_LOCAL_SYSTEM_RID);
        }
        $self->{sid} = $sid;
    }
    else{
         ($rc,$sid,$type) = SAPDB::Install::System::Win32::API::lookupAccount ($self->{bsname}->repr(),
                            undef);
        if ($rc == 0){
            $self->{sid} = $sid;
        }
    }
    return $sid;
}


sub newWin{
    my $self = shift->SUPER::new ();
    my ($name) = @_;
    if (!defined $name){
        $name = SAPDB::Install::System::Win32::API::GetCurrentUser ();
        if (!defined $name){
            $self->AddError ("Cannot get current user");
        }
    }
    $self->{name} = $name;
    $self->{bsname} = new BackSlasher ($name);
    $self->getSid ();

    if ($self->exists ()){
        my ($rc, $comment, $home);
        if ($self->{bsname}->{local}){
            (($rc, $comment, $home)) = SAPDB::Install::System::Win32::API::infoUser ($self->{bsname}->{second});
        }
        else{
            (($rc, $comment, $home)) = SAPDB::Install::System::Win32::API::infoUser ($self->{bsname}->{second},
                undef,undef,$self->{bsname}->{first});
        }
        if ($comment){
            $self->{comment} = $comment;
        }
        if ($home){
            $self->{home} = $home;
        }
    }
    return $self;
}


sub verifyPasswordUx{
    my ($self, $passwd) = @_;

    if (!defined $self->{crypted_password}){
        $self->AddMessage ("Cannot verify password: No crypted password");
        return undef;
    }
    if ($self->{crypted_password} eq 'x'){
        $self->AddMessage ("Cannot verify password: cannot read shadow file");
        return undef;
    }
    if ($self->{crypted_password} eq '*'){
        $self->AddMessage ("Cannot verify password: cannot get encrypted password (NIS+ or LDAP)");
        return undef;
    }

    if ($self->{crypted_password} eq 'VAS'){
        $self->AddMessage ("Cannot verify password: cannot get encrypted password (Quest Vintela Authentication Services)");
    return undef;
    }

    if (length ($self->{crypted_password}) < 13){
        $self->AddMessage ("Cannot verify password: cannot get encrypted password (invalid hash key length)");
        return undef;
    }

    # valid characters are A-Za-z0-9./$
    if ($self->{crypted_password} =~ /[^A-Za-z\d\$\/\.]/){
        $self->AddMessage ("Cannot verify password: cannot get encrypted password (invalid hash key character)");
        return undef;
    }

    utf8::encode ($passwd) if utf8::is_utf8 ($passwd);
    if (crypt ($passwd,$self->{crypted_password}) eq $self->{crypted_password}){
        return 1;
    }
    return 0;
}

sub verifyPasswordWin{
    my ($self, $passwd) = @_;
    $self->AddWarning ("Cannot verify password: Not yet implemented");
    return undef
}


sub existsUx{
    defined $_[0]->{uid};
}

sub existsWin{
    defined $_[0]->{sid};
}

sub createUx{
    my ($self,$id, $password, $comment, $group ,$home, $shell, $groups) = @_;
    if ($self->exists()){
        $self->AddMessage ("User \"$self->{name}\" already exists");
        return 1;
    }
    if (defined $password){
        $self->{crypt_password} = _hashPassword($password);
    }

    my $cmd ='useradd';
    my @args = ();
    if (defined $id){
        push @args, ('-u', $id, '-o');
    }
    if ($self->{crypt_password}){
        push @args, ('-p', $self->{crypt_password});
    }

    if (defined $comment){
        push @args, ('-c', $comment);
    }
    if (defined $group){
        push @args, ('-g', $group);
    }
    if (defined $shell){
        push @args, ('-s', $shell);
    }

    my $home_parent;

    if (defined $home){
        $home_parent = dirname ($home);
        push @args, ('-m');
        push @args, ('-d', $home);
    }

    if (defined $groups && @$groups){
        my $defaults = get_useradd_defaults();
        my $group_list;
        if (defined $defaults && $defaults->{'GROUPS'}){
            $group_list = $defaults->{'GROUPS'} . ',';
        }
        $group_list .= join (',', @$groups);
        push @args, ('-G', $group_list);
    }

    push @args, $self->{name};

    my $msg = $self->AddMessage ("Creating user \"$self->{name}\"");


    if (defined $home_parent && !-d $home_parent){
        my $messageText = "parent directory of home directory '$home_parent'";
        my $mkdir_msg = $msg->getSubMsgLst()->addMessage ("Creating $messageText");
        my $cfg = {'mode' => 0755};
        $mkdir_msg->getSubMsgLst()->injectIntoConfigHash ($cfg);
        if (!defined makedir ($home_parent, $cfg)){
            $self->setErrorMessage ("Cannot create $messageText", $mkdir_msg->getSubMsgLst());
            return undef;
        }
    }

    my $msglst = new SDB::Install::MsgLst ();

    my $rc;
    {
        local %ENV = %ENV;
        $ENV{'PATH'} = '/sbin:/usr/sbin:' . $ENV{PATH};

        $msglst->{fLogSubst} =
            sub {
                my ($txt) = @_;
                $txt =~ s/(-p\s+\S+\s+)/-p *** /;
                return $txt;
            };

        $rc = exec_program ($cmd, \@args, $msglst);
    }

    if ($rc != 0){
        $self->AddError ("Cannot create user", $msglst);
        $self->AddSubMsgLst ($msg, $msglst);
        return undef;
    }
    $self->AddSubMsgLst ($msg, $msglst);

    #
    # if nscd is runnig, it's cache
    # takes max. 15 sec to invalidate
    #

    my ($name,$passwd,$uid,$gid,$quota,$gcos,
                                $dir,$expire);
    my $tries = 15;

    while ($tries--){
        ($name,$passwd,$uid,$gid,$quota,$comment,$gcos,
            $dir,$shell,$expire) = getpwnam ($self->{name});
        last if defined $uid;
        sleep (1);
    }

    if (!defined $uid){
        $self->AddError ("Cannot create user");
        return undef;
    }

    $self->{uid} = $uid;
    $self->{gid} = $gid;
    $self->{group} = getgrgid ($gid);
    $self->{comment} = $gcos;
    $self->{home} = $dir;
    $self->{shell} = $shell;
    $self->{crypted_password} = $passwd;
    return 1;
}

sub _hashPassword {
    my ($password) = @_;
     my $hashFunction = '6'; # SHA-512
     my $salt = join '', ('.', '/', 0..9, 'A'..'Z', 'a'..'z')
         [rand 64, rand 64, rand 64, rand 64, rand 64, rand 64, rand 64, rand 64,
          rand 64, rand 64, rand 64, rand 64, rand 64, rand 64, rand 64, rand 64];

    utf8::encode ($password) if utf8::is_utf8 ($password);
    return SDB::Common::BuiltIn->get_instance()->crypt($password, '$'.$hashFunction.'$'.$salt);
}

sub createWin{
    my ($self, $id, $password, $comment, $group ,$home, $shell) = @_;

    if ($self->exists()){
        $self->AddMessage ("User \"$self->{name}\" already exists");
        return 1;
    }

    if (!defined $comment){
        $comment = '';
    }

    my ($rc, $errtxt) = SAPDB::Install::System::Win32::API::createUser ($self->{bsname}->{second},
                $password, $comment,undef, $self->{bsname}->{local} ? undef : $self->{bsname}->{first});
    if ($rc != 0){
        my $errlst = new SDB::Install::MsgLst ();
        $errlst->AddError ($errtxt);
        $self->AddError ("createUser error code: $rc", $errlst);
        return undef;
    }
    $self->getSid ();
    return 1;
}

sub isLocalUser{
    my ($self) = @_;

    if (!defined $self->{crypted_password} ||
        $self->{crypted_password} eq '*' ||
        $self->{crypted_password} eq 'VAS') {
        return 0;
    }
    return 1;
}


sub deleteUx{
    my ($self, $keepUserHomeDir) = @_;

    if (!$self->exists()){
        return 1;
    }

    if (!$self->isLocalUser()) {
        $self->AddMessage ("User \"$self->{name}\" is not a local user and will not be deleted");
        return 1;
    }

    my $cmd = 'userdel';
    my @args = ($keepUserHomeDir) ? ($self->{name})
                                  : ('-r', '-f', $self->{name});

    my $msg = $self->AddMessage ("Deleting user \"$self->{name}\"");

    my $msglst = new SDB::Install::MsgLst ();

    my $rc;
    {
        local %ENV = %ENV;
        $ENV{'PATH'} = '/sbin:/usr/sbin:' . $ENV{PATH};
        $rc = exec_program ($cmd, \@args, $msglst);
    }

    if ($rc != 0){
        $self->AddError ("Cannot delete user", $msglst);
        $self->AddSubMsgLst ($msg, $msglst);
        return undef;
    }

    $self->AddSubMsgLst ($msg, $msglst);

    undef $self->{uid};

    return 1;
}


sub deleteWin{
    if (SAPDB::Install::System::Win32::API::deleteUser ($_[0]->{bsname}->{second},
	             undef, $_[0]->{bsname}->{local} ? undef : $_[0]->{bsname}->{first}) == 0){

        if (SAPDB::Install::System::Win32::API::deleteProfile ($_[0]->{sid}) == 0){
            $_[0]->AddMessage ("User profile deleted");
        }
        else{
            $_[0]->AddMessage ("User profile could not be deleted");
        }
        undef $_[0]->{sid};
        return 1;
    }

    return undef;
}


sub getGroupsUx{
    my ($self, $nocache) = @_;
    if (defined $self->{groups} && !$nocache){
        return $self->{groups};
    }

    my @groups;

    if (defined $self->{group}){
        push @groups, $self->{group};
    }

    # secondary groups
    setgrent();
    while( my ($name,$passwd,$gid,$members) = getgrent () ){
        foreach my $member (split(' ',$members)){
            if ($member eq $self->{name}){
                push @groups, $name;
                last;
            }
        }
    }
    endgrent();
    $self->{groups} = \@groups;
    return $self->{groups};
}

sub getGroupsWin{
    my ($self, $nocache) = @_;

    if (defined $self->{groups} && !$nocache){
        return $self->{groups};
    }
    (my $rc, $self->{groups}) = SAPDB::Install::System::Win32::API::getGroups ($self->{bsname}->{concat});
    return $self->{groups};
}


sub hasgroupUx{
    my ($self,$group) = @_;

    foreach my $group_name (@{$self->getGroups ()}){
        if ($group_name eq $group){
            return 1;
        }
    }
    return 0;
}


sub hasgroupWin{
    my ($self, $group) = @_;
    my $bsgroup = new BackSlasher ($group);
    foreach my $group_name (@{$self->getGroups ()}){
        if (lc ($group_name) eq $bsgroup->{concat}){
            return 1;
        }
    }
    return 0;
}

sub addgroupUx{
my ($self,$group) = @_;

    if (!defined getgrnam ($group)){
        $self->AddError ("Cannot add user \"$self->{name}\" to group \"$group\": group doesn\'t exist");
        return undef;
    }


    if ($self->hasgroup ($group)){
        return 1;
    }

    my $cmd = 'groupmod';
    my @args = ('-A', $self->{name}, $group);

    my $msg = $self->AddMessage ("Adding user \"$self->{name}\" to group \"$group\"");

    my $msglst = new SDB::Install::MsgLst ();

    my $rc;
    {
        local %ENV = %ENV;
        $ENV{'PATH'} = '/sbin:/usr/sbin:'  . $ENV{PATH};
        $rc = exec_program ($cmd, \@args, $msglst);
    }

    if ($rc == 2){

        # "groupmod -A" (SuSE syntax) returned "wrong usage"
        # falling back to redhat/debian syntax "usermod -a -G"

        $cmd = 'usermod';
        @args = ('-a','-G', $group, $self->{name});
        $msglst = new SDB::Install::MsgLst ();
        local %ENV = %ENV;
        $ENV{'PATH'} = '/sbin:/usr/sbin:'  . $ENV{PATH};
        $rc = exec_program ($cmd, \@args, $msglst);
    }

    if ($rc != 0){
        $self->AddError ("Cannot add user \"$self->{name}\" to group \"$group\"", $msglst);
        $self->AddSubMsgLst ($msg, $msglst);
        return undef;
    }
    $self->AddSubMsgLst ($msg, $msglst);
    push @{$self->{groups}}, $group;
    return 1;
}

sub addgroupWin{
    my ($self,$group) = @_;
    my ($rc,$errtxt);
    my $bsgroup= new BackSlasher($group);

    if ($bsgroup->{local}){
        ($rc, $errtxt) = SAPDB::Install::System::Win32::API::addLocalGroupMember ($bsgroup->{second}, $self->{sid});
        if ($rc != 0){
            my $errlst = new SDB::Install::MsgLst ();
            $errlst->AddError ($errtxt);
            $self->AddError ("addLocalGroupMember() failed with error code $rc", $errlst);
        }
    }
    else{
        ($rc, $errtxt) = SAPDB::Install::System::Win32::API::addUserToGroup ($bsgroup->{name}, $self->{name}, undef, $bsgroup->{first});
        if ($rc != 0){
            my $errlst = new SDB::Install::MsgLst ();
            $errlst->AddError ($errtxt);
            $self->AddError ("addUserToGroup() failed with error code $rc", $errlst);
        }
    }
    return $rc == 0;
}

sub getInitialEnvironmentUx{
    my ($self) = @_;

    if (!$self->exists){
        $self->setErrorMessage ("User '$self->{name}' doesn't exist");
        return undef;
    }

    my $switchUser = $self->{uid} != $>;

    if ($> != 0 && $switchUser){
        $self->setErrorMessage ("Permission denied");
        return undef;
    }

    my $loginshell = $self->{shell};
    if (! -x $loginshell){
        $self->setErrorMessage ("No valid login shell ($loginshell)");
        return undef;
    }

    my $rc;
    my @args = ('-c', 'env');
    my $shellName = basename ($loginshell);

    if ($shellName eq 'sh' || $shellName eq 'bash'){
        unshift (@args, '-l');
    }

    my $msglst = $self->getMsgLst ();
    my $outbuffer;
    {
        local %ENV = ();
        $ENV{PATH} = '/bin:/usr/bin';
        $ENV{HOME} = $self->{home};
        $ENV{USER} = $self->{name};
        if ($switchUser){
            require SAPDB::Install::SetUser;
            $msglst->addMessage ("Switching to user '$self->{name}'");
            my ($suRc, $suErrorText) = SAPDB::Install::SetUser::SetUser ($self->{uid});
            if( $suRc != 0) {
                $self->setErrorMessage ("Cannot switch user: $suErrorText");
                return undef;
            }
        }

        $msglst->addMessage ("Perfoming loginshell ($shellName)");

        my $cfg = {'outLines' => []};

        my $execMsgLst = new SDB::Install::MsgLst;
        $execMsgLst->injectIntoConfigHash ($cfg);

        $rc = exec_program ($loginshell, \@args, $cfg);

        if ($switchUser){
            $msglst->addMessage ("Switching back to original user'");
            my ($suRc, $suErrorText) = SAPDB::Install::SetUser::SetUser ();
            if($suRc != 0) {
                $self->setErrorMessage ("Cannot switch user back: $suErrorText");
                return undef;
            }
        }
        if ($rc != 0){
            $self->setErrorMessage ("Error performing login shell", $execMsgLst);
            return undef;
        }
        $outbuffer = $cfg->{outLines};
    }

    my %env;
    my ($varName, $varValue);
    foreach my $line (@$outbuffer){
        ($varName, $varValue) = ($line =~ /^([^=]+)=(.*)/);
        if ($varName){
            $env{$varName} = $varValue;
        }
    }
    return \%env;
}

sub getInitialEnvironmentWin{
    $_[0]->setErrorMessage ("getInitialEnvironment() is not yet implemented on this platform");
    return undef;
}


sub isadminUx{
    return $_[0]->{uid} == 0;
}

sub isadminWin{
    return new SDB::Install::Group ('BUILTIN\Administrators')->has ($_[0]->{bsname}->{concat});
}

sub getnameUx {return $_[0]->{name};}

sub getnameWin{
    return $_[0]->{bsname}->{second};
}

sub addAccountRights{
    my ($self,$priv_list) = @_;
    my $rc = SAPDB::Install::System::Win32::API::addAccountRights ($self->{sid},$priv_list);
    if ($rc != 0){
        $self->AddError ("addAccountRights returned with rc $rc");
    }
    return $rc == 0;
}

if ($isWin){
    *new = \&newWin;
    *exists = \&existsWin;
    *create = \&createWin;
    *delete = \&deleteWin;
    *hasgroup = \&hasgroupWin;
    *addgroup = \&addgroupWin;
    *isadmin = \&isadminWin;
    *getInitialEnvironment = \&getInitialEnvironmentWin;
    *getname = \&getnameWin;
    *getGroups = \&getGroupsWin;
}
else{
    *new = \&newUx;
    *exists = \&existsUx;
    *create = \&createUx;
    *delete = \&deleteUx;
    *hasgroup = \&hasgroupUx;
    *addgroup = \&addgroupUx;
    *isadmin = \&isadminUx;
    *getname = \&getnameUx;
    *getInitialEnvironment = \&getInitialEnvironmentUx;
    *getGroups = \&getGroupsUx;
    sub id {return $_[0]->{uid};}
    sub group {return $_[0]->{group};}
    sub gid {return $_[0]->{gid};}
    sub  comment {return $_[0]->{comment};}
    sub  home {return $_[0]->{home};}
    sub  shell {return $_[0]->{shell};}
}

sub verifyPassword {
    my ($self, $passwd, $instconfig) = @_;

    if ($isWin) {
        return $self->verifyPasswordWin($passwd);
    }
    return $self->verifyPasswordUx($passwd, $instconfig);
}

1;
