package SDB::Install::Utils::DatabaseStudio::DatabaseStudioFeatureManager;

use SDB::Install::SysVars;
use SDB::Install::System qw (makedir);
use SDB::Install::Archive;
use SDB::Install::Utils::DatabaseStudio::DatabaseStudioFeatureDetector qw (LCM_FEATURE_ID);
use SDB::Install::Utils::DatabaseStudio::DatabaseStudioKnownFeatures qw (IsKnownFeatureId GetKnownFeatureIdByKey GetBatchKey GenerateFeatureDescriptionText);

use strict;

our @ISA = qw (SDB::Install::BaseLegacy);

sub new {
	my ( $package, $instconfig ) = @_;
	my $self = {};
	bless( $self, $package );
	$self->{instconfig} = $instconfig;
	$self->{cashedPath} = "";
	return $self;
}

sub setProgressHandler {
	my ( $self, $handler ) = @_;
	$self->{progressHandler} = $handler;
}

sub updateDefaultAndValidValues {
	my ( $self, $installationPath ) = @_;

	if ( !defined $installationPath || '' eq $installationPath || $installationPath eq $self->{cashedPath} ) {
		return 1;
	}
	
	my $message = "Updating default and valid values...";
	my $msg = $self->_addFeatureManagerLogMessage( $message );
	my $makeDirMsgLst = new SDB::Install::MsgLst ();
	my $cfg = {};
	$makeDirMsgLst->injectIntoConfigHash ($cfg);
	if ( !makedir( $installationPath, $cfg ) ) {
		$self->appendErrorMessage('Cannot create installation directory', $makeDirMsgLst);
		return 0;
	}

	my $errlst = new SDB::Install::MsgLst ();
	if ( ! $self->_retrieveFeaturesFromNewRepository( $installationPath, $errlst) ) {
		$self->PushError("Cannot retrieve features from source repository. Check your repository.", $errlst);
		return 0;
	}
	
	if ( $self->isUpdate() ) {
		$errlst = new SDB::Install::MsgLst ();
		if ( ! $self->_retrieveInstalledFeatures( $installationPath, $errlst) ) {
			$self->PushError("Cannot retrieve installed features", $errlst);
            return 0;
        }
	}
	
	if ( ! $self->_fillFeaturesDefaultAndValidValues( $installationPath ) ) {
        $self->PushError("Cannot retrieve installed features from $installationPath");
		return 0;
	}
    
    # Get messages from _retrieveFeaturesFromNewRepository and _fillFeaturesDefaultAndValidValues
	$self->AddSubMsgLst ($msg, $self->{detector});
	
	$self->{progressHandler}->StepFinished() if ( defined $self->{progressHandler} );
	$self->{cashedPath} = $installationPath;
	return 1;
}

sub _addFeatureManagerProgressMessage {
	my ( $self, $message ) = @_;
	
    if ( defined $self->{progressHandler} ) {
        $self->{progressHandler}->InitProgress();
        $self->{progressHandler}->SetProgress( $message );
    } else {
        print $message . "\n";
    }
}

sub _addFeatureManagerLogMessage {
    my ( $self, $message ) = @_;
    return $self->AddMessage( $message );
}

sub _retrieveInstalledFeatures {
	my ( $self, $installationPath, $errlst) = @_;
	my $message = "Retrieving list of installed features...";
    $self->_addFeatureManagerLogMessage( $message );
    $self->_addFeatureManagerProgressMessage( $message );
    $self->{detector}->setMsgLstContext ([undef, $errlst]);
    return $self->{detector}->retrieveInstalledFeatures( $installationPath ) ;
}

sub _retrieveFeaturesFromNewRepository {
	my ( $self, $installationPath, $errlst) = @_;
    
    my $message = "Retrieving list of features from new repository...";
    $self->_addFeatureManagerLogMessage( $message );
	$self->_addFeatureManagerProgressMessage( $message );
	
	my $repository = $self->{instconfig}->getValue ( 'SourceRepositoryUrl' );
    if ( ! $repository ) {
        $repository = $self->{instconfig}->getBatchValue('SourceRepositoryUrl');
    	if ( ! $repository ) {
        	$repository = $self->{instconfig}->getDefault('SourceRepositoryUrl');
    	}
    }

    $self->{detector} = new SDB::Install::Utils::DatabaseStudio::DatabaseStudioFeatureDetector( $self->_getKit(), $installationPath, $repository, $self->_getJavaVm );
    $self->{detector}->setMsgLstContext ([undef, $errlst]);
    my $rc = $self->{detector}->retrieveFeaturesFromNewRepository( $installationPath );

    return $rc;
}

sub _fillFeaturesDefaultAndValidValues {
	my ( $self, $installationPath) = @_;
	
	if ( $self->isUpdate() ) {
        $self->_fillFeaturesDefaultAndValidValuesUpdate();
    } else {
        $self->_fillFeaturesDefaultAndValidValuesInstall();
    }
    
    return 1;
}

sub _fillFeaturesDefaultAndValidValuesInstall {
	my ($self) = @_;

	if ( $self->{detector}->isOldFeaturesSetInRepository() ) {
        my $message = "Working with non separated to features repository...";
        $self->_addFeatureManagerLogMessage( $message );

		$self->_getSelectedFeaturesParam()->{hidden} = 1;
		$self->_getSelectedFeaturesParam()->{skip} = 1;
		return;
	}
	
	# new set of features
	my $features     = $self->{detector}->getRepositoryFeatures();
	my @valid_values = ("all");
	my @ui_values    = ("All features");

	$self->_pushValidValues($features, \@valid_values, \@ui_values);

	$self->_getSelectedFeaturesParam()->{'valid_values'} = \@valid_values;
	$self->_getSelectedFeaturesParam()->{'ui_values'}    = \@ui_values;
}

sub _fillFeaturesDefaultAndValidValuesUpdate {
	my ($self) = @_;
	$self->_determineInstallableAndUpdateFeatures();
	my $selectedFeaturesParam = $self->_getSelectedFeaturesParam();
	if ( $self->{detector}->isOldFeaturesSetInRepository() ) {
		$self->AddMessage( "Updating a non separated to features installation with a non separated to features repository..." );
		$selectedFeaturesParam->{hidden} = 1;
		$selectedFeaturesParam->{skip} = 1;
		return;
	}
	$self->AddMessage( "Updating with separated to features repository..." );

	my $featuresForUpdate = $self->{featuresForUpdate};
	my $installableFeatures = $self->{installableFeatures};
		
	if ( $self->{detector}->isOldFeaturesSetInstalled() ) {
		$selectedFeaturesParam->{'console_text'} = "\n".'HANA Studio has been divided into features. Select which features you would like to install:'."\n";
	}else {
		$selectedFeaturesParam->{'console_text'} = "\n".'Select additional features to install:'."\n";
	}
	if ( scalar @$featuresForUpdate != 0 ) {
		my $text = "The following features will be updated:";
		foreach my $feature (@$featuresForUpdate) {
				$text = $text . "\n". GenerateFeatureDescriptionText($feature);
		}
		$selectedFeaturesParam->{'console_text'} = $text . "\n" . $selectedFeaturesParam->{'console_text'};
	}
	my $numberOfFeaturesForInstall = scalar @$installableFeatures;
	if ( $numberOfFeaturesForInstall == 0 ) {
		$self->AddMessage ("No additional features available for installation");
		$selectedFeaturesParam->{hidden} = 1;
		$selectedFeaturesParam->{'default'} = 'none';
	} else {
		my @valid_values = ();
		my @ui_values    = ();
		if ( $numberOfFeaturesForInstall != 1 ) {
			push( @valid_values, "all" );
			push( @ui_values,    "All features" );
		}

		$self->_pushValidValues($installableFeatures, \@valid_values, \@ui_values);

		if ( $self->{detector}->isOldProductInstalled() ) {
			$selectedFeaturesParam->{'default'} = 'all';
		} else {
			push( @valid_values, "none" );
			push( @ui_values,    "No features" );
			$selectedFeaturesParam->{'default'} = 'none';
		}
		$selectedFeaturesParam->{'valid_values'} = \@valid_values;
		$selectedFeaturesParam->{'ui_values'} = \@ui_values;
	}
}

sub _determineInstallableAndUpdateFeatures{
	my ($self) = @_;
	my $featuresForUpdate  = [];
	my $installableFeatures = [];
	my $newFeatures        = $self->{detector}->getRepositoryFeatures();
	foreach my $newFeature (@$newFeatures) {
		if ( $self->{detector}->isInstalledFeature( $newFeature->{id} ) ) {
			my $installedFeature = $self->{detector}->getInstalledFeatureById($newFeature->{id});
			if( $newFeature->{'version'} ne $installedFeature->{'version'}){
				push @$featuresForUpdate, $newFeature;
			}else{
				$self->AddMessage("Skipping update of feature ".$installedFeature->{id}." version ".$installedFeature->{'version'}." already installed");
			}
		} else {
			push @$installableFeatures, $newFeature;
		}
	}
	$self->{featuresForUpdate}  = $featuresForUpdate;
	$self->{installableFeatures} = $installableFeatures;
}

sub getFeaturesForUpdate {
	my ($self) = @_;
	return $self->{featuresForUpdate};
}

sub getAllInstallableFeatures {
	my ($self) = @_;
	return $self->{installableFeatures};
}

sub getSelectedFeaturesIds {
    my ($self) = @_;	
    my $result = [];	
	my $selectedFeaturesCsv = $self->_getSelectedFeaturesParam()->{value};
    for my $featureBatchValue (split(/\s*,\s*/, $selectedFeaturesCsv)) {
        if ("all" eq $featureBatchValue) {
        	if ($self->isUpdate()) {
                my $installableFeatures = $self->getAllInstallableFeatures();
                my $featureIdsForInstall = [];
                foreach my $feature (@$installableFeatures) {
                    push @$featureIdsForInstall, $feature->{id};
                }                        		
        		return $featureIdsForInstall;        		
        	} else {
                return $self->{detector}->getRepositoryFeatureIds();        		
        	}
        }   
        if ("none" eq $featureBatchValue) {
            return $result;	
        }
        my $featureId = GetKnownFeatureIdByKey($featureBatchValue);
        if ($featureId) {
            push @$result, $featureId;        	
        } else {
            push @$result, $featureBatchValue;          
        }
    }
    return $result;
}

#--Methods used by kit

sub isOldFeaturesSetInstalled {
	my( $self ) = @_;
	
	$self->{detector}->isOldFeaturesSetInstalled()
}

sub getFeaturesToUninstall {
    my( $self ) = @_;
    if( ! $self->isUpdate ){
    	#nothing to uninstall on a new installation
    	return undef;
    }
    
    if ($self->{detector}->isOldFeaturesSetInstalled()) {
        my @result = ($self->{detector}->getInstalledProduct(), LCM_FEATURE_ID);
    	
    	#for revision 70 & 71 when separate features contained in the product are installed 
	    my $featuresForUpdate = $self->getFeaturesForUpdate();
	    if (scalar @$featuresForUpdate != 0) {
	        my $featureIdsForUpdate = [];
	    	foreach my $feature (@$featuresForUpdate) {
	     		push @result, $feature->{id};
	    	}        	
	    }
	     
        my $uninstallString = join (', ', @result);
    	return $uninstallString;
    } 
    
    
    my $uninstallIdList = [];
    #when adding features to existing installation via update, do not unninstall product feature
#	my $productId = undef;
	
	my $installedProduct = $self->{detector}->getInstalledFeatureById( $self->{detector}->getInstalledProduct() );
	my $repositoryProduct = $self->{detector}->getNewestRepositoryProductFeature();
  	if ( defined $installedProduct && defined $repositoryProduct &&  
  			($installedProduct->{id} ne $repositoryProduct->{'id'} || $installedProduct->{'version'} ne $repositoryProduct->{'version'}) ) {
  		push @$uninstallIdList, $self->{detector}->getInstalledProduct();
  	}
  	
  	#This feature may be left behind previous update procedures and may or may not be existing
  	my $repoLCMFeature = $self->{detector}->getRepositoryFeatureById(LCM_FEATURE_ID);
  	my $installedLCMFeature = $self->{detector}->getInstalledFeatureById(LCM_FEATURE_ID);
  	my $isOnlyInstalled = defined $installedLCMFeature  && !defined $repoLCMFeature;
  	my $isForUpdate = !$isOnlyInstalled && defined $installedLCMFeature && $installedLCMFeature->{'version'} ne $repoLCMFeature->{'version'}; 
  	if($isOnlyInstalled || $isForUpdate ){
  		push @$uninstallIdList, $installedLCMFeature->{id};
  	};
  	
    my $featuresForUpdate = $self->getFeaturesForUpdate();
    if (scalar @$featuresForUpdate != 0) {
    	foreach my $feature (@$featuresForUpdate) {
     		push @$uninstallIdList, $feature->{id};
    	}
    }   
    return join (', ', @$uninstallIdList);
}

sub getFeaturesToInstall {
    my ($self) = @_;
    if ($self->{detector}->isOldFeaturesSetInRepository()) {
    		#this should always be old product ID. Query it in case of custom repositories
    		my $productId = $self->{detector}->getNewestProductIdInRepository();
    		if( $self->{detector}->isInRepository(LCM_FEATURE_ID) ){
	    		return $productId.", ".LCM_FEATURE_ID;
    		}
    		#in repisitories 70 & 71 there is no LCM feature but still the product contains the separated features
    		return $productId; 
    }
    if ($self->isUpdate()) {
    	return $self->_generateInstallFeaturesStringForUpdate();
    } 
    #install case
    my $selectedFeatures = $self->getSelectedFeaturesIds(); 
    return $self->{detector}->getNewestProductIdInRepository().", ".join (', ', @$selectedFeatures);
}

sub _generateInstallFeaturesStringForUpdate {
	my ( $self ) = @_;
	my @result = ( $self->{detector}->getNewestProductIdInRepository() );
    my $featuresForUpdate = $self->getFeaturesForUpdate();
    if (scalar @$featuresForUpdate != 0) {
    	my $featureIdsForUpdate = [];
        foreach my $feature (@$featuresForUpdate) {
        	push @$featureIdsForUpdate, $feature->{id};
        }            	
        @result = ( @result, @$featureIdsForUpdate );
    }                    	
    my $selectedFeatures = $self->getSelectedFeaturesIds(); 
    if (scalar @$selectedFeatures != 0) {
		@result = ( @result, @$selectedFeatures );
	}      
	my $installString = join (', ', @result);
	return $installString;
}

sub isUpdate {
	return $_[0]->{instconfig}->isUpdate();
}

sub _getSelectedFeaturesParam {
	return $_[0]->{instconfig}->{params}->{'SelectedFeatures'};
}

sub _getKit {
	return $_[0]->{instconfig}->{kit};
}
sub _getJavaVm {
    return $_[0]->{instconfig}->getBatchValue ('JavaVm');
}

sub _pushValidValues {
	my ($self, $features, $valid_values, $ui_values) = @_;

	foreach my $feature (@$features) {
		my $key = IsKnownFeatureId($feature->{id}) 
			? GetBatchKey($feature->{id})
			: $feature->{id};
		my $text = GenerateFeatureDescriptionText($feature);

		push( @$valid_values, $key );
		push( @$ui_values,    $text );

	}
}

1;
